#ifdef USE_VGALIB
  #include <vga.h>
#else
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/extensions/XShm.h>
#include <X11/cursorfont.h>
#include <sys/shm.h>
#include "synchronizer.h" /* for ReadAll */

Display* display;
Drawable window;
GC gc;
XImage* image;    
Visual* visual;
Colormap colormap=0;
int depth,real_depth;
XColor colors[256];

#endif

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include "primitives.h"

unsigned char *doubleBuffer; //mask[MAX_X * MAX_Y];
int screenWidth, screenHeight;
XShmSegmentInfo shmInfo;

static void CreateImg (void)
{
  int major, minor, ignore;
  Bool pixmaps;
      
  if (XQueryExtension (display, "MIT-SHM", &ignore, &ignore, &ignore) &&
  XShmQueryVersion( display, &major, &minor, &pixmaps ) == True) {
    //if (shmInfo.shmid == 0) printf ("Using MIT-SHM extension\n");
    shmInfo.shmid = shmget (IPC_PRIVATE, screenWidth * screenHeight,
      IPC_CREAT | 0777);
    if (shmInfo.shmid==-1) { perror ("MIT-SHM failed"); exit (15); }
    doubleBuffer = (unsigned char*)shmat (shmInfo.shmid, 0, 0);
    shmInfo.shmaddr = (char *)doubleBuffer;
    image = XShmCreateImage (display, visual, depth, ZPixmap,
      shmInfo.shmaddr, &shmInfo, screenWidth, screenHeight);
    shmInfo.readOnly = False;
    XShmAttach (display, &shmInfo);
    XSync (display, False);
  }
  else {
    shmInfo.shmaddr = NULL;
    doubleBuffer = (unsigned char*)malloc (screenWidth * screenHeight);
    image = XCreateImage(display,visual,depth,ZPixmap,0,(char*)doubleBuffer,
      screenWidth, screenHeight, 8, screenWidth * real_depth);
  }
}

void DestroyImg (void)
{
  if (image) XDestroyImage(image);
  if (!shmInfo.shmaddr) free (doubleBuffer);
  else {
    XShmDetach (display, &shmInfo);
    shmdt (shmInfo.shmaddr);
    shmctl (shmInfo.shmid, IPC_RMID, 0);
  }
}

void OpenDisplay (const char *title, int argc, char *argv[])
{
  #ifdef USE_VGALIB
    vga_disabledriverreport ();
    vga_setmode (G320x200x256);
    //graph_mem = (unsigned char*)malloc (64000);
    screenWidth = vga_getxdim ();
    screenHeight = vga_getydim ();
    doubleBuffer = malloc (screenHeight * screenWidth);
    //vga_setlinearaddressing ();
  #else
  int nxpfv, nscreen, i;
  char *geometry="512x384";
  
  display = XOpenDisplay(0);
  if(!display) {
    fprintf (stderr, "Cannot open display\n");
    exit(12);
  }
  nscreen = DefaultScreen(display);
  
  for (i=1;i<argc;i++) {
    if (strcmp (argv[i], "-geometry") == 0) geometry = argv[i+1];
  }
  screenWidth = atoi (geometry);
  screenHeight = screenWidth*3/4; /* Force 0.75 ratio */
  window = XCreateSimpleWindow (display,RootWindow(display,nscreen), 0, 0,
    screenWidth,screenHeight, 0, 0, 0);
  XSetStandardProperties(display,window,title,title,0,0,0,0);
  XMapWindow (display, window);
  if ((geometry = strchr (geometry, '+')) != NULL) {
    XMoveWindow (display, window, atoi (geometry + 1),
      atoi (strchr (geometry + 1, '+') + 1));
  }
  XDefineCursor (display, window, XCreateFontCursor(display,XC_draft_small));
  gc = XCreateGC (display, window, 0, NULL);
  depth = DefaultDepth (display, nscreen);
  XPixmapFormatValues* xpfv = XListPixmapFormats (display, &i);
  for (i--; i >= 0; i--) {
    if (depth == xpfv[i].depth) real_depth = xpfv[i].bits_per_pixel / 8;
  }
  //printf("display depth=%d, bytes per pixel=%d\n",depth,real_depth);
  visual = DefaultVisual(display, nscreen);
  XSetWindowAttributes xswa;
  xswa.event_mask = KeyPressMask | KeyReleaseMask | StructureNotifyMask;
  XChangeWindowAttributes(display,window, CWEventMask, &xswa );

  if (visual->c_class == TrueColor | visual->c_class == StaticColor |
     visual->c_class == StaticGray ) {
    fprintf (stderr, "Only 8 bits per pixel supported\n"
                      "Start XFree with \"startx -- -bpp 8\"\n");
    exit (13);
    #if 0
    colormap = XCreateColormap(display,window,visual,AllocNone);
    int r,g,b;
    for( r=0; r<max_rgb; r++ ) {
      for( g=0; g<max_rgb; g++ ) {
        printf("Allocating colors, %d of %d...             \r", 
               r*max_rgb*max_rgb+g*max_rgb+b,max_rgb*max_rgb*max_rgb);
        for( b=0; b<max_rgb; b++ ) {
   	  XColor foo;
	  foo.red=r<<(16-rgb_bits);
 	  foo.green=g<<(16-rgb_bits);
	  foo.blue=b<<(16-rgb_bits);
	  foo.flags=DoRed|DoGreen|DoBlue;
          XAllocColor(display,colormap,&foo);
          rgb_table[r][g][b] = foo.pixel;
        }	 
      }
    }
    printf("                                              \r");
    #endif
  }
  else {
    colormap = XCreateColormap(display,window,visual,AllocAll);
  }
  CreateImg ();
  #endif
}

void CloseDisplay(void)
{
  #ifdef USE_VGALIB
    vga_setmode (TEXT);
    free (doubleBuffer);
  #else
    DestroyImg ();
    XDestroyWindow(display,window);
    XFreeGC(display,gc);
    XCloseDisplay(display);
  #endif
}

void StartFrame (void)
{
  //memset (mask, 0, sizeof (mask));
  #ifdef USE_VGALIB
    memcpy (graph_mem, doubleBuffer, screenHeight * screenWidth);
  #else
  if (shmInfo.shmaddr) {
    XShmPutImage (display, window, gc, image,
     0, 0, 0, 0, screenWidth, screenHeight, False);
  }
  else {
    XPutImage (display, window, gc, image, 0, 0, 0, 0,
      screenWidth, screenHeight);
  }
  XSync (display, False);
  XEvent ev;
  if (XCheckWindowEvent (display, window, StructureNotifyMask, &ev) &&
  ev.xconfigure.type == ConfigureNotify) {
    DestroyImg ();
    screenWidth = ev.xconfigure.width&~3;
    /* xfree86 is dependant on line length being a multiple of 4 */
    screenHeight = screenWidth*3/4;
    CreateImg ();
  }
  //memset(doubleBuffer, 0, screenHeight * screenWidth);
  #endif
}

keyEventType GetKey (KeySym *nextKey)
{
  XEvent event;
  
  if (!XCheckWindowEvent (display, window, KeyPressMask | KeyReleaseMask,
  &event)) return noKeyEvent;

  *nextKey = XLookupKeysym (&event.xkey, 0);
  //printf ("%d %d keycode=%d, keysym=%x (%s)\n",
  //  event.xkey.type, event.xkey.keycode, keysym, XKeysymToString(keysym));
  return event.xkey.type == KeyRelease ? keyReleaseEvent : keyPressEvent;
}

#define COMBO(x,y) (((x)<<16)|((y)&0xFFFF))
/* This combo thing allows us to store both an x and a y value in one */
/* integer and have one addition were two would normally be necessary. */
/* The carry over should not be significant. */

void MapTriangle (unsigned char *source, point *a, point *b, point *c,
  int flags)
{
  rational lStart, lStartAdj, lStop, lStopAdj, y;
  rational txStart, tyStart; /* These determine */
  /* were we are in the texture at the beginning of the line. */
  rational xtx, ytx, xty, yty, det;
  /* A move of one unit in the y direction is equivalent to a move of */
  /* (ytx,yty) in texture space. */
  unsigned xIncCombo, yIncCombo, combo;
  point *top, *tmp, *left, *right;
  int x, x2, count;
  unsigned char *dest, *destLine;//, *maskPtr, *maskLine;
  char pixel;
  
  top = a;
  if (*b < *top) top = b;
  if (*c < *top) top = c;
  while (a != top) { tmp=a; a=b; b=c; c=tmp; }
  left = top;
  right = top;
  
  det = mul (c->sx - a->sx, b->sy - a->sy) - mul (c->sy-a->sy, b->sx-a->sx);
  if (det < 0) {
    fprintf (stderr, "Reverse\n");
    return; /* We don't do triangles from the back */
  }
  if (det < 0x100) det = 0x100; /* Prevent div by (almost) 0 */
  xtx = divide (
    mul (c->tx-a->tx, b->sy-a->sy) - mul (c->sy-a->sy, b->tx-a->tx), det);
  xty = divide (
    mul (c->ty-a->ty, b->sy-a->sy) - mul (c->sy-a->sy, b->ty-a->ty), det);
  ytx = divide (
    mul (c->sx-a->sx, b->tx-a->tx) - mul (c->tx-a->tx, b->sx-a->sx), det);
  yty = divide (
    mul (c->sx-a->sx, b->ty-a->ty) - mul (c->ty-a->ty, b->sx-a->sx), det);
  
  y = (top->sy&~0xFF)+0x100;
  destLine = doubleBuffer + (Trunc(y) - 1) * screenWidth;
  txStart = top->tx - mul (xtx, top->sx) + mul (ytx, y-top->sy);
  tyStart = top->ty - mul (xty, top->sx) + mul (yty, y-top->sy);
  //maskLine = mask + (y>>8)*screenWidth;
  xIncCombo = COMBO (xtx, xty);
  while (1) {
    if (y > left->sy) {
      tmp = left == top ? b : c;
      /* tmp is the new left point. We want to trace the line on the */
      /* screen from left to tmp. */
      if (y > tmp->sy) {
        if (y > c->sy) break; /* if y is beneath all the points */
        left = tmp;
        tmp = c;
      }
      /* tmp is the new left point. We want to trace the line on the */
      /* screen from left to tmp. */
      lStartAdj = divide(tmp->sx - left->sx, tmp->sy - left->sy);
      lStart = mul (lStartAdj, y - left->sy) + left->sx;
      left = tmp;
    }
    if (y > right->sy) {
      tmp = right == top ? c : b;
      /* tmp is the new left point. We want to trace the line on the */
      /* screen from left to tmp. */
      if (y > tmp->sy) {
        if (y > b->sy) break; /* if y is beneath all the points */
        right = tmp;
        tmp = b;
      }
      /* tmp is the new left point. We want to trace the line on the */
      /* screen from left to tmp. */
      lStopAdj = divide(tmp->sx - right->sx, tmp->sy - right->sy);
      lStop = mul (lStopAdj, y - right->sy) + right->sx;
      right = tmp;
    }
    if (y >= 0 && Trunc(y) < screenHeight) {
      x = Trunc (lStart);
      if (x < 0) x = 0;
      
      x2 = Trunc (lStop);
      count = ((x2 >= screenWidth) ? screenWidth : x2) - x;
      if (count > 0) {
        dest = x + destLine;
        x++;
        combo = COMBO (txStart + x*xtx, tyStart + x*xty);
        #ifndef __GNUC__
        #define DO_ROW(x,y) \
          for (; count > 0; count--) { \
            pixel = source[(combo&0xFF00)|((combo>>24)&0xFF)]; \
            x \
            dest++; \
            combo += xIncCombo;\
          } /* for each pixel */
        #else
        #define DO_ROW(x,y) \
        __asm__ (\
          "1:movl  %%eax,%%ebx\n\t" \
            "shrl  $24,%%ebx\n\t" \
            "movb  %%ah,%%bh\n\t" \
            "addl  %%esi,%%ebx\n\t" \
            "movb  (%%ebx),%%bl\n\t" \
            y \
            "movb  %%bl,(%%edi)\n\t" \
          "2:add   %%edx,%%eax\n\t" \
            "inc   %%edi\n\t" \
            "dec   %%ecx\n\t" \
            "jnz   1b\n\t" \
          : \
          : "a" (combo), "c" (count), "d" (xIncCombo), \
            "S" (source), "D" (dest) \
          : "ebx" \
        );
        #endif
        if (flags&1) {
          DO_ROW(
            if (pixel) *dest = pixel;,
            "or    %%bl,%%bl\n\t"
            "je    2f\n\t"
          )
        }
        else {
          DO_ROW(,)
        }
      } /* if x is in range */
    } /* if y is in range */
    destLine += screenWidth;
    //maskLine += screenWidth;
    txStart += ytx;
    tyStart += yty;    
    lStart += lStartAdj;
    lStop += lStopAdj;
    y += 0x100;
  }
}

unsigned char *ReadBmpFromFd (int hand)
/* Reads an 8 bit file and sets the palette */
{
  int i;
  unsigned char *mem;
  struct {
    char b,m,unk1[12];
    short type, unk2, wid, hei, unk3, bits;
    unsigned char pal[256][3];
  } head;
  
  //CloseDisplay();
  if (ReadAll (hand, &head, sizeof (head)) != sizeof (head) ||
  head.b != 'B' || head.m != 'M' ||
  head.type != 12 || head.bits != 8) {
    return NULL;
  }
  head.wid = (head.wid+3)&~0x3; /* Force multiple of 4. See BMP spec */
  if ((mem = (unsigned char *)malloc (head.hei * head.wid)) == NULL) {
    return NULL;
  }
  #ifdef USE_VGALIB
  for (i=0;i<256;i++) {
    vga_setpalette (i, head.pal[i][2]>>2, head.pal[i][1]>>2,
    head.pal[i][0]>>2);
  }
  #else
  for (i=0;i<256;i++) {
    colors[i].pixel=i;
    colors[i].red=head.pal[i][2]<<8;
    colors[i].green=head.pal[i][1]<<8;
    colors[i].blue=head.pal[i][0]<<8;
    colors[i].flags=DoRed|DoGreen|DoBlue;
  }
  XStoreColors(display,colormap,colors,256);
  XSetWindowColormap(display,window,colormap);
  #endif
  for (i = head.hei - 1; i >= 0; i--) {
    ReadAll (hand, mem + i*head.wid, head.wid);
  }
  return mem;
}

unsigned char *ReadBmp (const char *fname)
{
  int hand;
  unsigned char *mem;

  if ((hand = open (fname, O_RDONLY)) == -1) return NULL;
  mem = ReadBmpFromFd (hand);
  close (hand);
  return mem;
}

