#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <unistd.h>
#include <fcntl.h>
#include <time.h>
#include <sys/ioctl.h>
#include <sys/soundcard.h>
#include "primitives.h"
#include "synchronizer.h"
#include "textureParams.h"

#define MAX_PLAYERS 16
#define XFIG_SCALER 30

#define TANK_LEN    ((SCR_WID-1)/8)
#define TANK_WID    (TANK_LEN/2)
#define TANK_SPEED  50
#define ADJ_TANK_SPEED (TANK_SPEED * SYNCHRONIZER_MS)

#define BUL_LEN     (TANK_LEN/2)
#define BUL_WID     (BUL_LEN/2)
#define BULLET_SPEED 90

#define XPLD_LEN    (TANK_WID) /* always a square */

#define CHECK_ALLOC(statement) if ((statement)==NULL) \
  { fprintf (stderr, "Out of memory\n"); exit (14); }

typedef double scalar;
#define SCR_WID RES_INDEP_SCREEN_WIDTH

struct vect {
  scalar x, y;
  vect () {}
  vect (scalar newx, scalar newy) { x=newx; y=newy; }
};
inline vect operator+(vect a, vect b)
{
  return vect (a.x + b.x, a.y + b.y);
}
inline vect operator-(vect a, vect b)
{
  return vect (a.x - b.x, a.y - b.y);
}
inline void operator+=(vect &a, vect b)
{
  a.x += b.x;
  a.y += b.y;
}
inline vect operator*(vect a, scalar b)
{
  return vect (a.x * b, a.y * b);
}

inline int operator==(vect a, vect b)
{
  return a.x == b.x && a.y == b.y;
}

int Intersect (vect e, vect f, vect g, vect *intersectPoint)
{ /* Returns 0 if no intersection, 1 if the intersection had a clockwise */
  /* nature, and -1 if it had an anticlockwise nature. */
  /* If e=0 and f=0, there may be problems. */
  scalar det, det2;
  int retCode = 1;
  
  det = e.x * f.y - e.y * f.x; /* Ja Frans, ek gebruik ook Kramer se */
  if (det < 0) { /* reel (Gauss eliminasie is moeilik) */
    g.x = -g.x;
    g.y = -g.y; /* Strangely the game works perfectly without */
    det = -det; /* forcing the dets positive. */
    retCode = -1;
  }
  det2 = g.x * e.y - g.y * e.x;
  if (0 <= det2 && det2 <= det) {
    det2 = g.x * f.y - g.y * f.x;
    if (0 <= det2 && det2 <= det) {
      if (intersectPoint) {
        if (det == 0) *intersectPoint = vect (0, 0);
        else *intersectPoint = e * (det2 / det);
      }
      return retCode;
    }
  }
  return 0;
}

#define MAX_VERT_PER_MODEL 4
typedef struct {
  int nvert;
  vect vert[MAX_VERT_PER_MODEL]; /* relative to 0,0 */
} entityModel;
/* The new location of an enity will always be calculated from its model, */
/* so that round of errors does not accumulate. */

typedef enum {
  playerModel, bulletModel, explosionModel,
  maxModelNumber, wallModel,
} modelNumber;

entityModel model[maxModelNumber] = {
{ 4, vect(-TANK_LEN/2, -TANK_WID/2), vect(TANK_LEN/2, -TANK_WID/2),
     vect(TANK_LEN/2, TANK_WID/2), vect(-TANK_LEN/2, TANK_WID/2) },
{ 3, vect(-BUL_LEN/2, -BUL_WID/2),  vect(BUL_LEN/2, 0),
     vect(-BUL_LEN/2, BUL_WID/2), vect(0, 0) },
{ 4, vect(-XPLD_LEN/2, -XPLD_LEN/2), vect(XPLD_LEN/2, -XPLD_LEN/2),
     vect(XPLD_LEN/2, XPLD_LEN/2), vect(-XPLD_LEN/2, XPLD_LEN/2) },
};

typedef struct entityStruct {
  int nvert; /* Must be first in this struct (DownloadNew) */
  entityStruct *next, **prev;
  vect pos;
  vect vel; /* Added to pos at the end of each frame. */
  scalar dir, angular; /* dir in radians. angular in radians per frame */
  modelNumber m;
  unsigned textureOffset;
  int p, frame; /* Which player owns this entity, animation count */
  vect upLeftBoundBox, lowRightBoundBox; /*To test if they collide with us*/
  vect vert[1]; /* GDB dumps core if this array has size 0 */
} entity;
/* An entity is something that has a position, a direction, a velocity and */
/* angular velocity. We keep a list of entities for collision detection */
/* purposes. */

/* We compensate for server speed when we handle the keys. */

entity lastEntity, *firstEntity = &lastEntity;
/* The effects are added to the end of the list, so to access it we have */
/* lastEntity which is dummy. The list is therefore not NULL terminated, */
/* and this simplifies the Add and Delete functions. (x->next->prev = ...) */

struct {
  unsigned rnd; /* random number */
  textureParams texture;
  vect startPoint[20];
  int  numOfStarts;
  vect startFinishA, startFinishB;
} param;

int FindTexture (const char *name) /* returns the offset */
{
  int i;
  for (i = 1; i < sizeof (param.texture)/sizeof(param.texture[0]) &&
  param.texture[i].name[0] != '\0' &&
  strcmp (param.texture[i].name, name) <= 0; i++) {}
  return param.texture[i-1].offset;
}

double SynchronizedRandom (void) // returns value in 0 to 1 range.
{
  param.rnd = param.rnd * 12343 % 58679; // Like they used in Kerberos 4? Ha.
  return param.rnd / 58679.0;
}

void AddEntity (entity *e)
{
  e->next = firstEntity;
  firstEntity->prev = &e->next;
  e->prev = &firstEntity;
  firstEntity = e;
}

void DeleteEntity (entity *e)
{
  e->next->prev = e->prev;
  *e->prev = e->next;
  if (e->m != playerModel) free (e);
}

entity *NewEntity (int nvert)
{
  entity *e;
  
  CHECK_ALLOC(e=(entity*)malloc (sizeof (*e) + sizeof(e->vert[0]) * nvert));
  e->nvert = nvert;
  return e;
}

typedef enum {
  playerForward, playerBackward, playerLeft, playerRight, playerFire,
  numberOfPlayerKeys
};
#define PLAYER_FORWARD    (1<<playerForward)
#define PLAYER_BACKWARD   (1<<playerBackward)
#define PLAYER_LEFT       (1<<playerLeft)
#define PLAYER_RIGHT      (1<<playerRight)
#define PLAYER_FIRE       (1<<playerFire)
#define OTHER_PLAYER      (numberOfPlayerKeys)
/* Shift count for other player. See sameKeyboard. */
KeySym keyTable[numberOfPlayerKeys*2]={ /* defaults */
  XK_Up, XK_Down, XK_Left, XK_Right, XK_Shift_R,
  'd', 'c', 'x', 'v', 'z'
};

typedef struct {
  entity e;
  vect v[MAX_VERT_PER_MODEL];
  char name[10];
  int health;
  int frags;
  int controls, fireWait;
  scalar energy;
  int ipAddress; /* Should have been a name :-( */
  int theOtherPlayer; /* for samekeyboard */
} playerType;

playerType player[MAX_PLAYERS]; /* Limit the amount of players */

vect    pointOfHit, *lineHitStart, *lineHitEnd;
entity *entityHit;
int  TestHit (vect *start, vect *end, entity *exclude)
/* Goes throu all the lines of all the entities except *exclude to find */
/* an intersection with the line from *start to *stop. Returns 0 if no */
/* intersections are found. Returns 1 if one is found, pointOfImpact is */
/* the intersection point, and entityHit is the entity containing the */
/* other line. Be careful, because it does not give the hit nearest to */
/* start. */
{
  entity *itr;
  vect   *begin;
  vect   ulBBox, lrBBox;
  int i;
  
  if (start->x < end->x) { ulBBox.x = start->x; lrBBox.x = end->x; }
  else { lrBBox.x = start->x; ulBBox.x = end->x; }
  if (start->y < end->y) { ulBBox.y = start->y; lrBBox.y = end->y; }
  else { lrBBox.y = start->y; ulBBox.y = end->y; }
  
  for (itr = firstEntity; itr != &lastEntity; itr = itr->next) {
    if (ulBBox.x > itr->lowRightBoundBox.x ||
    ulBBox.y > itr->lowRightBoundBox.y ||
    lrBBox.x < itr->upLeftBoundBox.x ||
    lrBBox.y < itr->upLeftBoundBox.y) continue;
    if (itr == exclude) continue;
    
    begin = itr->vert + 0;
    for (i = itr->nvert - 1;i >= 0; i--) {
      if (Intersect (*end - *start, itr->vert[i] - *begin,
      *begin - *start, &pointOfHit)) {
        pointOfHit += *start;
        entityHit = itr;
        lineHitEnd = itr->vert + i;
        lineHitStart = begin;
        return 1;
      }
      begin = itr->vert + i;
    }
  }
  return 0;
}

void SetBoundBox (entity *e)
{
  vect *v;
  int i;
  
  e->upLeftBoundBox = e->vert[0];
  e->lowRightBoundBox = e->vert[0];
  for (i = 1, v = e->vert + 1; i < e->nvert; i++, v++) {
    if (v->x < e->upLeftBoundBox.x) e->upLeftBoundBox.x = v->x;
    if (v->y < e->upLeftBoundBox.y) e->upLeftBoundBox.y = v->y;
    if (v->x > e->lowRightBoundBox.x) e->lowRightBoundBox.x = v->x;
    if (v->y > e->lowRightBoundBox.y) e->lowRightBoundBox.y = v->y;
  }
}

void PlaceEntity (entity *e)
{ /* Used for teleporting (telefrag !), and placing new entities */
  entityModel *m;
  scalar cosine, sine;
  vect *begin;
  int i;
  
  m = model + e->m; /* This must exsist */
  for (i=0;i<m->nvert;i++) {
    cosine = cos (e->dir);
    sine = sin (e->dir);
    e->vert[i].x = e->pos.x + cosine * m->vert[i].x - sine * m->vert[i].y;
    e->vert[i].y = e->pos.y + sine * m->vert[i].x + cosine * m->vert[i].y;
  }
  e->nvert = m->nvert;
  
  if (e->m == explosionModel) e->lowRightBoundBox.x = 0; /* No hits */
  else {
    SetBoundBox (e);  
    begin = e->vert;
    for (i = e->nvert - 1; i>=0;i--) {
      while (TestHit (begin, e->vert + i, e)) { // Handle a double telefrag
        if (entityHit->m == bulletModel) {
          DeleteEntity (entityHit);
          continue;
        }
        if (entityHit->m != playerModel) { /* Hit a wall */
          DeleteEntity (e);
          return;
        }
        if (e->m != playerModel) {
          DeleteEntity (e);
          return;
        }
        /* telefrag ! */
        //printf ("%d has telefraged %d\n",((playerType*)e)-player,
        //  ((playerType*)entityHit) - player);
        ((playerType*)e)->frags++;
       
        ((playerType*)entityHit)->health = 0;
        *entityHit->prev = entityHit->next;
      }
      begin = e->vert + i;
    } /* For each borderline */
  } /* if it occupies space */
} /* PlaceEntity */

void MoveEntities (void)
{
  entity *itr, *next;
  playerType *p;
  entityModel *m;
  scalar t, dt, sine, cosine, det, det2;
  int  collision, i, j;
  vect origin, delta, newv[MAX_VERT_PER_MODEL], *startVert;

  for (i = 0, p = player; i < MAX_PLAYERS; i++, p++) {
    if (p->health <= 0) continue;
    
    sine = sin (p->e.angular);
    cosine = cos (p->e.angular);
    p->e.vel.x = p->e.vel.x * cosine - p->e.vel.y * sine;
    p->e.vel.y = p->e.vel.y * cosine + p->e.vel.x * sine;

    dt = SYNCHRONIZER_MS * 2 * M_PI / (2000+
      (p->e.vel.x*p->e.vel.x+p->e.vel.y*p->e.vel.y)/
      SYNCHRONIZER_MS/SYNCHRONIZER_MS);
    if (p->controls & PLAYER_LEFT) p->e.angular = -dt;
    else if (p->controls & PLAYER_RIGHT) p->e.angular = dt;
    else p->e.angular = 0;
    p->e.vel = p->e.vel * 0.9;
    if (p->controls & PLAYER_FORWARD) {
      p->e.vel.x += ADJ_TANK_SPEED/10 * cos (p->e.dir);
      p->e.vel.y += ADJ_TANK_SPEED/10 * sin (p->e.dir);
    }
    else if (p->controls & PLAYER_BACKWARD) {
      p->e.vel.x -= ADJ_TANK_SPEED/10 * cos (p->e.dir);
      p->e.vel.y -= ADJ_TANK_SPEED/10 * sin (p->e.dir);
    }
    
    if (p->fireWait > 0) p->fireWait -= SYNCHRONIZER_MS;
    if (p->energy < 20) p->energy += 0.001 * SYNCHRONIZER_MS;
    /* Every second he gets energy for 1 bullet */
    if ((p->controls & PLAYER_FIRE) && p->fireWait <= 0 && p->energy >= 1) {
      p->energy -= 1;
      p->fireWait = (BUL_LEN+BULLET_SPEED*SYNCHRONIZER_MS) /
        (BULLET_SPEED - TANK_SPEED);
      /* Give this bullet time to get away */
      itr = NewEntity (model[bulletModel].nvert);
      AddEntity (itr);
      sine = sin(p->e.dir);
      cosine = cos(p->e.dir);
      itr->pos.x = p->e.pos.x-cosine*model[playerModel].vert[0].x*1.51;
      itr->pos.y = p->e.pos.y-sine*model[playerModel].vert[0].x*1.51;
      itr->vel.x = cosine*BULLET_SPEED*SYNCHRONIZER_MS;
      itr->vel.y = sine*BULLET_SPEED*SYNCHRONIZER_MS;
      itr->dir = p->e.dir;
      itr->angular = 0;
      itr->m = bulletModel;
      itr->p = i;
      itr->textureOffset = FindTexture ("bullet");
      PlaceEntity (itr);
    } /* if shooting */
  } /* for each player */
  for (itr = firstEntity; itr != &lastEntity; itr = next) {
    next = itr->next; /* If we free itr, we must still be able to advance */
    if (itr->m == explosionModel) {
      char str[15];
      sprintf (str, "explode%d", itr->frame); /* animate the explosion */
      if (itr->textureOffset == (i=FindTexture (str))) DeleteEntity (itr);
      itr->textureOffset = i; /* Test if there is another one */
      itr->frame++;
      continue;
    }
    if (itr->m >= maxModelNumber ||
    (itr->angular == 0 && itr->vel.x == 0 && itr->vel.y ==0)) continue;
    
    
    m = model + itr->m;
    for (t = 0, dt = 1; dt > 0 && t < 1; dt /= 2) {
    /* conventional binary search start with dt=0.5, but the propability */
    /* of hitting something is so small, we try to make t=1 */
    
      if (dt < 0.1) dt = 0; /* Revert to a no collision spot */
      
      sine = sin (itr->dir + itr->angular * (t + dt));
      cosine = cos (itr->dir + itr->angular * (t + dt));
      origin = itr->pos + itr->vel * (t + dt);
      
      collision = 0;
      for (i = 0; i < itr->nvert; i++) {
        /* Calculate the newv */
        newv[i].x = origin.x + cosine * m->vert[i].x - sine * m->vert[i].y;
        newv[i].y = origin.y + sine * m->vert[i].x + cosine * m->vert[i].y;
        /* See if the line from the new to the old vert cuts something */
        collision = collision || TestHit (newv + i, itr->vert + i, itr);
      }
      startVert = newv + itr->nvert - 1;
      for (i = 0; !collision && i < itr->nvert; i++) {
        /* See if the line from the new start to the new end cuts something*/
        collision = TestHit (startVert, newv + i, itr);
        startVert = newv + i;
      }
      if (!collision) t += dt;
    } /* find t = the greatest/last time without a collision */
    
    delta = itr->vel * t;
    if (itr->m == playerModel) { /* Test for lap completed */
      p = (playerType *) itr;
      switch (Intersect (param.startFinishB - param.startFinishA, delta,
      itr->pos - param.startFinishA, NULL)) {
        case  1:
          if (p->frags < 0) break; /* Disallow less than -5 frags. */
          p->frags -= 10; /* fallthrou for printing */
        case -1:
          p->frags += 5;
          //printf ("Player %d now has %d points\n", p - player, p->frags);
          break;
      }
    }
    
    itr->pos += delta;
    itr->dir += itr->angular * t;
    memcpy (itr->vert, newv, itr->nvert * sizeof (itr->vert[0]));
    SetBoundBox (itr);
    if (t < 1) { /* if we hit something. */
      if (itr->m == bulletModel || entityHit->m == bulletModel) {
        entity *e = NewEntity (model[explosionModel].nvert);
        e->next = &lastEntity;
        *lastEntity.prev = e; /* Set .next of whoever was last */
        e->prev = lastEntity.prev;
        lastEntity.prev = &e->next;
        // set vel, angular ?
        e->p = itr->p;
        e->pos = pointOfHit;
        e->dir = SynchronizedRandom () * 2 * M_PI;//0;//itr->dir;
        e->m = explosionModel;
        e->textureOffset = FindTexture ("explode"); 
        e->frame = 0; // This will cause a find texture
        PlaceEntity (e);
      }
      if (itr->m == bulletModel) {
        if (entityHit->m == playerModel) {
          p = (playerType *)entityHit;
          p->health -= 10;
          if (p->health <= 0) {
            DeleteEntity (entityHit);
            player[itr->p].frags++;
            //printf("Player %d now has %d frags\n", itr->p,
            //  player[itr->p].frags);
          }
        }
        DeleteEntity (itr);
      }
      else if (entityHit->m == bulletModel) {
        if (itr->m == playerModel) {
          p = (playerType *)itr;
          p->health -= 10;
          if (p->health <= 0) {
            DeleteEntity (itr);
            player[entityHit->p].frags++;
            //printf("Player %d now has %d frags\n", entityHit->p,
            //  player[entityHit->p].frags);
          }
        }
        DeleteEntity (entityHit);
      }
      else {
        itr->vel.x = -itr->vel.x/2;
        itr->vel.y = -itr->vel.y/2;
        itr->angular = 0;
      }
    } /* if we hit something. */
  } /* for all movable entities */
} /* MoveEntities */

unsigned char *textureBase;

void PutString (int x, int y, char *s)
{
  point a, b, c, d;
  while (*s >= ' ') {
    b.tx = a.tx = ((*s - ' ') % 42) * 6 * 256;
    c.ty = a.ty = ((*s - ' ') / 42) * 10 * 256;
    c.tx = d.tx = a.tx + 6 * 256;
    b.ty = d.ty = a.ty + 10 * 256;
    b.sx = a.sx = x * 6 * 256;
    c.sy = a.sy = y * 10 * 256;
    c.sx = d.sx = a.sx + 6 * 256;
    b.sy = d.sy = a.sy + 10 * 256;
    MapTriangle (textureBase + FindTexture ("ascii"), &a, &b, &c, 1);
    MapTriangle (textureBase + FindTexture ("ascii"), &c, &b, &d, 1);
    s++;
    x++;
  }
}

int sameKeyboard = 0; /* This indicates we will not generate input */
int soundPipe = -1;

typedef struct {
  int length, controls;
} regularMsgType;

void Show (void)
/* Texture space is multiplied by 2 so that if the screen is 512x384, */
/* the texture is not zoomed. */
{
  entity *itr;
  int i;
  point a, b, c;
  entityModel *m;
  rational
    povx = (rational)player[myClientNo].e.pos.x - RES_INDEP_SCREEN_WIDTH/2,
    povy = (rational)player[myClientNo].e.pos.y - RES_INDEP_SCREEN_HEIGHT/2;
  
  MakeResolutionDependant (&a, 0, 0);
  a.tx = povx<<1;
  a.ty = povy<<1;
  MakeResolutionDependant (&b, RES_INDEP_SCREEN_WIDTH, 0);
  b.tx = a.tx + RES_INDEP_SCREEN_WIDTH*2;
  b.ty = a.ty;
  MakeResolutionDependant (&c, 0, RES_INDEP_SCREEN_HEIGHT);
  c.tx = a.tx;
  c.ty = a.ty + RES_INDEP_SCREEN_HEIGHT*2;
  MapTriangle (textureBase + FindTexture ("background"), &a, &c, &b, 0);
  
  MakeResolutionDependant (&a, RES_INDEP_SCREEN_WIDTH,
    RES_INDEP_SCREEN_HEIGHT);
  a.tx = b.tx;
  a.ty = c.ty;
  MapTriangle (textureBase + FindTexture ("background"), &b, &c, &a, 0);
  
  for (itr = firstEntity; itr != &lastEntity; itr = itr->next) {
    MakeResolutionDependant (&a, (rational)itr->vert[0].x - povx,
      (rational)itr->vert[0].y - povy);
    MakeResolutionDependant (&c, (rational)itr->vert[1].x - povx,
      (rational)itr->vert[1].y - povy);
    m = itr->m < maxModelNumber ? model + itr->m : (entityModel*) NULL;
    if (m) {
      a.tx = 0;//(rational) m->vert[0].x;
      a.ty = 0;//(rational) m->vert[0].y;
      c.tx = (rational) (m->vert[1].x - m->vert[0].x) << 1;
      c.ty = (rational) (m->vert[1].y - m->vert[0].y) << 1;
    }
    else {
      a.tx = (rational) itr->vert[0].x << 1;
      a.ty = (rational) itr->vert[0].y << 1;
      c.tx = (rational) itr->vert[1].x << 1;
      c.ty = (rational) itr->vert[1].y << 1;
    }
    for (i = 2; i < itr->nvert; i++) {
      memcpy (&b, &c, sizeof (b));
      MakeResolutionDependant (&c, (rational) itr->vert[i].x - povx,
        (rational) itr->vert[i].y - povy);
      if (m) {
        c.tx = (rational) (m->vert[i].x - m->vert[0].x) << 1;
        c.ty = (rational) (m->vert[i].y - m->vert[0].y) << 1;
      }
      else {
        c.tx = (rational) itr->vert[i].x << 1;
        c.ty = (rational) itr->vert[i].y << 1;
      }
      MapTriangle (textureBase + itr->textureOffset, &a, &c, &b,
        itr->m != explosionModel ? 0 : 1);
    }
  } /* For each entity */
  for (i = 0; i < MAX_PLAYERS && player[i].ipAddress; i++) {
    PutString (0, i, player[i].name);
    char frags[5];
    sprintf (frags, "%d", player[i].frags);
    PutString (sizeof (player[i].name), i, frags);
  }
  StartFrame();
    
  keyEventType ket;
  KeySym key;
  static regularMsgType regularMsg = { sizeof(regularMsg), 0 };
  
  while ((ket = GetKey (&key)) != noKeyEvent) {
    //printf ("%s\n", XKeysymToString (key));
    for (i = 0; i < sizeof (keyTable) / sizeof (keyTable[0]); i++) {
      if (key == keyTable[i]) {
        if (ket == keyPressEvent)   regularMsg.controls |= (1<<i);
        if (ket == keyReleaseEvent) regularMsg.controls &= ~(1<<i);
        break;
      }
    }
    if (key == XK_Escape) {
      //printf("Enjoy your Linux !\n");
      exit(0);
    }
  }
  if (!sameKeyboard) Broadcast (&regularMsg);
} /* Show */

void ReadFig (const char *filename)
{ /* May core dump if the fig file is incorrect. */
  FILE *in;
  char line[80], *str;
  int d, type, count, i, entityCount=0, dir;
  double doubleDummy, det;
  vect swapVert;
  entity *e;
  
  param.numOfStarts = 0;
  param.startFinishA.x = param.startFinishA.y = param.startFinishB.x = 0;
  param.startFinishB.y = 1; /* Initialize to defaults */
  if ((in=fopen (filename, "r")) == NULL) {
    perror (filename);
    exit(13);
  }
  while (fgets(line, sizeof(line), in) != NULL) {
    if (sscanf (line, "%d %d %d %d %d %d %d %d %d %lf %d %d %d %d %d %d",
    &d,&d,&d,&d,&type,&d,&d,&d,&d,&doubleDummy,&d,&d,&d,&d,&d,&count)!=16) {
      if (sscanf (line, "%d %d %d %d %d %d %d %lf %d %d %d %lf %lf",
      &d,&d,&d,&d,&d,&d,&d,&doubleDummy,&d,&d,&d,
      &swapVert.x, &swapVert.y) == 13) {
        swapVert.x *= XFIG_SCALER;
        swapVert.y *= XFIG_SCALER;
        if (strchr (line, 'S') && param.numOfStarts <
        sizeof (param.startPoint) / sizeof (param.startPoint[0])) {
          param.startPoint[param.numOfStarts] = swapVert;
          param.numOfStarts++;
        }
        if (strchr (line, 'A')) param.startFinishA = swapVert;
        if (strchr (line, 'B')) param.startFinishB = swapVert;
      }
      continue;
    }
    count--; /* Get number of DISTINCT vertexes */
    e = NewEntity (count);
    AddEntity (e);
    e->m = wallModel;
    if (type == -1) type = 0;
    char textureName[20];
    sprintf (textureName, "wall%d", type);
    e->textureOffset = FindTexture (textureName);
    i = 0;
    entityCount++;
    while (i <= count && fgets (line, sizeof(line), in) != NULL) {
      /* Currently we only read convex polys with there points in */
      /* clockwise order. */
      for (str = strtok (line, " \t"); str; str = strtok (NULL," \t")) {
        if (i < count) {
          e->vert[i].x = atof (str)*XFIG_SCALER;
          e->vert[i].y = atof (strtok (NULL," \t"))*XFIG_SCALER;
          if (i >= 1 && e->vert[i-1] == e->vert[i]) {
            fprintf (stderr, "Warning: Poly %d visits (%d,%d) twice\n",
              entityCount, int(e->vert[i].x/XFIG_SCALER),
              int(e->vert[i].y/XFIG_SCALER));
          }
          if (i >= 2) {
            det = (e->vert[1].x-e->vert[0].x)*(e->vert[i].y-e->vert[0].y)-
            (e->vert[i].x-e->vert[0].x)*(e->vert[1].y-e->vert[0].y);
            if (i == 2) dir = det > 0;
            else if (i > 2 && dir != (det > 0)) {
              fprintf (stderr, "Warning: Poly %d non-convex\n", entityCount);
            }
          } /* if a triangle or larger */
        } /* if not the last point */
        else if (e->vert[0].x != atof (str)*XFIG_SCALER ||
        e->vert[0].y != atof (strtok (NULL," \t"))*XFIG_SCALER) {
          fprintf (stderr, "Warning : Unclosed entity\n");
        }
        i++;
      } /* for each vertex */
    } /* for each line of the vertexes declaration */
    if (!dir) { /* Force clockwise */
      for (i=0,count--;i<count;i++,count--) {
        swapVert = e->vert[i];
        e->vert[i] = e->vert[count];
        e->vert[count] = swapVert;
      }
    }
    SetBoundBox (e);
  } /* while not eof */
} /* ReadFig */

void ReadConfig (void)
{
  FILE *configFile;
  char line[80], *str;
  int i, lno = 0;
  const static char *keyDesc[] = {
    "Forward", "Backward", "Left", "Right", "Fire",
    "OtherForward", "OtherBackward", "OtherLeft", "OtherRight", "OtherFire"
  };

  if ((str=getenv("HOME")) == NULL) return;
  sprintf (line, "%s/.dlrc", str);
  if ((configFile = fopen (line, "r")) != NULL) {
    while (fgets (line, sizeof (line), configFile) != NULL) {
      lno++;
      str = strtok (line, " \t\n");
      if (!str || str[0] == '#') continue;
      
      for (i = 0; i < sizeof (keyDesc) / sizeof (keyDesc[0]); i++) {
        if (strcmp (keyDesc[i], str) == 0) {
          str = strtok (NULL, " \t\n");
          if (str && XStringToKeysym (str) != NoSymbol) {
            keyTable[i] = XStringToKeysym (str);
            str = strtok (NULL, " \t\n");
          }
          break;
        }
      }
      if (str) fprintf(stderr,"Unknown %s in line %d of .dlrc\n", str, lno);
    }
    fclose (configFile);
  }
}

void TestRespawn (playerType *msgPlayer)
{
  if (msgPlayer->health <= 0 &&
  (msgPlayer->controls & PLAYER_FORWARD)) {
    //printf ("Player %d respawns\n", msgPlayer - player);
    msgPlayer->health = 100;
    SynchronizedRandom ();
    msgPlayer->e.pos.x = param.startPoint[param.rnd % param.numOfStarts].x;
    msgPlayer->e.pos.y = param.startPoint[param.rnd % param.numOfStarts].y;
    msgPlayer->e.dir = msgPlayer->e.vel.x = msgPlayer->e.vel.y = 0;
    msgPlayer->energy = 20;
    msgPlayer->fireWait = 0;
    AddEntity (&msgPlayer->e);
    PlaceEntity (&msgPlayer->e);
  } /* if respawn */
}

typedef enum {
  broadcastHello = 1,
  broadcastSameKeyboard,
};

#define FILE_PATH "/usr/lib/games/"

int DownloadNew (int sock, int numberOfNewClient)
{
  char buf[512];
  int bmpFile, code, len;
  entity *ent;
  
  if (numberOfNewClient >= MAX_PLAYERS)
    return 0; /* Just incase this game gets that popular */
  write (sock, &param, sizeof (param));
  write (sock, player, sizeof (player));
  for (ent = firstEntity; ent != &lastEntity; ent = ent->next) {
    if (ent->m != playerModel) {
      write (sock, ent, sizeof (*ent) + ent->nvert * sizeof (ent->vert[0]));
    }
    else {
      code = -ent->p;
      write (sock, &code, sizeof (code));
    }
  }
  code = -MAX_PLAYERS;
  write (sock, &code, sizeof (code));
  if ((bmpFile = open (FILE_PATH "dl.bmp", O_RDONLY)) == -1) return 0;
  while ((len = read (bmpFile, buf, sizeof (buf))) > 0) {
    write (sock, &buf, len);
  }
  close (bmpFile);
  //read (sock, &code, sizeof (code));
  return 1;
}

int main (int argc, char *argv[])
{
  rxMsgHeader *start, *stop;
  int i, downl;
  playerType *msgPlayer;
  char *figFile = FILE_PATH "dl.fig";
  typedef struct {
    int length, type, ipAddress;
    char name[sizeof (player[0].name)];
  } startupType;
  startupType startup;
  typedef struct {
    int length, type, keyboardClient;
  } sameKeyboardType;
  sameKeyboardType sameKeyboardMsg;

  lastEntity.prev = &firstEntity;
  
  strncpy (startup.name, getlogin (), sizeof (startup.name));
  for (i = 1; i < argc; i++) {
    if (argv[i][0] != '-' || strcmp (argv[i], "-geometry") == 0) continue;
    if (argv[i][1] == 'm') {
      i++;
      figFile = argv[i];
      continue;
    }
    if (strcmp (argv[i], "-sameKeyboard") == 0) {
      sameKeyboard = 1;
      usleep (200000); /* Give the server time to listen on the port */
      continue;
    }
    if (argv[i][1] == 'n') {
      i++;
      strncpy (startup.name, argv[i], sizeof (startup.name));
      continue;
    }
    /* Unknown option */
    fprintf (stderr,"\
To run as server for up to %d clients : %s\n\
To connect to a server : %s host.running.deadline.server.com\n\
To run single player (Are you bored ?) : %s alone\n\
You can add '-m[ap] someMap.fig' to the end of the commandline\n\
You set your name with -n[ame] \"Billy G\"\n\
The only supported Xt option is -geometry WxH+X+Y\n\
\n\
If two persons want to play on the same X11 display (and therefore the\n\
same keyboard) start the first client/server, run a second client with\n\
-sameKeyboard as a parameter and give focus to the first client/server.\n\
The second client will try to get his input from the first client.\n",
    MAX_PLAYERS, argv[0], argv[0], argv[0], argv[0]);
    return 0;
  } /* for each argument */
  StartCommunications (argc > 1 && argv[1][0] != '-' ? argv[1] : "", 20052,
    &downl);
  OpenDisplay ("Deadline 2 (Killing butterfly)", argc, argv);
  if (downl == -1) { /* If we serve */
    param.rnd = (time (NULL)&0xfff)+1;
    i = open (FILE_PATH TEXTUREPARAMS_FILE, O_RDONLY);
    read (i, param.texture, sizeof (param.texture));
    close (i);
  
    ReadFig (figFile);
    for (i = 0; i < MAX_PLAYERS; i++) {
      player[i].health = 0;
      player[i].frags = 0;
      player[i].controls = 0;
      char skinStr[7];
      sprintf (skinStr, "skin%d", i);
      if ((player[i].e.textureOffset = FindTexture (skinStr)) == 0) {
        sprintf (skinStr, "skin%d", i % 4);
        player[i].e.textureOffset = FindTexture (skinStr);
      }
      player[i].e.m = playerModel;
      player[i].e.p = i;
      player[i].theOtherPlayer = -1;
      player[i].ipAddress = 0;
    }
    textureBase = ReadBmp (FILE_PATH "dl.bmp");
  }
  else { /* Get our download from the server */
    if (ReadAll (downl, &param, sizeof (param)) < sizeof (param)) {
      fprintf (stderr, "Error. The server is probably full\n");
      exit (15);
    }
    ReadAll (downl, player, sizeof (player));
    int code;
    while (ReadAll (downl, &code, sizeof (code)) == sizeof (code) &&
    code > -MAX_PLAYERS) {
      entity *e;
      if (code <= 0) {
        e = &player[-code].e;
      }
      else {
        e = NewEntity (code); /* nvert = code */
        read (downl, sizeof (e->nvert) + (char*)e,
          sizeof (*e) + sizeof (e->vert[0]) * code - sizeof (e->nvert));
      }
      e->next = &lastEntity; /* Add at the end */
      *lastEntity.prev = e; /* Set .next of whoever was last */
      e->prev = lastEntity.prev;
      lastEntity.prev = &e->next;
    }
    textureBase = ReadBmpFromFd (downl);
    //write (downl, &param.rnd, sizeof (param.rnd));
  } /* Get our download from the server */
  #if 0
  if ((soundPipe = open ("/dev/dsp", O_WRONLY)) != -1) {
    int play_precision = 16, play_stereo = 1, play_rate = 44100,
      fragmentsize=(4<<16)|12;
  
    if(ioctl(soundPipe, SNDCTL_DSP_SAMPLESIZE, &play_precision) == -1 ||
       ioctl(soundPipe, SNDCTL_DSP_STEREO, &play_stereo) == -1 ||
       ioctl(soundPipe, SNDCTL_DSP_SPEED, &play_rate) == -1
    || ioctl(soundPipe, SNDCTL_DSP_SETFRAGMENT, &fragmentsize) == -1
    ){
      fprintf (stderr, "Device can't play sound in this format\n");
      close (soundPipe);
      soundPipe = -1;
    }
    else {
      printf ("Using /dev/dsp for stereo sound effects\n");
    }
  }
  #endif
  
  ReadConfig ();

  startup.name[sizeof (startup.name) - 1] = '\0'; // Force ASCIIZ
  startup.length = sizeof (startup);
  startup.type = broadcastHello;
  startup.ipAddress = gethostid ();
  Broadcast (&startup);

  if (sameKeyboard) { /* if some client also runs on this machine ... */
    for (i=0;;i++) {
      if (i >= myClientNo) {
        fprintf (stderr,
          "No other player on this computer and sameKeyboard is set\n");
        exit (12);
      }
      if (player[i].ipAddress == startup.ipAddress) {
        sameKeyboardMsg.length = sizeof (sameKeyboardMsg);
        sameKeyboardMsg.type = broadcastSameKeyboard;
        sameKeyboardMsg.keyboardClient = i;
        Broadcast (&sameKeyboardMsg);
        break;
      }
    }
  } // If sameKeyboard
  while (1) {
    Show ();
    Synchronize (&start, &stop, DownloadNew);
    //if ((char*)stop - (char*)start > 20)
    //  i=0;
    for (; start < stop; start=NEXT_MSG (start)) {
      msgPlayer = player + start->clientNo;
      if (start->len == sizeof (regularMsgType)) {
        memcpy (&msgPlayer->controls, start->data,
          sizeof (msgPlayer->controls));
        TestRespawn (msgPlayer);
        if (msgPlayer->theOtherPlayer >= 0) {
          player[msgPlayer->theOtherPlayer].controls =
            msgPlayer->controls >> OTHER_PLAYER;
          TestRespawn (player + msgPlayer->theOtherPlayer);
        }
      } /* if a regular message */
      else if (start->len == sizeof (startupType) &&
      MSG_TO(startupType,start)->type == broadcastHello) {
        msgPlayer->ipAddress = MSG_TO(startupType,start)->ipAddress;
        memcpy (msgPlayer->name, MSG_TO(startupType,start)->name,
          sizeof (msgPlayer->name));
      }
      else if (start->len == sizeof (sameKeyboardType) &&
      MSG_TO(sameKeyboardType,start)->type == broadcastSameKeyboard) {
        player[MSG_TO(sameKeyboardType,start)->keyboardClient].
          theOtherPlayer = start->clientNo;
      }
    } /* for each message */
    MoveEntities ();
  } /* while it is still dark outside (and the deathmatch continues ... */
} /* main */

