/* x11.c - X11 output routines for CodeWar
 *
 * Rhett D. Jacobs <rdj@cea.com.au>
 * GNU Public Licence, 1996.
 *
 * Last Modified: <rdj>
 */

#include <stdio.h>
#include <string.h>

#include "x11.h"
#include "graphics.h"
#include "timers.h"
#include "x11_misc.h"
#include "config.h"
#include "weapon.h"
#include "explosion.h"
#include "messages.h"
#include "misc.h"


extern Player *play_list;
extern Weapon_List *weapon_list;
extern Explosion_List *explosion_list;
extern Config config;
extern Channel channel;
extern msg_statistics msg_stats;
extern Resource_Defines rd;

game_window battle;
channel_window ch_win;

static char output_buffer[255];

/* internal functions */
static Window x11int_open_window(int x, int y, int width, int height, 
				 int border_width, unsigned long foreground,
				 int unsigned long background, 
				 char *window_name, char *icon_name);
static GC x11int_createGC(Window window);
static void x11int_setup_backdrop(void);
static void x11int_setup_sprite(Sprite *sprite, char *filename);
static void x11int_setup_rotated_sprite(Sprite *sprite, Sprite *source_spr,
					char sprite_colours[3][16], int reset);
static void x11int_setup_weapons(void);
static void x11int_draw_sprite(Sprite *sprite, int x, int y);
static void x11int_setup_explosions(void);
static void x11int_setup_xcraft(Player *plr);
static void x11int_setup_channel(void);
static Window x11int_open_window(int x, int y, int width, int height, 
				 int border_width, unsigned long foreground,
				 int unsigned long background, 
				 char *window_name, char *icon_name)
{
  Window working_window;
  XSizeHints size_hints;
  char path_string[255];
  
  working_window = XCreateSimpleWindow(battle.display,
				       DefaultRootWindow(battle.display),
				       x, y, width, height, border_width,
				       foreground, background);

  size_hints.flags = PPosition | PMinSize | PSize;
  size_hints.x = x;
  size_hints.y = y;
  size_hints.width = width;
  size_hints.height = height;
  size_hints.min_width = 50;
  size_hints.min_height = 50;
  
  strcpy(path_string, rd.x11_pixmap_path);
  strcat(path_string, rd.x11_icon);

  XSetStandardProperties(battle.display, working_window, window_name, 
			 icon_name,
			 ReadXPM(battle.display, battle.window, battle.screen,
				 path_string, &battle.attributes), NULL, 
			 (int)NULL, &size_hints);

  return(working_window);
}


static GC x11int_createGC(Window window)
{
  XGCValues vals;

  return(XCreateGC(battle.display, window, 0, &vals));
}


static void x11int_setup_sprite(Sprite *sprite, char *filename)
{
  sprite->image = ReadXPM(battle.display, battle.window, 
			  battle.screen, filename,
			  &sprite->attributes);
  
  sprite->mask = CreateMask(sprite->image, battle.display,
			    battle.screen, sprite->attributes.width,
			    sprite->attributes.height);
}


static void x11int_setup_rotated_sprite(Sprite *sprite, Sprite *source_spr,
					char sprite_colours[3][16], int reset)
{
  sprite->image = Rotate_Pixmap(source_spr->image, battle.display,
				battle.window, battle.screen, 
				source_spr->attributes.width,
				source_spr->attributes.height,
				&sprite->attributes,
				sprite_colours,
				reset);
  
  sprite->mask = CreateMask(sprite->image, battle.display,
			    battle.screen, sprite->attributes.width,
			    sprite->attributes.height);
}


static void x11int_setup_weapons(void)
{
  char path_string[255];

  strcpy(path_string, rd.x11_pixmap_path);
  strcat(path_string, rd.x11_cannon);
  x11int_setup_sprite(&battle.cannon, path_string);

  strcpy(path_string, rd.x11_pixmap_path);
  strcat(path_string, rd.x11_missile);
  x11int_setup_sprite(&battle.missile, path_string);

  strcpy(path_string, rd.x11_pixmap_path);
  strcat(path_string, rd.x11_charge);
  x11int_setup_sprite(&battle.charge, path_string);

  strcpy(path_string, rd.x11_pixmap_path);
  strcat(path_string, rd.x11_bomb);
  x11int_setup_sprite(&battle.bomb, path_string);

  strcpy(path_string, rd.x11_pixmap_path);
  strcat(path_string, rd.x11_atomic);
  x11int_setup_sprite(&battle.atomic, path_string);

  strcpy(path_string, rd.x11_pixmap_path);
  strcat(path_string, rd.x11_slow);
  x11int_setup_sprite(&battle.slow, path_string);

  strcpy(path_string, rd.x11_pixmap_path);
  strcat(path_string, rd.x11_server_slow);
  x11int_setup_sprite(&battle.server_slow, path_string);
}


static void x11int_setup_explosions(void)
{
  int i;
  char image_file[255];
  
  for (i = 0; i < X11_EXPLOSION_MAX; i++) {
    sprintf(image_file,"%s%s%d.xpm", 
	    rd.x11_pixmap_path, rd.x11_anim_explosion, 
	    X11_EXPLOSION_MAX - i);
    x11int_setup_sprite(&battle.explosion[i], image_file);
  }
}


void X11_remove_player(int socket)
{
  Player *plr;

  if (find_player(socket, &plr, play_list))
    if (!(plr->graphics.first))
      XDestroyWindow(battle.display, plr->graphics.window);
}


static void x11int_setup_xcraft(Player *plr)
{
  int i;
  char path_string[255];
  
  /* setup X-Craft window */
  sprintf(output_buffer,"%s - 0x%X", plr->name, plr->socket);
  plr->graphics.window = x11int_open_window(1, 1,
					    rd.x11_xcraft_win_width,
					    rd.x11_xcraft_win_height,
					    1, battle.foreground,
					    battle.background,
					    output_buffer,
					    output_buffer);

  XMapWindow(battle.display, plr->graphics.window);

  strcpy(path_string, rd.x11_pixmap_path);
  strcat(path_string, rd.x11_xcraft);
  
  x11int_setup_sprite(&plr->graphics.xcraft[0], path_string);
  x11int_setup_rotated_sprite(&plr->graphics.xcraft[1], 
			      &plr->graphics.xcraft[0],
			      plr->graphics.xcraft_colour, 1);
  for (i = 2; i <= X11_XCRAFT_MAX; i++) {
    x11int_setup_rotated_sprite(&plr->graphics.xcraft[i%X11_XCRAFT_MAX], 
				&plr->graphics.xcraft[i-1],
				plr->graphics.xcraft_colour, 0);
  }

  return;
}


static void x11int_setup_channel(void)
{
  ch_win.window = x11int_open_window(1, 1,
				     rd.x11_channel_win_width,
				     rd.x11_channel_win_height,
				     1, battle.foreground,
				     battle.background,
				     "Open Channel",
				     "Open Channel");
  
  XMapWindow(battle.display, ch_win.window);
  
  return;
}


static void x11int_setup_backdrop(void)
{
  static int first = 1;
  int x, y, x_times, y_times;
  char path_string[255];
  
  if (first) {
    strcpy(path_string, rd.x11_pixmap_path);
    strcat(path_string, rd.x11_backdrop);
    
    /* load original xpm image */
    battle.backdrop_tile.image = ReadXPM(battle.display, battle.window, 
					 battle.screen, path_string, 
					 &battle.backdrop_tile.attributes);
  }
  
  /* deterime size necesary of backdrop pixmap */
  x_times = battle.width / battle.backdrop_tile.attributes.width + 1;
  y_times = battle.height / battle.backdrop_tile.attributes.height + 1;
  
  if (!(first))
    XFreePixmap(battle.display, battle.backdrop.image);
  else
    first = 0;
  
  /* create backdrop pixmap */
  battle.backdrop.image = XCreatePixmap(battle.display,
					RootWindow(battle.display, 
						   battle.screen),
					x_times * 
					battle.backdrop_tile.attributes.width,
					y_times * 
					battle.backdrop_tile.attributes.height,
					DefaultDepth(battle.display, 
						     battle.screen));
  
  /* tile image into new pixmap */
  for (y = 0; y < y_times; y++)
    for (x = 0; x < x_times; x++)
      XCopyArea(battle.display, battle.backdrop_tile.image, 
		battle.backdrop.image, battle.gc, 1, 1, 
		battle.backdrop_tile.attributes.width,
		battle.backdrop_tile.attributes.height, 
		1 + x * battle.backdrop_tile.attributes.width - x,
		1 + y * battle.backdrop_tile.attributes.height - y);
}


void X11_refresh(void)
{
  int i, j, k, offset = 10;
  static int first = 1;
  char temp_buffer[255];
  char build_buffer[255];
/*  XEvent event;*/
  XWindowAttributes war;
  static int destination_speed;
  static int count, last_count = 0;
  static timer_val refresh_timer = {0,0};
  static char output_string[255];

  if (first) {
    destination_speed = (int)(1000000.0/(float)config.simulate_int+0.5);
    first = 0;
  }

  /*
     if (XCheckWindowEvent(battle.display, battle.window, ButtonReleaseMask,
     &event))
     battle.size_change = TRUE;
     */

  XGetWindowAttributes(battle.display, battle.window, &war);

  if (war.width != battle.width ||
      war.height != battle.height) {
    battle.size_change = TRUE;
    battle.width = war.width;
    battle.height = war.height;
  } 
  
  if (!(battle.size_change)) {
    XCopyArea(battle.display, battle.render_window, 
	      battle.window, battle.gc, 1, 1, battle.width,
	      battle.height, 1, 1);

    XCopyArea(battle.display, battle.backdrop.image, battle.render_window,
	      battle.gc, 1, 1, battle.width, battle.height, 1, 1);
    
    XDrawImageString(battle.display, battle.render_window, battle.text_gc,
		     10, 10, output_string, strlen(output_string));

    XSync(battle.display, FALSE);
  } else {
    XFreePixmap(battle.display, battle.render_window);
    battle.render_window = XCreatePixmap(battle.display,
					 RootWindow(battle.display, 
						    battle.screen),
					 battle.width, battle.height,
					 DefaultDepth(battle.display,
						      battle.screen));

    battle.scale[X_AXIS] = (float)battle.width / (float)config.field[X_AXIS];
    battle.scale[Y_AXIS] = (float)battle.height / (float)config.field[Y_AXIS];

    x11int_setup_backdrop();
    
    XCopyArea(battle.display, battle.backdrop.image, battle.render_window,
	      battle.gc, 1, 1, battle.width, battle.height, 1, 1);
    
    battle.size_change = FALSE;
  }

  count++;

  if (check_timer(&refresh_timer)) {
    set_timer(&refresh_timer, (long)1000000);
    last_count = count;
    count = 0;
  }

  if (config.server_slow)
    x11int_draw_sprite(&battle.slow, 10, 10);

  if (last_count < destination_speed)
    x11int_draw_sprite(&battle.server_slow, 
		       battle.width-battle.server_slow.attributes.width - 10,
		       10);


  for (i = 0; i < rd.channel_bandwidth; i += 16) {

    output_buffer[0] = (char) NULL;
    
    for (j = 0; j < 4; j++) {
      
      build_buffer[0] = (char)NULL;

      for (k = 0; k < 4; k++) {
	  sprintf(temp_buffer,"%02X", channel.data[i+4*j+k]);
	  
	strcat(build_buffer, temp_buffer);
      }

      strcat(output_buffer, build_buffer);
      strcat(output_buffer, " ");
    }

    XDrawImageString(battle.display, ch_win.window, battle.text_gc,
		     10, offset, output_buffer, strlen(output_buffer));
    offset += 12;    
  }

  if (last_count != 0)
    sprintf(output_string, 
	    "updates %d => %3.3fms  W %d H %d Dest %d Elasped Time: %0.3f", 
	    last_count, 1.0/(float)last_count, battle.width, battle.height,
	    destination_speed, config.elapsed_time);
  
  
  return;
}


void X11_setup_display(void)
{
  unsigned long valuemask;
  XGCValues values;

  if ((battle.display = XOpenDisplay(NULL)) == NULL) {
    fprintf(stderr,"error: could not open graphics display.\n");
    exit(0);
  }

  battle.width = rd.gfx_width;
  battle.height = rd.gfx_height;
  battle.size_change = FALSE;

  battle.scale[X_AXIS] = (float)battle.width / (float)config.field[X_AXIS];
  battle.scale[Y_AXIS] = (float)battle.height / (float)config.field[Y_AXIS];

  battle.screen = DefaultScreen(battle.display);

  /* use default foreground and background colours for now. */
  battle.foreground = WhitePixel(battle.display, battle.screen);
  battle.background = BlackPixel(battle.display, battle.screen);
  
  sprintf(output_buffer,"CodeWar v%0.1f - rdj@cea.com.au", rd.version);
  battle.window = x11int_open_window(1, 1, (int)battle.width, 
				     (int)battle.height,
				     1, battle.foreground, battle.background,
				     output_buffer,
				     "CodeWar");
  battle.clear_gc = x11int_createGC(battle.window);
  battle.text_gc = x11int_createGC(battle.window);
  battle.gc = x11int_createGC(battle.window);
  battle.green_gc = x11int_createGC(battle.window);
  battle.red_gc = x11int_createGC(battle.window);
  battle.blue_gc = x11int_createGC(battle.window);
  battle.orange_gc = x11int_createGC(battle.window);
  battle.cyan_gc = x11int_createGC(battle.window);
  battle.passive_gc = x11int_createGC(battle.window);
  battle.title_gc = x11int_createGC(battle.window);

  XSetForeground(battle.display, battle.clear_gc, battle.background);
  XSetForeground(battle.display, battle.text_gc, battle.foreground);
  XSetForeground(battle.display, battle.gc, battle.foreground);

  XSetForeground(battle.display, battle.green_gc,
		 palette_colour_set("green", battle.display, battle.screen));
  XSetForeground(battle.display, battle.red_gc,
		 palette_colour_set("red", battle.display, battle.screen));
  XSetForeground(battle.display, battle.blue_gc,
		 palette_colour_set("blue", battle.display, battle.screen));
  XSetForeground(battle.display, battle.orange_gc,
		 palette_colour_set("orange", battle.display, battle.screen));
  XSetForeground(battle.display, battle.cyan_gc,
		 palette_colour_set("cyan", battle.display, battle.screen));

  XSetForeground(battle.display, battle.passive_gc,
		 palette_colour_set("grey75", battle.display, battle.screen));
  XSetBackground(battle.display, battle.passive_gc,
		 palette_colour_set("grey30", battle.display, battle.screen));

  XSetForeground(battle.display, battle.title_gc,
		 palette_colour_set("white", battle.display, battle.screen));
  XSetBackground(battle.display, battle.title_gc,
		 palette_colour_set("grey30", battle.display, battle.screen));

  XSetBackground(battle.display, battle.clear_gc, battle.background);
  XSetBackground(battle.display, battle.text_gc, battle.background);
  XSetBackground(battle.display, battle.gc, battle.background);
  XSetBackground(battle.display, battle.green_gc, battle.background);
  XSetBackground(battle.display, battle.red_gc, battle.background);
  XSetBackground(battle.display, battle.blue_gc, battle.background);
  XSetBackground(battle.display, battle.orange_gc, battle.background);
  XSetBackground(battle.display, battle.cyan_gc, battle.background);

  XSelectInput(battle.display, battle.window, ButtonReleaseMask);
  
  valuemask =  GCGraphicsExposures;
  values.graphics_exposures = FALSE;
  XChangeGC(battle.display, battle.gc, valuemask, &values);

  valuemask = GCFunction;
  values.function = GXset;
  XChangeGC(battle.display, battle.text_gc, valuemask, &values);

  XMapWindow(battle.display, battle.window);

  battle.render_window = XCreatePixmap(battle.display,
				       RootWindow(battle.display, 
						  battle.screen),
				       battle.width, battle.height,
				       DefaultDepth(battle.display,
						    battle.screen));
  
  x11int_setup_backdrop();
  x11int_setup_weapons();
  x11int_setup_explosions();
  x11int_setup_channel();

  XCopyArea(battle.display, battle.backdrop.image, battle.render_window,
	    battle.gc, 1, 1, battle.width, battle.height, 1, 1);
  return;
}


void X11_draw_field_border(void)
{
  return;
}


void X11_printxy(char *buffer, int x, int y)
{
  return;
}


static void x11int_draw_sprite(Sprite *sprite, int x, int y)
{
  XSetClipOrigin(battle.display, battle.gc, x, y);
  XSetClipMask(battle.display, battle.gc, sprite->mask);
  XCopyArea(battle.display, sprite->image,
	    battle.render_window, battle.gc, 1, 1, 
	    sprite->attributes.width, sprite->attributes.height,
	    x, y);
  XSetClipMask(battle.display, battle.gc, None); 
}


void X11_draw_player_positions(void)
{
  Player *plr;
  int frame_index;

  if (play_list != NULL) {
    plr = play_list;
    while (plr != NULL) {

      /* if new player, set up sprite */
      if (plr->graphics.first) {
	x11int_setup_xcraft(plr);
	plr->graphics.first = 0;
      }
      
      if (check_timer(&plr->graphics.frame_timer)) {
	set_timer(&plr->graphics.frame_timer, plr->graphics.frame_delay);
	plr->graphics.current_frame =
	  (plr->graphics.current_frame+1)%(X11_XCRAFT_MAX);
      }
      
      frame_index = plr->graphics.current_frame;

      x11int_draw_sprite(&plr->graphics.xcraft[frame_index],
			 (int)(battle.scale[X_AXIS] *
			       (float)plr->status.position[X_AXIS]) - 
			 plr->graphics.xcraft[frame_index].attributes.width/2,
			 (int)(battle.scale[Y_AXIS] * 
			       (float)plr->status.position[Y_AXIS]) - 
			 plr->graphics.xcraft[frame_index].attributes.height/2);
      
      plr = plr->nextplr;
    }
  }
  
  X11_draw_combat();
  X11_draw_explosions();

  X11_draw_player_status();

  return;
}


void X11_clear_window(int min_x, int min_y, int max_x, int max_y)
{
  return;
}


void X11_close_display(void)
{
  XFreeGC(battle.display, battle.gc);
  XFreePixmap(battle.display, battle.render_window);
  XCloseDisplay(battle.display);
  
  return;
}


void X11_draw_player_borders(void)
{
  return;
}


void X11_draw_box(int min_x, int min_y, int max_x, int max_y)
{
  return;
}


void X11_field_display(float x, float y, char *buffer)
{
  return;
}


void X11_refresh_plr(void)
{
  return;
}


void X11_clear_player_area(void)
{
  return;
}


void X11_draw_combat(void)
{
  Weapon_List *ptr;
  Sprite *spr = (Sprite *)NULL;

  if (weapon_list != NULL) {
    ptr = weapon_list;
    while (ptr != NULL) {
      switch(ptr->weapon) {
      case WP_MISSILE:
	spr = &battle.cannon; break;
      case WP_CHARGE:
	spr = &battle.charge; break;
      case WP_ATOMIC:
	spr = &battle.atomic; break;
      case WP_CANNON:
	spr = &battle.cannon; break;
      case WP_BOMB:
	spr = &battle.bomb; break;
      };

      x11int_draw_sprite(spr, 
			 (int)(battle.scale[X_AXIS] * ptr->position[X_AXIS] -
			       spr->attributes.width / 2),
			 (int)(battle.scale[Y_AXIS] * ptr->position[Y_AXIS] -
			       spr->attributes.height / 2));

      ptr = ptr->nextptr;
    }
  }

  return;
}


void X11_explode(int id)
{
  Explosion_List *ptr;
  Sprite *spr = (Sprite *)NULL;
  int frame_no;

  if (find_explosion(id, &ptr, explosion_list)) {

    frame_no = (int)(remaining_timer(&ptr->remove) / 
		     (long)(EXP_DISPLAY_TIME/X11_EXPLOSION_MAX+1));
    
    if (frame_no >= X11_EXPLOSION_MAX)
      frame_no = X11_EXPLOSION_MAX-1;
    
    switch(ptr->weapon) {
    case WP_CANNON:
      spr = &battle.explosion[frame_no]; break;
    case WP_MISSILE:
      spr = &battle.explosion[frame_no]; break;
    case WP_CHARGE:
      spr = &battle.explosion[frame_no]; break;
    case WP_BOMB:
      spr = &battle.explosion[frame_no]; break;
    case WP_ATOMIC:
      spr = &battle.explosion[frame_no]; break;
    }

    x11int_draw_sprite(spr, 
		       (int)(battle.scale[X_AXIS] * 
			     (float)ptr->position[X_AXIS]) - 
		       spr->attributes.width / 2,
		       (int)(battle.scale[Y_AXIS] * 
			     (float)ptr->position[Y_AXIS]) - 
		       spr->attributes.height / 2);
  }

  return;
}


void X11_draw_explosions(void)
{
  Explosion_List *ptr;
  
  if (explosion_list != NULL) {
    ptr = explosion_list;
    while (ptr != NULL) {
      if (check_timer(&ptr->remove))
	remove_explosion(ptr->id, &explosion_list);
      else
	X11_explode(ptr->id);
      ptr = ptr->nextptr;
    }
  }

  return;
}


void X11_field_display_offset(float x, float y, int off_x, int off_y, 
			       char *buffer)
{
  return;
}


void X11_show_name_sock(Player *plr, int x, int y)
{
  sprintf(output_buffer,"%s [0x%X] %0.0f %0.0f        ",
	  plr->name, plr->socket, 
	  plr->status.position[X_AXIS], plr->status.position[Y_AXIS]);
  XDrawImageString(battle.display, plr->graphics.window, battle.text_gc,
		   x, y, output_buffer, strlen(output_buffer));
  return;
}


void X11_show_vel_acc(Player *plr, int x, int y)
{
  sprintf(output_buffer,"Vl: %0.2f Ac: %0.2f        ",
	  pythagoras(plr->status.velocity[X_AXIS],
		     plr->status.velocity[Y_AXIS]),
	  pythagoras(plr->status.acc_actual[X_AXIS],
		     plr->status.acc_actual[Y_AXIS]));
  
  XDrawImageString(battle.display, plr->graphics.window, battle.text_gc,
		   x, y, output_buffer, strlen(output_buffer));
  return;
}


void X11_show_debug(Player *plr, int x, int y)
{
  sprintf(output_buffer,"%d", msg_stats.max_port_size);
  sprintf(output_buffer,"P: %d | M: %d ", msg_stats.port_errors,
	  msg_stats.msg_errors);
  sprintf(output_buffer,"%d", plr->port_buffer_size);

  return;
}


void X11_show_can_miss(Player *plr, int x, int y)
{
  sprintf(output_buffer, "Cn: %ld Ms: %d        ",
	  plr->weapons.cannons, plr->weapons.missiles);
  
  XDrawImageString(battle.display, plr->graphics.window, battle.text_gc,
		   x, y, output_buffer, strlen(output_buffer));
  
  return;
}


void X11_show_bomb_atomic(Player *plr, int x, int y)
{
  sprintf(output_buffer, "Bm: %d At: %d        ",
	  plr->weapons.bombs, plr->weapons.atomics);

  XDrawImageString(battle.display, plr->graphics.window, battle.text_gc,
		   x, y, output_buffer, strlen(output_buffer));
  
  return;
}


void X11_show_head_scan(Player *plr, int x, int y)
{
  sprintf(output_buffer,"Hd: %.0f Sn: %.2f Sd: %.2f        ",
	  rad_2_deg(plr->status.acc_heading),
	  plr->status.scan,
	  rad_2_deg(plr->status.scan_dir));

  XDrawImageString(battle.display, plr->graphics.window, battle.text_gc,
		   x, y, output_buffer, strlen(output_buffer));
  
  return;
}


void X11_show_damage_shields(Player *plr, int x, int y)
{
  GC output_gc;

  if (plr->status.damage != 0)
    output_gc = battle.red_gc;
  else
    if (plr->status.shields < 100)
      output_gc = battle.orange_gc;
    else
      output_gc = battle.green_gc;

  sprintf(output_buffer,"Dm: %3d Sh: %3d En: %3d        ", plr->status.damage,
	  plr->status.shields, plr->status.energy);
  
  XDrawImageString(battle.display, plr->graphics.window, output_gc,
		   x, y, output_buffer, strlen(output_buffer));
  
  return;
}


void X11_show_message(Player *plr, int x, int y)
{
  if (plr->message[0]) {
    sprintf(output_buffer,"\"%-.26s\"                                        ",
	    plr->message);
    
    XDrawImageString(battle.display, plr->graphics.window, battle.cyan_gc,
		     x, y, output_buffer, strlen(output_buffer));
  }
  
  return;
}


void X11_draw_static_sprite(Player *plr, int x, int y)
{
  Sprite *sprite = &plr->graphics.xcraft[plr->graphics.current_frame];

  XSetClipOrigin(battle.display, battle.gc, x, y);
  XSetClipMask(battle.display, battle.gc, sprite->mask);
  XCopyArea(battle.display, sprite->image,
	    plr->graphics.window, battle.gc, 1, 1, 
	    sprite->attributes.width, sprite->attributes.height,
	    x, y);
  XSetClipMask(battle.display, battle.gc, None); 
  
  return;
}


void X11_draw_player_status(void)
{
  Player *plr;
  XWindowAttributes war;
  int x_pos, y_pos, y_inc;

  x_pos = 10;
  y_inc = 13;
  
  if (play_list != NULL) {
    plr = play_list;
    while (plr != NULL) {
      y_pos = 10;

      /*XClearWindow(battle.display, plr->graphics.window);*/
      
      X11_show_name_sock(plr, x_pos, y_pos);
      y_pos += y_inc;
      
      X11_show_vel_acc(plr, x_pos, y_pos);
      y_pos += y_inc;

      X11_show_head_scan(plr, x_pos, y_pos); 
      y_pos += y_inc;

      X11_show_damage_shields(plr, x_pos, y_pos);
      y_pos += y_inc;

      X11_show_can_miss(plr, x_pos, y_pos);
      y_pos += y_inc;

      X11_show_bomb_atomic(plr, x_pos, y_pos);
      y_pos += y_inc;

      X11_show_message(plr, x_pos, y_pos);

      XGetWindowAttributes(battle.display, plr->graphics.window, &war);
      X11_draw_static_sprite(plr, war.width - 30, 5);

      plr = plr->nextplr;
    }
  }
}
