/* weapon.c - weapon routines for CodeWar
 *
 * Rhett D. Jacobs <rdj@cea.com.au>
 * GNU Public Licence, 1996.
 *
 * Last Modified: 9/3/96 <rdj>
 */

#include <stdlib.h>

#include "weapon.h"
#include "explosion.h"
#include "misc.h"
#include "damage.h"

extern Config config;

Weapon_List *weapon_list = (Weapon_List *)NULL;


/* combat - main call routine to handle weapon simulation.
 */
void combat(void)
{
  Weapon_List *ptr;
  int remove;

  if (weapon_list != NULL) {
    ptr = weapon_list;
    while (ptr != NULL) {
      remove = FALSE;
      if (!(weapon_update(ptr)))
	remove = TRUE;
      if (remove)
	remove_weapon(ptr->id, &weapon_list);
      else
/*	if (check_timer(&ptr->detinate))*/
	if (check_sim_timer(&ptr->detinate))
	  weapon_detinate(ptr->id, ptr->position[X_AXIS], 
			  ptr->position[Y_AXIS]);
      ptr = ptr->nextptr;
    }
  }

  return;
}


/* weapon_detinate - handles the weapon detination at the current location
 * of the weapon.  Then adds the explosion to the explosion list for output
 * to the console and removes the weapon entry.
 */
void weapon_detinate(int id, float x, float y)
{
  Weapon_List *ptr;

  if (find_weapon(id, &ptr, weapon_list)) {
    add_explosion(ptr->weapon, x, y, EXP_DISPLAY_TIME);
    damage_check_players(ptr->weapon, x, y);
    remove_weapon(id, &weapon_list);
  }
}


int weapon_update(Weapon_List *ptr)
{
  int i, ret_val = TRUE;
  float vel_old[2];

  for (i = 0; i < 2; i++) {
    vel_old[i] = ptr->velocity[i];
    ptr->velocity[i] = vel_old[i] + ptr->acceleration[i] *
      (float)config.time_int;

    ptr->position[i] = vel_old[i] * config.time_int + 0.5 *
      ptr->acceleration[i] * config.time_int * config.time_int +
	ptr->position[i];
  }
  if (!(weapon_check_player_bounds(ptr->position)))
    ret_val = FALSE;

  return(ret_val);
}


int weapon_check_player_bounds(float pos[])
{
  int ret_val = TRUE;

  if (!(pos[X_AXIS] <= config.field[X_AXIS] &&
      pos[Y_AXIS] <= config.field[Y_AXIS] &&
      pos[X_AXIS] >= (float)0 &&
      pos[Y_AXIS] >= (float)0))
    ret_val = FALSE;
  
  return(ret_val);
}


/* get_unique_id - finds a unique id for the system to use in assigning
 * a new weapon to the weapon list.
 */
int get_unique_id(Weapon_List *search_list)
{
  static int ret_val = 0;
  return(ret_val++%65535);
}


/* add_weapn - adds a weapon to the weapon list for simulation.
 */
void add_weapon(short weapon_type, float velocity[2], float position[2],
		float acceleration, float direction, float detinate)
{
  Weapon_List *new_weapon;

  new_weapon = (Weapon_List *)calloc(1, sizeof(Weapon_List));
  /*  set_timer(&new_weapon->detinate, (float)1000000 * (float)detinate);*/
  set_sim_timer(&new_weapon->detinate, (long)(1000000.0 * detinate));

  new_weapon->weapon = weapon_type;
  new_weapon->velocity[X_AXIS] = velocity[X_AXIS];
  new_weapon->velocity[Y_AXIS] = velocity[Y_AXIS];
  new_weapon->position[X_AXIS] = position[X_AXIS];
  new_weapon->position[Y_AXIS] = position[Y_AXIS];

  new_weapon->acceleration[X_AXIS] = 
    (float)((double)acceleration *
	    /*cos_val((float)direction));*/
	    cos((double)direction));
  new_weapon->acceleration[Y_AXIS] = 
    (float)((double)acceleration *
	    /*sin_val((float)direction));*/
	    sin((double)direction));

  new_weapon->id = get_unique_id(weapon_list);
  add_weapon_list(new_weapon, weapon_list);

 return;
}


/* find_weapon - finds a weapon entry in the weapon list.
 */
int find_weapon(int id, Weapon_List **weapon_ptr, Weapon_List *search_list)
{
  Weapon_List *tmp;

  if (search_list == NULL)
    return(FALSE);
  else {
    tmp = search_list;
    while (tmp != NULL) {
      if (tmp->id == id) {
	*weapon_ptr = tmp;
	return(TRUE);
      }
      tmp = tmp->nextptr;
    }
  }
  return(FALSE);
}


void add_weapon_list(Weapon_List *new, Weapon_List *list)
{
  Weapon_List *tmp;

  if (list == NULL)
    weapon_list = new;
  else {
    tmp = list;
    while (tmp->nextptr != NULL)
      tmp = tmp->nextptr;
    tmp->nextptr = new;
  }
  return;
}


/* remove_weapon - Removes a weapon entry from the weapon list.
 */
int remove_weapon(int id, Weapon_List **search_list)
{
  Weapon_List *tmp, *tmp2;

  if (*search_list == NULL) {
    return(FALSE);
  } else {
    tmp = *search_list;
    if (tmp->id == id) {
      if (tmp->nextptr == NULL) {
	*search_list = NULL;
	free(tmp);
      } else {
	*search_list = tmp->nextptr;
	free(tmp);
      }
    } else {
      tmp2 = tmp;
      tmp = tmp->nextptr;
      while (tmp != NULL) {
	if (tmp->id == id) {
	  tmp2->nextptr = tmp->nextptr;
	  free(tmp);
	  return(TRUE);
	} else {
	  tmp2 = tmp;
	  tmp = tmp->nextptr;
	}
      }
    }
  }

  return(FALSE);
}
