/* simulate.c - CodeWar processing system for player update and battle
 *
 * Last Modified: 10/12/95 <rdj>
 */

#include "simulate.h"
#include "graphics.h"

extern Player *play_list;
extern Config config;


/* simulate - this is the main processing routine called from the simulation
 * timer expiring.  It is responsible for updating all player positions, 
 * weapons, and resulting consequences.
 */
void simulate(void)
{
  Player *plr;

  config.elapsed_time += config.time_int;

  if (play_list != NULL) {
    plr = play_list;
    while (plr != NULL) {
      sim_update_position(plr);
      plr = plr->nextplr;
    }
  }
}

/* sim_update_position - this routine works out where the player's X-Craft is
 * based upon their last "cw_power()" call and their current position,
 * heading etc.  Acceleration is generated by the config.h setting for the
 * maximum X-Craft force (F) and the weight of the X-Craft, which is related
 * to the number of weapons used.
 *
 * Main equations:    Force = Mass * Acceleration
 *                 Velocity = Initial_Velocity + Acceration * Time_Interval
 *     Pos_Final - Pos_Init = 1/2 * (Init_Vel + Final_Vel) * Time_Interval
 */
void sim_update_position(Player *plr)
{
  int i;
  float max_acc, max_axis_acc[2], vel_old[2], pos_old[2];

  max_acc = (float)config.xcraft_force/(float)plr->status.mass;
  max_axis_acc[X_AXIS] = (float)max_acc * 
    (float)cos((double)plr->status.acc_heading);
  max_axis_acc[Y_AXIS] = (float)max_acc * 
    sin((double)plr->status.acc_heading);
  
  for (i = 0; i < 2; i++) {
    /* calculate current acceleration */
    if (fabs((double)plr->status.acc_desired[i]) > 
	fabs((double)max_axis_acc[i])) {
      if (plr->status.acc_desired[i] >= 0.0)
	plr->status.acc_actual[i] = max_axis_acc[i];
      else
	plr->status.acc_actual[i] = -max_axis_acc[i];
    } else
      plr->status.acc_actual[i] = plr->status.acc_desired[i];
    
    vel_old[i] = plr->status.velocity[i];
    plr->status.velocity[i] = vel_old[i] + plr->status.acc_actual[i] * 
      (float)config.time_int;
    
    pos_old[i] = plr->status.position[i];
    plr->status.position[i] = vel_old[i] * config.time_int + 0.5 *
      plr->status.acc_actual[i] * config.time_int * config.time_int +
	pos_old[i];
  }
  
  sim_check_player_bounds(plr->socket, plr->status.position,
			  plr->status.velocity, plr->status.acc_actual,
			  pos_old, vel_old);
}


void sim_check_player_bounds(int socket, float pos[], float vel[], 
			     float acc_actual[], float pos_old[],
			     float vel_old[])
{
  int i;
  
  if (pos[X_AXIS] >= config.field[X_AXIS] ||
      pos[Y_AXIS] >= config.field[Y_AXIS] ||
      pos[X_AXIS] < 0.0 || pos[Y_AXIS] < 0.0) {
    
    for (i = 0; i < 2; i++) {
      switch(config.wall_type) {
      case RUBBER:
	if (pos[i] >= config.field[i] || pos[i] < 0.0) {
	  vel[i] = vel_old[i];
	  pos[i] = vel_old[i] + pos_old[i];
	  vel[i] = -vel[i];
	  pos[i] = (vel_old[i] + vel[i]) * config.time_int * 0.5 + pos_old[i];
	} 
	break;
      case STONE:
	if (pos[i] >= config.field[i]) {
	  pos[i] = config.field[i]-1.0;
	  vel[i] = 0.0;
	}
	if (pos[i] < 0.0) {
	  pos[i] = 0.0;
	  vel[i] = 0.0;
	}
	break;
      case ABYSS:
	if (pos[i] >= config.field[i] || pos[i] < 0.0) {
	  remove_player(socket, &play_list);
	  graphics_refresh_plr();
	}
	break;
      case TWIGHLIGHT:
	if (pos[i] >= config.field[i])
	  pos[i] = 0.0;
	else if (pos[i] < 0.0)
	  pos[i] = config.field[i];
	break;
      }
    }
   } 

  return;
}

/* rebound_time - rubber wall deflection routines.  These rely on solving
 * the equation: 
 *   Pos_Final - Pos_Initial = Vel_Initial * Time + 1/2 * Acc * Time^2
 * for the parameter Time to evaluate at what point the X-Craft
 * intercepted with the wall and began to rebound.  Time to reach a 
 * boundary is returned.
 */
float rebound_time(float X, float X0, float V0, float A)
{
  float ret_val, t1, t2;
  double inside;

  inside = (double)(V0 * V0 + 2 * A * (X - X0));
  
  if (inside >= (double)0.0) {
    t1 = (-V0 + (float)sqrt(inside)) / A;
    t2 = (-V0 - (float)sqrt(inside)) / A;
    
    if (t1 < 0.0)
      ret_val = t2;
    else
      if (t2 < 0.0)
	ret_val = t1;
      else
	if (t1 < t2)
	  ret_val = t1;
	else
	  ret_val = t2;
  } else
    ret_val = 0.0;

  return(ret_val);
}

