/* server.c - networking routines for CodeWar server
 *
 * Rhett D. Jacobs <rdj@cea.com.au>
 * GNU Public Licence, 1996.
 *
 * Last Modified: <rdj>
 */

#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
        
#include "server.h"
#include "weapon.h"
#include "messages.h"
#include "graphics.h"
#include "srv_rqs.h"
#include "config.h"
#include "simulate.h"
#include "misc.h"

extern Player *play_list;
extern Config config;
extern msg_statistics msg_stats;
extern Resource_Defines rd;

int server_socket;
fd_set client;
timer_val screen_update;
timer_val simulate_timer;

/* cw_server_init - sets up CodeWar main server at TCP port specified in 
 * config.h.
 */
int cw_server_init(void)
{
  int length;
  int one=1;
  struct sockaddr_in server;

  /* create socket */
  if ((server_socket = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
    perror("server - socket");
    return(FALSE);
  }
  
  /* name socket using wildcards */
  server.sin_family = AF_INET;
  server.sin_addr.s_addr = INADDR_ANY;
  server.sin_port = htons(RENDEZVOUS);

  setsockopt(server_socket, SOL_SOCKET, SO_REUSEADDR, (char *)&one,
       sizeof(one));
  
  if (bind(server_socket, (struct sockaddr *)&server, sizeof(server)) < 0) {
    perror("server - bind");
    return(FALSE);
  }
  
  /* find assigned port number and echo it to stderr */
  length = sizeof server;
  if (getsockname(server_socket, (struct sockaddr *)&server, &length) < 0) {
    perror("server - getsockname");
    return(FALSE);
  }
  
  listen(server_socket, 5);
  cw_set_server_flags();

  return(TRUE);
}


void cw_set_server_flags(void)
{
  Player *tmp;

  FD_ZERO(&client);
  FD_SET(server_socket, &client);
  FD_SET(STDIN_FILENO, &client);
  
  if (play_list != NULL) {
    tmp = play_list;
    while (tmp != NULL) {
      FD_SET(tmp->socket, &client);
      tmp = tmp->nextplr;
    }
  }

  memset((void*)&msg_stats, 0, sizeof(msg_statistics)); 

  return;
}


int cw_server_process(void)
{
  int fd, new_sock, buf_size, exit_flag = 0, client_removed = FALSE;
  fd_set ready;
  struct timeval to;
  Player *tmp;
  unsigned char buffer[MAX_BUF_SIZE];

  to.tv_sec = SERVER_SEC;
  to.tv_usec = SERVER_USEC;

  cw_server_init_periodics();

  FD_ZERO(&ready);  

  while(!(exit_flag)) {
    memcpy((void *)&ready, (void *)&client, (size_t)sizeof(fd_set));
    fd = select(FD_SETSIZE, &ready, NULL, NULL, &to);
    
    switch(fd) {
    case -1:
      perror("server - select");
      client_removed = TRUE;
      break;
    case 0: 
      break;
    default:
      if (FD_ISSET(server_socket, &ready)) {
	switch(new_sock = 
	       accept(server_socket, (struct sockaddr *)0, (int *)0)) {
	case -1:
	  perror("server - accept");
	  client_removed = TRUE;
	  break;
	case 0:
	  break;
	default:
	  if (new_sock) {
	    FD_SET(new_sock, &client);
	    add_player("Unknown", new_sock);
	    graphics_refresh_plr();
	  }
	  break;
	}
      }
      if (play_list != NULL) {
	tmp = play_list;
	while (tmp != NULL) {
	  if (FD_ISSET(tmp->socket, &ready)) {
	    buf_size = read(tmp->socket, buffer, MAX_BUF_SIZE);
	    switch(buf_size) {
	    case -1:
	      perror("server - read");
	      remove_player(tmp->socket, &play_list);
	      client_removed = TRUE;
	      break;
	    case 0:
	      break;
	    default:
	      append_port_buff(tmp->port_buffer, &tmp->port_buffer_size,
			       buffer, buf_size);
	      break;
	    }
	  }
	  tmp = tmp->nextplr;
	}
      }
    }

    if (client_removed) {
      cw_set_server_flags();
      client_removed = FALSE;
    }

    cw_server_periodics();
  }
  
  return(TRUE);
}


void cw_signal_handler(int sig)
{
  static int too_slow = 0;
  static int count = 0, last_count = 0;
  static timer_val refresh_timer = {0,0};
  struct itimerval timer;

  memset((void *)&timer, 0, sizeof(struct itimerval));
  timer.it_value.tv_usec = config.simulate_int;
  signal(SIGALRM, cw_signal_handler);
  setitimer(ITIMER_REAL, &timer, 0);

  count++;

  if (check_timer(&refresh_timer)) {
    set_timer(&refresh_timer, (long)1000000);
    last_count = count;
    count = 0;
/*    fprintf(stderr,"Updates: %d %f\n ", last_count, 1.0/(float)last_count);*/

    if ((float)last_count < 1.0/(float)(config.time_int))
      config.server_slow = TRUE;
    else
      config.server_slow = FALSE;
  }

  if (too_slow)
    fprintf(stderr,"aggghh..too slow\n");

  too_slow = 1;
  service_msg_queues();
  simulate();
  combat();
  too_slow = 0;
}


void cw_server_periodics(void)
{
  static int count = 0, last_count = 0;
  static timer_val refresh_timer = {0,0};

  if (check_timer(&simulate_timer)) {
    set_timer(&simulate_timer, config.simulate_int);
    service_msg_queues();
    simulate();
    combat();
    
    count++;
    
    if (check_timer(&refresh_timer)) {
      set_timer(&refresh_timer, (long)1000000);
      last_count = count;
      count = 0;
      /*      fprintf(stderr,"Updates: %d %f\n ", last_count, 1.0/(float)last_count);*/
      
      if (last_count < (int)(1.0/(float)(config.time_int)+0.5))
	config.server_slow = TRUE;
      else
	config.server_slow = FALSE;
    }
  }

  if (check_timer(&screen_update)) {
    set_timer(&screen_update, config.screen_update);
    graphics_draw_player_positions();
    graphics_refresh();
  }
}


void cw_server_init_periodics(void)
{
  set_timer(&simulate_timer, config.simulate_int);
  set_timer(&screen_update, config.screen_update);
}


void cw_server_close(int sig)
{
  Player *plr;

  graphics_close_display();

  close(server_socket);
  if (play_list != NULL) {
    plr = play_list;
    while (plr != NULL) {
      remove_player(plr->socket, &play_list);
      graphics_refresh_plr();
      plr = plr->nextplr;
    }
  }
  printf("\n");
  exit(1);

  return;
}


void service_msg_queues(void)
{
  Player *tmp;
  
  if (play_list != NULL) {
    tmp = play_list;
    while (tmp != NULL) {
      if (tmp->port_buffer_size) {
	process_message_buffer(tmp->socket, tmp->port_buffer,
			       &tmp->port_buffer_size);
	srv_Send_Acknowledge(tmp->socket);
      }
      tmp = tmp->nextplr;
    }
  }
  
  return;
}


void append_port_buff(unsigned char *port_buff, int *port_buff_size,
		      unsigned char *append_buff, int append_buff_size)
{
  if (*port_buff_size + append_buff_size < MAX_PORT_BUFF) {
    memcpy((void *)port_buff, (void *)append_buff, append_buff_size);
    (*port_buff_size) += append_buff_size;

    if (*port_buff_size > msg_stats.max_port_size)
      msg_stats.max_port_size = *port_buff_size;
    
  } else 
    msg_stats.port_errors++;

  return;
}
