/* messages.c - routines to handle TCP/IP message parsing/passing.
 *
 * Rhett D. Jacobs <rdj@cea.com.au>
 * GNU Public Licence.
 *
 * Last Modified: 3/12/95 <rdj>
 */

#include "messages.h"
#include "srv_rqs.h"
#include "graphics.h"
#include "srv_ops.h"
#include "clt_ops.h"
#include "cw.h"
#include "cw_client.h"
#include "cw_server.h"

extern Player *play_list;
msg_statistics msg_stats;


/* shift_down - slides the contents of an array down by the number of
 * characters specified.
 */
int shift_down(unsigned char *destination, int num_characters, 
	       int total_length)
{
  int new_length;
  unsigned char temp_buffer[MAX_BUF_SIZE+1024];
  
  new_length = total_length - num_characters;
  
  /* need to do this as a 2 step process as memcpy() has undefined
   * behaviour when memory regions overlap.
   */
  memcpy((void *)&temp_buffer, (void *)destination, total_length);
  memcpy((void *)destination, (void *)&temp_buffer[num_characters], 
	 new_length);

  return(new_length);
}


/* extract_message - removes from the buffer the message type specified.
 * This message MUST be at the beginning of the buffer (i.e. its first
 * character needs to be buffer[0]).
 */
int extract_message(unsigned char *buffer, int *buf_size,
		    unsigned char *dest_buffer, int extract_size)
{
  memcpy((void *)dest_buffer, (void *)buffer, extract_size);
  *buf_size = shift_down(buffer, extract_size, *buf_size);

  return(*buf_size);
}


int send_message(int sock, unsigned char *data_type, int data_size)
{
  int ret_val = 0;
  
  if (write(sock, data_type, data_size))
    ret_val = 1;
  
  return(ret_val);
}


int get_message_size(int current_message)
{
  int ret_val = 0;
 
  switch(current_message) {
  case MT_REGISTER_PROGRAM:
    ret_val = sizeof(Mess_Register_Program); break;
  case MT_PRINT_BUFFER:
    ret_val = sizeof(Mess_Print_Buffer); break;
  case MT_PLAYER_STATUS:
    ret_val = sizeof(Mess_Player_Status); break;
  case MT_CONFIG_STATUS:
    ret_val = sizeof(Mess_Config_Status); break;
  case MT_CHANNEL_STATUS:
    ret_val = sizeof(Mess_Channel_Status); break;
  case MT_POWER:
    ret_val = sizeof(Mess_Power); break;
  case MT_SCAN:
    ret_val = sizeof(Mess_Scan); break;
  case MT_MISSILE:
    ret_val = sizeof(Mess_Missile); break;
  case MT_CHARGE:
    ret_val = sizeof(Mess_Charge); break;
  case MT_ATOMIC:
    ret_val = sizeof(Mess_Atomic); break;
  case MT_CANNON:
    ret_val = sizeof(Mess_Cannon); break;
  case MT_BOMB:
    ret_val = sizeof(Mess_Bomb); break;
  case MT_ACKNOWLEDGE:
    ret_val = sizeof(Mess_Acknowledge); break;
  case MT_BOOST_SHIELDS:
    ret_val = sizeof(Mess_Boost_Shields); break;
  case MT_TX_CHANNEL:
    ret_val = sizeof(Mess_Tx_Channel); break;

  case MT_GET_STATUS:
    ret_val = sizeof(Mess_Get_Status); break;
  case MT_GET_CONFIG:
    ret_val = sizeof(Mess_Get_Config); break;
  case MT_GET_CHANNEL:
    ret_val = sizeof(Mess_Get_Channel); break;
  }
  
  return(ret_val);
}


void process_request(int current_message, void *extracted_message, int sock)
{
  switch(current_message) {
  case MT_REGISTER_PROGRAM:
    srv_Register_Program(extracted_message, sock); break;
  case MT_PRINT_BUFFER:
    srv_Print_Buffer(extracted_message, sock); break;
  case MT_PLAYER_STATUS:
    clt_Player_Status(extracted_message, sock); break;
  case MT_CONFIG_STATUS:
    clt_Config_Status(extracted_message, sock); break;
  case MT_CHANNEL_STATUS:
    clt_Channel_Status(extracted_message, sock); break;
  case MT_POWER:
    srv_Power(extracted_message, sock); break;
  case MT_SCAN:
    srv_Scan(extracted_message, sock); break;
  case MT_MISSILE:
    srv_Missile(extracted_message, sock); break;
  case MT_CHARGE:
    srv_Charge(extracted_message, sock); break;
  case MT_ATOMIC:
    srv_Atomic(extracted_message, sock); break;
  case MT_CANNON:
    srv_Cannon(extracted_message, sock); break;
  case MT_BOMB:
    srv_Bomb(extracted_message, sock); break;
  case MT_BOOST_SHIELDS:
    srv_Boost_Shields(extracted_message, sock); break;
  case MT_TX_CHANNEL:
    srv_Tx_Channel(extracted_message, sock); break;

  case MT_ACKNOWLEDGE:
    clt_Acknowledge(); break;
  case MT_GET_STATUS:
    srv_Get_Status(extracted_message, sock); break;
  case MT_GET_CONFIG:
    srv_Get_Config(extracted_message, sock); break;
  case MT_GET_CHANNEL:
    srv_Get_Channel(extracted_message, sock); break;
  }
  
  return;
}


int process_message(int sock, unsigned char *buffer, int buf_size)
{
  static short int current_message = 0;
  static int local_buffer_length = 0;
  int required_length, exit_flag = FALSE, ret_val = TRUE;
  static unsigned char local_buffer[MAX_BUF_SIZE+1024];
  unsigned char extracted_message[MAX_BUF_SIZE];

  memcpy((void *)&local_buffer[local_buffer_length], (void *)buffer,
	 buf_size);
  local_buffer_length += buf_size;
  
  while (!(exit_flag)) {
    if (!(current_message))
      memcpy((void *)&current_message, (void *)&local_buffer[0],
	     sizeof(short int));

    if ((required_length = get_message_size(current_message))) {
      if (required_length <= local_buffer_length) {
	if (!(extract_message(local_buffer, &local_buffer_length,
			      extracted_message, required_length)))
	  exit_flag = TRUE;
	
	process_request(current_message, extracted_message, sock);
	current_message = 0;
      } else
	exit_flag = TRUE;
    } else {
      graphics_printxy("*** client unknown message ***",1,10);
      error_print("unknown message type",0,0);
      current_message = local_buffer_length = 0;
      ret_val = FALSE;
      exit_flag = TRUE;
    }
  }

  return(ret_val);
}


int process_message_buffer(int sock, unsigned char *buffer, int *buf_size)
{
  short int current_message = 0;
  int required_length, ret_val = TRUE;
  unsigned char extracted_message[MAX_BUF_SIZE];

  memcpy((void *)&current_message, (void *)buffer, sizeof(short int));

  if ((required_length = get_message_size(current_message))) {
    if (required_length <= *buf_size) {
      extract_message(buffer, buf_size, extracted_message,
		      required_length);
      process_request(current_message, extracted_message, sock);
    }
  } else {
    graphics_printxy("*** unknown message type ***",1,10);
    error_print("unknown message type.",0,0);
    msg_stats.msg_errors++;
    *buf_size = 0;
    ret_val = FALSE;
  }

  return(ret_val);
}





