/* main.c - silverune v0.4
 *
 * Rhett D. Jacobs <rdj@cea.com.au>
 * GNU Public Licence, 1996.
 *
 * Silverune is fairly simple X-Craft as a "proof of concept" for the CodeWar
 * system.  I am still yet to fully document it, which may be part of the
 * CodeWar manual as a guide to X-Craft creation.
 *
 * Essentially, silverune simply scans Xtrapolate until it finds a target.
 * At this point, it attempts to lock onto the target and continually 
 * fire missiles into it until it has been destroyed.  Once this is 
 * complete, it continues searching and the process continues.
 *
 * Silverune also performs the housekeeping function of ensuring that it
 * always has its shields up to full if it detects damage has occurred.
 * At this point it also attempts evasive action (as it is probably begin
 * attacked) for a few moments, before attempts to stop itself and begin
 * rescanning.  There are still bugs in this section of the X-Craft.
 *
 * It should also be noted that I've implemented a fairly simple priority
 * scheduler to perform these tasks.  This tends to work well for tasks
 * such as those required by an X-Craft, as actions such as determing
 * shield status are not required as often as scans of certain areas of the
 * battlefield.
 *
 * Last Modified: <rdj>
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "codewar.h"
#include "main.h"
#include "schedule.h"


void(*func_ptr)(void) = NULL;
char output_buffer[255];


/* scan_field - simple target strategy of performing a circular scan of
 * the entire playfield and shooting at any tagets found.  To speed up the
 * operation a fixed interval of scan is given.
 */
int scan_field(void)
{
  int ret_val = TRUE;
  float distance;
  static float range = 5.0;
  static float direction = 0.0;
  static int iq = 0;

  if ((distance = cw_scan(direction, range)) > 0.0) {
    iq = TRUE;
    cw_missile(100.0, direction, (float)(sqrt((double)(2.0*distance)))/10.0);
  }

  if (iq) {
    direction -= range;

    if (direction < 0.0)
      direction += 360.0;
    
    iq = FALSE;
  } else {
    direction += range;
    
    if (direction > 360.0)
      direction -= 360.0;
  }

  return(ret_val);
}


/* check_shields - this function is called periodically to ensure that the
 * X-Craft's shields are in good condition.  When the X-Craft no longer 
 * has any energy reserves, a FALSE return value is given so that the 
 * function no longer wastes time checking.  The energy variable is 
 * not retrieved from the CodeWar server each time as this is a waste of
 * "bandwidth" when it can be more quickly keep track of by the X-Craft.
 */
int check_shields(void)
{
  int shields, damage, ret_val = TRUE;

  static int first = TRUE;
  static int energy;
  static int prev_shields;
  static int prev_damage;
  static float direction = -1.0;

  if (first) {
    first = FALSE;
    energy = cw_get_energy();
    prev_shields = cw_get_shields();
    prev_damage = cw_get_damage();
  }

  if (direction != -1.0) {
    cw_power(100.0, direction+180.0);
    cw_print_buffer("Stop");
    direction = -1.0;
    
    shields = cw_get_shields();
    
    if (shields < 100) {
      cw_boost_shields(100-shields);
      energy -= (100-shields);
    }
  } else {
    cw_power(0.0, 0.0);
    cw_print_buffer("Done");
    
    damage = cw_get_damage();
    
    shields = cw_get_shields();
    
    if (shields < 100) {
      cw_boost_shields(100-shields);
      energy -= (100-shields);
    }
    
    if (prev_shields != shields ||
	prev_damage != damage) {
      direction = (float)(rand()%360);
      cw_power(100.0, direction);
      cw_print_buffer("Start");
      
      prev_shields = shields;
      prev_damage = damage;
    }
  }
  
  return(ret_val);
}
  

/* service_function_list - to facilitate a range of functions being 
 * periodically called, the func_ptr is set to point to this function
 * which in turn calls the functions registered in its f_list[].  These
 * functions are called as priority scheduled using routines from 
 * schedule.c.
 */
void service_function_list(void)
{
  static f_type f_list[] = {{10, 10, scan_field}, {1, 1, check_shields}};
  static f_size = sizeof(f_list) / sizeof(f_type);
  
  schedule_function_list(f_list, f_size);
}


/* init - simple setup routine.  Assigns a periodic handler pointer for
 * use by the utils.c functions, and registers the X-Craft with the 
 * CodeWar server.
 */
int init(int argc, char **argv)
{
  char *hostname = (char *)NULL;
  int ret_val = FALSE;

  func_ptr = service_function_list;
  sprintf(output_buffer,"Silverune v%0.1f", VERSION);

  if (argc > 1)
    hostname = argv[1];

  if (cw_register_network_program(output_buffer, hostname))
    ret_val = TRUE;

  return(ret_val);
}


void main(int argc, char **argv)
{
  if (init(argc, argv)) {
    for (;;) {
      func_ptr();
    }
  }
}
