extern "C" {
#include "tcpip.h"
#include <curses.h>
#include <sys/file.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/time.h>
#include <fcntl.h>
#include <netinet/in.h>

#ifndef SYS5
#define gettimeofday(tv) ( gettimeofday((tv), 0) )
#endif
}

#include <string.h>
#include "common.h"
#include "rcvbytes.h"
#include "wui.h"

CWormUI::CWormUI()
: nUserInputDelay(1) {
}

void CWormUI::InitKeyMap()
{
	unsigned char uc = 255;
	do { m_szCharMap[uc] = uc; } while (uc--);

	m_pcInputBuffer = m_szInputBuffer;
}

void CWormUI::Remap(unsigned char ucOldChar, unsigned char ucNewChar) 
{
	m_szCharMap[ucNewChar] = ucOldChar;
}

unsigned char CWormUI::TtyToProtocolKey(int nTerminalInput) 
{
	switch (nTerminalInput) {
		case KEY_LEFT: return 'h';
		case KEY_DOWN: return 'j';
		case KEY_UP: return 'k';
		case KEY_RIGHT: return 'l';
	}
	return m_szCharMap[(unsigned char)(nTerminalInput & 0xff)];
}

//*****************************************************************************
//*****************************************************************************

void CWormUI::ReadInput()
{
	for (;;) {
		int nRead = getch();
		if(nRead == ERR) return;

		char c = TtyToProtocolKey(nRead);
		if (c == 0x0c) { // ^L
			wrefresh(curscr);
		}
		else {
			*(m_pcInputBuffer++) = c;
			return;
		}
		
	}
}

int CWormUI::PrintUsage()
{
	printf(
		"Usage: boa <server>[:<port>] [--<digit>] [oldchar=newchar ...]\n\n"

		"\t<digit> is 1 by default, it gives the # of tenths of a second\n"
		"\t\tyou are given to enter the input after the screen update\n"

		"\tIf no remapping of the form oldchar=newchar is given, the "
			"following\n"
		"\tcontrols are in effect during the game (like in vi(1)):\n"
		"\t\tLeft -- cursor left, or 'h'\n"
		"\t\tDown -- cursor down, or 'j'\n"
		"\t\tUp -- cursor up, or 'k'\n"
		"\t\tRight -- cursor right, or 'l'\n"
		"\t\tQuit -- 'q'\n\n"

		"\tOldchar is any of [hjklq], newchar is any single char\n"
		"\tExample:\n\n"
		"\t\t\tboa wormserver h=o j=a k=q l=p q=Q\n\n"
		"\t\tenables the following controls:\n\n"
		"\t\t\tLeft -- 'o'\n"
		"\t\t\tDown -- 'a'\n"
		"\t\t\tUp -- 'q'\n"
		"\t\t\tRight -- 'p'\n"
		"\t\t\tQuit -- 'Q'\n"
	);
	return 1;
}

void CWormUI::Run(int argc, char** argv)
{
	if(argc < 2)
	{
		PrintUsage();
		return;
	}

	InitKeyMap();

	int nPort = DEFAULT_PORT;
	ParseArg(argv[1], nPort);

	for (int n = 2; n < argc; n++) 
	{
		const signed char* c_szOption = (signed char*) argv[n];
		if (
			(c_szOption[1] != '=' 
				&& (c_szOption[0] != '-' || c_szOption[1] != '-')
			) || c_szOption[3])
		{
			PrintUsage();
			return;
		}
		if (c_szOption[1] == '=')
			Remap(c_szOption[0], c_szOption[2]);
		else {
			int nSuggestedDelay = c_szOption[2] - '0';
			if (nSuggestedDelay > 1 && nSuggestedDelay < 10)
				nUserInputDelay = nSuggestedDelay;
		}
	}

	CWormClient::Run(argv[1], nPort);
}

bool CWormUI::Initialize()
{
	// Create and set up the main window
	// =================================

	p_winMain = initscr();
	noecho();
	cbreak();
	clear();
	nodelay(p_winMain, TRUE);
	typeahead(-1);

	leaveok(p_winMain, TRUE);
	keypad(p_winMain, TRUE);
	if (curs_set(0) == ERR) {
		// hack around, try VT220 cursor disable seq.
		printf ("\x1b[?25l");
	}
	refresh();

	return true;
}

bool CWormUI::Uninitialize()
{
	endwin();
	printf ("\x1b[?25h");
	return true;
}
	
bool CWormUI::OnRefresh()
{
	refresh();
	timerInputDelay.CheckPoint();
	return true;
}
	
bool CWormUI::OnDeath(int nPlayer)
{
	beep();
	return true;
}

bool CWormUI::OnScore(int nPlayer, long lScore)
{
	mvprintw(0, 10*nPlayer+1, "%c:%-8ld", 'A'+nPlayer, lScore);
	return true;
}

bool CWormUI::OnDisconnect(int nPlayer)
{
	mvprintw(0, 10*nPlayer+2, "!");
	mvprintw(23, 10*nPlayer+1, "%%%%%%%%%%%%%%%%%%%%");
	return (nPlayer != m_nMyId);
}	

bool CWormUI::OnLives(int nPlayer, long lLives)
{
	mvprintw(23, 10*nPlayer + 1, "%c:%-8ld", 'A'+nPlayer, lLives);
	return true;
}

bool CWormUI::OnStoppedDueTo(int nAppearedPlayer)
{ 
	mvprintw(23, 0, "%c", nAppearedPlayer + 'a');
	DiscardTypedAhead();
	return true; 
}

unsigned char CWormUI::GetInput()
{
	if (!IsInputPending()) {
		long lTimeSinceUpdatedScreen = timerInputDelay.Elapsed();
 		if (lTimeSinceUpdatedScreen < 200) {
			halfdelay(nUserInputDelay); // .1 sec
			ReadInput();
			nodelay(p_winMain, TRUE);
 		}
 		else
 			ReadInput();
	}
	if(!IsInputPending()) 
	{
		return 0;
	}
	else
	{
		mvprintw(23, 0, "%%");
		char ch;
		if ((ch = m_szInputBuffer[0]) == 'q') return -1;

		memcpy(m_szInputBuffer, m_szInputBuffer+1, 
				(m_pcInputBuffer-m_szInputBuffer)-1);
		m_pcInputBuffer--;
		return ch;
	}
}

bool CWormUI::OnCellEmpty(int nX, int nY)
{
	Draw(nY, nX, ' ');
	return true;
}

bool CWormUI::OnCellPlayer(int nX, int nY, int nPlayer)
{
	Draw(nY, nX, 'A' + nPlayer);			
	return true;
}
bool CWormUI::OnCellPrize(int nX, int nY, int nValue)
{
	Draw(nY, nX, '0' + nValue);			
	return true;
}

bool CWormUI::OnCellWall(int nX, int nY)
{
	Draw(nY, nX, '%');
	return true;
}

void CWormUI::Draw(int nY, int nX, char ch)
{
	mvaddch(nY, nX, ch);
}


int main(int argc, char** argv)
{
	CWormUI WormUI;

	WormUI.Run(argc, argv);
	return 1;
}
