extern "C" {
#include "tcpip.h"
#include <sys/file.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/time.h>
#include <fcntl.h>

#include <string.h>
#include <stdio.h>
#include <stdlib.h>

#ifndef SYS5
#include <unistd.h>
#define gettimeofday(tv) ( gettimeofday((tv), 0) )
#endif
}

#include "common.h"
#include "rcvbytes.h"
#include "boabot.h"

#define LMASK_ALL 0x000000C0
#define PROPERTY_OFFSET 20000

void CWormBot::CToVisitQueue::AddPlace(int nX, int nY)
{
	//printf("Adding\n");
	m_nLast++;
	if(m_nLast == MAX_QUEUE) m_nLast = 0;
	m_aToVisit[m_nLast].nX = nX;
	m_aToVisit[m_nLast].nY = nY;
	m_bEmpty = false;
}

void CWormBot::CToVisitQueue::AddAllNeighbors(int j, int i)
{
	AddPlace(j-1, i);
	AddPlace(j+1, i);
	AddPlace(j, i-1);
	AddPlace(j, i+1);
}

bool CWormBot::CToVisitQueue::NextPlace(int& nX, int& nY)
{
	// printf("Get %d\n", m_bEmpty);
	if(m_bEmpty) return false;
	nX = m_aToVisit[m_nFirst].nX;
	nY = m_aToVisit[m_nFirst].nY;
	// printf("Res: %d,%d\n", nX, nY);

	m_nFirst++;
	if(m_nFirst == MAX_QUEUE) m_nFirst = 0;

	if(m_nFirst == m_nLast) 
	{
		m_bEmpty = true;
		m_nFirst = 1; m_nLast = 0;
	}
	return true;
}


//***************************************************************************
void CWormBot::CopyField(FIELDTYPE& field1, FIELDTYPE& field2)
{
	for(int i = 0; i < FIELD_HEIGHT; i++)
	{
		memcpy(field2[i], field1[i], FIELD_WIDTH * (sizeof(int)));
	}
}

int CWormBot::PrintUsage()
{
	printf(
		"Usage: boabot <server>[:<port>]\n\n"
		"Must be run before the other clients connect.\n"
		"DO NOT FORGET TO KILL YOUR BOABOT AFTER YOU LEAVE!!!\n"
	);
	return 1;
}

bool CWormBot::IsOpposite(char ch1, char ch2)
{
	if(ch1 == 'h' && ch1 == 'l') return true;
	if(ch1 == 'l' && ch1 == 'h') return true;
	if(ch1 == 'j' && ch1 == 'k') return true;
	if(ch1 == 'k' && ch1 == 'j') return true;
	return false;
}

#define MAXDIST 10000
#define CELLDIST(FIELD, X,Y) (FIELD[Y][X] >> 8)
#define ISEMPTY(FIELD, X, Y) ((FIELD[Y][X] & LMASK_ALL) == MASK_EMPTY)
#define SETCELLDIST(FIELD, X, Y, VAL) \
	FIELD[Y][X] = (FIELD[Y][X] & 0x000000FF) | ((VAL) << 8);

#define TESTMIN(VAR, Y, X) \
	if(CELLDIST(field, X, Y) < VAR) VAR = CELLDIST(field, X, Y);



void CWormBot::PrintTrace(FIELDTYPE& field)
{
		for(int i = 0; i < FIELD_HEIGHT; i++)
		{
			for(int j = 0; j < FIELD_WIDTH; j++)
			{
				if((field[i][j] & LMASK_ALL) == MASK_EMPTY)
				{
					if(field[i][j] >> 8 == MAXDIST) printf(" ");
					else printf("%d", (field[i][j] >> 8)%10);
				}
				else printf(".");
			}
			printf("\n");
		}
}

void CWormBot::PrintProperty(FIELDTYPE& field)
{
		for(int i = 0; i < FIELD_HEIGHT; i++)
		{
			for(int j = 0; j < FIELD_WIDTH; j++)
			{
				if((field[i][j] & LMASK_ALL) == MASK_EMPTY)
				{
					if((field[i][j] >> 8) > PROPERTY_OFFSET) printf("+");
					else printf("-");
				}
				else printf(".");
			}
			printf("\n");
		}
}

void CWormBot::ClearDist(FIELDTYPE& field)
{
	int i, j;

	for(i = 0; i < FIELD_HEIGHT; i++)
		for(j = 0; j < FIELD_WIDTH; j++)
			SETCELLDIST(field, j, i, MAXDIST);
}

void CWormBot::TraceDist(FIELDTYPE& field, int nToX, int nToY, bool bPrecise)
{
	// Clear the array
	// ===============

	ClearDist(field);

	m_Queue.Reset();
	m_Queue.AddAllNeighbors(nToX, nToY);

	// Clear the endpoint temporarily
	// ==============================

	SETCELLDIST(field, nToX, nToY, 1);

	// Trace it
	// ========

	TraceDistSet(field, bPrecise);
}

void CWormBot::TraceDistSet(FIELDTYPE& field, bool bPrecise)
{
	int i, j;
	while(m_Queue.NextPlace(j, i))
	{
		// printf("Do %d,%d\n", j, i);
		if(ISEMPTY(field, j, i))
		{
			if(!bPrecise && CELLDIST(field, j, i) < MAXDIST) continue;

			// printf("Ex: %d,%d: ", j, i);
			int nMinD = CELLDIST(field, j, i)-1;
			int nSaveD = nMinD;
			TESTMIN(nMinD, i-1, j);
			TESTMIN(nMinD, i+1, j);
			TESTMIN(nMinD, i, j-1);
			TESTMIN(nMinD, i, j+1);

			if(nMinD < nSaveD) 
			{
				// printf("(%d,%d) %d -> %d\n", j, i, nSaveD, nMinD);
				m_Queue.AddAllNeighbors(j, i);
				SETCELLDIST(field, j, i, (nMinD+1));
				if(i == 0 || j == 0) {PrintTrace(field); getchar();}
			}
		}
	}

/*
	printf("Traced\n");

	PrintTrace(field);
	printf("Done\n");
	getchar();
*/
}

void CWormBot::MapAllWorms(FIELDTYPE& field)
{
	CopyField(m_field, field);
	ClearDist(field);
	m_Queue.Reset();

	for(int i = 0; i < 100; i++)
		if(m_aWorms[i].nX != -1)
		{
			m_Queue.AddAllNeighbors(m_aWorms[i].nX, m_aWorms[i].nY);
			SETCELLDIST(field, m_aWorms[i].nX, m_aWorms[i].nY, 1);
		}

	TraceDistSet(field, true);
}


void CWormBot::MakeIntoProperty(FIELDTYPE& fieldW)
{
	// Compare this map to the others in place
	// =======================================

	for(int i = 0; i < FIELD_HEIGHT; i++)
		for(int j = 0; j < FIELD_WIDTH; j++)
			if(ISEMPTY(m_field, j, i))
			{
				int nNewVal = CELLDIST(fieldW, j, i) + 
								PROPERTY_OFFSET - CELLDIST(m_field, j, i);
				SETCELLDIST(m_field, j, i, nNewVal);
			}
	// PrintProperty(m_field); getchar();
}

void CWormBot::AddChange(int nX, int nY, int nValue)
{
	m_aChanged[m_nChanged].nX = nX;
	m_aChanged[m_nChanged].nY = nY;
	m_aChanged[m_nChanged].nVal = m_field[nY][nX];
	m_field[nY][nX] = nValue;
	m_nChanged++;
}
	
void CWormBot::CheckHead(int nX, int nY)
{
	if((m_field[nY][nX] & LMASK_ALL) == MASK_EMPTY)
	{
		for(int i = 0; i < 100; i++)
		{
			if(m_aWorms[i].nX != -1)
			{
				if(
					(m_aWorms[i].nX == nX && abs(m_aWorms[i].nY-nY) <= 1) ||
					(m_aWorms[i].nY == nY && abs(m_aWorms[i].nX-nX) <= 1)
				)
				{
					printf("Protect head at %d,%d\n", nX, nY);
					AddChange(nX, nY, MASK_OTHER);
					break;
				}
			}
		}
	}
}

void CWormBot::UndoChanges()
{
	for(int i = 0; i < m_nChanged; i++)
	{
		m_field[m_aChanged[i].nY][m_aChanged[i].nX] = m_aChanged[i].nVal;
	}
}

void CWormBot::Pace()
{
	// printf("No way\n");
	if(m_chNextMove != 0)
	{
		m_chMove = m_chNextMove;
		m_chNextMove = 0;
	}
	else if((m_field[m_nY+m_nDY][m_nX+m_nDX] & LMASK_ALL) == MASK_EMPTY)
	{
		// Continue
		// ========
	}
	else if((m_field[m_nY+1][m_nX] & LMASK_ALL) == MASK_EMPTY)
	{
		m_chMove = 'j';
		if(m_chPrevMove == 'l') m_chNextMove = 'h';
		else m_chNextMove = 'l';
	}
	else if((m_field[m_nY-1][m_nX] & LMASK_ALL) == MASK_EMPTY)
	{
		m_chMove = 'k';
		if(m_chPrevMove == 'l') m_chNextMove = 'h';
		else m_chNextMove = 'l';
	}
	else if((m_field[m_nY][m_nX+1] & LMASK_ALL) == MASK_EMPTY)
	{
		m_chMove = 'l';
		if(m_chPrevMove == 'k') m_chNextMove = 'j';
		else m_chNextMove = 'k';
	}
	else if((m_field[m_nY][m_nX-1] & LMASK_ALL) == MASK_EMPTY)
	{
		m_chMove = 'h';
		if(m_chPrevMove == 'k') m_chNextMove = 'j';
		else m_chNextMove = 'k';
	}
}

bool CWormBot::AssertMove(char ch)
{
	switch(ch)
	{
		case 'h': m_nDX = -1; m_nDY = 0; break;
		case 'l': m_nDX = 1; m_nDY = 0; break;
		case 'j': m_nDX = 0; m_nDY = 1; break;
		case 'k': m_nDX = 0; m_nDY = -1; break;
		default: return false;
	}
	return true;
}

int CWormBot::CountFreeNeighbors(int nX, int nY)
{
	int nSum = 0;
	if(ISEMPTY(m_field, nX+1, nY)) nSum++;
	if(ISEMPTY(m_field, nX-1, nY)) nSum++;
	if(ISEMPTY(m_field, nX, nY-1)) nSum++;
	if(ISEMPTY(m_field, nX, nY+1)) nSum++;
	return nSum;
}

void CWormBot::PickAnyMove()
{
	//printf("Just one choice at all\n");
	if(ISEMPTY(m_field, m_nX+1, m_nY)) m_chMove = 'l';
	if(ISEMPTY(m_field, m_nX-1, m_nY)) m_chMove = 'h';
	if(ISEMPTY(m_field, m_nX, m_nY+1)) m_chMove = 'j';
	if(ISEMPTY(m_field, m_nX, m_nY-1)) m_chMove = 'k';
}
	
int CWormBot::PickClosestMove()
{
	int nMin = MAXDIST;

	if(ISEMPTY(m_field, m_nX+1, m_nY) &&
		CELLDIST(m_field, m_nX+1, m_nY) < nMin)
	{
		nMin = CELLDIST(m_field, m_nX+1, m_nY);
		m_chMove = 'l';
	}
	if(ISEMPTY(m_field, m_nX-1, m_nY) &&
		CELLDIST(m_field, m_nX-1, m_nY) < nMin)
	{
		nMin = CELLDIST(m_field, m_nX-1, m_nY);
		m_chMove = 'h';
	}
	if(ISEMPTY(m_field, m_nX, m_nY-1) &&
		CELLDIST(m_field, m_nX, m_nY-1) < nMin)
	{
		nMin = CELLDIST(m_field, m_nX, m_nY-1);
		m_chMove = 'k';
	}
	if(ISEMPTY(m_field, m_nX, m_nY+1) &&
		CELLDIST(m_field, m_nX, m_nY+1) < nMin)
	{
		nMin = CELLDIST(m_field, m_nX, m_nY+1);
		m_chMove = 'j';
	}

/*
	// Check if the continuation will do
	// =================================

	int nDX, nDY;
	GetMoveShifts(m_chPrevMove, nDX, nDY);
	if(CELLDIST(m_field, m_nX+nDX, m_nY+nDY) == nMin)
	{
		m_chMove = m_chPrevMove;
	}
*/

	if(nMin == MAXDIST) PickAnyMove();
		
	return nMin;
}

int CWormBot::ComputeTerritory(FIELDTYPE& field)
{
	int nMine = 0;
	for(int i = 0; i < FIELD_HEIGHT; i++)
		for(int j = 0; j < FIELD_WIDTH; j++)
			if(ISEMPTY(field, j, i) && CELLDIST(field, j, i) > PROPERTY_OFFSET)
				nMine++;
	return nMine;
}

	
void CWormBot::PrepareMove()
{
	if (bStoppedWithoutRivals) {
		// Do not move
		m_chMove = 0;
	}
	else {
		int nHscore = 0, nLscore = 0, nJscore = 0, nKscore = 0;

		// Count initial free cells
		// ========================

		int nFree = CountFreeNeighbors(m_nX, m_nY);
		if(nFree == 0) return;
		// printf("Free = %d\n", nFree);

		// Mark off all those head-dangerous
		// =================================

		m_nChanged = 0;
		CheckHead(m_nX+1, m_nY);
		CheckHead(m_nX-1, m_nY);
		CheckHead(m_nX, m_nY+1);
		CheckHead(m_nX, m_nY-1);
		if(m_nChanged >= nFree-1) 
		{
			PickAnyMove();
			UndoChanges();
			return;
		}
		nFree -= m_nChanged;
		// printf("Free from head= %d\n", nFree);

		// Map reachability of tail
		// ========================

		TraceDist(m_field, m_nTailX, m_nTailY, false);
		int nInaccessible = 0;
		if(ISEMPTY(m_field, m_nX+1, m_nY) &&
			CELLDIST(m_field, m_nX+1, m_nY) == MAXDIST) nInaccessible++;
		if(ISEMPTY(m_field, m_nX-1, m_nY) &&
			CELLDIST(m_field, m_nX-1, m_nY) == MAXDIST) nInaccessible++;
		if(ISEMPTY(m_field, m_nX, m_nY+1) &&
			CELLDIST(m_field, m_nX, m_nY+1) == MAXDIST) nInaccessible++;
		if(ISEMPTY(m_field, m_nX, m_nY-1) &&
			CELLDIST(m_field, m_nX, m_nY-1) == MAXDIST) nInaccessible++;

		if(nFree == nInaccessible)
		{
			Pace();
			UndoChanges();
			return;
		}

		// Wall off tail-inaccessible moves
		// ================================

		if(CELLDIST(m_field, m_nX+1, m_nY) == MAXDIST) 
			AddChange(m_nX+1, m_nY, MASK_OTHER);
		if(CELLDIST(m_field, m_nX-1, m_nY) == MAXDIST) 
			AddChange(m_nX-1, m_nY, MASK_OTHER);
		if(CELLDIST(m_field, m_nX, m_nY+1) == MAXDIST) 
			AddChange(m_nX, m_nY+1, MASK_OTHER);
		if(CELLDIST(m_field, m_nX, m_nY-1) == MAXDIST) 
			AddChange(m_nX, m_nY-1, MASK_OTHER);
		
		if(nFree == nInaccessible+1)
		{
			//printf("Just one choice to tail\n");
			PickClosestMove();
			UndoChanges();
			return;
		}
		nFree -= nInaccessible;
		// printf("Free to tail= %d\n", nFree);

		//=========================================================
		// If here, there are nFree tail-accessible head-safe moves

		// Map property
		// ============

		FIELDTYPE fieldW;
		MapAllWorms(fieldW);
		int nArea, nAreaH, nAreaL, nAreaJ, nAreaK;

		// Determine if there is any sense in going for it.
		// ================================================

		int nSave = m_field[m_nY][m_nX];
		m_field[m_nY][m_nX] = MASK_EMPTY;
		TraceDist(m_field, m_nPrizeX, m_nPrizeY);

		int nDist = PickClosestMove();

		TraceDist(m_field, m_nX, m_nY);
		MakeIntoProperty(fieldW);

		if(CELLDIST(m_field, m_nPrizeX, m_nPrizeY) > PROPERTY_OFFSET)
		{
			// Take the closest one
			// ====================

			//printf("Have advantage of %d\n", CELLDIST(m_field, m_nPrizeX, m_nPrizeY) - PROPERTY_OFFSET);
			m_field[m_nY][m_nX] = nSave;
			// printf("Distance to tail = %d\n", nDist);
			UndoChanges();
			return;
		}

		m_field[m_nY][m_nX] = nSave;
			
		// No sense running after it
		// =========================

		//printf("Don't go there!\n");

		// Find the move maximizing territory
		// ==================================
		
		int nMax = 0;
		if(ISEMPTY(m_field, m_nX+1, m_nY))
		{
			TraceDist(m_field, m_nX+1, m_nY);
			MakeIntoProperty(fieldW);
			nAreaL = ComputeTerritory(m_field);
			if(nAreaL > nMax)
			{
				nMax = nAreaL;
				m_chMove = 'l';
			}
		}
		if(ISEMPTY(m_field, m_nX-1, m_nY))
		{
			TraceDist(m_field, m_nX-1, m_nY);
			MakeIntoProperty(fieldW);
			nAreaL = ComputeTerritory(m_field);
			if(nAreaL > nMax)
			{
				nMax = nAreaL;
				m_chMove = 'h';
			}
		}
		if(ISEMPTY(m_field, m_nX, m_nY+1))
		{
			TraceDist(m_field, m_nX, m_nY+1);
			MakeIntoProperty(fieldW);
			nAreaL = ComputeTerritory(m_field);
			if(nAreaL > nMax)
			{
				nMax = nAreaL;
				m_chMove = 'j';
			}
		}
		if(ISEMPTY(m_field, m_nX, m_nY-1))
		{
			TraceDist(m_field, m_nX, m_nY-1);
			MakeIntoProperty(fieldW);
			nAreaL = ComputeTerritory(m_field);
			if(nAreaL > nMax)
			{
				nMax = nAreaL;
				m_chMove = 'k';
			}
		}

		UndoChanges();
	}
}
			
void CWormBot::Run(int argc, char** argv)
{
	if(argc < 2)
	{
		PrintUsage();
		return;
	}

	int nPort = DEFAULT_PORT;
	ParseArg(argv[1], nPort);

	CWormClient::Run(argv[1], nPort);
}

bool CWormBot::Initialize()
{
	// Initialize the field
	// ====================

	for(int i = 0; i < FIELD_HEIGHT; i++)
		for(int j = 0; j < FIELD_WIDTH; j++)
			m_field[i][j] = 0;

	m_chMove = m_chPrevMove = m_chNextMove = 0;

	return true;
}

bool CWormBot::Uninitialize()
{
	return true;
}
	
bool CWormBot::OnRefresh()
{
	m_Timer.CheckPoint();
	PrepareMove();
	AssertMove(m_chMove);
	//printf("Move took %d\n", m_Timer.Elapsed());
	m_Timer.CheckPoint();
	return true;
}

bool CWormBot::OnDeath(int nPlayer)
{
	printf("%c died!!\n", 'A'+nPlayer);
	if(nPlayer != m_nMyId)
	{
		m_aWorms[nPlayer].nX = -1;
		m_aWorms[nPlayer].nY = -1;
	}
	return true;
}

bool CWormBot::OnStoppedDueTo(int nPlayer)
{
	if (!m_a_bConnectedAfter[nPlayer]) {
		if (nPlayer == m_nMyId)
			bStoppedWithoutRivals = !nLaterPlayers;
		else {
			m_a_bConnectedAfter[nPlayer] = true;
			nLaterPlayers++;
			bStoppedWithoutRivals = false;
		}
	}
	return true;
}

bool CWormBot::OnScore(int nPlayer, long lScore)
{
	if(nPlayer == m_nMyId) printf("My score is now %d\n", lScore);
	return true;
}

bool CWormBot::OnDisconnect(int nPlayer)
{
	if (m_a_bConnectedAfter[nPlayer]) {
		m_a_bConnectedAfter[nPlayer] = false;
		nLaterPlayers--;
	}
	return (nPlayer != m_nMyId);
}	

bool CWormBot::OnLives(int nPlayer, long lLives)
{
	return true;
}

unsigned char CWormBot::GetInput()
{
	//printf("Had spare: %d\n", m_Timer.Elapsed());
	m_Timer.CheckPoint();
	m_chPrevMove = m_chMove;
	return m_chMove;
}

bool CWormBot::OnCellEmpty(int nX, int nY)
{
	if((m_field[nY][nX] & 0xFF)  == (MASK_PLAYER | m_nMyId))
	{
		m_nTailX = nX;
		m_nTailY = nY;
	}
	m_field[nY][nX] = MASK_EMPTY;
	return true;
}

bool CWormBot::OnCellPlayer(int nX, int nY, int nPlayer)
{
	//printf("In player %d. Me = %d\n", nPlayer, m_nMyId);
	m_field[nY][nX] = MASK_PLAYER | nPlayer;
	
	if(nPlayer == m_nMyId)
	{
		m_nX = nX; 
		m_nY = nY;
		//printf("Set my to %d,%d\n", nX, nY);
	}
	else
	{
		m_aWorms[nPlayer].nDX = nX-m_aWorms[nPlayer].nX;
		m_aWorms[nPlayer].nDY = nY-m_aWorms[nPlayer].nY;
		m_aWorms[nPlayer].nX = nX;
		m_aWorms[nPlayer].nY = nY;
	}
	return true;
}

bool CWormBot::OnCellPrize(int nX, int nY, int nValue)
{
	m_nPrizeX = nX; m_nPrizeY = nY;
	//printf("Prize is at %d,%d\n", nX, nY);
	m_field[nY][nX] = MASK_EMPTY;
	return true;
}

bool CWormBot::OnCellWall(int nX, int nY)
{
	m_field[nY][nX] = MASK_OTHER;
	return true;
}

int main(int argc, char** argv)
{
	/*
	if (fork()) return 0;
	setpgid(0, 0);
	*/
	CWormBot WormBot;

	WormBot.Run(argc, argv);
	return 1;
}
