/*

	splumber.c

	main() de Space Plumber

	(C) Angel Ortega 1997

*/

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>
#include <string.h>
#include <sys/wait.h>

#include "sp_supp.h"
#include "sp_types.h"
#include "sp_grx.h"
#include "sp_sb.h"
#include "sp_kbd.h"
#include "sp_ray.h"
#include "sp_map.h"
#include "sp_play.h"
#include "sp_maze.h"
#include "sp_param.h"

extern int errno;


/* mostrar men s/no */
int _menu=1;

/* laberinto personalizado s/no */
int _custom=0;

/* mostrar intro s/no */
int _intro=1;

/* mscara del men */
char * _menu_mask;

/* cursor */
#define CURSOR_SIZE 32*32
char cursor[32][32];

/* posiciones x,y del cursor */
int cursor_pos_x[5]={143,58,89,160,171};
int cursor_pos_y[5]={16,49,83,117,151};

int menu_opt=0;

/* buffer de texto para TextMask */
char _text_mask[80*25];

/* sonidos para la msica y el agua */
sp_sound * music;
sp_sound * water;

int _text_mask_x=-1;

int _tty_under_water=0;


/* mensaje de inicio (con sistema y versin del compilador) */

#ifdef DJGPP
#define STARTUP_MSG "Startup - gcc %s (DJGPP) [%s]"
#endif

#ifdef LINUX_SVGALIB
#define STARTUP_MSG "Startup - gcc %s (linux + svgalib) [%s]"
#endif

#ifdef UNIX_X11
 #ifdef linux
  #define STARTUP_MSG "Startup - gcc %s (linux + X11) [%s]"
 #else
  #define STARTUP_MSG "Startup - gcc %s (unix + X11) [%s]"
 #endif
#endif


/********************
	Cdigo
*********************/


void TextMask(char * screen)
/* hace una mscara con el texto en _text_mask */
{
	int n,y;
	char line[80];
	char * ptr;

	ptr=_text_mask;

	for(y=1;y<25;y++)
	{
		for(n=0;*ptr!='\n' && *ptr!='\0' && n<sizeof(line);n++,ptr++)
			line[n]=*ptr;
		line[n]='\0';

		PrintString(screen,_text_mask_x,y,
			y!=1 && _text_mask_x==-1?199:255,line);

		if(*ptr=='\0')
			break;
		else
			ptr++;
	}
}


/** menu **/

void MenuStartup(void)
/* comienzo en modo men */
{
	SuppStartup();
	GrxStartup();
	KbdStartup();
	SoundStartup();
	RayStartup();
	MapStartup();
	PlayerStartup();

	_menu_mask=(char *) a_malloc(SCREEN_X_SIZE*SCREEN_Y_SIZE);

	/* msica */
	music=LoadSound("sound/holst.wav");
	water=LoadSound("sound/water1.wav");

	SetSound(music,32,32,1);
	SetSound(water,5,5,1);

	StartSound(music);
	StartSound(water);
}


void MenuShutdown(void)
/* cierra el modo men */
{
	RayShutdown();
	SoundShutdown();
	KbdShutdown();
	SuppShutdown();
	PlayerShutdown();

	SetTextMode();
}


void MenuMask(char * screen)
/* mscara de la pantalla virtual con el men */
{
	int n,m;
	char * org,*des;
	int xcursor,ycursor;

	/* dibuja el texto */
	for(n=SCREEN_X_SIZE*SCREEN_Y_SIZE,
		org=_menu_mask,des=screen;n;n--,org++,des++)
	{
		if(*org)
			*des=*org;
	}

	/* dibuja el cursor */
	xcursor=cursor_pos_x[menu_opt];
	ycursor=cursor_pos_y[menu_opt];

	des=screen + xcursor + (ycursor * SCREEN_X_SIZE);

	for(n=0;n<32;n++)
	{
		for(m=0;m<32;m++)
		{
			if(cursor[n][m])
				*(des+m)=cursor[n][m];
		}

		des+=SCREEN_X_SIZE;
	}
}


int Tty(char * message)
/* muestra en la pantalla virtual el mensaje, como si fuese un terminal */
{
	clock_t c;
	time_t t;
	char * ptr;

	_text_mask[0]='\0';
	ptr=_text_mask;

	t=0;

	DrawMask=TextMask;

	_text_mask_x=1;

	for(;;)
	{
		c=clock();
		while(c==clock());

		if(_tty_under_water)
			DrawWaterVirtualScreen();
		else
			DrawVirtualScreen();

		KbdPoll();

		if(_user_cancel || _joy_button_B)
		{
			while(_user_cancel || _joy_button_B)
				KbdPoll();

			return(0);
		}

		if(_user_enter || _joy_button_A)
		{
			while(_user_enter || _joy_button_A)
				KbdPoll();

			return(1);
		}

		if(t==0)
		{
			if(*message=='\a' || *message=='\0')
				t=time(NULL)+2;
			else
			{
				*ptr=*message;
				message++;
				ptr++;
				*ptr='\0';
			}
		}
		else
		if(time(NULL)>t)
		{
			t=0;

			if(*message=='\0')
				break;
			else
				message++;
		}
	}

	t=time(NULL)+3;
	while(time(NULL)<t)
	{
		if(_tty_under_water)
			DrawWaterVirtualScreen();
	}

	return(1);
}


void Fork(char * addcmds)
/* se ejecuta a s mismo para el juego */
{
	int ret;
	char cmd[4096];

	for(;;)
	{
		KbdShutdown();
		SoundShutdown();

#ifdef linux
		SetTextMode();
#endif
		sprintf(cmd,"./splumber %s menu=0 level=%d ",_main_argv,_level);
		if(addcmds!=NULL)
			strcat(cmd,addcmds);

		ret=system(cmd);

		KbdStartup();
		SoundStartup();

#ifdef SPLUMBER_UNIX

#ifdef LINUX_SVGALIB
		SetGraphMode();
#endif

		if(WIFEXITED(ret))
			ret=WEXITSTATUS(ret);
#endif

		if(ret==SPLUMBER_SYSTEMERR)
		{
			/* error ejecutando */
			bang("Fork",strerror(errno));
		}
		else
		if(ret==SPLUMBER_DEAD)
		{
			/* ha muerto: volver a lanzarlo */
		}
		else
		if(ret==SPLUMBER_RESOLVED)
		{
			/* ha resuelto el nivel: seguir adelante */
			if(_level!=0)
				_level++;
		}
		else
		if(ret==SPLUMBER_WIN)
		{
			/* ha resuelto el ltimo nivel */
			_level=1;
			break;
		}
		else
		if(ret==SPLUMBER_CANCEL)
		{
			/* ha cancelado */
			break;
		}
		else
		if(ret==SPLUMBER_BANG)
		{
			/* splumber di un bang */
			bang("Fork","Fatal error in splumber.log");
		}

		if(_level==0)
			break;
	}

#ifdef DJGPP
	SetGraphMode();
#endif

	StartSound(music);
	StartSound(water);
}


int GetNumber(char * strmask, int max)
/* pide un valor numrico */
{
	char str_num[30];

	while(_user_enter)
		KbdPoll();

	DrawMask=TextMask;

	/* pide el nmero */
	str_num[0]='\0';
	for(;;)
	{
		sprintf(_text_mask,strmask,str_num);

		DrawWaterVirtualScreen();

		KbdGetNumber(str_num,max);

		if(_user_enter)
		{
			while(_user_enter)
				KbdPoll();

			break;
		}
	}

	return(atoi(str_num));
}


void GoLevel(void)
/* pide un nivel y su cdigo de acceso y lo lanza si es vlido */
{
	int n_password;

	_level=GetNumber("GO LEVEL\n\n\n\n\nLEVEL NUMBER: %s\n",30);

	if(_level==0)
	{
		DrawMask=NULL;
		return;
	}

	if(_level==1)
	{
		Fork(NULL);
		DrawMask=NULL;

		return;
	}

	n_password=GetNumber("GO LEVEL\n\n\n\n\nACCESS CODE: %s\n",999998);

	ReadLevel();

	if(n_password!=170868 && _password!=n_password)
	{
		_level=1;

		strcpy(_text_mask,"GO LEVEL\n\n\n\n\nWRONG ACCESS CODE\n");

		for(;;)
		{
			DrawWaterVirtualScreen();

			KbdPoll();

			if(_user_cancel || _user_enter)
				break;
		}

		while(_user_cancel || _user_enter)
			KbdPoll();
	}
	else
		Fork(NULL);

	DrawMask=NULL;
}


void Custom(void)
/* laberinto personalizado */
{
	char cmdline[1024];
	int __ambient_light;
	int __water_speed;
	int __num_pumps;
	int __num_consoles;
	int __maze_steps;
	int __broken_light;
	int __area_num;

	__num_pumps=GetNumber(
		"CUSTOM MAZE\n\n"
		"NUMBER OF PUMPS THAT\n"
		"WILL BE GENERATED\n"
		"\n"
		"\n\nNUMBER OF PUMPS: [1..5]: %s\n",5);

	if(__num_pumps==0)
		__num_pumps=1;

	__num_consoles=GetNumber(
		"CUSTOM MAZE\n\n"
		"NUMBER OF CONSOLES THAT\n"
		"WILL BE GENERATED\n"
		"\n"
		"\n\nNUMBER OF CONSOLES: [0..5]: %s\n",5);

	__ambient_light=GetNumber(
		"CUSTOM MAZE\n\n"
		"THE AMBIENT LIGHT DEFINES HOW\n"
		"DARK (4) OR BRIGHT (9)\n"
		"WILL BE THE MAZE.\n"
		"\n\nAMBIENT LIGHT [4..9]: %s\n",9);

	if(__ambient_light<4)
		__ambient_light=_ambient_light;

	__maze_steps=GetNumber(
		"CUSTOM MAZE\n\n"
		"THE MAZE STEPS DEFINES\n"
		"THE TOTAL SIZE OF THE MAZE.\n"
		"THE GREATER THE NUMBER OF\n"
		"PUMPS & CONSOLES, THE GREATER\n"
		"THIS NUMBER MUST BE.\n"
		"\n\nMAZE STEPS [15..50]: %s\n",50);

	if(__maze_steps<15)
		__maze_steps=15;

	__maze_steps+=__num_pumps*5;

	if(__num_consoles)
		__maze_steps+=__num_consoles*5;

	if(__maze_steps>100)
		__maze_steps=100;

	__area_num=GetNumber(
		"CUSTOM MAZE\n\n"
		"THE AREA NUMBER DEFINES\n"
		"THE GRAPHICS USED IN MAKING\n"
		"THE MAZE. 1 IS 'CORRIDORS',\n"
		"2 IS 'URANIUM SILO', AND SO ON."
		"\n\nAREA NUMBER [1..6]: %s\n",6);

	if(__area_num<1)
		__area_num=1;

	__broken_light=GetNumber(
		"CUSTOM MAZE\n\n"
		"THE BROKEN LIGHT MAKES\n"
		"THE AMBIENT LIGHT BLINK.\n"
		"\n"
		"\n\nBROKEN LIGHT [0..1]: %s\n",1);

	sprintf(cmdline,
		"custom=1 num_pumps=%d num_consoles=%d "
		"ambient_light=%d maze_steps=%d area_num=%d "
		"broken_light=%d",
		__num_pumps, __num_consoles, __ambient_light,
		__maze_steps, __area_num, __broken_light);

	_level=0;

	Fork(cmdline);
}


void Intro(void)
/* intro */
{
	time_t t;
	sp_palette * p;

	/* desactiva la mscara */
	DrawMask=NULL;

	/* carga el logo */

#ifdef STRATOS
	LoadScreen(_virtual_screen,"extern/logo.pcx");
	p=LoadPalette("extern/logo.pal");
#else
	LoadScreen(_virtual_screen,"graph/aolabs.pcx");
	p=LoadPalette("graph/aolabs.pal");
#endif

	if(_use_gamma_correct)
		p=BrightPalette(p,_gc_level);

	SetPalette(p);

	for(t=time(NULL);t+10>time(NULL);)
	{
		DrawVirtualScreen();

		KbdPoll();

		if(_user_cancel || _joy_button_B)
		{
			while(_user_cancel || _joy_button_B)
				KbdPoll();

			return;
		}

		if(_user_enter || _joy_button_A)
		{
			while(_user_enter || _joy_button_A)
				KbdPoll();

			break;
		}
	}

	CleanVirtualScreen();
	DrawVirtualScreen();

	/* pantalla inicial */
	LoadScreen(_virtual_screen,"graph/logo.pcx");
	p=LoadPalette("graph/logo.pal");

	if(_use_gamma_correct)
		p=BrightPalette(p,_gc_level);

	SetPalette(p);

	for(t=time(NULL);t+15>time(NULL);)
	{
		DrawWaterVirtualScreen();

		KbdPoll();

		if(_user_cancel || _joy_button_B)
		{
			while(_user_cancel || _joy_button_B)
				KbdPoll();

			return;
		}

		if(_user_enter || _joy_button_A)
		{
			while(_user_enter || _joy_button_A)
				KbdPoll();

			break;
		}
	}

	/* pantalla inicial */
	LoadScreen(_virtual_screen,"graph/planet.pcx");
	p=LoadPalette("graph/planet.pal");

	SetPalette(p);

	if(!Tty("SPACE PLUMBER MISSION BRIEFING\n\n\nPLANET: X-239-LQ (TSUNAMI XXVI)\a\nSURFACE: 100% WATER\a\n"
	    "RESOURCES: URANIUM AND THALIUM\a\nEXPLOITED BY:\n"
	    "  98% AQUA ENTERPRISES, INC.\n  02% OTHER COMPANIES.\n"))
		return;

	LoadScreen(_virtual_screen,"graph/planet.pcx");

	Tty("PROBLEM DESCRIPTION:\a\n  MAIN PROCESSING PLANT\n"
	    "  PRESSURE SYSTEM MALFUNCTION.\a\n"
	    "  ALL TECHNICIANS EVACUATED.\a\n\n"
	    "MISSION: ACTIVATE ALL WATER\n"
	    "  EXTRACTION PUMPS.\a\n"
	    "  DEACTIVATE SECURITY CONSOLES\n"
	    "  WHERE AVAILABLE.\n");
}


void Menu(void)
/* muestra el men inicial */
{
	sp_palette * p;
	int up,down;
	int reload;

	/* no hay logger */
	_use_logger=0;

	MenuStartup();

	if(_intro)
		Intro();

	_text_mask_x=-1;

	reload=1;
	up=down=0;

	p=LoadPalette("graph/menu.pal");

	while(!_user_cancel)
	{
		if(reload)
		{
			reload=0;

			DrawMask=NULL;

			CleanVirtualScreen();
			DrawVirtualScreen();

			/* carga la mscara, la paleta y el agua del men */
			LoadScreen(_menu_mask,"graph/menu.pcx");
			LoadScreen(_virtual_screen,"graph/wmenu.pcx");

			/* carga el cursor */
			LoadPcx((char *) cursor,"graph/cursor.pcx",CURSOR_SIZE);

			DrawMask=MenuMask;

			InvalidPalette();
			SetPalette(p);
		}

		DrawWaterVirtualScreen();

		KbdPoll();

		if(_kbd_actions & ACT_FORWARD)
		{
			down=0;

			if(up==0)
			{
				up=1;

				menu_opt--;

				if(menu_opt==-1)
					menu_opt=4;
			}
		}
		else
			up=0;

		if(_kbd_actions & ACT_BACK)
		{
			up=0;

			if(down==0)
			{
				down=1;

				menu_opt++;

				if(menu_opt==5)
					menu_opt=0;
			}
		}
		else
			down=0;

		if(_user_cancel)
			break;

		if(_user_enter || _joy_button_A)
		{
			reload=1;

			switch(menu_opt)
			{
			case 0:
				if(_level==0)
					_level=1;

				Fork(NULL);

				break;

			case 1:

				GoLevel();
				break;

			case 2:
				Custom();
				break;
			case 3:

				while(_user_enter || _joy_button_A)
					KbdPoll();

				DrawMask=NULL;

				LoadScreen(_virtual_screen,"graph/help.pcx");
				SetPalette(LoadPalette("graph/help.pal"));

				DrawVirtualScreen();

				while(!_user_enter && !_user_cancel &&
				      !_joy_button_A)
					KbdPoll();

				while(_user_enter || _user_cancel ||
				      _joy_button_A)
					KbdPoll();

				break;

			case 4:
				_user_cancel=1;
				break;
			}
		}
	}

	MenuShutdown();

	if(_user_cancel)
	{
		printf("	+--------------------------------------------------+\n");
		printf("	| Thank you for playing Space Plumber.		   |\n");
		printf("	+--------------------------------------------------+\n");
		printf("	| Space Plumber is FREE and comes with NO WARRANTY |\n");
		printf("	|						   |\n");
		printf("	| Developed by Angel Ortega during 1997 	   |\n");
		printf("	| comments to email: angel@triptico.com 	   |\n");
		printf("	|						   |\n");
		printf("	| Visit the Space Plumber Home Page at: 	   |\n");
		printf("	|						   |\n");
		printf("	| http://www.triptico.com			   |\n");
		printf("	|						   |\n");
		printf("	| (C) Angel Ortega 1997 			   |\n");
		printf("	+--------------------------------------------------+\n");

	}
}


/** juego **/

void GameCore(int x, int y, int z, int a)
/* ncleo del juego */
{
	int cycle_count=0;

	while(!_user_cancel && !_last_level_resolved)
	{
		KbdPoll();

		if(DrawNewFrame())
		{
			if(!_user_pause)
			{
				TimeEvents(y);

				Motion(&x,&y,&z,&a,_kbd_actions);

				cycle_count++;

				if(!(cycle_count & 0x01) && y < _water_level)
					continue;
			}
			else
				SetMessage("","","PAUSED...","",1);

			Objects();

			Render(x,y,z,a);
		}
	}

	/* si se ha resuelto el ltimo nivel, mensaje */
	if(_last_level_resolved)
	{
		_tty_under_water=1;

		SetSound(_snd_underwater,0,0,0);
		SetSound(_snd_water_go,0,0,0);
		SetSound(_snd_water,15,15,1);

		LoadScreen(_virtual_screen,"graph/end.pcx");
		SetPalette(LoadPalette("graph/end.pal"));

		Tty("SPACE PLUMBER MISSION REPORT\a\n\n"
		    "ALL LEVELS HAS BEEN SECURED.\a\n"
		    "PRESSURE SYSTEM RESTARTED.\a\n"
		    "AQUA ENTERPRISES, INC. MAIN\n"
		    "PLANT IS AGAIN FULLY FUNCTIONAL.\a\n\n"
		    "WHAT ABOUT SOME DAYS OF VACATION?\n");

		LoadScreen(_virtual_screen,"graph/end.pcx");
		
		Tty("RESPONSE FROM HIGH COMMAND\a\n\n"
		    "CONGRATULATIONS.\a\n\n"
		    "VACATION REQUESTS MUST BE MADE\n"
		    "BY FILLING THE FORM TT-243/F.\a\n"
		    "THIS FORM IS UNAVAILABLE FOR NOW.\a\n\n"
		    "END OF TRANSMISSION.\n");
	}
}


void GameStartup(void)
/* inicia todo lo necesario en una sesin de juego */
{
	SuppStartup();
	ParseParam(ReadLevel());

	GrxStartup();
	KbdStartup();
	SoundStartup();
	RayStartup();
	MapStartup();
	PlayerStartup();
	MazeStartup();

	logger(_game_name,s_sprintf(STARTUP_MSG,__VERSION__,__DATE__));

	logger(_game_name,s_sprintf("Startup - %d bytes total memory used",_memory_used));
}


void GameShutdown(void)
/* cierra todo en una sesin de juego */
{
	RayShutdown();
	SoundShutdown();
	KbdShutdown();
	SuppShutdown();
	PlayerShutdown();

#ifdef LINUX_SVGALIB
	DrawMask=NULL;

	CleanVirtualScreen();
	DrawVirtualScreen();
#endif

	SetTextMode();

	logger(_game_name,
		s_sprintf("Shutdown - Frames per sec.: %d / %d",
			_avg_frames_per_sec, _max_frames_per_sec));
}


void Game(void)
/* inicia el modo juego */
{
	int x,y,z,a;
	char msg0[80];
	char msg1[80];
	char msg2[80];
	char msg3[80];

	GameStartup();

	x=16384+64;
	z=16384+64;
	y=16384;
	a=ANGLE(0);

	if(_level==0)
		strcpy(msg0,"LEVEL: CUSTOM");
	else
		sprintf(msg0,"LEVEL: %d",_level);

	sprintf(msg1,"AREA: %s",_area_name);
	sprintf(msg2,"PUMPS: %d / CONSOLES: %d",_num_pumps,_num_consoles);
	sprintf(msg3,"TOTAL SINKING ESTIMATED: %02d:%02d",_left_minutes,_left_seconds);

	SetMessage(msg0,msg1,msg2,msg3,10);

	GameCore(x,y,z,a);

	GameShutdown();
}


int main(int argc, char * argv[])
{
	int ret;

	ParseFile("splumber.rc");
	ParseMain(argc,argv);

	/* si se ha seleccionado algn nivel, no se pide el men */
	if(_level!=0 || _custom!=0)
		_menu=0;

	if(_menu)
	{
		Menu();

		ret=0;
	}
	else
	{
		Game();

		/* cdigos de retorno */

		if(_last_level_resolved)
			ret=SPLUMBER_WIN;
		else
		if(_level_resolved)
			ret=SPLUMBER_RESOLVED;
		else
		if(_player_dead)
			ret=SPLUMBER_DEAD;
		else
			ret=SPLUMBER_CANCEL;
	}

	return(ret);
}
