/*

	sp_supp.c

	Rutinas diversas de soporte.

	Space Plumber - Angel Ortega

*/

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <unistd.h>

#ifdef DJGPP
#include <dir.h>
#include <io.h>
#include <dpmi.h>
#endif

#ifdef linux
#define O_BINARY 0
#endif

#include "sp_types.h"
#include "sp_supp.h"
#include "sp_grx.h"


/* memoria usada */
long _memory_used=0;

/* si est a 1, fuerza a pack_fopen a intentar leer el fichero desde
   fuera del pack, permitiendo extensiones al pack principal */
int _pack_extensions=0;

/* switch de uso del logger */
int _use_logger=1;

/* switch de uso de stderr para el logger */
#ifdef UNIX_X11
int _log_to_stderr=1;
#else
int _log_to_stderr=0;
#endif


/* el fichero pack tiene la siguiente estructura por cada elemento:

   long name_size;		tamao del nombre del elemento
   char name[name_size];	nombre del elemento (includo el \0)
   long data_size;		tamao del elemento
   char data[data_size];	el elemento

*/

/* fichero pack estndar */
char _std_pack_file[160]="splumber.pck";

/* fichero pack alternativo */
char _alt_pack_file[160]="";

/* directorio donde se puede escribir */
#ifdef linux
char _writable_dir[100]="/tmp/";
#else
char _writable_dir[100]="./";
#endif

/* ficheros temporales */
char _tmp_pack_file[160];
char _tmp_pack2_file[160];
char _tmp_pack3_file[160];

/* fichero del logger */
char _logger_file[160];


/* cach de ficheros */

#define FILE_CACHE_NAME_SIZE 60
struct file_cache
{
	char name[FILE_CACHE_NAME_SIZE];
	void * data;
	struct file_cache * next;
};

static struct file_cache * _file_cache=NULL;


/* nmeros aleatorios */
unsigned long _random=0;


/********************
	Cdigo
*********************/

char * s_sprintf(char * fmt, ...)
/* sprintf con buffer temporal */
{
	static char sp_buf[4096];
	va_list argptr;

	va_start(argptr, fmt);
	vsprintf(sp_buf, fmt, argptr);
	va_end(argptr);

	return(sp_buf);
}


void logger(char * where, char * msg)
/* escribe la cadena en el fichero de log */
{
	FILE * f;

	if(_use_logger)
	{
		if(_log_to_stderr)
			fprintf(stderr,"%s: %s\n",where,msg);
		else
		if((f=fopen(_logger_file,"a"))!=NULL)
		{
			fprintf(f,"%s: %s\n",where,msg);

			fclose(f);
		}
	}
}


void volatile bang(char * where, char * msg)
/* sale con error */
{
	logger(where,msg);

	SetTextMode();

	printf("\nBANG! %s: %s\n",where,msg);
	fflush(stdout);

	exit(SPLUMBER_BANG);
}


void * a_malloc(size_t size)
/* Pide memoria con accounting. Aborta si no se puede obtener */
{
	void * ptr;

	ptr=malloc(size);

	_memory_used+=size;

	if(!ptr)
	{
		bang("a_malloc",s_sprintf("You need at least %d bytes to run this program.",
			_memory_used));
	}

	return(ptr);
}


void a_free(void * ptr, size_t size)
/* libera memoria, tenindola en cuenta en el accounting */
{
	_memory_used-=size;

	free(ptr);
}


static FILE * pack_extract(char * packfile, char * filename, char * mode)
/* extrae un fichero de un pack y lo abre */
{
	long size;
	int i,o,c;
	FILE * f;
	FILE * p;
	char pack_filename[250];
	char * buffer;
	int zpck=0;

	/* abre el pack */
	if((i=open(packfile,O_RDONLY|O_BINARY))==-1)
		bang("pack_fopen",packfile);

	/* lee la firma del fichero */
	read(i,pack_filename,6);

	/* si es <zpck>, se activa el flag de compresin */
	if(memcmp(pack_filename,"<zpck>",6)==0)
		zpck=1;
	else
	/* si no es <xpck>, fuera */
	if(memcmp(pack_filename,"<xpck>",6)!=0)
		bang("pack_fopen","Bad format");


	/* crea el fichero temporal */
	if((o=open(_tmp_pack_file,O_CREAT|O_TRUNC|O_WRONLY|O_BINARY,
		S_IREAD|S_IWRITE))==-1)
		bang("pack_fopen","_tmp_pack_file");

	for(;;)
	{
		/* lee el tamao de un elemento */
		if(read(i,&size,sizeof(size))!=sizeof(size))
		{
			close(i);
			close(o);
			return(NULL);
		}

		/* lee tantos caracteres como diga size */
		read(i,pack_filename,size);

		/* lee, adems, el tamao del elemento en s */
		read(i,&size,sizeof(size));

		/* es ste? */
		if(strcmp(pack_filename,filename)==0)
			break;

		/* no es: saltarse el elemento */
		lseek(i,size,SEEK_CUR);
	}

	/* ya est: pedir memoria */
	buffer=(char *) a_malloc(size);

	/* lee todo el elemento y lo escribe sobre el temporal */
	read(i,buffer,size);
	write(o,buffer,size);

	/* libera el bloque y lo resta de _memory_used */
	a_free(buffer,size);

	close(i);
	close(o);

	if(zpck)
	{
		sprintf(pack_filename,"gzip -c -d %s",_tmp_pack_file);

		if((p=popen(pack_filename,"rb"))==NULL)
			bang("pack_extract","gzip");

		f=fopen(_tmp_pack3_file,"wb");

		while((c=getc(p))!=EOF)
			fputc(c,f);

		fclose(f);
		pclose(p);

		if((f=fopen(_tmp_pack3_file,mode))==NULL)
			bang("pack_fopen","tmp3 file (?)");

	}
	else
	/* lo abre ahora en el modo pedido */
	if((f=fopen(_tmp_pack_file,mode))==NULL)
		bang("pack_fopen","tmp file (?)");

	return(f);
}


FILE * pack_fopen(char * filename, char * mode)
/* Abre un fichero que est dentro de un pack (o como un fichero
   convencional si est activado _pack_extensions) */
{
	FILE * f;

	/* si estn permitidas las extensiones, se intenta
	   abrir el fichero directamente */
	if(_pack_extensions)
	{
		if((f=fopen(filename,mode))!=NULL)
			return(f);
	}

	/* extrae primero del pack alternativo */
	if(_alt_pack_file[0]!='\0')
	{
		if((f=pack_extract(_alt_pack_file,filename,mode))!=NULL)
			return(f);
	}

	/* extrae del pack principal */
	if((f=pack_extract(_std_pack_file,filename,mode))!=NULL)
		return(f);

	bang("pack_fopen",filename);
}


FILE * pack_dup(FILE * f, char * mode)
/* Duplica el fichero temporal primero en el segundo. Es un hack
   para evitar (levemente) la no-reentrancia de pack_fopen */
{
	int c;
	FILE * o;

	if((o=fopen(_tmp_pack2_file,"wb"))==NULL)
		bang("pack_dup",_tmp_pack2_file);

	while((c=getc(f))!=EOF)
		putc(c,o);

	fclose(f);
	fclose(o);

	if((o=fopen(_tmp_pack2_file,mode))==NULL)
		bang("pack_dup","fopen ?");

	return(o);
}


#ifdef DJGPP
static void SmartdrvFlush(void)
/* obliga al Smartdrv a vaciar los buffers */
{
	__dpmi_regs regs;

	regs.x.ax=0x4a10;
	regs.x.bx=0x0001;

	__dpmi_int(0x2f, &regs);
}
#endif


/** cach **/

void * SeekCache(char * name)
/* busca el fichero en el cache */
{
	struct file_cache * cache;

	cache=_file_cache;

	while(cache!=NULL)
	{
		if(strcmp(name,cache->name)==0)
			return(cache->data);

		cache=cache->next;
	}

	return(NULL);
}


void AddCache(char * name, void * data)
/* aade el fichero al cache */
{
	struct file_cache * c;

	c=(struct file_cache *) a_malloc(sizeof(struct file_cache));

	strcpy(c->name,name);
	c->data=data;
	c->next=_file_cache;

	_file_cache=c;
}


unsigned long sp_random(void)
/* devuelve un nmero aleatorio */
{
	_random=(_random * 58321) + 11113;

	return(_random >> 16);
}


void sp_randomize(int seed)
/* define la semilla de sp_random */
{
	_random=seed;
}


long RANDOM(int range)
{
#ifdef STD_RANDOM
	return(random()%(range));
#else
	return(sp_random()%(range));
#endif
}


void RANDOMIZE(long seed)
/* define la semilla */
{
#ifdef STD_RANDOM
	srandom(seed);
#else
	sp_randomize(seed);
#endif
}


void SuppStartup(void)
/* inicializa las rutinas de soporte */
{
#ifdef DJGPP
	__dpmi_version_ret vr;
#endif

	/* crea el directorio de escritura */
	mkdir(_writable_dir,666);

	/* construye los nombres de los ficheros */
	sprintf(_logger_file,"%ssplumber.log",_writable_dir);
	sprintf(_tmp_pack_file,"%ssp_0000.tmp",_writable_dir);
	sprintf(_tmp_pack2_file,"%ssp_0001.tmp",_writable_dir);
	sprintf(_tmp_pack3_file,"%ssp_0002.tmp",_writable_dir);

	if(_use_logger)
		unlink(_logger_file);

	/* a partir de aqu, ya hay logger */
	logger("Space Plumber","boot...");

#ifdef DJGPP
	/* borra el fichero .BAT de instalacin
	   (esto no es muy elegante, pero tampoco lo es
	   dejar el .BAT ah) */
	unlink("kbackrun.bat");

	SmartdrvFlush();

	__dpmi_get_version(&vr);

	logger("SuppStartup",s_sprintf("DPMI %d.%d %s - x86==%d",
		vr.major, vr.minor, (vr.flags&0x1?"32bit":"16bit"),vr.cpu));
#endif
}


void SuppShutdown(void)
/* cierra el sistema de soporte */
{
	unlink(_tmp_pack_file);
	unlink(_tmp_pack2_file);
	unlink(_tmp_pack3_file);
}
