/*

	sp_grx.c

	Rutinas de video: paleta, grficos, carga.

	Space Plumber - Angel Ortega

*/

#include <stdio.h>
#include <string.h>
#include <math.h>

#ifdef DJGPP
#include <dpmi.h>
#include <pc.h>
#include <sys/movedata.h>
#endif

#ifdef LINUX_SVGALIB
#include <vga.h>
#endif

#ifdef UNIX_X11
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#endif

#include "sp_supp.h"
#include "sp_types.h"
#include "sp_grx.h"


/* flag de uso del retrazado vertical */
int _use_wait_retrace=0;

/* paleta en curso */
sp_palette * _actual_palette=NULL;

/* paleta normal y submarina */
sp_palette * _normal_palette=NULL;
sp_palette * _water_palette=NULL;

/* paletas con gamma-correct */
sp_palette * _gc_normal_palette=NULL;
sp_palette * _gc_water_palette=NULL;

/* flag de uso de gamma-correct */
int _use_gamma_correct=0;

/* nivel de gamma-correct */
int _gc_level=2;

/* pantallas virtuales */
char * _virtual_screen=NULL;
char * _water_virtual_screen=NULL;


/* efectos de ondas de agua: */

/* bitmaps del agua */

/* agua original (cargada de disco) */
sp_texture * _original_water_texture;

/* puntero al bitmap del agua a dibujar */
sp_texture * _water_texture;

/* punteros a buffers donde se efecta la ondulacin del agua */
sp_texture * _wave_water_texture[2];

/* _wave_water_texture en uso */
int _wave_water_texture_index=0;

/* tablas de senos y cosenos para el bitmap del agua */
int _water_texture_sin_table[TEXTURE_Y_SIZE];
int _water_texture_cos_table[TEXTURE_X_SIZE];

/* perodo de ondulacin */
int _water_texture_period=0;

/* pantalla virtual de agua */

/* tablas de senos y cosenos */
int _water_screen_sin_table[SCREEN_Y_SIZE];
int _water_screen_cos_table[SCREEN_X_SIZE];

/* perodo de ondulacin */
int _water_screen_period=0;

/* textos */

/* el font */
unsigned char * _system_font;

/* tamao en lneas verticales del font */
int _system_font_size;

/* funcin de dibujo de la mscara */
void (* DrawMask)(char *);

/* el fichero que contiene el font */
char _font_file[150]="etc/v-dub.ktl";


#ifdef UNIX_X11

/* Diferentes variables de control para X11 */

Display * _x_display;
int _x_screen;
XVisualInfo _x_visualinfo;
Window _x_window;
Colormap _x_colormap;
Visual * _x_visual;
GC _x_gc;
XEvent _x_event;
XImage * _x_image;
XColor _x_palette[256];

char _x_display_name[150]="";

/* hack para conocer si estamos en modo men o juego */
extern int _menu;

#endif


/********************
	Cdigo
*********************/


void SetGraphMode(void)
/* pone el vdeo en modo 320x200x256 */
{
#ifdef DJGPP
	__dpmi_regs regs;

	regs.x.ax=0x13;
	__dpmi_int(0x10, &regs);
#endif

#ifdef LINUX_SVGALIB
	vga_setmode(G320x200x256);
#endif

#ifdef UNIX_X11
	unsigned long attrmask;
	XSetWindowAttributes attrs;
	XGCValues xgcvalues;

	_x_display=XOpenDisplay(_x_display_name);

	if(!_x_display)
		bang("SetGraphMode","Bad display");

	_x_screen=DefaultScreen(_x_display);

	if(!XMatchVisualInfo(_x_display,_x_screen,8,
		PseudoColor,&_x_visualinfo))
		bang("SetGraphMode","Display must be 8bpp (256 col.)");

	_x_visual=_x_visualinfo.visual;

	_x_colormap=XCreateColormap(_x_display,
		RootWindow(_x_display,_x_screen), _x_visual, AllocAll);

	attrmask=CWEventMask|CWColormap|CWBorderPixel;
	attrs.event_mask=KeyPressMask|KeyReleaseMask|ExposureMask;
	attrs.colormap=_x_colormap;
	attrs.border_pixel=0;

	_x_window=XCreateWindow(_x_display,
		RootWindow(_x_display, _x_screen),
		0, 0, SCREEN_X_SIZE, SCREEN_Y_SIZE, 0, 8,
		InputOutput, _x_visual, attrmask, &attrs);

	xgcvalues.graphics_exposures=False;

	_x_gc=XCreateGC(_x_display, _x_window, GCGraphicsExposures,
		&xgcvalues);

	XMapWindow(_x_display, _x_window);

	for(;;)
	{
		XNextEvent(_x_display, &_x_event);

		if(_x_event.type==Expose && !_x_event.xexpose.count)
			break;
	}

	_x_image=XCreateImage(_x_display, _x_visual,
		8, ZPixmap, 0, _virtual_screen,
		SCREEN_X_SIZE, SCREEN_Y_SIZE, 8, SCREEN_X_SIZE);

	XStoreName(_x_display, _x_window,
		_menu==1?"Space Plumber Menu":"Space Plumber " SP_VERSION);
#endif
}


void SetTextMode(void)
/* pone el vdeo en modo texto */
{
#ifdef DJGPP
	__dpmi_regs regs;

	regs.x.ax=0x3;
	__dpmi_int(0x10, &regs);
#endif

#ifdef LINUX_SVGALIB
	vga_setmode(TEXT);

	/* enva el comando ANSI (estamos en la consola)
	   que borra la pantalla */
	printf("\x1b[H\x1b[J\n");
#endif
}


void WaitRetrace(void)
/* espera el retrazado vertical */
{
	if(_use_wait_retrace)
	{

#ifdef DJGPP
		while (!(inportb(0x3DA)&8));
		while ((inportb(0x3DA)&8));
#endif

#ifdef LINUX_SVGALIB
		vga_waitretrace();
#endif

	}
}

/** paleta **/

void InvalidPalette(void)
/* invalida la paleta definida */
{
	_actual_palette=NULL;
}


void SetPalette(sp_palette * pal)
/* activa la paleta */
{
	int n;

	if(pal==_actual_palette)
		return;

#ifdef DJGPP
	outportb(0x3c8,0);

	for(n=0;n<256;n++)
	{
		outportb(0x3c9,((unsigned char)pal->r)>>2);
		outportb(0x3c9,((unsigned char)pal->g)>>2);
		outportb(0x3c9,((unsigned char)pal->b)>>2);

		pal++;
	}
#endif

#ifdef LINUX_SVGALIB
	for(n=0;n<256;n++)
		vga_setpalette(n, pal[n].r>>2, pal[n].g>>2, pal[n].b>>2);
#endif

	_actual_palette=pal;

#ifdef UNIX_X11
	for(n=0;n<256;n++)
	{
		_x_palette[n].pixel=n;
		_x_palette[n].flags=DoRed|DoGreen|DoBlue;
		_x_palette[n].red=((short)pal[n].r)<<8;
		_x_palette[n].green=((short)pal[n].g)<<8;
		_x_palette[n].blue=((short)pal[n].b)<<8;
	}

	XStoreColors(_x_display, _x_colormap, _x_palette, 256);
#endif
}


sp_palette * LoadPalette(char * palfile)
/* carga la paleta */
{
	int n;
	FILE * f;
	char lin[80];
	unsigned int r,g,b;
	sp_palette * pal;
	sp_palette * save_pal;

	pal=(sp_palette *) a_malloc(sizeof(sp_palette)*256);
	save_pal=pal;

	f=pack_fopen(palfile,"r");

	/* lee la primera lnea (JASC-PAL), la segunda
	   (versin) y la tercera (256) */
	fgets(lin,sizeof(lin)-1,f);
	fgets(lin,sizeof(lin)-1,f);
	fgets(lin,sizeof(lin)-1,f);

	for(n=0;n<256;n++)
	{
		if(fscanf(f,"%u %u %u\n",&r,&g,&b)!=3)
			break;

		pal->r=r;
		pal->g=g;
		pal->b=b;

		pal++;
	}

	for(;n<256;n++)
	{
		pal->r=pal->g=pal->b=0;
		pal++;
	}

	fclose(f);

	return(save_pal);
}


sp_palette * WaterPalette(sp_palette * orgpal)
/* crea una paleta 'submarina' respecto a orgpal */
{
	int water;
	int n;
	sp_palette * despal;
	sp_palette * pal;

	despal=(sp_palette *) a_malloc(sizeof(sp_palette)*256);
	pal=despal;

	for(n=0;n<256;n++)
	{
		water=orgpal->r;
		water/=3;

		pal->r=water;

		water=orgpal->g;
		water/=3;

		pal->g=water;

		pal->b=orgpal->b;

		pal++;
		orgpal++;
	}

	return(despal);
}


sp_palette * BrightPalette(sp_palette * orgpal, int factor)
/* crea una paleta ms brillante */
{
	int n,i;
	sp_palette * despal;
	sp_palette * pal;

	factor*=24;

	despal=(sp_palette *) a_malloc(sizeof(sp_palette)*256);
	pal=despal;

	for(n=0;n<256;n++)
	{
		i=(int)orgpal->r + factor;
		if(i>255)
			i=orgpal->r;
		pal->r=i;

		i=(int)orgpal->g + factor;
		if(i>255)
			i=orgpal->g;
		pal->g=i;

		i=(int)orgpal->b + factor;
		if(i>255)
			i=orgpal->b;
		pal->b=i;

		pal++;
		orgpal++;
	}

	return(despal);
}


/** texturas **/

void LoadPcx(char * pcx,char * pcxfile,int size)
/* carga un pcx */
{
	int n,m;
	unsigned char c;
	FILE * f;

	f=pack_fopen(pcxfile,"rb");

	/* se salta la cabecera */
	fseek(f,128,SEEK_SET);

	n=0;
	while(n<size)
	{
		c=fgetc(f);

		if(c>0xC0)
		{
			/* es un run-length */
			m=c & 0x3F;

			c=fgetc(f);
		}
		else
			m=1;

		while(m)
		{
			pcx[n++]=c;
			m--;
		}
	}

	fclose(f);
}


void LoadScreen(char * screen,char * pcxfile)
/* carga una pantalla */
{
	LoadPcx(screen,pcxfile,SCREEN_X_SIZE*SCREEN_Y_SIZE);
}


sp_texture * LoadTexture(char * textfile)
/* Carga una textura. Las texturas soportadas son ficheros PCX de 128x128
   con la paleta normalizada, y se rotan 90 a la izquierda */
{
	int n,m;
	unsigned char c;
	sp_texture * text;
	FILE * f;
	int x,y;

	if((text=(sp_texture *)SeekCache(textfile))!=NULL)
		return(text);

	text=(sp_texture *) a_malloc(TEXTURE_SIZE);

	x=0;
	y=TEXTURE_Y_SIZE-1;

	f=pack_fopen(textfile,"rb");

	/* se salta la cabecera */
	fseek(f,128,SEEK_SET);

	n=0;
	while(n<TEXTURE_SIZE)
	{
		c=fgetc(f);

		if(c>0xC0)
		{
			/* es un run-length */
			m=c & 0x3F;

			c=fgetc(f);
		}
		else
			m=1;

		while(m)
		{
			text[x+y*TEXTURE_X_SIZE]=c;

			y--;

			if(y==-1)
			{
				x++;
				y=(TEXTURE_Y_SIZE)-1;
			}

			m--;
			n++;
		}
	}

	fclose(f);

	AddCache(textfile,text);

	return(text);
}


/** pantallas virtuales **/

void DumpScreen(char * screen)
/* vuelca la pantalla virtual sobre la pantalla real */
{
	WaitRetrace();

#ifdef DJGPP
	dosmemput(screen,SCREEN_X_SIZE*SCREEN_Y_SIZE,0xa0000);
#endif

#ifdef LINUX_SVGALIB
	memcpy(graph_mem,screen,SCREEN_X_SIZE*SCREEN_Y_SIZE);
#endif

#ifdef UNIX_X11

	_x_image->data=screen;

	XPutImage(_x_display, _x_window, _x_gc,
		_x_image, 0, 0, 0, 0, SCREEN_X_SIZE, SCREEN_Y_SIZE);

	XSync(_x_display, False);
#endif
}


static void AllocVirtualScreens(void)
/* habilita memoria para las pantallas virtuales */
{
	_virtual_screen=(char *) a_malloc(SCREEN_X_SIZE*SCREEN_Y_SIZE);
	_water_virtual_screen=(char *) a_malloc(SCREEN_X_SIZE*SCREEN_Y_SIZE);
}


void CleanVirtualScreen(void)
/* limpia la memoria virtual */
{
	memset(_virtual_screen,'\0',SCREEN_X_SIZE*SCREEN_Y_SIZE);
}


void DrawVirtualScreen(void)
/* dibuja la pantalla virtual sobre la pantalla */
{
	if(DrawMask!=NULL)
		DrawMask(_virtual_screen);

	DumpScreen(_virtual_screen);
}


/** ondulaciones de agua **/

/* bitmap del agua */

void WaveWaterTexture(void)
/* genera la ondulacin sobre el bitmap */
{
	register sp_texture c;
	register sp_texture * optr;
	register sp_texture * dptr;
	int x,y,sx;
	int a,b,fsx;

	optr=_original_water_texture;
	dptr=_water_texture=_wave_water_texture[_wave_water_texture_index];

	for(y=0;y<TEXTURE_Y_SIZE;y++)
	{
		a=_water_texture_sin_table[(y+_water_texture_period)&TEXTURE_Y_MASK];

		for(x=0;x<TEXTURE_X_SIZE;x++)
		{
			b=_water_texture_cos_table[(x+_water_texture_period)
				&TEXTURE_X_MASK];

			fsx=(a+b)>>10;
			sx=(x+fsx)&TEXTURE_X_MASK;

			c=*(optr+sx);
			*dptr=c;
			dptr++;
		}

		optr+=TEXTURE_X_SIZE;
	}

	_wave_water_texture_index^=1;
	_water_texture_period++;
}


void FadeEdges(void)
/* oscurece los bordes de la pantalla */
{
	unsigned char * ptr;
	register unsigned char * ptr2;
	int n;

	for(ptr=_water_virtual_screen+(SCREEN_X_SIZE-15),n=0;
		n<SCREEN_Y_SIZE;n++,ptr+=SCREEN_X_SIZE)
	{
		ptr2=ptr;

		(*ptr2)>>=1; ptr2++;
		(*ptr2)>>=1; ptr2++;
		(*ptr2)>>=1; ptr2++;
		(*ptr2)>>=1; ptr2++;
		(*ptr2)>>=1; ptr2++;

		(*ptr2)>>=2; ptr2++;
		(*ptr2)>>=2; ptr2++;
		(*ptr2)>>=2; ptr2++;
		(*ptr2)>>=2; ptr2++;
		(*ptr2)>>=2; ptr2++;

		(*ptr2)>>=3; ptr2++;
		(*ptr2)>>=3; ptr2++;
		(*ptr2)>>=3; ptr2++;
		(*ptr2)>>=3; ptr2++;
		(*ptr2)>>=3; ptr2++;


		(*ptr2)>>=3; ptr2++;
		(*ptr2)>>=3; ptr2++;
		(*ptr2)>>=3; ptr2++;
		(*ptr2)>>=3; ptr2++;
		(*ptr2)>>=3; ptr2++;

		(*ptr2)>>=2; ptr2++;
		(*ptr2)>>=2; ptr2++;
		(*ptr2)>>=2; ptr2++;
		(*ptr2)>>=2; ptr2++;
		(*ptr2)>>=2; ptr2++;

		(*ptr2)>>=1; ptr2++;
		(*ptr2)>>=1; ptr2++;
		(*ptr2)>>=1; ptr2++;
		(*ptr2)>>=1; ptr2++;
		(*ptr2)>>=1; ptr2++;
	}
}


void DrawWaterVirtualScreen(void)
/* genera la ondulacin sobre la pantalla virtual */
{
	register unsigned char c;
	register unsigned char * optr;
	register unsigned char * dptr;
	register int o;
	int x,y,sx;
	int a,b,fsx;

	optr=_virtual_screen;
	dptr=_water_virtual_screen;

	for(y=0;y<SCREEN_Y_SIZE;y++)
	{
		o=(y+_water_screen_period)%SCREEN_Y_SIZE;

		a=_water_screen_sin_table[o];

		for(x=0;x<SCREEN_X_SIZE;x++)
		{
			o=(x+_water_screen_period)%SCREEN_X_SIZE;

			b=_water_screen_cos_table[o];

			fsx=(a+b)>>10;
			sx=(x+fsx)%SCREEN_X_SIZE;

			c=*(optr+sx);
			*dptr=c;
			dptr++;
		}

		optr+=SCREEN_X_SIZE;
	}

	_water_screen_period++;

	FadeEdges();

	/* vaca la primera lnea, que parece estar
	   llenndose de mierda */
	memset(_water_virtual_screen,'\0',SCREEN_X_SIZE);

	if(DrawMask!=NULL)
		DrawMask(_water_virtual_screen);

	DumpScreen(_water_virtual_screen);
}


void InitWater(void)
/* inicializa el sistema de ondulacin de agua */
{
	double f;
	int n;

	/* rellena las tablas de senos y cosenos del bitmap */
	for(n=0;n<TEXTURE_Y_SIZE;n++)
	{
		f=((double) n)*6.283184/TEXTURE_Y_SIZE;
		f=sin(f)*(1<<13);
		_water_texture_sin_table[n]=(int) f;
	}

	for(n=0;n<TEXTURE_X_SIZE;n++)
	{
		f=((double) n)*6.283184/TEXTURE_X_SIZE;
		f=cos(f)*(1<<13);
		_water_texture_cos_table[n]=(int) f;
	}

	/* rellena las tablas de senos y cosenos de la pantalla */
	for(n=0;n<SCREEN_Y_SIZE;n++)
	{
		f=((double) n)*6.283184/SCREEN_Y_SIZE;
		f=sin(f)*(1<<13);
		_water_screen_sin_table[n]=(int) f;
	}

	for(n=0;n<SCREEN_X_SIZE;n++)
	{
		f=((double) n)*6.283184/SCREEN_X_SIZE;
		f=cos(f)*(1<<13);
		_water_screen_cos_table[n]=(int) f;
	}

	/* pide memoria para los dos buffers */
	_wave_water_texture[0]=(unsigned char *)a_malloc(TEXTURE_SIZE);
	_wave_water_texture[1]=(unsigned char *)a_malloc(TEXTURE_SIZE);
}


/** textos **/

void LoadFont(char * filename)
/* carga un font. Los fonts soportados son los .KTL de KGE,
   por tanto, no proporcionales y de un byte de ancho */
{
	unsigned char c;
	FILE * f;

	f=pack_fopen(filename,"rb");

	/* obtiene el tamao del font */
	c=fgetc(f);
	_system_font_size=(int) c;

	/* desprecia los 7 bytes restantes de la cabecera */
	fseek(f,7,SEEK_CUR);

	_system_font=(unsigned char *) a_malloc(_system_font_size*256);

	/* lee todo el font */
	fread(_system_font,256,_system_font_size,f);

	fclose(f);
}


void PrintString(unsigned char * screen, int x, int y,
	unsigned char color, unsigned char * string)
/* Imprime una cadena de caracteres en la pantalla virtual */
{
	unsigned char * screen2;
	int n,m;
	unsigned char mask;
	unsigned char c;
	unsigned char color2;

	if(screen==NULL || string==NULL)
		return;

	color2=color>>1;

	if(x==-1)
	{
		/* si x es -1, se centra */
		x=(SCREEN_X_SIZE/9)/2-strlen(string)/2;
	}
	else
	if(x==-2)
	{
		/* si x es -2, se justifica a la derecha */
		x=(SCREEN_X_SIZE/9)-strlen(string);
	}

	screen+= (x*9 + y*SCREEN_X_SIZE*_system_font_size);

	for(m=0;m<_system_font_size;m++)
	{
		screen2=screen;

#ifdef GRADIENT_FONT
		if(m>=(_system_font_size-6))
			color>>=1;
#endif

		for(n=0;string[n];n++)
		{
			c=*(_system_font + (string[n]*_system_font_size) + m);

			for(mask=0x80;mask;mask>>=1)
			{
				if(mask & c)
				{
					*screen2=color;
					*(screen2+SCREEN_X_SIZE+1)=color2;
				}

				screen2++;
			}

			screen2++;
		}

		screen+=SCREEN_X_SIZE;
	}
}


void GrxStartup(void)
/* inicializa la seccin de grficos */
{
	AllocVirtualScreens();

	InitWater();

	LoadFont(_font_file);

	SetGraphMode();
}
