// texture.cc

/*
   Sofie, a real time 3d engine / Copyright (C) 1997 Stephan Schiessling
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/


#include <iostream.h>
#include <procbuf.h> // this is GNU specific for usinf pipes
#include <fstream.h>
#include "texture.h"
#include "graphics.h"

///
extern Graphics graphics;

// needed to calculate colormap-cell from pixel values
#ifdef HICOLOR
double rw1=1.0/2080;
double rw2=2*rw1;
#else
double rw1=1.0/10923;
double rw2=1.0/9363;
#endif 


/// convert integer in string
String i2string(unsigned int i) {
  if (!i) return String("0");
  char buf[13];
  buf="000000000000";
  char *pos=buf;
  while (*++pos); // search the end
  do
    *--pos = i% 10+'0';
  while(i /=10);
  return String(pos);
};

///
Pixel RGB_to_Pixel(unsigned short red, unsigned short green, unsigned short blue) {
  Pixel pixel;
#ifdef HICOLOR
  // first is red
  if (red>=32768) red-=1040;
  pixel=((Pixel) (((double) red)*rw1))<<6;
  // then green
  pixel|=(Pixel) (((double) green)*rw2);
  pixel<<=5;
  // then blue
  if (blue>=32768) blue-=1040;
  pixel|=(Pixel) (((double) blue)*rw1);
#else  //for PseudoColor / 8bit
  // first is red
  pixel=((Pixel) (((double) red)*rw1))*42;
  // then green
  pixel+=(Pixel) (((double) green)*rw2)*6;
  // then blue
  pixel+=(Pixel) (((double) blue)*rw1);
#endif
  return pixel;
};



//===============================


Texture::Texture (void) { 
  width=height=0; 
  content=NULL; 
  tex=NULL;
};

// test if image file exists. If not return false.
bool Texture::is_existing (char const * filename) {
  String real_filename(filename);
  real_filename+=".ppm.gz";
  ifstream test_if_existing((const char *)real_filename);
  if (!test_if_existing.is_open()) 
    return false; // not existing  
  return true;
};

// load a compressed image file, where filename 
// is without suffix (.ppm.gz). prt decides if messages are printed or not.
// return value is true iff texture was successfully loaded.
bool Texture::load (char const * filename, bool prt=true) {

  if (!is_existing(filename)) return false;
  // This procedure is only necessary, because gzip would complain
  // about a non existing file. This looks ugly.
  
  
  // now image file exists
  if (prt)
    cout << "Loading texture " << filename;
  String command("gzip -d -c -S .ppm.gz ");
  command+=filename;
  procbuf buf((const char *)command,ios::in); // open pipe
  istream inp(&buf);
  return load (inp,filename,prt);
};

bool Texture::load (istream &inp,char const * filename=NULL, bool prt=true) {

  if (inp.eof()) {
    if (prt) cout << '\r' << "ERROR: texture " << filename << " not found!\n";
    return false;
  };
  
  char c;  
  inp.get(c);
  
  if (c != 'P') {
    if (prt) cout << '\r' << "ERROR: texture " << filename << " does not have signature P !\n";
    return false;
  };
  Pixel *start;
  Colormap * colormap=graphics.get_colormap();
  int max_val;
  inp.get(c);
  switch (c) {
  case '3':
    if (prt) cout << "   type: P3, size: ";
    inp >> width;
    inp >> height;
    if (prt) cout << width << "x" << height << " ";
    inp >> max_val;
    content=new (Pixel)[width*height];
    start=content;
    for (int h=0; h<height; h++) {
      for (int w=0; w<width; w++) {
	unsigned short red,green,blue;

	Pixel pixel;
	inp >> red;
	inp >> green;
	inp >> blue;

#ifdef HICOLOR
        // first is red
	if (red>=32768) red-=1040;
	pixel=((Pixel) (((double) red)*rw1))<<6;
	// then green
	pixel|=(Pixel) (((double) green)*rw2);
	pixel<<=5;
	// then blue
	if (blue>=32768) blue-=1040;
	pixel|=(Pixel) (((double) blue)*rw1);
#else  //for PseudoColor / 8bit
        // first is red
	pixel=((Pixel) (((double) red)*rw1))*42;
	// then green
	pixel+=(Pixel) (((double) green)*rw2)*6;
	// then blue
	pixel+=(Pixel) (((double) blue)*rw1);
#endif

	*start=pixel;
	start++;
      };
    };
    break;
  case '6':
    if (prt) cout << "   type: P6, size: ";
    inp >> width;
    inp >> height;
    if (prt) cout << width << "x" << height << " ";
    inp >> max_val;
    inp.get(c);
    content=new (Pixel)[width*height];
    start=content;
    for (int h=0; h<height; h++) 
      for (int w=0; w<width; w++) {
	unsigned char red,green,blue;
	inp.get(red);
	if (red>=32) red=red >>3;
	inp.get(green);
	if (green>=64) green=green >>2;
	inp.get(blue);
	if (blue>=32) blue=blue >>3;
	*start=(((red << 6) | green) << 5) | blue;
	start++;
      };
      break;
  default:
    if (prt) cout << '\r' << "ERROR: texture " << filename << " does not have signature P 3 or P6!\n";
    return false;
    break;
  }; // end switch
  if (prt) cout << "done.\n";

  // precalculate tex, to make texture-mapping faster
  tex=new (Pixel*)[height];
  for (int u=0; u<height; u++)
    tex[u]=content+u*width;

  return true;
};

///
void Texture::del(void) {
  if (content != NULL) {
    delete[] content;
    delete[] tex;
    width=0;
    height=0;
    content=NULL;
    tex=NULL;
  };
};


///
void Texture::draw(int x=0, int y=0) {
  Pixel * tstart=content;
  Pixel * pstart=graphics.framebuffer+y*graphics.width+x;
  int wd=width;
  int ht=height;
  if (graphics.width<wd+x) wd=graphics.width-x;
  if (graphics.height<ht+y) ht=graphics.height-y;

  for (int h=0; h<ht; h++) {
    for (int w=0; w<wd; w++) 
      *(pstart+w)=*(tstart+w);
    pstart+=graphics.width;
    tstart=tex[h];
  };
};

/********************************
 END    struct Texture
 ********************************/


/*******************************
    struct Mipmap
 ******************************/
Mipmap::Mipmap (void) { 
  nr_of_textures=0; 
};

///
void Mipmap::load (char const * filename) {
  if (nr_of_textures != 0) {
    cout << "ERROR: mipmap already loaded!\n";
    exit(-1);
  };
  cout << "Loading Mipmap " << filename << " size  ";
  cout.flush();
  String s;
  for (;;) {
    String real_filename(filename);
    real_filename+=".";
    String size=i2string(nr_of_textures);
    real_filename+=size;
    cout << size << ' ';
    cout.flush();
    if (!texture[nr_of_textures].load((const char *)real_filename,false)) break;
    s+=size;
    s+=" ";
    nr_of_textures++;
    if (nr_of_textures>MAX_NR_OF_TEXTURES) break;
  };
  if (nr_of_textures==0) { 
    cout << '\r' << "ERROR: Mipmap " << filename << " not found!\n";
    return;
  }
  else
    cout << '\r' << "Loading Mipmap " << filename << " size  " << s << " done.\n";

  // calculate scaling factors
  factor[0]=1.0;
  double den=1.0/((double) texture[0].width);
  for (int i=nr_of_textures-1; i>0; i--)
    factor[i]=((double) texture[i].width)*den;
};

///
void Mipmap::del(void) {
};
/********************************
 END    struct Mipmap
 ********************************/







