// scrman.cc

/*
   Sofie, a real time 3d engine / Copyright (C) 1997 Stephan Schiessling
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/


// in this file the class Screen_Manager will be defined
// look there to get an impression on the subject

#include "scrman.h"
#include "graphics.h"

///
extern Graphics graphics;

///
bool Interval::is_in(int p) {
  if (from>p) return false;
  if (to<p) return false;
  return true;
};

///
bool Interval::intersects (Interval &iv) {
  if (is_in( iv.from )) {
    if (! is_in( iv.to ))  iv.to=to;
    return true;
  };
  if (is_in( iv.to )) {
    iv.from=from;
    return true;
  };
  return false;
};

///
bool Interval::adapts (Interval &iv,Interval &sec) {
  if (is_in( iv.from )) {
    if (! is_in( iv.to )) {
      iv.to=to;
      to=iv.from-1;
      return true;
    };
    // (*iv) lies in (*this), therefore (*sec) should not be empty
    sec.from=iv.to+1;
    sec.to=this->to;
    to=iv.from-1;
    return true;
  };
  if (is_in( iv.to )) {
    iv.from=from;
    from=iv.to+1;
    return true;
  };
  if (iv.from<from && iv.to>to) {
    iv.from=from;
    iv.to=to;
    to=from-1;
    return true;
  };
  
  return false;
};


/***********************************************/

Row_Manager::Row_Manager(void) {
  init();
};

///
void Row_Manager::init (void) {
  nr_of_contents=0;    
  interval[0].from=0;  // row should have only free pixels.
  interval[0].to=frustrum.scr_width-1;
};

///
void Row_Manager::init (Row_Manager &rm) {
  nr_of_contents=rm.nr_of_contents;
  for (int nr=nr_of_contents; nr>=0; nr--)
    interval[nr]=rm.interval[nr];
};

///
void Row_Manager::init (Texture &t, int row, Pixel blend_out) {
  Pixel * tstart=t.content+t.width*row;
  int column=0;
  int max_column=frustrum.scr_width;
  nr_of_contents=-1;
  // searches next free pixel
init1:
  if (*tstart!=blend_out) {
  init3:
    tstart++;
    column++;
    if (column<max_column) goto init1;
    if (nr_of_contents<0) {
      nr_of_contents=-1;    
      interval[0].from=0;
      interval[0].to=-1; //scr_width-1;
    };
    return; // finished
  };
  // now tstart points to the next free pixel
  nr_of_contents++;
  interval[nr_of_contents].from=column;
init2:
  tstart++;
  column+=1;
  if (column>=max_column) {
    interval[nr_of_contents].to=column-1;
    return; // finished
  };
  if (*tstart==blend_out) goto init2;
  // now tstart points to a occupied pixel
  interval[nr_of_contents].to=column-1;
  goto init3;
};

///
bool Row_Manager::get (Interval &iv) {
  for (int i=nr_of_contents; i>=0; i--) {
    if (interval[i].adapts(iv,sec)) {
      if (interval[i].is_empty()) {
	interval[i].from=interval[nr_of_contents].from;
	interval[i].to=interval[nr_of_contents].to;
	nr_of_contents--;
      };
      // if (iv) lies in the "middle" of interval[i], then we need a new interval
      if (! sec.is_empty()) { 
	nr_of_contents++; 
#ifdef SAFE
	if (nr_of_contents>=max_intervals) {
	  cout << "ERROR! Screen Manager needs more max_intervals!\n";
	  int *schluss=NULL;
	  *schluss=0; // provoke SEGMENTATION FAULT 
	  exit(-1);
	};
#endif
	interval[nr_of_contents].from=sec.from;
	interval[nr_of_contents].to=sec.to;
	sec.to=sec.from-1; // make *sec again an empty interval
      };
      return true;
    };
  };
  return false;
};


///
bool Row_Manager::is_free (int coord) {
  for (int i=nr_of_contents; i>=0; i--) {
    if (interval[i].is_in(coord)) return true;
  };
  return false;
};



/********************************************/


Screen_Manager::Screen_Manager (Texture &t,Pixel blend_out) {
  init(t,blend_out);
};

///
void Screen_Manager::init (Texture &t,Pixel blend_out) {
  int col, row;
  if (t.width>=frustrum.scr_width) col=frustrum.scr_width;
  else col=t.width;
  if (t.height>=frustrum.scr_height) row=frustrum.scr_height;
  else row=t.height;
  for (int r=row-1; r>=0; r--)
    content[r].init(t,r,blend_out);
};

///
void Screen_Manager::init(void) {
  for (int r=frustrum.scr_height-1; r>=0; r--) 
    content[r].init();
};

///
void Screen_Manager::init(Screen_Manager &sm) {
  for (int r=frustrum.scr_height-1; r>=0; r--) 
    content[r].init(sm.content[r]);    
};

///
void Screen_Manager::clear_remaining (Pixel color=0) {
  Interval iv;
  Pixel * pstart;
  Pixel * pend;
  Pixel * buffer=graphics.framebuffer;
  iv.from=0;
  iv.to=graphics.width-1;

  for (int irow=0; irow<graphics.height; irow++) {
    Row_Manager *rm=&content[irow];
    for (int i=0; i<=rm->nr_of_contents; i++) {
      pstart=buffer+irow*graphics.width+rm->interval[i].from;
      pend=pstart+(rm->interval[i].to-rm->interval[i].from);
      
    loop:
      *pstart=color;
      pstart++;
      if (pstart<=pend) goto loop;
      iv.from=0;
      iv.to=frustrum.scr_width-1;
    }; // end for
  }; // end for
};

/***************************************/






