// polyon.h

/*
   Sofie, a real time 3d engine / Copyright (C) 1997 Stephan Schiessling
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/


#ifndef F_POLYGON
#define F_POLYGON

#include "points.h"
#include "tpointset.h"
#include "texture.h"
#include "scrman.h"
#include "globals.h"



extern bool use_mipmaps;



/// to classify the BSP tree
enum cls { COINCIDENT  = 0,
	   IN_BACK_OF  = 1,
	   IN_FRONT_OF = 2,
	   SPANNING    = 3 };


/** a plane is give in the Hesse form, ie. a plane is uniquely given
 * by its normal vector n and the minimal distance d to the origin.
 * a point P is contained in this plane iff n*P=d (* is the euklidean scalarproduct) */
class Plane {
public:
  /// the normal vector
  Point n;        
  /// minimal distance of plane to origin
  double d;            
  //-------------------------------
  /// constructor to create some plane
  Plane (void) {};
  /// constructor to create the plane with given normal and distance
  Plane(const Point &,const double);
  /// constructor to create the plane in which the three points are contained (if possible!)
  Plane(const Point &,const Point &,const Point &);
  /// tests of the point is contained in the plane
  bool is_in(const Point &) const;
  /// function to determine plane in which the 3 points lie (if possible!)
  void calc_plane(const Point &,const Point &,const Point &); 
  /// classifies the point (see cls).  
  double classify (const Point &) const;
};

#ifndef ACCURATE
void init_not_accurate_texture_mapping(void);
#endif

class Polygon;

///
struct returnvalue {
  Polygon * front; 
  Polygon * back; 
};


/// a convex (!) polygon with a color
class Polygon {
  // polygons should always have less then max_vertices-2 vertices
  // but sometimes they will be split, nr_of_vertices could increase!!!!
  public:
    const int max_vertices=20;
    /// it is clear, for a triangle this is 3
  protected:
    int nr_of_vertices;
    /** this is the list of the vertices (actually pointers to them) of the polygon.
       To make live easier the last is the same as the first to get a cycle. */
    TPoint* vertex[max_vertices];
    /// normal vector of the polygon
    TPoint * plane_n;
    /// distance of the plane in which the polygon is contained from the origin
    double plane_d;
  public:
    /// for every polygon there should be assigned a color	
    Pixel color;
    /** true iff polygon has only one side (ie only one side is visible, the other not).
       * the reason for that is simply gain of speed. It is easy to decide what side the
       * observer can see. This is known as backface sculling */
    bool backface;
#ifdef DEBUG
    /// a string as signature, helps to identify the polygon, usefull for debugging
    char * signature;  
#endif
    //--------------------------------
    /// constructor to build an empty polygon
    Polygon (void);
    /// constructor to build the polygon given by 3 vertices (a triangle)
    Polygon (Point &,Point &,Point &);
    /// constructor to build the polygon given by 4 vertices
    Polygon (Point &,Point &,Point &,Point &);
    /// constructor to build the polygon given by 5 vertices
    Polygon (Point &,Point &,Point &,Point &,Point &);
    /// constructor to build the polygon given by 6 vertices
    Polygon (Point &,Point &,Point &,Point &,Point &,Point &);
    /// given nr vertices in a field, this constructor builds the polygon with the vertices in the field
    Polygon (Point *field, int nr); 
    /// constructor to simply copy data from another polygon (vertices are copied not only their pointers) 
    Polygon (const Polygon * other);
    /// a virtual constructor
    virtual Polygon * new_Polygon(void);
    /// a virtual constructor
    virtual Polygon * new_Polygon(Point *field, int nr);
    /// a virtual constructor
    virtual Polygon * new_Polygon(const Polygon * other) const;
    ///
    virtual void copy_rest (Polygon *) const;
    /// draws the polygon to the framebuffer (but cares of the screenmanager)
    virtual void draw(View *v=NULL); // v is only used in childs
    /// pointer to first vertex of polygon in world coordinates
    Point * first_vertex_w(void); 
    /// pointer to normal of polygon in relative coordinates
    Point * normal_r(void);
    /// pointer to normal of polygon in world coordinates
    Point * normal_w(void);
    /// distance of the plane which contains polygon to origin
    double distance (void);
    /// is only defined if Polygon is a MPolygon 
    virtual void replace_mipmap_by(Mipmap &m1,Mipmap &m2);
    /// does nothing if makro DEBUG is set
    void set_signature (const char * sig);
    /// does nothing if makro DEBUG is set
    char * get_signature (void);
    /// Translation of relative coordinates, by point &
    void translate (Point &);
    ///
    cls classify (Plane &plane) const;
    /** a sphere given by a point P and radius flies with velocity vel in space. This function 
       * calculates the minimal timeinterval until the sphere collides with the polygon (may be infinity) */
    double collision_time(Point &vel,Point &P,double radius);
    /** splits polygon in at most 2 polygons, where the split is given by the plane;
       * here we use Polygon **, because front and back are return values, they are changed
       * by the methode (we want Polygon * as return value, but then Polygon * should be a reference) */
    virtual returnvalue split (const Plane &part);
    /// registrate the vertices and direction vectors of the polygon
    virtual void registrate_points(TPoint_Set &,TPoint_Set &);
  protected:
    /// the observer sees only in positive direction. This function cuts of the negative part of the polygon
    void clipping(void);
    /// calculates some constants of polygon, which are necessary for drawing
    void calc_constants (void);
};  


/// a polygon with a mipmap
class MPolygon : public Polygon {
  public:
    /// false iff mipmap instead of one color should be used
    bool usecolor;
    /// pointer to the origin of the texture
  protected:
    TPoint  * texture_origin; 
    /// pointer to the first texture coordinate
    TPoint  * texture_right;  
    /// pointer to the second texture coordinate
    TPoint  * texture_up;     
    /// pointer to the associated mipmap
    Mipmap * mipmap; 
    //------------------------------------------
  public:
    /// constructor to build an empty polygon
    MPolygon (void);
    /// constructor to build the polygon given by 3 vertices (a triangle)
    MPolygon (Point &a,Point &b,Point &c);
    /// constructor to build the polygon given by 4 vertices
    MPolygon (Point &,Point &,Point &,Point &);
    /// constructor to build the polygon given by 5 vertices
    MPolygon (Point &,Point &,Point &,Point &,Point &);
    /// constructor to build the polygon given by 6 vertices
    MPolygon (Point &,Point &,Point &,Point &,Point &,Point &);
    /// given nr vertices in a field, this constructor builds the polygon with the vertices in the field
    MPolygon (Point *field, int nr); 
    /// constructor to simply copy data from another polygon (vertices are copied not only their pointers) 
    MPolygon (const Polygon * other);
    /// a virtual constructor
    virtual Polygon * new_Polygon(void);
    /// a virtual constructor
    virtual Polygon * new_Polygon(Point *field, int nr);
    /// a virtual constructor
    virtual Polygon * new_Polygon(const Polygon * other) const;
    /// 
    virtual void copy_rest (Polygon *) const;
    /// to assign a special mipmap to the polygon, the points are the texture coordinates
    void attach_mipmap(Mipmap &,Point& ,Point& ,Point& );
    /// to change the current mipmap, but it has to be of the same size
    void attach_mipmap(Mipmap &);
    /// polygon gets the same mipmap as the other polygon has
    void attach_same_mipmap(const MPolygon *);
    /// if polygon has mipmap m1, then it will be changed to  mipmap m2 
    virtual void replace_mipmap_by(Mipmap &m1,Mipmap &m2);
    /// draws the polygon
    /// splits polygon in at most 2 polygons, where the split is given by the plane
    virtual returnvalue split (const Plane &part);
    /// registrate the vertices and direction vectors of the polygon
    virtual void registrate_points(TPoint_Set &,TPoint_Set &);
    /// draws the polygon
    virtual void draw(View *v=NULL); // v is not used
  protected:  
    /// to assign a special mipmap to the polygon, the points are the texture coordinates
    void _attach_mipmap(Mipmap &,Point& ,Point& ,Point& );
#ifndef ACCURATE
    void draw_mode_2(void); // mipmapping 
#endif
    void draw_mode_2a(void); // accurate mipmapping 
};

/// a Wrapper class for polygon, to create a list of Polygons (Polygon_List)
class W_Polygon {
public:
  /// polygon assigned to the wrapper class
  Polygon * polygon;
  // pointer to next wrapper class
  W_Polygon * next;
};

/// a list of polygons
class Polygon_List {
  public:
    W_Polygon * first;
    W_Polygon * last;
    //------------------
  public:
    /// constructor creates an empty list
    Polygon_List (void);
    /// constructor copies an other polygon list
    Polygon_List (const Polygon_List * other);
    /// destructor of polygon list
    ~Polygon_List (void);
    /// a virtual constructor
    virtual Polygon_List * new_Polygon_List(void);
    /// a virtual constructor
    virtual Polygon_List * new_Polygon_List(const Polygon_List *other);
    /// copies other polygon list into this polygon list  
    void copy (Polygon_List *);
    /// produces a copy of this object
    virtual Polygon_List * clone (void);
    /** after Polygon_List is set up, this function should be called; 
       * it needs two TPoint_Sets to registrate polygon points in case a polygon must be splited */
    virtual void build (void);
    /// true iff polygon list is empty
    bool is_empty(void) const;
    /// add polygon to polygon list. The polygon is not copied, it is directly taken!
    void add(const Polygon *);
    /// add a whole polygon list to polygon list. Here each polygon is copied.
    void add(const Polygon_List * other);
    /// gives the number of polygons contained in the polygon list
    int nr_of_polygons(void) const;
    /// change the order of the polygons randomly
    void shuffle(void);
    /// for each polygon with mipmap m1 change mipmap by m2 
    void replace_mipmap_by(Mipmap &m1,Mipmap &m2);
    void print (void);
    /// draws each polygon of the polygon list
    virtual void draw(View *v=NULL); // v is not used here
    /// draws each polygon of the polygon list
    //virtual void merge_draw(View *v=NULL,); // v is not used here
    /** a sphere given by a point P and radius flies with velocity vel in space. This function 
       * calculates the minimal timeinterval  until the sphere collides with a polygon of the 
       * polygon list (may be infinity) */
    double collision_time(Point &vel,Point &P,double radius);
    /// registrate the vertices and direction vectors of the each polygon in Polygon_List
    virtual void registrate_points(TPoint_Set &,TPoint_Set &);
  protected:
    /// removes W_Polygon from List, but does not delete Polygon which W_Polygon represents
    void remove(W_Polygon *);
};


#endif F_POLYGON





