// polygon.cc

/*
   Sofie, a real time 3d engine / Copyright (C) 1997 Stephan Schiessling
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/


#include "polygon.h"
#include <math.h>
#include <stream.h>
#include <String.h>
#include <stdlib.h>
#include "graphics.h"


extern Graphics graphics;  
extern Screen_Manager screen_manager;


#ifndef ACCURATE
double rez[20000];
double *prez;
void init_not_accurate_texture_mapping(void) {
  double c=-9999;
  for (int j=0; j<20000; j++) {
    if (c!=0) rez[j]=1/c;
    else rez[j]=INFINITY;
    c+=1.0;
  };
};
#endif

/// some global variables for draw_mode_
static Interval iv,iv_sp;
static double denom;
static double nenner;
static double add_nenner;
static double rez_add_nenner;
static double nenner_glob;
static double zu;
static double zv;
static double from;
static double to;
static double c;
static double cu;
static double cv;
static double dsp;
static double dsp_scale;
static double cu1;
static double cv1;
static double factor;
static Pixel * pstart;
static Pixel * pend;
static Pixel * buffer;
static Pixel ** tex;
static Texture * texture;
static int tw,th;

// some constants
static double max_column;
static double min_column;
static double max_row;
static double min_row;

static TPoint sp1; 
static TPoint sp2;

static Point sp3;  
static Point sp4; 

static double rows[Polygon::max_vertices];       
static TPoint* vertex_ac[Polygon::max_vertices]; 
static int nr_of_vertices_ac;

// column gives the column-plane equation:
// nc Q=dc    where dc=0 and nc=(column,dvp,0)
// calculate   P=P0-(nc*P0)/(nc*(P1-P0)) *(P1-P0)
/// = intersectionpoint (=ip) P of the column-plane with edge P0-P1
void calc_ip(Point &P,double &column,Point &p0,Point &p1) {
  double x,y,lambda;
  lambda=(column*p0.x-frustrum.distance_to_vp*p0.y)/(column*(p1.x-p0.x)-frustrum.distance_to_vp*(p1.y-p0.y));
  P.x=p0.x-lambda*(p1.x-p0.x);
  P.y=p0.y-lambda*(p1.y-p0.y);
  P.z=p0.z-lambda*(p1.z-p0.z);
};

/// true  iff  the slice is not empty
bool get_slice(double row) {
  int i=0;
  if (row<min_row) return false;
  if (row>max_row) return false;

  if (rows[0]<=row) {
  nxt0:
    i++;
    if (rows[i]<row) goto nxt0;
    // edge with intersectionpoint found
    calc_ip(sp3,row,vertex_ac[i-1]->v,vertex_ac[i]->v);
  nxt1:
    i++;
    if (rows[i]>row) goto nxt1;
    // another edge with intersectionpoint found
    calc_ip(sp4,row,vertex_ac[i-1]->v,vertex_ac[i]->v);
    return true;
  };
nxt2:
  i++;
  if (rows[i]>row) goto nxt2;
  // edge with intersectionpoint found
  calc_ip(sp3,row,vertex_ac[i-1]->v,vertex_ac[i]->v);
nxt3:
  i++;
  if (rows[i]<row) goto nxt3;
  // another edge with intersectionpoint found
  calc_ip(sp4,row,vertex_ac[i-1]->v,vertex_ac[i]->v);
  return true;
};

//=============================================

///
Plane::Plane(const Point &P,const double a) {
  n.init(P);  
  d=a; 
};

Plane::Plane(const Point& a,const Point& b,const Point& c) {
  calc_plane(a,b,c);
};

///
bool Plane::is_in(const Point & P) const {
  double sp=P*n-d;
  if (IS_ZERO(sp))  return true;
  return false;
}; 

///
void Plane::calc_plane(const Point& a,const Point& b,const Point& c) {
  Point D,E;
  D=b-a;
  E=c-a;
  VECTOR_PRODUCT(n,D,E);
  n.normalize();
  d=n*a;    
};

///
double Plane::classify (const Point &P) const {
  double sp=n*P-d;
  if (IS_ZERO(sp)) return 0; // because of rounding problem
  return sp;
};

//==================================================

///
Polygon::Polygon (void) {
  for (int i=0; i<max_vertices;i++) vertex[i]=NULL;
  nr_of_vertices=0;
  color=0;
  backface=true;
#ifdef DEBUG
  signature=NULL;
#endif
};

///
Polygon::Polygon (Point &pa,Point &pb,Point &pc) {
  if ((pa==pb) || (pa==pc) || (pb==pc)) {
    cout << "ERROR: 2 vertices of the polygon are equal!\n";
    exit(-1);
  };
  Plane plane(pa,pb,pc);
  plane_n=new TPoint(plane.n);
  plane_d=plane.d;
  vertex[0]=new TPoint(pa); 
  vertex[1]=new TPoint(pb); 
  vertex[2]=new TPoint(pc); 
  vertex[3]=vertex[0]; // one vertex more to get a cycle
  nr_of_vertices=3; // 0..2
  color=0;
  backface=true;
#ifdef DEBUG
  signature=NULL;
#endif
};

// DOIT! the edges need not cross!
/// DOIT! check if there are 3 points on a line
Polygon::Polygon (Point &pa,Point &pb,Point &pc,Point &pd) {
  if ((pa==pb) || (pa==pc) || (pa==pd) || (pb==pc) || (pb==pd) || (pc==pd)) {
    cout <<   "ERROR: 2 vertices of the polygon are equal!\n";
    exit(-1);
  };
  Plane plane(pa,pb,pc);
  plane_n=new TPoint(plane.n);
  plane_d=plane.d;
  if (!plane.is_in(pd)) {
    cout << "ERROR: the 4 vertices are not in one plane!\n";
    exit(-1);
  };
  vertex[0]=new TPoint(pa); 
  vertex[1]=new TPoint(pb); 
  vertex[2]=new TPoint(pc); 
  vertex[3]=new TPoint(pd);
  vertex[4]=vertex[0];
  nr_of_vertices=4; // 0..3
  color=0;
  backface=true;
#ifdef DEBUG
  signature=NULL;
#endif
};

///
Polygon::Polygon (Point &pa,Point &pb,Point &pc,Point &pd,Point &pe) {
  Plane plane(pa,pb,pc);
  plane_n=new TPoint(plane.n);
  plane_d=plane.d;
  vertex[0]=new TPoint(pa); 
  vertex[1]=new TPoint(pb); 
  vertex[2]=new TPoint(pc); 
  vertex[3]=new TPoint(pd);
  vertex[4]=new TPoint(pe);
  vertex[5]=vertex[0];
  nr_of_vertices=5; // 0..4
  color=0;
  backface=true;
#ifdef DEBUG
  signature=NULL;
#endif
};

///
Polygon::Polygon (Point &pa,Point &pb,Point &pc,Point &pd,Point &pe,Point &pf) {
  Plane plane(pa,pb,pc);
  plane_n=new TPoint(plane.n);
  plane_d=plane.d;
  vertex[0]=new TPoint(pa); 
  vertex[1]=new TPoint(pb); 
  vertex[2]=new TPoint(pc); 
  vertex[3]=new TPoint(pd);
  vertex[4]=new TPoint(pe);
  vertex[5]=new TPoint(pf);
  vertex[6]=vertex[0];
  nr_of_vertices=6; // 0..5
  color=0;
  backface=true;
#ifdef DEBUG
  signature=NULL;
#endif
};


/// used to split a polygon (necessary for construction of BSP tree)
Polygon::Polygon (Point *field, int nr) {
  if (nr<3) {
    cout << "ERROR: a polygon is defined at least by 3 points!\n";
    exit(-1);
  };
  Plane plane(field[0],field[1],field[2]);
  plane_n=new TPoint(plane.n);
  plane_d=plane.d;
  for (int i=nr-1; i>=0; i--)
    vertex[i]=new TPoint(field[i]);
  vertex[nr]=vertex[0];
  nr_of_vertices=nr; 
  color=0;
  backface=true;
#ifdef DEBUG
  signature=NULL;
#endif
};

///
Polygon::Polygon (const Polygon * other) {
  nr_of_vertices=other->nr_of_vertices;
  for (int i=0; i<nr_of_vertices; i++) { 
    vertex[i]=new TPoint(*other->vertex[i]);
  };
  vertex[nr_of_vertices]=vertex[0]; // the last vertex is also the first (cyclic stuff)
  plane_n=new TPoint(*other->plane_n);
  plane_d=other->plane_d;
  color=other->color;
  backface=other->backface;
#ifdef DEBUG
  signature=NULL;
  set_signature(other->signature);
#endif
};


Polygon * Polygon::new_Polygon (void) {
  return new Polygon ();
};

Polygon * Polygon::new_Polygon (Point *field, int nr) {
  return new Polygon (field,nr);
};

Polygon * Polygon::new_Polygon (const Polygon * other) const {
  return new Polygon (other);
};


void Polygon::copy_rest (Polygon * t) const {};

///
void Polygon::set_signature(const char *sig) {
#ifdef DEBUG
  if (signature!=NULL) {
    cout << "ERROR: signature of this polygon is already assigned!\n";
    exit (-1);
  };
  if (sig == NULL) {
    signature=NULL;
    return;
  };
  int count=0;
  while (sig[count]!=0) {count++;};
  count++; // with 0-Ending of string
  signature=new char[count];
  count=0;
  while (sig[count]!=0) {signature[count]=sig[count]; count++;};  
  signature[count]=0;
  cout << "remark: polygon received signature " << sig << "\n";
#endif
};

///
char * Polygon::get_signature (void) {
#ifdef DEBUG
  return signature;
#else 
  return NULL;
#endif
};

///
void Polygon::draw(View *v=NULL) { // v is not used
  
  // some preliminary functions
  clipping();
  calc_constants();
  
  // backface removal
  if (backface) 
    if (plane_d >= 0) return;
  if (nr_of_vertices_ac==0) return; // Polygon is behind the viewer
  
  buffer=graphics.framebuffer;
  Pixel col;

  double min_r=(double) ((int) min_row);
  double max_r=(double) ((int) max_row);


  if (color==0) col=rand() %32*2048; 
  else {
    // darken the polygon if it is far away
    register double red=(double) (color>>11);
    register double green=(double) ((color>>5) & 0x3f);
    register double blue=(double) (color & 0x1f);
    register double distance=vertex[0]->v.x*vertex[0]->v.x+vertex[0]->v.y*vertex[0]->v.y+vertex[0]->v.z*vertex[0]->v.z;
    distance=8000.0/distance;
    if (distance > 1.0) distance=1.0;
    red*=distance;
    green*=distance;
    blue*=distance;
    col=((      
	  (((Pixel) red)<<6)     |     ((Pixel) green)       
	  ) <<5)         |       ((Pixel) blue); 
  };

  for (double row=min_r; row<=max_r; row+=1) {
    int irow=(int) row+frustrum.scr_half_height;
    if (get_slice(row)) {
      
      if (sp3.x!=0)
	from=frustrum.distance_to_vp*sp3.z/sp3.x;
      else from=INFINITY*sp3.z;
      if (sp4.x!=0)
	to=frustrum.distance_to_vp*sp4.z/sp4.x;
      else to=INFINITY*sp4.z;
      
      // to must be greater or equal iv.from 
      if (to<from) { // swap from and to 
	double sp=to;
	to=from;
	from=sp;
      };
      iv.from=((int) ((from+frustrum.scr_half_width-1)*frustrum.vh_scale));
      iv.to=((int) ((to+frustrum.scr_half_width-1)*frustrum.vh_scale));
      iv_sp=iv;

      while (screen_manager.content[irow].get(iv)) {

	from=((double) iv.from)*frustrum.rez_vh_scale-frustrum.scr_half_width+1;
	  //	  if (from<-frustrum.scr_half_height+1) from=frustrum.scr_half_height+1;
	  //	  if (from>frustrum.scr_half_height) from=frustrum.scr_half_height;
	
	to=((double) iv.to)*frustrum.rez_vh_scale-frustrum.scr_half_width+1;
	//	  if (to<-frustrum.scr_half_height+1) to=frustrum.scr_half_height+1;
	//	  if (to>frustrum.scr_half_height) to=frustrum.scr_half_height;
      pstart=buffer+irow*frustrum.scr_width+iv.from;
      pend=pstart+(iv.to-iv.from);
      register co=col;

  print_pixel0:
      *pstart=co;
      //==   *pstart=*(texture->content+u*texture->width+v);
      pstart++;
      if (pstart<=pend) goto print_pixel0;
 
      iv=iv_sp;
      };
    }; 
  };
}; // end of Polygon::draw

/*
void Polygon::to_world(View &view) {
  for (int i=nr_of_vertices-1; i>=0; i--)
    vertex[i]->to_world(view);
  plane_n->to_world(view);
  if (mipmap != NULL) { // if there is no mipmap, then there are also no mipmap coordinates
    texture_origin->to_world(view);
    texture_up->to_world(view);
    texture_right->to_world(view);
  };
};

void Polygon::to_view(View &view) {
  for (int i=nr_of_vertices-1; i>=0; i--)
    vertex[i]->to_view(view);
  plane_n->to_view(view);
  if (mipmap != NULL) { // mipmap coordinates are only necessary if there is a mipmap
    texture_origin->to_view(view);
    texture_up->to_view(view);
    texture_right->to_view(view);
  };
};
*/

// intersects the polygon with the region (x>=0)
// the variables with ending _ac will be calculated (ac=after clipping)
// it is obvious that the clipped polygon has at least one more vertex
// and at most 2 new different vertices (therefore we need sp1 and sp2)
// more explicit:
// if the first vertex is contained in (x<0), then find the next
// vertex in (x>=0). If there is no such vertex, then let nr_of_vertices_ac=0
// and finish. Otherwise calculate the intersection point of the edge
// (first edge in (x>=0) to their previous vertex) with the viewers-plane (x==0).
// Put a pointer of this intersection point in the array vertex_ac.
// Then put all following vertices in (x>=0) in the array vertex_ac.
// Then calculate the intersection point of the edge (next vertex in (x<0) with
// previous vertex) with the viewers-plane (x==0). Put this intersection point
// again in verttex_ac. 
// Finally put vertex_ac[0] again in vertex_ac (make vertices cyclic --
// algorithms will be easier with that trick)  
/// The case first vertex in (x>=0) is analogous....

void Polygon::clipping(void) {
  double lambda;
  nr_of_vertices_ac=0;
  int i=0;

  if (vertex[0]->v.x<0) { 
    // if the first vertex is not visible, then
    // step over all next vertices in (x<0)
    for (;;) {
      i++;
      if (i>=nr_of_vertices) return;  // therefore the polygon is behind the viewer
      if (vertex[i]->v.x>=0) break;
    };
      
    // now the vertex is again in (x>=0)
    // calculate intersection point of the edge with (x==0) 
    // This is shall be the first vertex of the polygon
    if (vertex[i]->v.x>0) { 
      lambda=vertex[i-1]->v.x/(vertex[i]->v.x-vertex[i-1]->v.x);       
      sp1.v.x=0; 
      sp1.v.y=vertex[i-1]->v.y-lambda*(vertex[i]->v.y-vertex[i-1]->v.y);
      sp1.v.z=vertex[i-1]->v.z-lambda*(vertex[i]->v.z-vertex[i-1]->v.z);
      vertex_ac[nr_of_vertices_ac]=&sp1;
      nr_of_vertices_ac++;
    };
    // the case vertex[i]==0 will be done at the same time
    for (;;) {
      vertex_ac[nr_of_vertices_ac]=vertex[i];
      nr_of_vertices_ac++;
      i++;
      if (vertex[i]->v.x<0) break; // vertex[nr_of_vertices]->vx<0  !
    };
    // calculate intersection point of the edge with (x==0)
    lambda=vertex[i-1]->v.x/(vertex[i]->v.x-vertex[i-1]->v.x);      
    sp2.v.x=0;
    sp2.v.y=vertex[i-1]->v.y-lambda*(vertex[i]->v.y-vertex[i-1]->v.y);
    sp2.v.z=vertex[i-1]->v.z-lambda*(vertex[i]->v.z-vertex[i-1]->v.z);
    vertex_ac[nr_of_vertices_ac]=&sp2;
    nr_of_vertices_ac++;   
    vertex_ac[nr_of_vertices_ac]=vertex_ac[0];
    return;
  }; // end of the case that the first vertex is not visible
  // begin of the case that the first vertex is in (x>=0) 
  for (;;) {
    vertex_ac[nr_of_vertices_ac]=vertex[i];
    nr_of_vertices_ac++;
    i++;
    if (i>=nr_of_vertices) {
      vertex_ac[nr_of_vertices_ac]=vertex_ac[0];
      return; // polygon is fully contained in (x>=0)
    };
    if (vertex[i]->v.x<0) break; 
  };
  // there exists an intersection with (x<0)
  // calculate intersection point of the edge with (x==0)
  lambda=vertex[i-1]->v.x/(vertex[i]->v.x-vertex[i-1]->v.x);      
  sp1.v.x=0;
  sp1.v.y=vertex[i-1]->v.y-lambda*(vertex[i]->v.y-vertex[i-1]->v.y);
  sp1.v.z=vertex[i-1]->v.z-lambda*(vertex[i]->v.z-vertex[i-1]->v.z);
  vertex_ac[nr_of_vertices_ac]=&sp1;
  nr_of_vertices_ac++;   
  
  // step over all next vertices in (x<0)
  for (;;) {
    i++;
    if (vertex[i]->v.x>=0) break; // vertex[nr_of_vertices]>=0   !
  };

  // vertex is again in (x>=0)
  // calculate intersection point of the edge with (x==0)
  lambda=vertex[i-1]->v.x/(vertex[i]->v.x-vertex[i-1]->v.x);      
  sp2.v.x=0;
  sp2.v.y=vertex[i-1]->v.y-lambda*(vertex[i]->v.y-vertex[i-1]->v.y);
  sp2.v.z=vertex[i-1]->v.z-lambda*(vertex[i]->v.z-vertex[i-1]->v.z);
  vertex_ac[nr_of_vertices_ac]=&sp2;
  nr_of_vertices_ac++;   

  for (;;) {
    vertex_ac[nr_of_vertices_ac]=vertex[i];
    nr_of_vertices_ac++;
    i++;
    if (i>=nr_of_vertices) {
      vertex_ac[nr_of_vertices_ac]=vertex_ac[0];
      return; // polygon is fully contained in (x>=0)
    };
  };

};

///
void Polygon::replace_mipmap_by (Mipmap &t1,Mipmap &t2) {
};

///
returnvalue Polygon::split (const Plane &part) {
  returnvalue pieces;
  int count = nr_of_vertices;
  int out_c = 0;
  int in_c = 0;
  Point ptA,ptB,outpts[max_vertices],inpts[max_vertices];
  double sideA,sideB;
  ptA.init(vertex[count-1]->r);
  sideA=part.classify(ptA);
  for (short i=-1;++i<count;) {
    ptB.init(vertex[i]->r);
    sideB=part.classify(ptB);
    if (sideB>0) {
      if (sideA<0) {
	Point sp;
	sp=ptB-ptA;
	double sect=-part.classify(ptA) / (part.n*sp);
	inpts[in_c]=outpts[out_c]=ptA+sect*sp;
	in_c++;
	out_c++;
      };
      outpts[out_c++].init(ptB);
    }
    else if (sideB<0) {
      if (sideA>0) {
	Point sp;
	sp=ptB-ptA;
	double sect=-part.classify(ptA) / (part.n*sp);
	inpts[in_c]=outpts[out_c]=ptA+sect*sp;
	in_c++;
	out_c++;
      };
      inpts[in_c++].init(ptB);
    }
    else {
      inpts[in_c]=outpts[out_c]=ptB;
      in_c++;
      out_c++;
    };
    ptA.init(ptB);
    sideA=sideB;
  };
  pieces.front=new_Polygon (outpts,out_c); // virtual constructor used
  pieces.front->color=color;
  pieces.front->backface=backface;
#ifdef DEBUG
  if (signature==NULL) {
    signature="-";
  };
  String sig(signature);
  sig+=".split";
  pieces.front->set_signature((const char *)sig);
#endif
  pieces.back=new_Polygon (inpts,in_c); // virtual constructor used
  pieces.back->color=color;
  pieces.back->backface=backface;
#ifdef DEBUG
  pieces.back->set_signature((const char *)sig);
#endif

  return pieces;
};    


// calculates:
// distance of the plane which is defined by the polygon to the origin zero (view coordinate system)
// min_column,max_column,min_row,max_row,
// (influenced by the screen coordinates)
// If the polygon is not visible, then the function sets
/// nr_of_vertices_ac=0. (draw does not paint such polygons)
void Polygon::calc_constants(void) {
// the constant plane_d
  plane_d=plane_n->v*vertex[0]->v;

  if (nr_of_vertices_ac==0) return;  // can forget polygons behind the viewer 

  // calculate columns and min_column,max_column 
  min_column=INFINITY;
  max_column=-INFINITY;
  min_row=INFINITY;
  max_row=-INFINITY;
  for (int i=nr_of_vertices_ac-1; i>=0; i--) {
    double mc;
    if (vertex_ac[i]->v.x!=0) { 
      double sp=frustrum.distance_to_vp/vertex_ac[i]->v.x;
      rows[i]=sp*vertex_ac[i]->v.y;
      mc=sp*vertex_ac[i]->v.z;
    }
    else {
      rows[i]=vertex_ac[i]->v.y*INFINITY;
      mc=vertex_ac[i]->v.z*INFINITY;
    };
    if (min_row>rows[i]) min_row=rows[i];
    if (max_row<rows[i]) max_row=rows[i]; 
    if (mc>max_column) max_column=mc;
    if (mc<min_column) min_column=mc;
  };
  rows[nr_of_vertices_ac]=rows[0];
  // if the polygon is not visible, then forget this polygon
  if (min_column>=frustrum.scr_half_width || max_column<-frustrum.scr_half_width) nr_of_vertices_ac=0;
  if (min_column<-frustrum.scr_half_width) min_column=-frustrum.scr_half_width;
  if (max_column>=frustrum.scr_half_width) max_column=frustrum.scr_half_width-1;

  if (min_row>=frustrum.scr_half_height || max_row<-frustrum.scr_half_height) nr_of_vertices_ac=0;
  if (min_row<-frustrum.scr_half_height) min_row=-frustrum.scr_half_height;
  if (max_row>=frustrum.scr_half_height) max_row=frustrum.scr_half_height-1;
};

///
void Polygon::translate (Point &t) {
  for (int i=nr_of_vertices-1; i>=0; i--)
    vertex[i]->r=vertex[i]->r+t;
  //plane_n does not change;
  plane_d=(plane_n->r)*(vertex[0]->r);
};

///
cls Polygon::classify (Plane &plane) const {
  bool coincident=true;
  bool in_front_of=false;
  bool in_back_of=false;
  register double ref;
  for (int i=nr_of_vertices-1; i>=0; i--) {
    ref=plane.classify(vertex[i]->r);
    if (ref!=0) coincident=false;
    if (ref<0) in_back_of=true;
    if (ref>0) in_front_of=true;
  };
  if (in_back_of & in_front_of) return SPANNING;
  if (coincident) return COINCIDENT;
  if (in_front_of) return IN_FRONT_OF;
  return IN_BACK_OF;
};

/// plane_n and plane_d need to be calculated before  
double Polygon::collision_time(Point &vel,Point &P,double radius) {
  plane_d=plane_n->w*vertex[0]->w;
  double sc=plane_n->w*vel;
  if (IS_ZERO(sc)) return INFINITY; // no collision, because moving is parallel to wall
  double denom=1.0/sc;
  double dist;
  double scal=plane_n->w*P;
  if (scal<plane_d)
    dist=(plane_d+radius-scal)*denom;
  else
    dist=(plane_d-radius-scal)*denom;
  //if (1.0<=fabs(dist)) return INFINITY; // it is not really INFINITY, but ...
  double t=(plane_d-scal)*denom;
  // P=(0,0,0), P'=(px,py,pz)
  Point p;
  p=P+t*vel;
  int i=0;
  {
  next_loop:
    Point sp;
    sp=(vertex[i+1]->w-vertex[i]->w);
    Point n;
    n.x=sp.y*plane_n->w.z-sp.z*plane_n->w.y;
    n.y=sp.z*plane_n->w.x-sp.x*plane_n->w.z;
    n.z=sp.x*plane_n->w.y-sp.y*plane_n->w.x;
    // normalize
    n.normalize();
    // lambda=distance from P' to the edge (Q_i+1 - Q_i)
    double lambda=n*(p-vertex[i]->w);
    if (lambda>radius) return INFINITY;
    i++;
    if (i<nr_of_vertices) goto next_loop;
  };
  return dist;
};

void Polygon::registrate_points (TPoint_Set &t, TPoint_Set &dir_t) {
  TPoint *sp;
  for (int i=0; i< nr_of_vertices; i++) {
    // if address of vertex is already in t 
    // (polygon could be registrated before)
    // then don't delete, therefore manages() is needed
    if (!t.manages(vertex[i])) {
      sp=vertex[i];
      vertex[i]=t.add(sp);
      delete sp; // get rid of old TPoint
    };
  };
  vertex[nr_of_vertices]=vertex[0]; // to make a cycle
  sp=plane_n;
  plane_n=dir_t.add(plane_n);
  delete sp; // get rid of old TPoint
};


//-------------------------------------------------

MPolygon::MPolygon (void) : Polygon() { 
  usecolor=true;
  texture_origin=NULL;
  texture_right=NULL;
  texture_up=NULL;
  mipmap=NULL;
};

MPolygon::MPolygon (Point &pa,Point &pb,Point &pc) : Polygon(pa,pb,pc)  {
  usecolor=true;
  texture_origin=NULL;
  texture_right=NULL;
  texture_up=NULL;
  mipmap=NULL;
};

MPolygon::MPolygon (Point &pa,Point &pb,Point &pc,Point &pd) : Polygon (pa,pb,pc,pd) {
  usecolor=true;
  texture_origin=NULL;
  texture_right=NULL;
  texture_up=NULL;
  mipmap=NULL;
};

MPolygon::MPolygon (Point &pa,Point &pb,Point &pc,Point &pd,Point &pe) 
  : Polygon (pa,pb,pc,pd,pe) {
  usecolor=true;
  texture_origin=NULL;
  texture_right=NULL;
  texture_up=NULL;
  mipmap=NULL;
};

MPolygon::MPolygon (Point &pa,Point &pb,Point &pc,Point &pd,Point &pe,Point &pf) 
  : Polygon (pa,pb,pc,pd,pe,pf) {
  usecolor=true;
  texture_origin=NULL;
  texture_right=NULL;
  texture_up=NULL;
  mipmap=NULL;
};

MPolygon::MPolygon (Point *field, int nr) : Polygon (field,nr) {
  usecolor=true;
  texture_origin=NULL;
  texture_right=NULL;
  texture_up=NULL;
  mipmap=NULL;
};


MPolygon::MPolygon (const Polygon * other) : Polygon(other) {
  usecolor=true;
  texture_origin=NULL;
  texture_right=NULL;
  texture_up=NULL;
  mipmap=NULL;
  other->copy_rest(this);
};

/*
MPolygon::MPolygon (const MPolygon * other) : Polygon(other) {
  usecolor=other->usecolor;
  mipmap=NULL; // suffices to indicate that there was no mipmap before
  attach_same_mipmap(other);
};
*/

Polygon * MPolygon::new_Polygon (void) {
  return new MPolygon ();
};

Polygon * MPolygon::new_Polygon (Point *field, int nr) {
  return new MPolygon (field,nr);
};

Polygon * MPolygon::new_Polygon (const Polygon * other) const {
  return new MPolygon (other);
};


void MPolygon::copy_rest (Polygon * t) const {
  MPolygon *mp=(MPolygon *) t;
  mp->usecolor=usecolor;
  mp->mipmap=NULL; // suffices to indicate that there was no mipmap before
  mp->attach_same_mipmap(this);
};

/// 
void MPolygon::attach_mipmap(Mipmap &mip,Point &O,Point &R,Point &U) {
  _attach_mipmap(mip,O,R,U);
  texture_right->r.as_texture_coord();
  texture_up->r.as_texture_coord();  
};

/// 
void MPolygon::_attach_mipmap(Mipmap &mip,Point &O,Point &R,Point &U) {
  if (&mip==NULL) { // attach no mipmap
    mipmap=NULL;
    return;
  };
  if (mipmap==NULL) { // there was no mipmap attached before
    texture_origin=new TPoint(O);
    texture_right=new TPoint(R);
    texture_up=new TPoint(U);
  }
  else {
    texture_origin->init(O);
    texture_right->init(R);
    texture_up->init(U);
  };
  mipmap=&mip;
  if (mipmap != NULL) usecolor=false; 
};

///
void MPolygon::attach_mipmap(Mipmap &mip) {
  if (mipmap==NULL) { // there was no mipmap attached before
    cout << "ERROR: there was no texture attached before!\n";
    exit(-1);
  };
  mipmap=&mip;
  if (mipmap != NULL) usecolor=false; 
};

///
void MPolygon::attach_same_mipmap(const MPolygon *w) {
  _attach_mipmap(*w->mipmap,w->texture_origin->r,w->texture_right->r,w->texture_up->r);
};

///
void MPolygon::replace_mipmap_by (Mipmap &t1,Mipmap &t2) {
  if (mipmap==&t1) mipmap=&t2;
};



#ifndef ACCURATE
///
void MPolygon::draw_mode_2(void) {

  buffer=graphics.framebuffer;
  

  // for mipmapping
  if ((mipmap->nr_of_textures>1) & (use_mipmaps)) {
    Point D;
    double lenr=1.0/texture_right->v.length();
    D=(lenr*lenr)*texture_right->v; 
    double f1=frustrum.distance_to_vp/(vertex[0]->v.x+D.x);
    double f2=frustrum.distance_to_vp/(vertex[0]->v.x);
    double my=f1*(vertex[0]->v.y+D.y)-f2*vertex[0]->v.y;
    double mz=f1*(vertex[0]->v.z+D.z)-f2*vertex[0]->v.z;
    
    double mip=my*my+mz*mz;
    
    int mipstep=0;
    if (mip <= 0.25) 
      mipstep=1;
    if (mip <= 0.0625)
      mipstep=2;
    if (mip <= 0.015625)
      mipstep=3;
    if (mip <= 0.00390625)
      mipstep=4;
    texture=&mipmap->texture[mipstep]; // set right mipmap
    factor=mipmap->factor[mipstep];
  } else  { // polygon has only 1 texture, so no mipmap calculation
    texture=&mipmap->texture[0];
    factor=1.0;
  };
  
  

  tex=texture->tex;
  tw=texture->width-1;
  th=texture->height-1;


  // calculate some constants for the polygon
  nenner_glob=plane_n->v.x*frustrum.distance_to_vp;
  // 2 constants to put the texture on the polygon
  cu=((texture_origin->v)*(texture_right->v))*factor;
  cv=((texture_origin->v)*(texture_up->v))*factor;
  add_nenner=plane_n->v.z*frustrum.rez_vh_scale;
  rez_add_nenner=1.0/add_nenner;
  dsp=plane_d*rez_add_nenner;
  dsp_scale=dsp*frustrum.rez_vh_scale;
  cu1=dsp_scale*texture_right->v.z*factor;
  cv1=dsp_scale*texture_up->v.z*factor;



  double min_r=(double) ((int) min_row);
  double max_r=(double) ((int) max_row);


  for (double row=min_r; row<=max_r; row+=1) {
    int irow=(int) row+frustrum.scr_half_height;
    if (get_slice(row)) {
      
      if (sp3.x!=0)
	from=frustrum.distance_to_vp*sp3.z/sp3.x;
      else from=INFINITY*sp3.z;
      if (sp4.x!=0)
	to=frustrum.distance_to_vp*sp4.z/sp4.x;
      else to=INFINITY*sp4.z;
      
      // to must be greater or equal iv.from 
      if (to<from) { // swap from and to 
	double sp=to;
	to=from;
	from=sp;
      };
      iv.from=((int) ((from+frustrum.scr_half_width-1)*frustrum.vh_scale));
      iv.to=((int) ((to+frustrum.scr_half_width-1)*frustrum.vh_scale));
      iv_sp=iv;

      while (screen_manager.content[irow].get(iv)) {

	from=((double) iv.from)*frustrum.rez_vh_scale-frustrum.scr_half_width+1;
	  //	  if (from<-frustrum.scr_half_height+1) from=frustrum.scr_half_height+1;
	  //	  if (from>frustrum.scr_half_height) from=frustrum.scr_half_height;
	
	to=((double) iv.to)*frustrum.rez_vh_scale-frustrum.scr_half_width+1;
	//	  if (to<-frustrum.scr_half_height+1) to=frustrum.scr_half_height+1;
	//	  if (to>frustrum.scr_half_height) to=frustrum.scr_half_height;
	
	nenner=nenner_glob+plane_n->v.y*row+plane_n->v.z*from;

	//c=nenner*rez_add_nenner;
	prez=&rez[(int) (nenner*rez_add_nenner)+9999];

	zu=dsp*
	  (frustrum.distance_to_vp*texture_right->v.x+row*texture_right->v.y+
	   from*texture_right->v.z)*factor;

	zv=dsp*
	  (frustrum.distance_to_vp*texture_up->v.x+row*texture_up->v.y+
	   from*texture_up->v.z)*factor;
    
  
    	
      pstart=buffer+irow*frustrum.scr_width+iv.from;
      pend=pstart+(iv.to-iv.from);

      goto print_pixel2;		 

  special2:
      zu+=cu1;
      zv+=cv1;
      //c+=1.0;
      prez++;
      
  print_pixel2:
      //denom=1.0/c;
      denom=*prez;
      register int u=(int) (zu*denom-cu) & tw;     
      register int v=(int) (zv*denom-cv) & th;
      *pstart=*(tex[u]+v);
      //==   *pstart=*(texture->content+u*texture->width+v);
      pstart++;
      if (pstart<=pend) goto special2;
 
      iv=iv_sp;
      };
    }; 
  };
}; // end of Polygon::draw_mode_2
#endif


/// draw_mode_2 but accurate
void MPolygon::draw_mode_2a(void) {
  
  buffer=graphics.framebuffer;

  // for mipmapping
  if ((mipmap->nr_of_textures>1) & (use_mipmaps)) {
    Point D;
    double lenr=1.0/texture_right->v.length();
    D=(lenr*lenr)*texture_right->v; 
    double f1=frustrum.distance_to_vp/(vertex[0]->v.x+D.x);
    double f2=frustrum.distance_to_vp/(vertex[0]->v.x);
    double my=f1*(vertex[0]->v.y+D.y)-f2*vertex[0]->v.y;
    double mz=f1*(vertex[0]->v.z+D.z)-f2*vertex[0]->v.z;
    
    double mip=my*my+mz*mz;
    
    int mipstep=0;
    if (mip <= 0.25) 
      mipstep=1;
    if (mip <= 0.0625)
      mipstep=2;
    if (mip <= 0.015625)
      mipstep=3;
    if (mip <= 0.00390625)
      mipstep=4;
    texture=&mipmap->texture[mipstep]; // set right mipmap
    factor=mipmap->factor[mipstep];
  } else  { // polygon has only 1 texture, so no mipmap calculation
    texture=&mipmap->texture[0];
    factor=1.0;
  };
    
  tex=texture->tex;
  tw=texture->width-1;
  th=texture->height-1;


  // calculate some constants for the polygon
  nenner_glob=plane_n->v.x*frustrum.distance_to_vp;
  // 2 constants to put the texture on the polygon
  cu=((texture_origin->v)*(texture_right->v))*factor;
  cv=((texture_origin->v)*(texture_up->v))*factor;
  add_nenner=plane_n->v.z*frustrum.rez_vh_scale;
  rez_add_nenner=1.0/add_nenner;
  dsp=plane_d*rez_add_nenner;
  dsp_scale=dsp*frustrum.rez_vh_scale;
  cu1=dsp_scale*texture_right->v.z*factor;
  cv1=dsp_scale*texture_up->v.z*factor;



  double min_r=(double) ((int) min_row);
  double max_r=(double) ((int) max_row);


  for (double row=min_r; row<=max_r; row+=1) {
    int irow=(int) row+frustrum.scr_half_height;
    if (get_slice(row)) {
      
      if (sp3.x!=0)
	from=frustrum.distance_to_vp*sp3.z/sp3.x;
      else from=INFINITY*sp3.z;
      if (sp4.x!=0)
	to=frustrum.distance_to_vp*sp4.z/sp4.x;
      else to=INFINITY*sp4.z;
      
      // to must be greater or equal iv.from 
      if (to<from) { // swap from and to 
	double sp=to;
	to=from;
	from=sp;
      };
      iv.from=((int) ((from+frustrum.scr_half_width-1)*frustrum.vh_scale));
      iv.to=((int) ((to+frustrum.scr_half_width-1)*frustrum.vh_scale));
      iv_sp=iv;

      while (screen_manager.content[irow].get(iv)) {

	from=((double) iv.from)*frustrum.rez_vh_scale-frustrum.scr_half_width+1;
	  //	  if (from<-frustrum.scr_half_height+1) from=frustrum.scr_half_height+1;
	  //	  if (from>frustrum.scr_half_height) from=frustrum.scr_half_height;
	
	to=((double) iv.to)*frustrum.rez_vh_scale-frustrum.scr_half_width+1;
	//	  if (to<-frustrum.scr_half_height+1) to=frustrum.scr_half_height+1;
	//	  if (to>frustrum.scr_half_height) to=frustrum.scr_half_height;
	
	nenner=nenner_glob+plane_n->v.y*row+plane_n->v.z*from;

	c=nenner*rez_add_nenner;
	//prez=&rez[(int) (nenner*rez_add_nenner)+9999];
	
	zu=dsp*
	  (frustrum.distance_to_vp*texture_right->v.x+row*texture_right->v.y+
	   from*texture_right->v.z)*factor;

	zv=dsp*
	  (frustrum.distance_to_vp*texture_up->v.x+row*texture_up->v.y+
	   from*texture_up->v.z)*factor;
    
  
    	
      pstart=buffer+irow*frustrum.scr_width+iv.from;
      pend=pstart+(iv.to-iv.from);


      
      goto print_pixel2a;		 
      
  special2a:
      zu+=cu1;
      zv+=cv1;
      c+=1.0;
      //prez++;
            
  print_pixel2a:
      denom=1.0/c;
      // denom=*prez;
      register int u=(int) (zu*denom-cu) & tw;     
      register int v=(int) (zv*denom-cv) & th;
      *pstart=*(tex[u]+v);
      //==   *pstart=*(texture->content+u*texture->width+v);
      pstart++;
      if (pstart<=pend) goto special2a;
      

      
   
      /*
      register int count;
      register int u,v;
      goto first2a;		     

  special2a:
      zu+=cu1;
      zv+=cv1;
      //prez++;
            
  print_pixel2a:
      // denom=*prez;
      u=(int) (zu*denom-cu) & tw;     
      v=(int) (zv*denom-cv) & th;
      *pstart=*(tex[u]+v);
      pstart++;
      if (--count>0) goto special2a;
      
      c+=16.0;
      first2a:
      denom=1.0/c;
      count=16;
      if (pstart+16<=pend) goto print_pixel2a;
      */


      iv=iv_sp;
    };
    }; 
  };
}; /// end of MPolygon::draw_mode_2a


void MPolygon::draw(View *v=NULL) { // v is not used here

  // some preliminary functions
  clipping();
  calc_constants();
  
  // backface removal
  if (backface) 
    if (plane_d >= 0) return;
  if (nr_of_vertices_ac==0) return; // Polygon is behind the viewer
  if (usecolor) Polygon::draw();
  else {
#ifdef ACCURATE
    draw_mode_2a();
#else
    if (fabs(plane_n->v.z)<0.06) {
      draw_mode_2a();
    }
    else
      draw_mode_2(); 
#endif
  };
};




///
returnvalue MPolygon::split (const Plane &part) {
  returnvalue pieces=Polygon::split(part);
  MPolygon * front=(MPolygon *) pieces.front;
  MPolygon * back=(MPolygon *) pieces.back;
  if (pieces.front!=0x0)   {
    front->attach_same_mipmap(this);
    front->usecolor=usecolor;
  };
  if (pieces.back!=0x0)   {
    back->attach_same_mipmap(this);
    back->usecolor=usecolor;
  };
  return pieces;
};    

void MPolygon::registrate_points (TPoint_Set &t, TPoint_Set &dir_t) {
  Polygon::registrate_points(t,dir_t);
  if (mipmap!=NULL) {
    TPoint *sp;
    sp=texture_origin;
    texture_origin=t.add(texture_origin);
    delete sp;
    sp=texture_right;
    texture_right=dir_t.add(texture_right);
    delete sp;
    sp=texture_up;
    texture_up=dir_t.add(texture_up);
    delete sp;
  };
};


//===============================================

Polygon_List::Polygon_List (void) {
  first=last=NULL;
};

Polygon_List::Polygon_List (const Polygon_List * other) {
  first=last=NULL;
  W_Polygon * i=other->first;
  Polygon * w;
  while (i!=NULL) {
    w=new Polygon(i->polygon);
    add(w);
    i=i->next;
  };
};

///
Polygon_List::~Polygon_List (void) {
  while (first!=NULL) {
    last=first->next;
    delete first;
    first=last;
  };
};

/// a virtual constructor
Polygon_List * Polygon_List::new_Polygon_List(void) {
  return new Polygon_List;
};


/// a virtual constructor
Polygon_List * Polygon_List::new_Polygon_List(const Polygon_List *other) {
  return new Polygon_List(other);  
};

/// copies other Polygon_List to this polygon list 
void Polygon_List::copy (Polygon_List * other) {
  W_Polygon * run=other->first;
  while (run!=NULL) {
    add(run->polygon->new_Polygon (run->polygon)); // virtual constructor
    run=run->next;
  };  
};

///
Polygon_List * Polygon_List::clone (void) {
  Polygon_List * pl=new Polygon_List();
  pl->copy(this);
  return pl;
};

/// to build up everything
void Polygon_List::build (void) {
  // nothing to do (used only for childs) 
};



///
bool Polygon_List::is_empty(void) const {
  if (first==NULL) return true; 
  return false; 
};
///
void Polygon_List::add(const Polygon *W) {
  if (is_empty()) { 
    first=new(W_Polygon);
    last=first;
  } 
  else {
    last->next=new(W_Polygon);
    last=last->next;
  };
  last->polygon=W;
  last->next=NULL;
};

///
void Polygon_List::add(const Polygon_List * other) {
  W_Polygon * i=other->first;
  Polygon * w;
  while (i!=NULL) {
    w=i->polygon->new_Polygon (i->polygon); // virtual constructor
    add(w);
    i=i->next;
  };
};

///
int Polygon_List::nr_of_polygons(void) const {
  W_Polygon * run=first;
  int i=0;
  while (run!=NULL) {
    run=run->next;
    i++;
  };
  return i;
};

///
void Polygon_List::shuffle (void) {
  int nr_of_polys=nr_of_polygons();
  if (nr_of_polys<=1) return; // nothing to do
  Polygon ** parray=new (Polygon *)[nr_of_polys];
  int * perm= new int[nr_of_polys];
  W_Polygon * run=first;
  int i=0;
  while (run!=NULL) {
    parray[i]=run->polygon;
    perm[i]=i;
    run=run->next;
    i++;
  };
  // shuffle perm
  for ( i=nr_of_polys-1; i>=1; i--) {
    register int sp1=rand() % i;
    register int sp2=perm[i];
    perm[i]=perm[sp1];
    perm[sp1]=sp2;
  };
  run=first;
  i=0;
  while (run!=NULL) {
    run->polygon=parray[i];
    run=run->next;
    i++;
  };  
  delete[] parray;
  delete[] perm;
};

///
void Polygon_List::replace_mipmap_by (Mipmap &t1,Mipmap &t2) {
  W_Polygon * run=first;
  while (run!=NULL) {
    run->polygon->replace_mipmap_by(t1,t2);
    run=run->next;
  };
};

/*
void Polygon_List::to_world (View &view) {
  W_Polygon * run=first;
  while (run!=NULL) {
    run->polygon->to_world(view);
    run=run->next;
  };
};

void Polygon_List::to_view (View &view) {
  W_Polygon * run=first;
  while (run!=NULL) {
    run->polygon->to_view(view);
    run=run->next;
  };
};
*/


///
Point * Polygon::first_vertex_w (void) {
  return &vertex[0]->w;
};

///
Point * Polygon::normal_r (void) {
  return &plane_n->r;
};

///
Point * Polygon::normal_w (void) {
  return &plane_n->w;
};

///
double Polygon::distance (void) {
  return plane_d;
};


void Polygon_List::print (void) {
#ifdef DEBUG
  W_Polygon * run=first;
  cout << "  begin.list:\n";
  while (run!=NULL) {
    if (run->polygon->signature == NULL)
      cout << "    polygon.signature= NULL" << endl;
    else
      cout << "    polygon.signature= " << run->polygon->signature << "\n";
    run=run->next;
  };
  cout << "  end.list:\n";
#endif
};

/// registrate the vertices and direction vectors of each polygon in Polygon_List
void Polygon_List::registrate_points(TPoint_Set &t,TPoint_Set &dir_t) {
  W_Polygon * run=first;
  while (run!=NULL) {
    run->polygon->registrate_points(t,dir_t);
    run=run->next;
  };
};


///
void Polygon_List::draw(View *v=NULL) { // v is not used here
  W_Polygon * run=first;
  while (run!=NULL) {
    run->polygon->draw();
    run=run->next;
  };
};

///
double Polygon_List::collision_time(Point &vel,Point &P,double radius) {
  double time=INFINITY;
  W_Polygon * run=first;
  while (run!=NULL) {
    double sp=run->polygon->collision_time(vel,P,radius);
    if ((sp>=0) && (sp<time)) time=sp;
    run=run->next;
  };
  return time;
};

///
void Polygon_List::remove (W_Polygon *w) {

  if (first==NULL) return; // list empty, therefore do nothing
  
  if (first==w) { // found, it is the first entry
    // remove first entry
    if (first==last) last=first->next;
    first=first->next;
    delete w;
    return;
  };
  
  W_Polygon * previous=first; // is !=NULL  
  W_Polygon * run=first->next;
  
  while (run!=NULL) {
    if (run==w) { // found, then kill
      previous->next=run->next;
      if (last==w) last=previous; 
      delete w;
      return;
    };
    previous=run;
    run=run->next;
  };
  // not found
};


//===========================================
