// points.h

/*
   Sofie, a real time 3d engine / Copyright (C) 1997 Stephan Schiessling
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#ifndef F_POINTS
#define F_POINTS

/// class Point describes a point in 3-space
class Point {  
public:
  /// 1. coordinate of the point
  double x;
  /// 2. coordinate of the point
  double y;
  /// 3. coordinate of the point
  double z;
  //--------------
  /// constructor which does nothing
  Point(void) {};
  /// constructor which initializes the point by the 3 double values
  Point(double, double, double);
  /// initializes the point by three double values
  void init(double, double, double);
  /// initializes the point by another point
  void init(const Point &);
  /// calculates the lenght^2 of the vector (=point)
  double length2(void);
  /// calculates the lenght of the vector (=point)
  double length(void);
  /// normalizes the vector (=point)
  void normalize(void);
  /// changes lenght of the vector (=point) in a special way; is only used for texture coordinates
  void adjust(const Point &, const Point &, const double &);
  ///
  void adjust(const double len,const Point &from,const Point &to);
  /// changes the vector (=point) in a special way; is only used for texture coordinates 
  void as_texture_coord(void);  
  /// transforms the point to view coordinate system
  void to_view (const struct View &);
  /// add to vector (=points)
  Point & operator+=(const Point &);
  /// subtract one vector (=point) from the other
  Point & operator-=(const Point &);
  //-------------------
  /// multiplication of vector (=point) with a scalar
  friend Point operator*(double,const Point &);
  /// scalarmultiplication (euklidean)
  friend double operator*(const Point &,const Point &);
  /// add to vector (=points)
  friend Point operator+(const Point &,const Point &);
  /// subtract one vector (=point) from the other
  friend Point operator-(const Point &,const Point &);
  /// true iff if the two points are equal
  friend bool operator==(const Point &,const Point &);
  /// true iff if the two points are not equal
  friend bool operator!=(const Point &,const Point &);
}; 


/// class View describes observers coordinate system (the view coordinatesystem)
class View {
public:
  /// the origin, where the observer is
  Point O;
  /// the 1. coordinate of the observers coordinate system
  Point Xv;
  /// the 2. coordinate of the observers coordinate system
  Point Yv;
  /// the 3. coordinate of the observers coordinate system
  Point Zv;
  //-----------------------------------
  /// constructor for standart coordinate system
  View(void);
  /// initialize view coordinate system by this data, where the z coordinate is calculated by the others
  void init(const Point &origin,const Point &X,const Point &Y); // Z will be calculated (X,Y,Z) will be right handed
  /// initialize view coordinate system by this data
  void init(const Point &origin,const Point &X,const Point &Y,const Point &Z); 
  /// initialize the view coordinate system by the other view coordinate system
  void init(const View &);
  /// normalizes the coordinate vectors of the view coordinate system
  void normalize(void);
};



// is used for vertices of polygons
// there should be fixed coordinates (r), absolute coordinates in the world (w) and
// view coordinates relative to some observer (v)
// ideally there should be a coordinate system attached to each TPoint,
// but usually TPoints belong to some Object to which there is a coordinate
/// system attached. 
class TPoint {
public:
  /// view coordinates
  Point v;       
  /// world coordinates
  Point w;       
  /// relative coordinates
  Point r;
  //--------------------
  /// constructor to create empty TPoint
  TPoint(void) {};
  /// constructor which sets relative coordinates
  TPoint(double,double,double); 
  /// constructor which sets relative coordinates
  TPoint(const Point &);
  /// constructor which sets relative coordinates
  TPoint(const TPoint &);
  /// initializes relative coordinates of TPoint
  void init(double,double,double);
  /// initializes relative coordinates of TPoint
  void init(const Point &);
  /// initializes relative coordinates of TPoint
  void init(const TPoint &);
  /// transforms relative coordinates of TPoint to world coordinates by given view 
  void to_world(const View &);
  /// transforms relative coordinates of a direction vector to world coordinates by given view
  void dir_to_world(const View &);
  /// transforms world coordinates of a vector to view coordinates by given view
  void to_view(const View &);
  /// transforms world coordinates of a direction vector to view coordinates by given view
  void dir_to_view(const View &);
  //---------------------
  /// true iff relative coordinates of the TPoints are equal
  friend bool operator==(const TPoint &,const TPoint &); 
  /// true iff relative coordinates of the TPoints are not equal
  friend bool operator!=(const TPoint &,const TPoint &);
};


#endif F_POINTS
