// points.cc

/*
   Sofie, a real time 3d engine / Copyright (C) 1997 Stephan Schiessling
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <math.h>
#include <stream.h>
#include "globals.h"
#include "points.h"


///
Point::Point (double a,double b,double c) {
  init(a,b,c);
};

///
void Point::init(double a,double b,double c) {
  x=a;
  y=b;
  z=c;
};

///
void Point::init(const Point &P) {
  x=P.x;
  y=P.y;
  z=P.z;
};

///
double Point::length2(void) {
  return x*x+y*y+z*z;
};

///
double Point::length(void) {
  return sqrt(x*x+y*y+z*z);
};

///
void Point::normalize(void) {
  double len=length();
  if (len!=0) (*this)=(1.0/len)*(*this);
};

///
void Point::adjust(const Point &P, const Point &Q, const double &c) {
  double rezc;
  if (c==0) rezc=INFINITY;
  else rezc=1.0/c;
  (*this)=rezc*(Q-P);
}; 

///
void Point::adjust(const double len,const Point &from,const Point &to) {
  Point D;
  D=to-from;
  double l=D.length();
  (*this)=(len/l)*D;
};



///
void Point::as_texture_coord(void) {
  double l;
  l=length();    // doit!    do not use length, because we need only l*l, therefore no sqrt();
  if (l==0) {
    cout << "ERROR: coordinate vector for the texture is 0!\n";
    exit(-1);
  };
  l=1.0/(l*l);
  (*this)=l*(*this);
};


///
void Point::to_view(const View &view) {
  Point sp;
  sp=(*this)-view.O;
  x=view.Xv*sp; // Skalarprodukt
  y=view.Yv*sp; // Skalarprodukt
  z=view.Zv*sp; // Skalarprodukt
};

///
Point & Point::operator+=(const Point &Q) {
  x+=Q.x;
  y+=Q.y;
  z+=Q.z;
  return *this;
};

///
Point & Point::operator-=(const Point &Q) {
  x-=Q.x;
  y-=Q.y;
  z-=Q.z;
  return *this;
};


///
Point operator*(double m,const Point &P) {
  return Point(m*P.x,m*P.y,m*P.z);
};

///
double operator*(const Point &P,const Point &Q) {
  return double (P.x*Q.x+P.y*Q.y+P.z*Q.z);
};

///
Point operator+(const Point &P,const Point &Q) {
  return Point(P.x+Q.x,P.y+Q.y,P.z+Q.z);
};

///
Point operator-(const Point &P,const Point &Q) {
  return Point(P.x-Q.x,P.y-Q.y,P.z-Q.z);
};



///
bool operator==(const Point &P,const Point &Q) {
  if (P.x!=Q.x) return false;
  if (P.y!=Q.y) return false;
  if (P.z!=Q.z) return false;
  return true;
};

///
bool operator!=(const Point &P,const Point &Q) {
  if (P.x!=Q.x) return true;
  if (P.y!=Q.y) return true;
  if (P.z!=Q.z) return true;
  return false;
};

//====================================

View::View (void) {
  O.init(0,0,0);
  Xv.init(1.0,0,0);
  Yv.init(0,1.0,0);
  Zv.init(0,0,1.0);
};

///
void View::init(const Point &origin,const Point &X,const Point &Y) {
  O.init(origin);
  Xv.init(X);
  Yv.init(Y);
  VECTOR_PRODUCT(Zv,Xv,Yv);
};

///
void View::init(const Point &origin,const Point &X,const Point &Y,const Point &Z) {
  O.init(origin);
  Xv.init(X);
  Yv.init(Y);
  Zv.init(Z);
};

///
void View::init(const View &other) {
  O.init(other.O);
  Xv.init(other.Xv);
  Yv.init(other.Yv);
  Zv.init(other.Zv);
};

///
void View::normalize(void) {
  Xv.normalize();
  Yv.normalize();
  Zv.normalize();
};


//=======================================


///
TPoint::TPoint(double a,double b,double c) {
  init(a,b,c);
};

///
TPoint::TPoint(const Point &P) {
  init(P);
};

///
TPoint::TPoint(const TPoint &P) {
  init(P);
};

///
void TPoint::init(double a,double b,double c) {
  r.init(a,b,c);
};

///
void TPoint::init(const Point &P) {
  r.init(P);
};

///
void TPoint::init(const TPoint &P) {
  r.init(P.r);
};

///
void TPoint::to_world(const View &view) {
  w=r.x*view.Xv+r.y*view.Yv+r.z*view.Zv;
  w=w+view.O;
};

///
void TPoint::dir_to_world(const View &view) {
  w=r.x*view.Xv+r.y*view.Yv+r.z*view.Zv;
  // no translation!
};

///
void TPoint::to_view(const View &view) {
  Point sp;
  sp=w-view.O;
  v.x=view.Xv*sp; // Skalarprodukt
  v.y=view.Yv*sp; // Skalarprodukt
  v.z=view.Zv*sp; // Skalarprodukt
};

///
void TPoint::dir_to_view(const View &view) {
  Point sp;
  v.x=view.Xv*w; // Skalarprodukt
  v.y=view.Yv*w; // Skalarprodukt
  v.z=view.Zv*w; // Skalarprodukt
};

///
bool operator==(const TPoint &P,const TPoint &Q) {
  if (P.r!=Q.r) return false;
  return true;
};

///
bool operator!=(const TPoint &P,const TPoint &Q) {
  return ~(P==Q);
};


//=================================

