// pilot.cc

/*
   Sofie, a real time 3d engine / Copyright (C) 1997 Stephan Schiessling
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include "pilot.h"
#include "console.h"
#include "objects.h"
#include <math.h>
#include <stdlib.h>


///
extern Console console;
/// inputs which can be used by everyone
static Input zero_input;  /// no input, so not change
static Input shot_input; /// input used to steer shots
static Input common_input; /// if input changes, use common_input

Pilot::Pilot (void) {
  target=NULL;
};

Input * Pilot::process(const View &view) {
  return &zero_input;
};

void Pilot::set_target (String *t) {
  target=t;
};

String * Pilot::current_target (void) {
  return target;
};

//===================================

///
Input * Console_Pilot::process(const View &view)  {  
  return &console.input;
};

//===================================

///
Input * MConsole_Pilot::process(const View &view)  {  

  if (console.input.x != 0) {
    if (console.input.x > 0) { 
      speed+=0.05;
      if (speed > 0.8) speed=0.8;
    } else {
      speed-=0.05;
      if (speed <= 0) speed=0;
    };
  };
  common_input.x=speed;
  common_input.y=console.input.y;
  common_input.z=console.input.z;
  common_input.angley=console.input.angley;
  common_input.anglez=console.input.anglez;
  common_input.anglex=console.input.anglex;
  common_input.shoot=console.input.shoot;
  
  return &common_input;
};

//====================================

RPilot::RPilot(String &filename) {
  rec.open((const char *)filename);
  cout << "recorder on\n";
};


RPilot::~RPilot (void) {
  rec.close();
  cout << "recorder off\n";
};

///
Input * RPilot::process(const View &view)  {  
  rec << console.input.x << " " << console.input.y << " " << console.input.z << " " 
      <<  console.input.anglex << " " << console.input.angley << " " << console.input.anglez << '\n';
  return &console.input;
};

//============================================


PPilot::PPilot(String &filename, bool * s) {
  status=s;
  recplay.open((const char *)filename);
  cout << "player on\n";
};

PPilot::~PPilot (void) {
  recplay.close();
  cout << "player off\n";
};

///
Input * PPilot::process(const View &view)  {  
  if (recplay.eof()) {
    //recplay.seekg(0L,ios::beg); doit!  does not work!
    //cout << "repeat\n";
    if (status != NULL)
      *status=true; // tell that player finished
    return &zero_input;
  };
  recplay >> common_input.x; 
  recplay >> common_input.y; 
  recplay >> common_input.z; 
  recplay >> common_input.anglex; 
  recplay >> common_input.angley; 
  recplay >> common_input.anglez;

  return &common_input;
};

//=============================================

Input * Shot_Pilot::process(const View &view) {
  shot_input.x=2.0; // better not here
  shot_input.y=0; //-0.05;
  return &shot_input;
};

//=============================================

///
Basic_CPilot::Basic_CPilot (void) : Pilot() {
  targets=NULL;
};

///
void Basic_CPilot::set_targets (Regex *t) {
  targets=t;
};


///
String * Basic_CPilot::current_target (void) {
  return target;
};

///
Input * Basic_CPilot::process(const View &view) {

  Radar radar;

  if (target != NULL) {
    radar.get_radar(view,target);
    if (radar.nr_of_objects == 0) {
      target=NULL;
      radar.get_radar(view,targets);      
    }
  }  else {
    radar.get_radar(view,targets);
  };

  if (radar.nr_of_objects==0) 
    return &zero_input;

  if (target == NULL)
    target=radar.object->id; // the nearest

  
  Radar_Object *ro=radar.object;



  //cout << ro->distance << "    " << ro->y << "      " << ro->x << "\n";
  
  
  if (ro->in_front_of) { 
    if (ro->distance > 30) 
      common_input.x=0.1;
    else
      common_input.x=0;    
  }    
  else
    common_input.x=0;    

    
  common_input.y=0;
  common_input.z=0;
  common_input.angley=0;
  common_input.anglez=asin(ro->x); // too slowly ???
  common_input.anglex=asin(ro->y);
  /*
     if (common_input.anglex > 0.8) common_input.anglex=0.8;
     if (common_input.anglex < -0.8) common_input.anglex=-0.8;
     if (common_input.anglez > 0.8) common_input.anglez=0.8;
     if (common_input.anglez < -0.8) common_input.anglez=-0.8;
  */
  //    if ((ro->y>-0.003*ro->distance) & (ro->y<0.003*ro->distance))
  //      common_input.anglez=ro->x*0.01;
  
  
  if ((ro->distance<140) & (ro->x < 0.1 ) & (ro->x > -0.1) & (ro->y < 0.1 ) & (ro->y > -0.1))
    common_input.shoot=true;
  else   common_input.shoot=false;
  
return &common_input;
};

//=========================================

Mine_Pilot::Mine_Pilot (void) : Basic_CPilot() {
  set_wait(100);
};

void Mine_Pilot::set_wait (int w) {
  wait=w;
  cwait=rand() & 127; // random , so that not all mines fire at the same time
};

///
Input * Mine_Pilot::process(const View &view) {

  Radar radar;

  target=NULL;
  
  if (target != NULL) {
    radar.get_radar(view,target);
    if (radar.nr_of_objects == 0) {
      target=NULL;
      radar.get_radar(view,targets);      
    }
  }  else {
    radar.get_radar(view,targets);
  };

  if (radar.nr_of_objects==0) 
    return &zero_input;

  if (target == NULL)
    target=radar.object->id; // the nearest

  
  Radar_Object *ro=radar.object;

  common_input.x=0;
  common_input.y=0;
  common_input.z=0;
  common_input.angley=0;
  common_input.anglez=asin(ro->x); // too slowly ???
  common_input.anglex=asin(ro->y);
    
  if (cwait < 0) {
    if (ro->distance<150) {
      common_input.shoot=true;
      cwait=wait;
    };
  } else {
    common_input.shoot=false;
    cwait--;
  };

  if (ro->distance<40) 
    common_input.shoot=true;
  
  return &common_input;
};


//=========================================

Maneuver::Maneuver (void) : Basic_CPilot () {
  init();
};

void Maneuver::init (void) {
  live_time=20;
};

int Maneuver::how_good_am_i (const View &view) {
  return 10;
};

Input * Maneuver::process (const View &view) {
  if (live_time<0) return NULL;
  live_time--;
  return &zero_input;
};

//=========================================

ManeuverA::ManeuverA (void) : Maneuver () {
  init();
};

void ManeuverA::init (void) {
  live_time=100;
};

int ManeuverA::how_good_am_i (const View &view) {
  return rand() & 0x1f;
};

Input * ManeuverA::process (const View &view) {
  if (live_time<0) return NULL;
  live_time--;

  return Basic_CPilot::process(view);
};

//=========================================

ManeuverB::ManeuverB (void) : Maneuver () {
  init();
};

void ManeuverB::init (void) {
  live_time=50;
};

int ManeuverB::how_good_am_i (const View &view) {
  
  Radar radar;
  radar.get_radar(view,target);
  
  if (radar.nr_of_objects==0) return 0;
  
  Radar_Object *ro=radar.object;
  
  if (!ro->in_front_of) return (10+(rand() & 0x7));

  return 3;
};

Input * ManeuverB::process (const View &view) {
  if (live_time<0) return NULL;
  live_time--;
  
  common_input.x=1;
  common_input.y=0;
  common_input.z=0;
  common_input.angley=0;
  common_input.anglez=0; 
  common_input.anglex=-0.5;

  return &common_input;
};

//=========================================

ManeuverC::ManeuverC (void) : Maneuver () {
  init();
};

void ManeuverC::init (void) {
  live_time=50;
};

int ManeuverC::how_good_am_i (const View &view) {
  return rand() & 0x1f;
};

Input * ManeuverC::process (const View &view) {
  if (live_time<0) return NULL;
  live_time--;

  common_input.x=0.6;
  common_input.y=0;
  common_input.z=0;
  common_input.angley=0;
  common_input.anglez=0.1; 
  common_input.anglex=0;

  return &common_input;
};


//=========================================

CPilot::CPilot (void) : Basic_CPilot () {
  current=NULL;
  nr_of_maneuvers=0;
  maneuver=NULL;
};

Input * CPilot::process(const View &view) {
  if (current==NULL) 
    init_maneuver(view);
  Input * sp=current->process(view);
  if (sp==NULL) { // maneuver finished
    init_maneuver(view);
    sp=current->process(view);
    if (sp==NULL) {
      cout << "ERROR: CPilot could not find valid maneuver!\n";
      exit(-1);
    };
  };
  return sp;
};    

void CPilot::init_maneuver (const View &view) {
  int max=-1;
  Maneuver ** run=maneuver;
  for (int i=0; i< nr_of_maneuvers; i++) {
    int sp=(*run)->how_good_am_i(view);
    if (sp>max) {
      max=sp;
      current=*run;
    };
    run++;
  };
  if (current==NULL) {
    cout << "ERROR: CPilot has no Maneuver!\n";
    exit(-1);
  }
  else
    current->set_target(target);
    current->set_targets(targets);
    current->init();
  //cout << current << "Maneuver initialized\n";
};

void CPilot::add (const Maneuver &m) {
  Maneuver ** sp=new (Maneuver *)[nr_of_maneuvers+1];
  for (int i=0; i<nr_of_maneuvers;i++)
    sp[i]=maneuver[i];
  sp[nr_of_maneuvers]=&m;
  delete[] maneuver;
  maneuver=sp;
  nr_of_maneuvers++;
};


//=========================================

Player::Player (const String & name) {
  rec_mode=false;
  running=false;
  stopped=true;
  object=NULL;
  pilotsp=NULL;
  ppilot=NULL;
  rpilot=NULL;
  filename=new String(name);
};

Player::~Player (void) {
  delete filename;
  if (!stopped) {
    if (rec_mode)
      rpilot->~RPilot();
    else
      ppilot->~PPilot();
    object->pilot=pilotsp; // restore old pilot
  };
};

void Player::as_recorder (void) {
  if (running) return;
  rec_mode=true;
};

void Player::as_player (void) {
  if (running) return;
  rec_mode=false;
};

void Player::run (Object * o) {
  if (running) return;
  running=true;
  object=o;
  pilotsp=object->pilot;
  if (rec_mode) {
    object->pilot=rpilot=new RPilot(*filename);
    cout << "recorder on\n";
  } else {
    object->pilot=ppilot=new PPilot(*filename,&stopped);
    cout << "player on\n";
  };
  stopped=false;
};

void Player::stop (void) {
  stopped=true;
};

bool Player::on (void) {
  return running;
};

void Player::pswitch (Object *o) {
  if (running)
    stopped=true;
  else
    run(o);
};

void Player::update (void) {
  if (running & stopped) { 
    if (rec_mode)
      rpilot->~RPilot();
    else
      ppilot->~PPilot();      
    object->pilot=pilotsp; // restore old pilot
    stopped=true;
    running=false;
    object=NULL;
  };
};
