// objman.cc

/*
   Sofie, a real time 3d engine / Copyright (C) 1997 Stephan Schiessling
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include "objman.h"
#include <iostream.h>
#include <stdlib.h>


extern Screen_Manager screen_manager;




OMdata::OMdata (Object *o) {
  object=o;
  next=NULL;
  wnext=NULL;
  wunder=NULL;
};

// could be simplified by using recursion (also for next), 
// but this costs speed
OMdata * OMdata::touches_bigger (Object * o) {
  OMdata * run=this;
  while (run != NULL) {
    if (run->object->size_type > o->size_type) {
      
      if (run->wunder != NULL) {
	OMdata * sp=run->wunder->touches_bigger(o);
	if (sp != NULL) return sp;
      };
      if (run->object->touches(o)) return this;
    };
    // nothing is under
    run=run->wnext; // next of working list
  }; // end while
  
  return NULL;
};

//========================================

/// 
void Stack::push(BSP_Tree ** b) {
  if (next_free>=max_elements) {
    cout << "ERROR: Stack overflow!" << endl;
    exit (-1);
  };
  content[next_free]=b;
  next_free++;
};

///
BSP_Tree ** Stack::pop (void) {
  if (next_free==0) 
    return NULL;
  next_free--;
  return content[next_free];
};

///
bool Stack::not_empty (void) { 
  if (next_free > 0) return true;
  return false;
};

//=====================================

Object_Manager::Object_Manager (void) { 
  active=working=ns_waiting=waiting=ns_sleeping=sleeping=NULL; 
  nr_waiting=0;
};

void Object_Manager::add (Object * o) {
  if (o == NULL) return; // subobjects could be NULL
  
  OMdata * sp=new OMdata(o);
  sp->next=waiting;
  waiting=sp;
  nr_waiting++;
  
  
  // now we are able to determine size_radius of object
  o->calc_size_radius();
  /** a negative radius (per default) is only an indication
     * that the collision radius should be the same size as
     * the size_radius */
  if (o->radius < 0) o->radius=o->size();

  // finally add all subobjects (like shots)

  String * s;
  if (o->id != NULL) {
    s=new String(*o->id);
    (*s)+="S";
  } else
    s=NULL;
  
  for (int i=0; i<o->nr_of_shots; i++) {
    o->shots[i]->id=s;
    ns_add(o->shots[i]);
  };
  ns_add(o->explosion);

};

void Object_Manager::ns_add (Object * o) {
  if (o == NULL) return;
  
  OMdata * sp=new OMdata(o);
  sp->next=ns_waiting;
  ns_waiting=sp;

  // now we are able to determine size_radius of object
  o->calc_size_radius();
  /** a negative radius (per default) is only an indication
     * that the collision radius should be the same size as
     * the size_radius */
  if (o->radius < 0) o->radius=o->size();

  // finally add all subobjects (like shots)

  String * s;
  if (o->id != NULL) {
    s=new String(*o->id);
    (*s)+="S";
  } else
    s=NULL;

  for (int i=0; i<o->nr_of_shots; i++) {
    o->shots[i]->id=s;
    ns_add(o->shots[i]);
  };
  ns_add(o->explosion);

};

void Object_Manager::sort (void) {
  move_sleepers();
  move_non_sleepers();  // could be speeded up, because already moved objects could be skipped
  move_ns_sleepers();
  move_non_ns_sleepers();  // could be speeded up, because already moved objects could be skipped
};

// move sleepers of waiting list to sleeping list
void Object_Manager::move_sleepers (void) {
  OMdata * previous=NULL;
  OMdata * run=waiting;
  while (run != NULL) {
    OMdata * runn= run->next;
    if (run->object->power <= 0) { 
      // remove from waiting list
      nr_waiting--;
      if (previous == NULL) 
	waiting=run->next; // first entry of list
      else 
	previous->next=run->next;
      // add to sleeping list
      run->next=sleeping;
      sleeping=run;
    } else
      previous=run;
    run=runn;
  }; // end while
};

// move ns_sleepers of ns_waiting list to ns_sleeping list
void Object_Manager::move_ns_sleepers (void) {
  OMdata * previous=NULL;
  OMdata * run=ns_waiting;
  while (run != NULL) {
    OMdata * runn= run->next;
    if (run->object->power <= 0) { 
      // remove from ns_waiting list
      if (previous == NULL) 
	ns_waiting=run->next; // first entry of list
      else 
	previous->next=run->next;
      // add to ns_sleeping list
      run->next=ns_sleeping;
      ns_sleeping=run;
    } else 
      previous=run;
    run=runn;
  }; // end while
};



// move non sleepers of sleeping list to waiting list
void Object_Manager::move_non_sleepers (void) {
  OMdata * previous=NULL;
  OMdata * run=sleeping;
  while (run != NULL) {
    OMdata * runn= run->next;
    if (run->object->power > 0) { 
      // remove from sleeping list
      if (previous == NULL) 
	sleeping=run->next; // first entry of list
      else 
	previous->next=run->next;
      // add to waiting list
      run->next=waiting;
      waiting=run;
      nr_waiting++;
    } else 
      previous=run;
    run=runn;
  }; // end while
};

// move non ns_sleepers of ns_sleeping list to ns_waiting list
void Object_Manager::move_non_ns_sleepers (void) {
  OMdata * previous=NULL;
  OMdata * run=ns_sleeping;
  while (run != NULL) {
    OMdata * runn= run->next;
    if (run->object->power > 0) { 
      // remove from ns_sleeping list
      if (previous == NULL) 
	ns_sleeping=run->next; // first entry of list
      else 
	previous->next=run->next;
      // add to ns_waiting list
      run->next=ns_waiting;
      ns_waiting=run;
    } else
      previous=run;
    run=runn;
  }; // end while
};


OMdata * Object_Manager::touches_bigger (Object * o) {
  return waiting->touches_bigger (o);
};

Object * Object_Manager::active_object(void) {
  return active->object;
};

View * Object_Manager::active_view(void) {
  return &(active->object->view);
};


// search only in list start, and the same level
void Object_Manager::mark_as_active(Object &o) {
  OMdata * run=waiting;
  while (run!=NULL) {
    if (run->object==&o) { // new active object found
      // before switching to new active object,
      // take care of cockpit
      if (active != NULL) 
	if (active->object->cockpit!=o.cockpit) 
	  cockpit_on_screen=false;
      active=run;
      return;
    };
    run=run->next;
  };
  cout << "Error: new active object not present!\n";
  exit(-1);
};

void Object_Manager::collisions (void) {
  OMdata * runa=waiting;
  OMdata * runb;

  
  while (runa != NULL) {

    // collision of solid with solid
        
    runb=runa->next;
    while (runb != NULL) {
      // collision of solid objects
      
      runa->object->collision(*runb->object);
      runb=runb->next;
    }; // end while

    // collision of solid with non-solid
    

    runb=ns_waiting;

    while (runb != NULL) {
      // collision of solid object with non-solid object
      
      runa->object->collision(*runb->object);
      runb=runb->next;
    }; // end while
    
    runa=runa->next;
  }; // end while

  // non-solid with non-solid collisions not interesting

};





void Object_Manager::print(void) {
  cout << "print working:" << endl;
  OMdata * run=working;
  while (run!=NULL) {
    cout << "mass=" << run->object->mass << "power=" << run->object->power <<" id=";  
    if (run->object->id == NULL)
      cout << "id = (NULL) ";
    else
      cout << "id = " << (*run->object->id);
    cout << "d=" << run->distance2;
    if (run->wunder != NULL) cout << " under";
    cout << endl;
    run=run->wnext;
  };
};

void Object_Manager::printw(void) {
  cout << "print waiting:" << endl;
  OMdata * run=waiting;
  while (run!=NULL) {
    cout << "mass=" << run->object->mass << "power=" << run->object->power <<" id=";  
    if (run->object->id == NULL)
      cout << "id = (NULL) ";
    else
      cout << "id = " << (*run->object->id);
    cout << "d=" << run->distance2 << endl;
    run=run->next;
  };
  cout << "print ns_waiting:" << endl;
  run=ns_waiting;
  while (run!=NULL) {
    cout << "mass=" << run->object->mass << "power=" << run->object->power <<" id=";  
    if (run->object->id == NULL)
      cout << "id = (NULL) ";
    else
      cout << "id = " << (*run->object->id);
    cout << "d=" << run->distance2 << endl;
    run=run->next;
  };
};

void Object_Manager::prints(void) {
  cout << "print sleeping:" << endl;
  OMdata * run=sleeping;
  while (run!=NULL) {
    cout << "mass=" << run->object->mass << "power=" << run->object->power <<" id=";  
    if (run->object->id == NULL)
      cout << "id = (NULL) ";
    else
      cout << "id = " << (*run->object->id);
    cout << "d=" << run->distance2 << endl;
    run=run->next;
  };
  cout << "print ns_sleeping:" << endl;
  run=ns_sleeping;
  while (run!=NULL) {
    cout << "mass=" << run->object->mass << "power=" << run->object->power <<" id=";  
    if (run->object->id == NULL)
      cout << "id = (NULL) ";
    else
      cout << "id = " << (*run->object->id);
    cout << "d=" << run->distance2 << endl;
    run=run->next;
  };
};



void Object_Manager::to_world (void) {  
  OMdata * run=waiting;
  while (run!=NULL) {
    run->object->to_world();
    run=run->next;
  };
  //save locations of objects; necessary for radar
  objectlocs=save_locs();

  // now non solid objects
  run=ns_waiting;
  while (run!=NULL) {
    run->object->to_world();
    run=run->next;
  };
};

Object_Locs Object_Manager::save_locs (void) {
  Object_Locs locs;
  locs.nr_of_locs=nr_waiting;
  locs.locs=new Object_Loc[locs.nr_of_locs];
  OMdata * run=waiting;
  for (int i=0; i<locs.nr_of_locs; i++) {
    if (run->object->id != NULL) {
      // objects with no identification should not appear on radars
      locs.locs[i].loc.init(run->object->view.O);
      locs.locs[i].id=run->object->id;
    } else 
      locs.nr_of_locs--;
    run=run->next;
  };
  return locs;
};



void Object_Manager::process(void) {
  OMdata * run=waiting;
  while (run!=NULL) {
    run->object->process();
    run=run->next;
  };
  
  // noew non solid objects
  run=ns_waiting;
  while (run!=NULL) {
    run->object->process();
    run=run->next;
  };
};

///
void Object_Manager::process_velocity(void) {
  OMdata * run=waiting;
  while (run!=NULL) {
    run->object->view.O=run->object->view.O+run->object->velocity;
    run=run->next;
  };

  // now non solid objects
  run=ns_waiting;
  while (run!=NULL) {
    run->object->view.O=run->object->view.O+run->object->velocity;
    run=run->next;
  };
};

void Object_Manager::update_objects(void) {
  OMdata * run=waiting;
  View *v=active_view();
  while (run!=NULL) {
    run->object->update_object(*v);
    run=run->next;
  };
  // non solid objects can also explode
  run=ns_waiting;
  while (run!=NULL) {
    run->object->update_object(*v);
    run=run->next;
  };

};



void Object_Manager::distance_sort (void) {
  // first assign to every Node a distance
  OMdata * run=working;
  Point * V=&active->object->view.O;
  while (run != NULL) {
    Point S(run->object->view.O-(*V));
    run->distance2=S.x*S.x+S.y*S.y+S.z*S.z;
    run=run->wnext; // next of working list
  }; // end while

  
  
  // sorting (improve it!!!)
  
  bool flag=true;  
  while (flag) {
    
    flag=false;
    OMdata * previous=NULL;
    run=working;
    while (run->wnext != NULL) {
      if (run->wnext->distance2 < run->distance2) {
	// exchange
	flag=true;
	if (previous == NULL) {
	  working=run->wnext;
	  previous=working;
	} else {
	  previous->wnext=run->wnext;
	  previous=previous->wnext;
	};
	OMdata * sp=run->wnext->wnext;
	run->wnext->wnext=run;
	run->wnext=sp;
      } else {
	previous=run;
	run=run->wnext;
      };
    }; // end while
    
  }; // end while

  // now it is sorted
  
}; 



// working list = waiting list + ns_waiting list 
void Object_Manager::build_working (void) {
  OMdata * run=waiting;
  OMdata * previous=NULL;
  while (run != NULL) {
    previous=run;
    run=run->next;
  };
  // previous points to end of waiting.
  
  
  // connect is to make life easier, but don't forget to 
  // disconnect it at the end of this function
  previous->next=ns_waiting;

  OMdata ** for_disconnection=&previous->next;
  
  int size=0;
  
  
  run=waiting;
  while (run != NULL) {
    if (run->object->size_type > size) 
      size=run->object->size_type;
    run=run->next;
  };
  // size is now the maximal size of non sleeping objects
  
  
  working=NULL;
  OMdata * last=NULL;
  while (size > 0) {
    run=waiting;
    while (run != NULL) {
      if (run != active) { // do not include active object
	if (run->object->size_type == size) { 
	  if (run->object->visible_by_view(*active_view())) { 
	    run->object->to_view(*active_view()); // to view (all objects in working list could be visible)
	    OMdata * sp;
	    if ((sp=working->touches_bigger(run->object)) != NULL) {
	      // add run to working list
	      sp->wunder=run;
	      run->wnext=NULL;
	      // now merge 
	      merged.push(sp->object->shape->merge(run->object->shape,&run->object->view.O));
	    } else {
	      if (working == NULL) { // list empty
		working=run;
		last=working;
		last->wnext=NULL;
		last->wunder=NULL;
	      } else { // list not empty
		last->wnext=run;
		last=run;
		last->wnext=NULL;
		last->wunder=NULL;
	      };
	    };
	  };
	};
      };
      run=run->next;
    }; // end while
    size--;
  }; // end while
  
  // now disconnect
  *for_disconnection=NULL;

  // now, the working list is created
    

  if (working == NULL) return;
  
  // finally sort it
  // not correct, sort all lists under, too
  distance_sort();
 
};


void Object_Manager::draw (void) {
  View *view=active_view();
  build_working();
  OMdata * run=working;
  while (run != NULL) {
    //run->object->draw(view);    
    run->object->shape->draw(view);
    run=run->wnext;
  };

  // unmerge

  while (merged.not_empty())
    *(merged.pop())=NULL;
   
};
  

void Object_Manager::draw_cockpit(bool to_draw) {
  if (!to_draw) { // there shall be no cockpit visible
    cockpit_on_screen=false;
    screen_manager.init();
    return;
  };
  // there shall be a cockpit visible
  Object * ao=active_object();
  
  if (ao->cockpit == NULL) return; // object has no cockpit
  
  if (!cockpit_on_screen) { // cockpit is not already visible
    // therefore draw cockpit
    if (!ao->cockpit->is_ok()) { // is cockpit not in memory and of right size?
      // then load cockpit
      ao->cockpit->load();
    };	
    // now cockpit is in memory or there is no cockpit
    if (!ao->cockpit->in_memory) { // cockpit could not be loaded
      cockpit_on_screen=false;
      screen_manager.init();
      cout << "Sorry, there is no cockpit of this size!\n";
      return;
    }
    // cockpit was not on_screen, but is now in memory
    ao->cockpit->draw();
    cockpit_on_screen=true;
  };
  // now cockpit is on screen  
  screen_manager.init(ao->cockpit->sm);
  // draw radar
  Radar radar;
  radar.get_radar(ao->view);
  ao->cockpit->draw_radar(radar,ao->pilot->current_target());
  ao->cockpit->draw_powerdisplay(ao->power);
  ao->cockpit->draw_speeddisplay(100*ao->velocity.length2());
};


