// objects.h

/*
   Sofie, a real time 3d engine / Copyright (C) 1997 Stephan Schiessling
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#ifndef F_OBJECTS
#define F_OBJECTS


#include "points.h"
#include "polygon.h"
#include "bsp.h"
#include "pilot.h"
#include "cockpit.h"
#include "tpointset.h"

struct Pilot;
struct Cockpit;

/// class to describe an object in 3-space
class Object {
    ///
    double size_radius;
    /// a counter for delay of shots
    int shoot_cycle;
  public:
    ///
    int size_type;
    /// 
    bool solid;
    /// true iff objects presents every viewer the same face (eg. used for explosions)
    bool facing;
    /// true iff objects should be considered as sphere for collision detection
    bool treat_as_sphere;
    /// 0 iff object lives forever, else it lives livetime cycles
    int live_time;
    /// initial value of counter for delay of shots
    int shoot_cycle_init;
    /// mass energy of object
    float mass;
    /// power of object
    float power; 
    /** radius of object for collision (set it directly); 
       * if no value is assigned to radius, then it will be
       * the radius of a sphere which includes the whole object */
    double radius;
    /// each object has its own view (its own coordinate system)
    View view;
    /// the current velocity (derivation of objects trajectory)
    Point velocity;
    /// this class contains all TPoints which defines the shape
    TPoint_Set tpoints;
    /** this class contains all TPoints of the shape, which are
       * direction vectors */
    TPoint_Set dir_tpoints;
    /// this defines the shape of the object
    BSP_Tree * shape;
    /// a pilot decides how an objects moves and acts 
    Pilot * pilot;
    /// each object can have its own cockpit, which is simply an image, which is a mask for the screen)
    Cockpit * cockpit;
    /// an object can have so much shots
    int nr_of_shots;
    /// true iff object is currently not able to shoot, because its last shot was just launghed
    bool shot_lock;
    /// an object has to lasers, left and right; true iff next shoot from the right laser
    bool shoot_right;
    /// index of the next free shot (object)
    int next_shot;
    /// pointer tto array of assigned shots (shot objects)
    Object ** shots;
    /// pointer to following object after death
    Object * explosion;
    /// string to identify object
    String * id;
    /** regular expression to determine with which objects 
       * this object does not collide; if cid is NULL, then it collides
       * with all objects */
    Regex * cid;
    //--------------------------
    /// constructor
    Object (void);
    /// copy constructor
    Object (const Object * other);
    /// should not be used!!
    Object (const Object & other);
    /// destructor
    ~Object (void);
    //--------------------------
    // initializes Object by another Object
    void init (const Object * other);
    /** registers the points of the polygon (polygon points change!) and adds this polygon 
       * to Polygon_List polygons; the polygon is not copied! */
    void registrate(const Polygon *);
    /// copies the polygon list to Polygon_List polygons and then registers all points of polygons.
    void registrate(const Polygon_List *);
    /// build shape of object
    void build (void);
    /// attaches pilot to object
    void attach_pilot(Pilot &);
    /// attaches cockpit to object
    void attach_cockpit(Cockpit &);
    /// attach shot object to object
    void attach_shot(Object &);
    /// attaches explosion object to object
    void attach_explosion(Object &);
    /// processes explosion object of object
    void update_object(View &);
    /// mixes the Polygon_List of which shape of object consists
    void shuffle(void);
    /// this function is good to change painting of object
    void replace_mipmap_by(Mipmap &,Mipmap &); 
    /// transforms object to world coordinate system
    void to_world(void);
    /// transforms object to view coordinate system (it is assumed, that to_world() was used before)
    void to_view(View &);
    /** true iff object could be visible by other view. It decides whether object 
       * could be inside the others viewing frustrum */
    bool visible_by_view (View &);
    /// 
    void process(void);
    /// draws the object, which is observed from View
    void draw(View *);
    ///
    bool solid_and_draw (void);
    /// true iff spheres around objects with size_radius have non empty intersection
    bool touches (Object * o);
    /// destroys object 
    void destroy (void);
    ///
    void collision (Object & other);  
    ///
    double collision_time_ss(Object &other);
    ///
    double collision_time_s(Object &other);
    ///
    double collision_time_n(Object &other);
    ///
    double collision_time(Object &other);
    /** calculates radius of object (call only after all polygons 
       * are registrated by object !). This is necessary because
       * calc_radius uses tpoints data member. */
    void calc_size_radius (void);
    /// 
    double size (void) { return size_radius; };
};

#endif F_OBJECTS











