// objects.cc

/*
   Sofie, a real time 3d engine / Copyright (C) 1997 Stephan Schiessling
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include "objects.h"
#include <stdio.h>
#include <math.h>
#include <stdlib.h>

///
Object::Object (void) {
  size_type=1;
  radius=-1;
  power=0;
  mass=0;
  live_time=0;
  shoot_cycle_init=0;
  facing=false;
  treat_as_sphere=true;
  solid=true;
  velocity.init(0,0,0);
  nr_of_shots=0;
  shots=NULL;
  explosion=NULL;
  shot_lock=false;
  shoot_right=true;
  next_shot=0;
  pilot=NULL;
  cockpit=NULL;
  id=NULL;
  cid=NULL;
  shape=NULL;
};

///
Object::Object (const Object * other) {
  init(other);
};

///should not be used
Object::Object (const Object & other) {
  cout << "ERROR: don't uses that copy-constructor, argument should be a pointer to Object!\n";
  exit(-1);
};

///
Object::~Object (void) {
  // doit!
};


void Object::init (const Object * other) {
  size_type=other->size_type;;
  radius=other->radius;
  power=other->power;
  mass=other->mass;
  live_time=other->live_time;
  shoot_cycle_init=other->shoot_cycle_init;
  facing=other->facing;
  treat_as_sphere=other->treat_as_sphere;
  solid=other->solid;
  view.init(other->view);
  velocity.init(other->velocity);
  pilot=other->pilot;
  cockpit=other->cockpit;
  nr_of_shots=other->nr_of_shots;
  shot_lock=other->shot_lock;
  shoot_right=other->shoot_right;
  next_shot=other->next_shot;
  // copy shot objects, not only pointers of shot objects
  if (nr_of_shots>0) {
    Object *op= new Object[nr_of_shots];
    shots=new (Object*)[nr_of_shots];
    for (int i=0; i<nr_of_shots; i++) { 
      op[i].init(other->shots[i]);
      shots[i]=&op[i];
    };
  };
  if (other->explosion != NULL) 
    explosion=new Object(other->explosion);
  else
    explosion=NULL;
  id=NULL; // do not take the others id !
  cid=NULL;
  if (other->shape==NULL) shape=NULL;
  else {
    //shape=other->shape->new_Polygon_List(other->shape); // virtual constructor
    shape=(BSP_Tree *) other->shape->clone();
    shape->registrate_points(tpoints,dir_tpoints);
  };
};


///
void Object::registrate(const Polygon *W) {
  Polygon * poly=W->new_Polygon(W); // copy whole Polygon
  //poly->registrate_points(tpoints,dir_tpoints);
  shape->add(poly);
}; 


///
void Object::registrate(const Polygon_List * W) {
  shape->add(W);  
  //shape->registrate_points(tpoints,dir_tpoints);
};


///
void Object::build (void) {
  if (shape==NULL) {
    cout << "ERROR: no shape of object defined!\n";
    exit(-1);
  };
  shape->build();
  shape->registrate_points(tpoints,dir_tpoints);
};

///
void Object::attach_pilot(Pilot &pil) {
  pilot=&pil;
};

///
void Object::attach_cockpit(Cockpit &cock) {
  cockpit=&cock;
};

///
void Object::attach_shot(Object &shot) {
  Object **nshots=new (Object*)[nr_of_shots+1]; //bad
  for (int i=0; i<nr_of_shots; i++) 
    nshots[i]=shots[i];
  nshots[nr_of_shots]=&shot;
  nr_of_shots++;
  delete[] shots;
  shots=nshots;
};

///
void Object::attach_explosion(Object &explos) {
  explosion=&explos;
};


///
void Object::calc_size_radius (void) {
  if (&tpoints == NULL) {
    cout << "May be an ERROR: calc_radius() cannot work, because no TPoints are registrated!\n";
    return;
  };
  size_radius=tpoints.calc_max_length();
};


///
void Object::shuffle(void) {
  shape->shuffle();
};

///
void Object::replace_mipmap_by(Mipmap &t1,Mipmap &t2) {
  shape->replace_mipmap_by(t1,t2);
}; 

///
/*
void Object::calc_bsp_tree(void) {
  bsp.build(polygons);
};
*/

///
void Object::to_world(void) {
  tpoints.to_world(view);
  dir_tpoints.dir_to_world(view);
};

///
void Object::to_view(View &v) {
  tpoints.to_view(v);
  dir_tpoints.dir_to_view(v);
};

///
bool Object::visible_by_view (View &v) {
  Point sp(view.O);
  sp.to_view(v);
  return frustrum.intersects(sp,size_radius);
};



///
void Object::process(void) {
  Input * input=pilot->process(view);
  // translation first
  velocity=input->x*view.Xv+input->y*view.Yv+input->z*view.Zv;

  double x1,y1,z1;
  // then rotation around x-axis (in view coordinate system)
  // Xv does not change, 
  // Yv=(Xv,Yv,Zv)*(0,cos(anglex),sin(anglex))
  // Zv analogous, or with vector product (Xv,Yv are known)
  if (input->anglex!=0) {
    y1=cos(input->anglex);
    z1=sin(input->anglex);
    view.Yv=y1*view.Yv+z1*view.Zv;
    VECTOR_PRODUCT(view.Zv,view.Xv,view.Yv);
  };
  // rotation around y-axis
  if (input->angley!=0) {
    x1=cos(input->angley);
    z1=sin(input->angley);
    view.Xv=x1*view.Xv+z1*view.Zv;
    VECTOR_PRODUCT(view.Zv,view.Xv,view.Yv);
  };
  // rotation around z-axis
  if (input->anglez!=0) {
    x1=cos(input->anglez);
    y1=sin(input->anglez);
    view.Xv=x1*view.Xv+y1*view.Yv;
    VECTOR_PRODUCT(view.Yv,view.Zv,view.Xv);
  };
  // normalize, otherwise image is moving because of rounding faults
  view.normalize(); 

  //now the shots

  if (nr_of_shots == 0) return;

  if (input->shoot) {
    if (shoot_cycle <= 0) { // ok, ready
      if (shots[next_shot]->power <= 0) {
	shoot_cycle=shoot_cycle_init;
	shots[next_shot]->power=1.0; // a shots gets a power unit
	shots[next_shot]->view.init(view);
	if (shoot_right) {
	  shots[next_shot]->view.O=shots[next_shot]->view.O+shots[next_shot]->view.Zv+4*shots[next_shot]->view.Xv;
	  shoot_right=false;
	}
	else {
	  shots[next_shot]->view.O=shots[next_shot]->view.O-shots[next_shot]->view.Zv+4*shots[next_shot]->view.Xv;
	  shoot_right=true;
	};
	shots[next_shot]->live_time=90;
	next_shot++;
	if (next_shot>=nr_of_shots) next_shot=0;
      };
    };
  };

};

/*
///
void Object::update_shots(void) {
  shot_lock=false;
  for (int i=0; i<nr_of_shots;i++) {
    if (shots[i]->power > 0) {
      shots[i]->live_time--;
      if (shots[i]->live_time<0)  shots[i]->power=0; // loose all power
      if (shots[i]->live_time>80) {
	shot_lock=true;
      };
    };
  };
};
*/


///
void Object::update_object(View &v) {

  shoot_cycle--;
  if (shoot_cycle <= 0) shoot_cycle=0;
  if (live_time == 1 ) 
    power=0;
  else {
    live_time--;
    if (facing) { // facing object
      view.Xv=(-1)*v.Xv;
      view.Yv=v.Yv;
      view.Zv=(-1)*v.Zv;
    };
  };
  
/*  
  if (explosion == NULL) return;
  if (explosion->power > 0) {
    if ((explosion->live_time--) < 0) 
      explosion->power=0;
    // explosions are facing objects
    explosion->view.Xv=(-1)*v.Xv;
    explosion->view.Yv=v.Yv;
    explosion->view.Zv=(-1)*v.Zv;
  };
  */
};

///
void Object::draw(View *v) {
  if (visible_by_view(*v)) {
    // only in that case object could be visible by viewer
    to_view(*v);
    shape->draw(v);
  };
}; 

///
bool Object::touches (Object *o) {
  Point sp=o->view.O-view.O;
  double dist2=sp.x*sp.x+sp.y*sp.y+sp.z*sp.z;
  double sum=size_radius+o->size_radius;
  if (dist2<= (sum*sum)) return true;
  return false;
};



///
bool Object::solid_and_draw (void) {
  return ((power > 0) & solid);
};

void Object::destroy (void) {
  power=0;
  // initiate explosion
  if (id != NULL) 
    if (this->solid==true)
      cout << "Object " << *id << " destroyed!" << endl;
  if (explosion != NULL) { 
    explosion->power=100;
    if (solid)
      explosion->live_time=30;
    else
      explosion->live_time=4;
    explosion->view.O=view.O;
  };
  
  
};

///
void Object::collision (Object & other) {
  // do nothing if at least one of the two does not exist
  if (power <=0 ) return;
  if (other.power <=0 ) return;
  
  if (other.id == NULL) return; // no collisions 
  if (cid !=NULL)
      if (other.id->matches(*cid)) return;
  
  if (collision_time(other) < 1.0) { // collision
    float sp=power;
    power-=(other.power+other.mass);
    if (power <= 0) 
      destroy();
    other.power-=(sp+mass);
    if (other.power <= 0) 
      other.destroy();
    //cout << "COLLISION\n";
  }; // end if
};

///
double Object::collision_time_ss(Object &other) {
  Point P(other.view.O-view.O);
  Point v(other.velocity-velocity);
  double r=radius+other.radius;
  double s=P*v;
  double v2=v*v;
  double d=s*s-v2*((P*P)-r*r); // discriminate
  if (d<0) return INFINITY; // no collision
  double t1,t2;
  v2=1.0/v2;
  d=sqrt(d);
  t1=(-s+d)*v2;
  t2=(-s-d)*v2; // t1>=t2 always true
  if (t2>=0) return t2;
  if (t1>=0) return t1;
  return INFINITY;
};

/// treat other as sphere and *this as collection of polygons
double Object::collision_time_s(Object &other) {  
  return shape->collision_time(other.velocity,other.view.O,other.radius);
};

///
double Object::collision_time_n(Object &other) {
  return -1.0; // not jet implemented
};

///
double Object::collision_time(Object &other) {
  if (treat_as_sphere) {
    if (other.treat_as_sphere) return collision_time_ss(other);
    return other.collision_time_s(*this);
  }
  if (other.treat_as_sphere) return collision_time_s(other);
  return collision_time_n(other);
};

//=============================================










