// input.cc

/*
   Sofie, a real time 3d engine / Copyright (C) 1997 Stephan Schiessling
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include "input.h"
#include <math.h>

#ifdef JOYSTICK
#include <linux/joystick.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <iostream.h>
#include "globals.h"

///
struct JS_DATA_TYPE js0,js1;
int joy_fd0,joy_status0;
int joy_fd1,joy_status1;
double joy_maxx;
double joy_maxy;
double joy_maxz;
double joy_minx;
double joy_miny;
double joy_minz;
double joy_centerx;
double joy_centery;
double joy_centerz;

void init_joystick(void);
void calibrate_joystick(void);
void calc_joystick(Input &input);

#endif


Input::Input(void) {
  ///
  clear();
  ///
  for (int i=0; ///
 i<nr_of_shifts; ///
 i++) {
    shake_shift[i]=am_shift*sin(i*2*3.14159265/nr_of_shifts);
  };
};

///
void Input::clear(void) {
  anglex=0;
  angley=0;
  anglez=0;
  x=0;
  y=0;
  z=0;
  t_factor=1;
  g_factor=1;
  shoot=false;
  shake=false;
  timer=0;
};

///
void Input::incr(void) {
  t_factor*=1.02;
  g_factor*=1.1;
};

///
void Input::decr(void) {
  t_factor*=0.98039;
  g_factor*=0.90909;
};

// the next is used for shake
// function is executed each time
/// the keyboard is watched
void Input::pre(void) { 
  y=0;
};
  
///
void Input::calc(void) {
  anglex*=g_factor;
  if (anglex>0.1) anglex=0.1;
  if (anglex<-0.1) anglex=-0.1;
  angley*=g_factor;
  if (angley>0.1) angley=0.1;
  if (angley<-0.1) angley=-0.1;
  anglez*=g_factor;
  if (anglez>0.1) anglez=0.1;
  if (anglez<-0.1) anglez=-0.1;
  x*=t_factor;
  if (x>2) x=2;
  if (x<-2) x=-2;
  y*=t_factor;
  if (y>2) y=2;
  if (y<-2) y=-2;
  z*=t_factor;
  if (z>2) z=2;
  if (z<-2) z=-2;

  if (shake | (timer!=0)) {
    timer++;
    if (timer>=nr_of_shifts) timer=0;
    y+=shake_shift[timer];
    //printf("y= %f  shift= %f  \n",y,shift[timer]);
  }; // end shake

  
#ifdef JOYSTICK
    calc_joystick(*this);
#endif

};

///
void Input::shoot_on(void) {
  shoot=true;
};

///
void Input::stop_shoot(void) {
  shoot=false;
};

/// for x
void Input::incx(void) {
  x=DX;
};

///
void Input::decx(void) {
  x=-DX;
};

///
void Input::stopx(void) {
  x=0;
};

/// for y
void Input::incy(void) {
  y=DY;
};

///
void Input::decy(void) {
  y=-DY;
};

///
void Input::stopy(void) {
  y=0;
};

/// for z
void Input::incz(void) {
  z=DZ;
};

///
void Input::decz(void) {
  z=-DZ;
};

///
void Input::stopz(void) {
  z=0;
};

/// for anglex
void Input::incax(void) {
  anglex=DAX;
};

///
void Input::decax(void) {
  anglex=-DAX;
};

///
void Input::stopax(void) {
  anglex=0;
};

/// for angley
void Input::incay(void) {
  angley=DAY;
};

///
void Input::decay(void) {
  angley=-DAY;
};

///
void Input::stopay(void) {
  angley=0;
};

/// for anglez
void Input::incaz(void) {
  anglez=DAZ;
};

///
void Input::decaz(void) {
  anglez=-DAZ;
};

///
void Input::stopaz(void) {
  anglez=0;
};

//======================================

Global_Input::Global_Input(void) {
  clear();
};

///
void Global_Input::clear(void) {
  quit=false;
  special=NO;
  // for shake
  cockpit=false;
  con_notify=false;
};

// the next is used for shake
// function is executed each time
/// the keyboard is watched
void Global_Input::pre(void) { 
  special=NO;
};

//==========================================

#ifdef JOYSTICK
///
void init_joystick(void) {
  joy_fd0=open ("/dev/js0", O_RDONLY);
  joy_fd1=open ("/dev/js1", O_RDONLY);
  if ((joy_fd0 < 0) || (joy_fd1 < 0)) {
    cout << "ERROR: Joystick not available!\n";
    exit (-1);
  };
};

///
void calibrate_joystick(void) {
  // calibrate joystick
  joy_status0 = read (joy_fd0, &js0, JS_RETURN);
  if (joy_status0 != JS_RETURN) {
    cout << "ERROR: Joystick error!\n";
    exit (-1);
  };
  joy_status1 = read (joy_fd1, &js1, JS_RETURN);
  if (joy_status1 != JS_RETURN) {
    cout << "ERROR: Joystick error!\n";
    exit (-1);
  };
  while ((read (joy_fd0, &js0, JS_RETURN) > 0) && 
	 (read (joy_fd1, &js1, JS_RETURN) > 0) &&
	 js0.buttons != 0x00);
  joy_centerx=(double) js0.x;
  joy_centery=(double) js0.y;
  joy_centerz=(double) js1.y;
  //printf("%f  %f  %f\n",joy_centerx,joy_centery,joy_centerz);
  cout << "Move joystick to lower right corner and press either button\n";
  while ((read (joy_fd0, &js0, JS_RETURN) > 0) && 
	 (read (joy_fd1, &js1, JS_RETURN) > 0) && 
	 js0.buttons == 0x00);
  joy_maxx=(double) js0.x;
  joy_maxy=(double) js0.y;
  joy_maxz=(double) js1.y;
  //printf("%f  %f  %f\n",joy_maxx,joy_maxy,joy_maxz);
  while ((read (joy_fd0, &js0, JS_RETURN) > 0) && 
	 (read (joy_fd1, &js1, JS_RETURN) > 0) && 
	 js0.buttons != 0x00);
};  

///
void calc_joystick(Input &input) {
  joy_status0 = read (joy_fd0, &js0, JS_RETURN);
  if (joy_status0 != JS_RETURN) {
    cout << "ERROR: Joystick error!\n";
    exit (-1);
  };
  joy_status1 = read (joy_fd1, &js1, JS_RETURN);
  if (joy_status1 != JS_RETURN) {
    cout << "ERROR: Joystick error!\n";
    exit (-1);
  };
  if (js0.buttons & 1) input.shoot=true;
  if (js0.buttons & 2) input.angley=0;
  else input.angley=((double) js0.x-joy_centerx)*0.00002;
  input.anglex=((double) js0.x-joy_centerx)*0.00002;
  input.anglez=-((double) js0.y-joy_centery)*0.00002;
  input.x=-((double) js1.y-joy_centerz)*0.001;
};

#endif
