// cockpit.cc

/*
   Sofie, a real time 3d engine / Copyright (C) 1997 Stephan Schiessling
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include "cockpit.h"
#include <stdio.h>
#include <procbuf.h>
#include "graphics.h"

///
extern Graphics graphics;

///
Cockpit::Cockpit (void) {
  keep_in_memory=false;
  in_memory=false;
  friends=NULL;
  not_moving_enemies=NULL;
  fradar_x=0.5;
  fradar_y=0.83;
  fradar_size=0.08;
  fpowerdisplay_x=0.5;
  fpowerdisplay_y=0.65;;
  fpowerdisplay_size_x=0.1;
  fpowerdisplay_size_y=0.015;
  fspeeddisplay_x=0.3;
  fspeeddisplay_y=0.8;
  fspeeddisplay_size_x=0.05;
  fspeeddisplay_size_y=0.015;
  name=NULL;
};

///
Cockpit::Cockpit (char * filename, Pixel bo) {
  keep_in_memory=false;
  in_memory=false;
  blend_out=bo;
  name=filename;
  friends=NULL;
  not_moving_enemies=NULL;
  fradar_x=0.5;
  fradar_y=0.83;
  fradar_size=0.08;
  fpowerdisplay_x=0.5;
  fpowerdisplay_y=0.65;
  fpowerdisplay_size_x=0.15;
  fpowerdisplay_size_y=0.015;
  fspeeddisplay_x=0.3;
  fspeeddisplay_y=0.8;
  fspeeddisplay_size_x=0.05;
  fspeeddisplay_size_y=0.015;
  //cockpit.load(filename);
  //sm.init(cockpit,blend_out); 
};

///
Cockpit::~Cockpit (void) {
  //doit;
};


/// true iff in_memory and of the right size
bool Cockpit::is_ok(void) {
  if (!in_memory) return false;
  if (frustrum.scr_width != cockpit.width) return false;
  if (frustrum.scr_height != cockpit.height) return false;
  return true;
};


///
void Cockpit::load(void) {
  if (in_memory) return;
  set_radar(); // calculates the new coordinates for the radar
  set_powerdisplay(); // calculates the new coordinates for powerdiplay
  set_speeddisplay(); // calculates the new coordinates for speeddiplay
  String real_filename(name);
  real_filename+=".";
  real_filename+=i2string(frustrum.scr_width);
  real_filename+="x";
  real_filename+=i2string(frustrum.scr_height);
  String shorten_filename(real_filename);
  real_filename+=".ppm.gz";
  cout << "Try to find cockpit " << real_filename << " ....\n";
  if (cockpit.load((const char *)shorten_filename)) {
    // cockpit with this size exists
    in_memory=true;
    sm.init(cockpit,blend_out);
  }
  else { // cockpit of this size is not there
    cout << "Not found, try to generate ... ";
    String command("gzip -d -c -S .ppm.gz "); //pnmscale -width ");
    command+=name;
    command+=".320x200 | pnmscale -width ";
    command+=i2string(frustrum.scr_width);
    command+=" -height ";
    command+=i2string(frustrum.scr_height);
    //cout << command << endl; 
    procbuf buf((const char *)command,ios::in); // open pipe
    istream inp(&buf);
    if (cockpit.load (inp)) {    
      in_memory=true;
      sm.init(cockpit,blend_out);
    };
  };
};

///
void Cockpit::set_radar(void) {
  radar_x=(int) (((float) frustrum.scr_width)*fradar_x);
  radar_y=(int) (((float) frustrum.scr_height)*fradar_y);
  radar_size=(int) (((float) frustrum.scr_width)*fradar_size);
};

///
void Cockpit::draw_radar(Radar &radar,String * stid=NULL) {
  Pixel * pstart;
  Pixel * buffer=graphics.framebuffer;
  Radar_Object * st=NULL;
  unsigned short front_color;
  unsigned short back_color;
  unsigned short color;
  // clear radar
  for (int r=-radar_size; r<=radar_size;r++) {
    for (int c=-radar_size;c<=radar_size;c++) {
      pstart=buffer+(radar_y+r)*frustrum.scr_width+(radar_x+c);
      *pstart=0;
    };
  };

  for (int i=radar.nr_of_objects-1; i>=0; i--) {
    Radar_Object * ro=&radar.object[i];

    //st=ro; // remove
    
    
    front_color=63488;
    back_color=31;
    
    if (friends != NULL) {
      if (ro->id->matches(*friends)) {
	front_color=2016;
	back_color=2047;
      }
    }; 
    
    if (not_moving_enemies != NULL) {
      if (ro->id->matches(*not_moving_enemies)) {
	front_color=65535;
	back_color=65504;
      };
    };      

    if (stid != NULL) {
      if (ro->id->matches(*stid)) {
	st=ro;
      };
    };      
    
#ifdef HICOLOR 
    if (ro->in_front_of) color=front_color; 
    else color = back_color; 
    if (ro->distance > 500) {
      color>>=1;
      if (ro->distance >4000) {
	color>>=1;
      };
    };
    //color&=0xf81f;
#else
    // not such complicated in 8 bit colordepth
    if (ro->in_front_of) color=210; 
    else color = 15;
#endif;
    Pixel * pstart=buffer+(radar_y+(int)(radar_size*ro->x))*frustrum.scr_width+(radar_x+(int)(radar_size*ro->y));
    *pstart=color;
    pstart++;
    *pstart=color;
    pstart++;
    *pstart=color;
    pstart+=frustrum.scr_width;
    *pstart=color;
    pstart--;
    *pstart=color;
    pstart--;
    *pstart=color;
  };

  if (st==NULL) return;
  // there was a special target...

  if (st->in_front_of) color=65535; 
  else color = 65504; 
  
  for (int r=-radar_size; r<=radar_size;r++) {
    pstart=buffer+(radar_y+r)*frustrum.scr_width+(radar_x+(int)(radar_size*st->y));
    *pstart=color;
  };
  for (int c=-radar_size;c<=radar_size;c++) {
    pstart=buffer+(radar_y+(int)(radar_size*st->x))*frustrum.scr_width+(radar_x+c);
    *pstart=color;
  };
  
}

///
void Cockpit::set_powerdisplay(void) {
  powerdisplay_x=(int) (((float) frustrum.scr_width)*fpowerdisplay_x);
  powerdisplay_y=(int) (((float) frustrum.scr_height)*fpowerdisplay_y);
  powerdisplay_size_x=(int) (((float) frustrum.scr_width)*fpowerdisplay_size_x);
  powerdisplay_size_y=(int) (((float) frustrum.scr_height)*fpowerdisplay_size_y);
};

///
void Cockpit::draw_powerdisplay(float p) {
  Pixel * pstart;
  Pixel * buffer=graphics.framebuffer;
  unsigned short color;
  
  if (p < 0) p=0;
  else if (p > 100) p=100;

  int pp=(int) (((float) powerdisplay_size_x)*p*0.02)-powerdisplay_size_x;
  
  for (int r=-powerdisplay_size_y; r<=powerdisplay_size_y;r++) {
    for (int c=-powerdisplay_size_x;c<=pp;c++) {
      pstart=buffer+(powerdisplay_y+r)*frustrum.scr_width+(powerdisplay_x+c);
      *pstart=65504;
    };
    for (int c=pp+1;c<=powerdisplay_size_x;c++) {
      pstart=buffer+(powerdisplay_y+r)*frustrum.scr_width+(powerdisplay_x+c);
      *pstart=31;
    };
  };
}


///
void Cockpit::set_speeddisplay(void) {
  speeddisplay_x=(int) (((float) frustrum.scr_width)*fspeeddisplay_x);
  speeddisplay_y=(int) (((float) frustrum.scr_height)*fspeeddisplay_y);
  speeddisplay_size_x=(int) (((float) frustrum.scr_width)*fspeeddisplay_size_x);
  speeddisplay_size_y=(int) (((float) frustrum.scr_height)*fspeeddisplay_size_y);
};

///
void Cockpit::draw_speeddisplay(float p) {
  Pixel * pstart;
  Pixel * buffer=graphics.framebuffer;
  unsigned short color;
  
  if (p < 0) p=0;
  else if (p > 100) p=1.0;

  int pp=(int) (((float) speeddisplay_size_x)*p*0.02)-speeddisplay_size_x;
  
  for (int r=-speeddisplay_size_y; r<=speeddisplay_size_y;r++) {
    for (int c=-speeddisplay_size_x;c<=pp;c++) {
      pstart=buffer+(speeddisplay_y+r)*frustrum.scr_width+(speeddisplay_x+c);
      *pstart=65504;
    };
    for (int c=pp+1;c<=speeddisplay_size_x;c++) {
      pstart=buffer+(speeddisplay_y+r)*frustrum.scr_width+(speeddisplay_x+c);
      *pstart=31;
    };
  };
}

///
void Cockpit::draw(void) {
  Pixel * tstart=cockpit.content;
  Pixel * pstart=graphics.framebuffer;
  for (int co=frustrum.scr_width*frustrum.scr_height-1; co>=0; co--) {
    *pstart=*tstart;
    pstart++;
    tstart++;
  }; 
  if (!keep_in_memory) del();
}

/// unload cockpit-texture
void Cockpit::del(void) {
  cockpit.del();
  in_memory=false;
};



/* used to find easily the blend_out value
   printf("%d\n",*(cockpit.content+160));
   exit(-1);
   */

//=========================================
