// bsp.cc

/*
   Sofie, a real time 3d engine / Copyright (C) 1997 Stephan Schiessling
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/


// documentation for bsp-trees: look for the bsptree-faq (somewhere in the internet)


#include "bsp.h"
#include <stream.h>
#include <math.h>


///
BSP_Tree::BSP_Tree (void) : Polygon_List() {
  front=back=NULL;					     
};

///
BSP_Tree::BSP_Tree (const Polygon_List * w) : Polygon_List(w) {  
  front=back=NULL;
  build();
};

/// a virtual constructor
Polygon_List * BSP_Tree::new_Polygon_List(void) {
  return new BSP_Tree;
};


/// a virtual constructor
Polygon_List * BSP_Tree::new_Polygon_List (const Polygon_List *w) {
  return new BSP_Tree(w);
};

void BSP_Tree::copy (BSP_Tree * other) {
  Polygon_List::copy((Polygon_List *)other);
  if (other->front==NULL) front=NULL;
  else 
    front=(BSP_Tree *) other->front->clone();
  if (other->back==NULL) back=NULL;
  else 
    back=(BSP_Tree *) other->back->clone();
};

/// produces a copy of this object
Polygon_List * BSP_Tree::clone (void) {
  BSP_Tree *b=new BSP_Tree();
  b->copy(this);
  return b;
};


/// uses only relative coordinates, but plane_d (of polygons) have
/// to be calculated !! (for relative case) 
void BSP_Tree::build (void) {
  if (first==NULL) return; // polygon list empty
  if ((front!=NULL) || (back!=NULL)) {
    // BSP_Tree has already leafs
    // now we don't allow this, later it will be implemented
    cout << "ERROR: build BSP_Tree not possible, it has already leafs!\n";
    exit(-1);
  };
  
  // leafs are empty, initial Polygon_List not empty
  Plane partition;
  partition.n.init(*first->polygon->normal_r());
  partition.d=first->polygon->distance();
  front=new BSP_Tree;
  back=new BSP_Tree;
  Polygon *poly;
  W_Polygon * previous;
  W_Polygon * root=first; // != NULL
  while ((root=(previous=root)->next)!=NULL) {
    poly=root->polygon;
    switch (poly->classify(partition)) {
    case COINCIDENT:
      break; // leave this polygon in Polygon_List
    case IN_BACK_OF: 
      back->add(poly);
      remove(root); // remove W_Polygon from initial Polygon_List
      root=previous; // be carefull with previous, if previous==root before remove is called (here not possible)
      break;
    case IN_FRONT_OF:
      front->add(poly);
      remove(root); // remove W_Polygon from initial Polygon_List
      root=previous;
      break;
    case SPANNING:
      returnvalue pieces=poly->split(partition);
      back->add(pieces.back); 
      front->add(pieces.front); 
      remove(root); // remove W_Polygon from initial Polygon_List
      root=previous;
      delete poly; // and delete polygon 
      break;
    }; // end switch
  }; // end while  
  
  if (front->is_empty()) {
    delete front;
    front=NULL;
  } else 
    front->build();

  if (back->is_empty()) {
    delete back;
    back=NULL;
  } else 
    back->build();
  
};

///
BSP_Tree ** BSP_Tree::merge (BSP_Tree * other,Point * o) {
  BSP_Tree * run=this;
  BSP_Tree * previous=NULL;
  double sp;
  while (run != NULL) {  
    previous=run;
    sp=(*run->first->polygon->normal_w())*((*o)-*run->first->polygon->first_vertex_w());
    if (sp>=0) { // IN_FRONT_OF
      run=run->front;
    }
    else 
      run=run->back;
  };
  // previous is now the last Node != NULL
  
  if ( sp>=0 ) {
    previous->front=other; // problematic if other is only a Polygon_List!!!
    return &(previous->front);
  }
  previous->back=other; // problematic if other is only a Polygon_List!!!
  return &(previous->back);
};




/// registrate the vertices and direction vectors of the each polygon in Polygon_List
void BSP_Tree::registrate_points(TPoint_Set &t,TPoint_Set &dir_t) {
  Polygon_List::registrate_points(t,dir_t);
  if (front!=NULL)
    front->registrate_points(t,dir_t);
  if (back!=NULL) 
    back->registrate_points(t,dir_t);
};



///
void BSP_Tree::print (void) {
  cout << "\nList of BSP_Tree:\n";
  sub_print();
  cout << "End of List of BSP_Tree.\n\n";
};

///
void BSP_Tree::sub_print (void) {
  if (front!=NULL) {
    cout << " front:\n";
    front->sub_print();
  };
  if (back!=NULL)  {
    cout << " back:\n";
    back->sub_print();
  };
  cout << " this_node:\n";
  Polygon_List::print();
  cout << " up\n";
};


///
void BSP_Tree::draw(View *v=NULL) { // here v is used
  if (front==NULL && back==NULL) {
    Polygon_List::draw();
    return;
  };
  double sp=(*first->polygon->normal_w())*(v->O-*first->polygon->first_vertex_w());
  if (IS_ZERO(sp)) sp=0;
  // new
  if (sp>0) { // IN_FRONT_OF
    if (front!=NULL) front->draw(v);
    Polygon_List::draw();
    if (back!=NULL) back->draw(v);
  }
  else if (sp<0) { // IN_BACK_OF
    if (back!=NULL) back->draw(v);
    Polygon_List::draw();
    if (front!=NULL) front->draw(v);
  }
  else { // COINCIDENT: 
    if (front!=NULL) front->draw(v);
    if (back!=NULL) back->draw(v);
  };
};


