/*
 * Copyright (C) 1997 Tobias Gloth (gloth@unknown.westfalen.de)
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <ctype.h>
#include <dirent.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "xtoto/args.h"
#include "xtoto/avltree.h"
#include "xtoto/list.h"
#include "xtoto/stack.h"
#include "xtoto/xtoto.h"

#include "main.h"

#define _NAME "max"
#define _VERSION "0.1.0"
#define _COMMENT "helps to create large make-trees"
#define _HELP \
    "usage: " _NAME " [options] <definition>\n" \
    "  -?   --help             display usage help\n" \
    "  -L   --license          display license\n" \
    "  -V   --version          display version\n" \
    "  -v   --verbose          verbose mode on/off\n" \
    "\n" \
    "  -a   --autoconf         run autoconf when ready\n" \
    "  -b   --backup           create backup files\n" \
    "\n"

const char *NAME = _NAME;
const char *VERSION = _VERSION;
const char *HELP = _HELP;

const char *LICENSE =
    "\n"
    _NAME " v" _VERSION " - " _COMMENT "\n"
    "Copyright (C)97 Tobias Gloth (gloth@unknown.westfalen.de)\n"
    GNU_PUBLIC_LICENSE;

void get_args (int argc, char **argv);
void shut_down ();

typedef struct {
    int fix;
    char *name;
    char *NAME;
} SPECIAL;

#define SPECIALS 6
const SPECIAL special[SPECIALS] = {
    {0, "x11",     "X11"},
    {1, "net",     "NET"},
    {1, "threads", "THREADS"},
    {1, "dl",      "DL"},
    {1, "math",    "MATH"},
    {1, "crypt",   "CRYPT"}
};

typedef struct {
    char *name;
    struct LIST *need;
    struct LIST *like;
} APPLICATION;

typedef struct {
    char *name;
    char *NAME;
    int special;
    struct LIST *near;
    char *replace;
    struct LIST *need;
    struct LIST *alternative;
    struct LIST *lib;
    char *include;
    struct LIST *depend;
    struct LIST *pending;
    int resolved;
} LIBRARY;

typedef struct {
    char *name;
    char *NAME;
    char *alias;
    int failsafe;
    int hidden;
    struct LIST *need;
    struct LIST *like;
    struct LIST *depend;
    struct LIST *pending;
    int resolved;
} MODULE;

typedef struct {
    char *name;
    char *NAME;
    struct LIST *near;
    char *replace;
    struct LIST *alternative;
    char *file;
} PROGRAM;

typedef struct {
    char *name;
    char *NAME;
    char *cxx_suffix;
    char *depend;
    struct LIST *manual;
    struct LIST *language;
    int version_major;
    int version_minor;
    int patch_level;
} PROJECT;

PROJECT *project=0;

struct LIST *application=0;
APPLICATION *next_application=0;

struct LIST *library=0;
LIBRARY *next_library=0;

struct LIST *module=0;
MODULE *next_module=0;

struct LIST *program=0;
PROGRAM *next_program=0;

struct LIST *module_order=0;
struct LIST *library_order=0;

int backup = 0;
int autoconf = 1;
FILE *mk = 0;

#define txtcmp ((int(*)(const void *, const void*))strcmp)

void (*ident_insert) ();

int compare_library_name (const void *, const void *);
void copy_directory (const char *name);
void handle_unknown_dependency (char *name, struct LIST *depend, char *need);
int library_is_special (char *);
void parse_error (const char *);
void set_target (const char *, const char *, int mode);
int token_is_special (char *);
void touch_project_m4 ();
void translate (const char *);
void write_x11_libs ();
char *xX (const char *);
char *Xx (const char *);

/* add a like-dependency to current application */
void application_add_like () {
    list_insert (next_application->like, xX (yytext));
}

/* add a need-dependency to current application */
void application_add_need () {
    list_insert (next_application->need, xX (yytext));
}

/* create a new application */
void application_create () {
    next_application = (APPLICATION*) safe_malloc (sizeof (APPLICATION));
    next_application->name = safe_strdup (yytext);
    next_application->like = list_create ();
    next_application->need = list_create ();
}

/* destroy an application */
void application_destroy (void *ptr) {
    APPLICATION *app = (APPLICATION*) ptr;

    safe_free (app->name);
    list_destroy_all (app->like, safe_free);
    list_destroy_all (app->need, safe_free);
    safe_free (app);
}

/* destroy all applications in list */
void application_destroy_all () {
    list_destroy_all (application, application_destroy);
}

/* insert current application into list */
void application_end () {
    list_append (application, next_application);
}

/* check a condition in a template file */
int check_condition (char *cond) {

    /* check for languages */
    if (!strcmp (cond, "c") || !strcmp (cond, "c++")) {
        return list_search (project->language, cond, txtcmp) ? 1 : 0;
    }

    /* check manual type */
    if (!strcmp (cond, "man") || !strcmp (cond, "texinfo") ||
        !strcmp (cond, "doc")) {
        return list_search (project->manual, cond, txtcmp) ? 1 : 0;
    }

    /* check for dependency modes */
    if (!strcmp (cond, "smart") || !strcmp (cond, "traditional")) {
	char *tmp = project->depend ? project->depend : "traditional";
	return !strcmp (tmp, cond);
    }

    /* check for the presence of libraries */
    if (!strcmp (cond, "libs")) {
        return !list_is_empty (module);
    }

    /* check for special libraries */
    if (token_is_special (cond)) {
        return list_search (library, cond, compare_library_name) ? 1 : 0;
    }

    return -1;
}

/* make sure that no names were used more than once */
void check_names () {
    struct LIST_ITERATOR *iterator;
    struct AVLTREE *tree;
    MODULE *mod;
    LIBRARY *lib;
    PROGRAM *prog;

    tree = avltree_create (txtcmp);
    iterator = list_iterator_create ();

    /* insert the module names */
    list_iterator_reset (iterator, module);
    while ((mod = list_get_next (iterator)) != 0) {
        if (avltree_search (tree, mod->name)) {
	    error ("name %s used twice\n", mod->name);
	}
	avltree_insert (tree, mod->name);
    }
 
    /* insert the library names */
    list_iterator_reset (iterator, library);
    while ((lib = list_get_next (iterator)) != 0) {
        if (avltree_search (tree, lib->name)) {
	    error ("name %s used twice\n", lib->name);
	}
	avltree_insert (tree, lib->name);
    }
 
    /* insert the program names */
    list_iterator_reset (iterator, program);
    while ((prog = list_get_next (iterator)) != 0) {
        if (avltree_search (tree, prog->name)) {
	    error ("name %s used twice\n", prog->name);
	}
	avltree_insert (tree, prog->name);
    }
 
    /* ok, nothing suspicious found */
    list_iterator_destroy (iterator);
    avltree_unlink_all (tree);
}

int compare_library_name (const void *ptr1, const void *ptr2) {
    return strcmp (((LIBRARY*)ptr1)->name, (char*)ptr2);
}

int compare_program_name (const void *ptr1, const void *ptr2) {
    return strcmp (((PROGRAM*)ptr1)->name, (char*)ptr2);
}

void compose (const char *dir, const char *name, int mode) {
    FILE *in;
    char *source;
    char *line, *next, *first;
    int lineno=0, result, ch;
    struct STACK *visible;

    /* find the name of the source-file */
    source = safe_strcat (PREFIX "/lib/max/", dir, "/", name, (char*)0);
    verbose_printf ("%s -> %s/%s\n", source, dir, name);

    in = safe_fopen (source, "rt");
    set_target (dir, name, mode);

    /* create the stack */
    visible = stack_create ();

    /* copy the file line by line */
    while (1) {
        line = safe_input (in, '\n');
	if (!line) {
	    break;
	}
	lineno++;

	/* check for an escape sequence */
	if (line[0] == '@') {
	    if (!strncmp (line+1, "if !", 4)) {
	        if (!stack_get_size (visible) || stack_top (visible)) {
		    result = check_condition (line+5);
		} else {
		    result = 1;
		}
		if (result == -1) {
		    error ("invalid @if in line %d of \"%s\"\n",
		        lineno, source);
		}
	        stack_push (visible, (void*)!result);
	    } else
	    if (!strncmp (line+1, "if ", 3)) {
	        if (!stack_get_size (visible) || stack_top (visible)) {
		    result = check_condition (line+4);
		    if (result == -1) {
		        error ("invalid @if in line %d of \"%s\"\n",
		            lineno, source);
		    }
		} else {
		    debug_printf ("%s/%s: @if \"%s\" remains false\n",
		        dir, name, line+4);
		    result = 0;
		}
	        stack_push (visible, (void*)result);
	    } else
	    if (!strncmp (line+1, "else", 4)) {
		if (!stack_get_size (visible)) {
		    error ("invalid @else in line %d of \"%s\"\n",
		        lineno, source);
		}
		result = !(int)stack_pop (visible);
	        if (stack_get_size (visible) && !stack_top (visible)) {
		    result = 0;
		}
	        stack_push (visible, (void*)result);
	    } else
	    if (!strncmp (line+1, "end", 3)) {
		if (!stack_get_size (visible)) {
		    error ("invalid @end in line %d of \"%s\"\n",
		        lineno, source);
		}
	        stack_pop (visible);
	    } else {
		error ("unknown command in line %d of \"%s\"\n",
		    lineno, source);
	    }
	} else {
	    if (!stack_get_size (visible) || stack_top (visible)) {

		next = line;
		while (*next) {

		    /* skip all non-letters */
		    while (!isalpha (*next) && *next) {
		        fputc (*next, mk);
			next++;
		    }

		    /* at end of file? */
                    if (!*next) {
		        break;
		    }

		    first = next++;
		    while (isalpha (*next)) {
		        next++;
		    }

                    ch = *next;
		    *next = '\0';
		    translate (first);
		    *next = ch;
		}
		fputc ('\n', mk);
	    }
	}
	safe_free (line);
    }
    fclose (in);

    if (stack_get_size (visible)) {
         error ("open @if remains in line %d of \"%s\"\n",
             lineno, source);
    }

    stack_unlink_all (visible);

    safe_free (source);
}

void copy_directory (const char *name) {
    struct dirent *entry;
    char *dirname;
    DIR *dir;

    dirname = safe_strcat (PREFIX "/lib/max/", name, (char*)0);
    if (!(dir = opendir (dirname))) {
        error ("can't open directory \"%s\"\n", dirname);
    }

    safe_mkdir (name, 0755);

    while ((entry = readdir (dir))) {
	if (strcmp (entry->d_name, ".") && strcmp (entry->d_name, "..")) {
	    if (list_is_empty (module) && strstr (entry->d_name, "-sh")) {
	        continue;
	    }
            compose (name, entry->d_name, 0644);
	}
    }

    safe_free (dirname);
    closedir (dir);
}

void create_src_dirs () {
    struct LIST_ITERATOR *iterator, *iterator2;
    APPLICATION *app;
    MODULE *mod;
    char *like, *need;
    char *dirname, *alias;

    /* create the library-related directories */
    if (!list_is_empty (module)) {
        safe_mkdir ("include", 0755);
    }

    iterator = list_iterator_create ();
    iterator2 = list_iterator_create ();

    list_iterator_reset (iterator, module);
    while ((mod = list_get_next (iterator)) != 0) {
        alias = mod->alias ? mod->alias : mod->name;
        dirname = safe_strcat ("lib", alias, (char*)0);
	set_target (dirname, "GNUmakefile", 0644);
        if (list_search (project->language, "c", txtcmp)) {
	     fprintf (mk, "CFILES = $(wildcard *.c)\n");
	}
        if (list_search (project->language, "c++", txtcmp)) {
	     fprintf (mk, "CXXFILES = $(wildcard *.%s)\n",
	         project->cxx_suffix ? project->cxx_suffix : "cc");
	}
	fprintf (mk, "NAME = %s\n", mod->name);
	fprintf (mk, "alias = %s\n", mod->alias ? mod->alias : mod->name);
	fprintf (mk, "HIDDEN = %s\n", mod->hidden ? "YES" : "NO");
	fprintf (mk, "NEEDS = %s\n\n", mod->NAME);
	fprintf (mk, "%s_HOME = ..\n", project->NAME);
	fprintf (mk, "include $(%s_HOME)/rules/lib.mk\n", project->NAME);
	safe_free (dirname);

        dirname = safe_strcat ("include/", alias, (char*)0);
	safe_mkdir (dirname, 0755);
	safe_free (dirname);
    }

    list_iterator_reset (iterator, application);
    while ((app = list_get_next (iterator)) != 0) {
	set_target (app->name, "GNUmakefile", 0644);
        if (list_search (project->language, "c", txtcmp)) {
	     fprintf (mk, "CFILES = $(wildcard *.c)\n");
	}
        if (list_search (project->language, "c++", txtcmp)) {
	     fprintf (mk, "CXXFILES = $(wildcard *.%s)\n",
	         project->cxx_suffix);
	}
	fprintf (mk, "NAME = %s\n", app->name);

	if (list_search (project->manual, "man", txtcmp)) {
	     fprintf (mk, "MAN = %s.1\n", app->name);
	}

	/* write the like-dependencies */
	if (!list_is_empty (app->like)) {
	    fprintf (mk, "LIKES =");
	    list_iterator_reset (iterator2, app->like);
	    while ((like = list_get_next (iterator2)) != 0) {
	        fprintf (mk, " %s", like);
	    }
	    fprintf (mk, "\n");
	}

	/* write the need-dependencies */
	if (!list_is_empty (app->need)) {
	    fprintf (mk, "NEEDS =");
	    list_iterator_reset (iterator2, app->need);
	    while ((need = list_get_next (iterator2)) != 0) {
	        fprintf (mk, " %s", need);
	    }
	    fprintf (mk, "\n");
	}
	fprintf (mk, "\n");
	fprintf (mk, "%s_HOME = ..\n", project->NAME);
	fprintf (mk, "include $(%s_HOME)/rules/app.mk\n", project->NAME);
    }

    /* create the doc-directory */
    if (list_search (project->manual, "doc", txtcmp)) {
        compose ("doc", "GNUmakefile", 0644);
    }

    list_iterator_destroy (iterator);
    list_iterator_destroy (iterator2);
}

void find_alternatives () {
    struct LIST_ITERATOR *iterator;
    PROGRAM *prog, *prog2;
    LIBRARY *lib, *lib2;

    iterator = list_iterator_create ();
    list_iterator_reset (iterator, program);
    while ((prog = list_get_next (iterator)) != 0) {
        if (prog->replace) {
            if (!strcmp (prog->replace, prog->NAME)) {
	        error ("program %s can't replace itself\n", prog->name);
	    }
	    prog2 = list_search (program, prog->replace, compare_program_name);
	    if (!prog2) {
	        error ("program %s replaces unknown %s\n",
		    prog->name, prog->replace);
	    }
	    list_append (prog2->alternative, prog->NAME);
	}
    }

    list_iterator_reset (iterator, library);
    while ((lib = list_get_next (iterator)) != 0) {
        if (lib->replace) {
            if (!strcmp (lib->replace, lib->name)) {
	        error ("library %s can't replace itself\n", lib->name);
	    }
	    lib2 = list_search (library, lib->replace, compare_library_name);
	    if (!lib2) {
	        error ("library %s replaces unknown %s\n",
		    lib->name, lib->replace);
	    }
	    list_append (lib2->alternative, lib->NAME);
	}
    }
    list_iterator_destroy (iterator);
}

void handle_unknown_dependency (char *name, struct LIST *depend, char *need) {
    char *tmp;

    if (library_is_special (need)) {
	debug_printf ("adding unknown library %s\n", need);
        tmp = Xx (need);
        library_create (tmp);
	safe_free (tmp);
	library_end ();
	if (depend) {
	    list_append (depend, need);
	}
    } else {
        error ("%s has unknown dependency %s\n", name, need);
    }
}

void get_args (int argc, char **argv) {
    parse_command_line (argc, argv, ARG_ALL,
        ARG_SWITCH, "-a", "--autoconf", &autoconf,
        ARG_SWITCH, "-b", "--backup", &backup,
        ARG_END);
    if (fargc != 1) {
        show_help ();
    }
}

void ident_set_handler (void (*func) ()) {
    ident_insert = func;
}

void init_data () {
    library = list_create ();
    program = list_create ();
    module = list_create ();
    application = list_create ();
}

void library_add_near () {
    list_insert (next_library->near, safe_strdup (yytext));
}

void library_add_need () {
    list_append (next_library->need, xX (yytext));
}

void library_create (char *name) {
    next_library = (LIBRARY*) safe_malloc (sizeof (LIBRARY));
    next_library->special = 0;
    next_library->name = safe_strdup (name);
    next_library->NAME = xX (name);
    next_library->replace = 0;
    next_library->alternative = list_create ();
    next_library->near = list_create ();
    next_library->need = list_create ();
    next_library->lib = list_create ();
    next_library->include = 0;
    next_library->depend = list_create ();
    next_library->pending = 0;
    next_library->resolved = 0;
}

void library_destroy (void *ptr) {
    LIBRARY *lib = (LIBRARY*) ptr;
    safe_free (lib->name);
    safe_free (lib->NAME);
    safe_free (lib->replace);
    list_unlink_all (lib->alternative);
    list_destroy_all (lib->near, safe_free);
    list_destroy_all (lib->need, safe_free);
    list_destroy_all (lib->lib, safe_free);
    safe_free (lib->include);
    list_unlink_all (lib->depend);
    list_unlink_all (lib->pending);
    safe_free (lib);
}

void library_destroy_all () {
    list_destroy_all (library, library_destroy);
}

void library_end () {
    next_library->special = library_is_special (next_library->NAME);
    if (!library_is_special (next_library->NAME)) {
        if (!list_get_size (next_library->lib)) {
            parse_error ("no library file specified\n");
        }
        if (!next_library->include) {
            parse_error ("no include file specified\n");
        }
    }
    list_append (library, next_library);
}

/* check if "name" is a special library */
int library_is_special (char *name) {
    int i;

    for (i=0; i<SPECIALS; i++) {
        if (!strcmp (name, special[i].NAME)) {
	    return 1;
	}
    }
    return 0;
}

void library_set_include () {
    if (next_library->include) {
        parse_error ("include file already specified\n");
    }
    next_library->include = safe_strdup (yytext);
}

void library_add_lib () {
    list_append (next_library->lib, safe_strdup (yytext));
}

void library_set_replace () {
    if (next_library->replace) {
        parse_error ("replace library already specified\n");
    }
    next_library->replace = safe_strdup (yytext);
}

void module_create () {
    next_module = (MODULE*) safe_malloc (sizeof (MODULE));
    next_module->name = safe_strdup (yytext);
    next_module->NAME = xX (yytext);
    next_module->alias = 0;
    next_module->failsafe = 0;
    next_module->hidden = 0;
    next_module->need = list_create ();
    next_module->like = list_create ();
    next_module->depend = list_create ();
    next_module->pending = 0;
    next_module->resolved = 0;
}

void module_destroy (void *ptr) {
    MODULE *module = (MODULE*) ptr;
    safe_free (module->name);
    safe_free (module->NAME);
    safe_free (module->alias);
    list_destroy_all (module->like, safe_free);
    list_destroy_all (module->need, safe_free);
    list_unlink_all (module->depend);
    list_unlink_all (module->pending);
    safe_free (module);
}

void module_destroy_all () {
    list_destroy_all (module, module_destroy);
}

void module_end () {
    list_append (module, next_module);
}

void module_add_like () {
    list_append (next_module->like, xX (yytext));
}

void module_add_need () {
    list_append (next_module->need, xX (yytext));
}

void module_set_alias () {
    if (next_module->alias) {
        parse_error ("module alias already specified\n");
    }
    next_module->alias = safe_strdup (yytext);
}

void module_set_private () {
    next_module->hidden = 1;
}

int module_search (const void *ptr1, const void *ptr2) {
    return strcmp (((MODULE*)ptr1)->NAME, ptr2);
}

void parse_error (const char *err_msg) {
    error ("line %d: %s\n", yylineno, err_msg);
}

void project_create () {
    PROJECT *next_project;

    if (project) {
        parse_error ("project declared twice\n");
    }
    next_project = (PROJECT*) safe_malloc (sizeof (PROJECT));
    next_project->language = list_create ();
    next_project->name = safe_strdup (yytext);
    next_project->NAME = xX (yytext);
    next_project->cxx_suffix = 0;
    next_project->depend = 0;
    next_project->manual = list_create ();
    next_project->version_major = 0;
    next_project->version_minor = 0;
    next_project->patch_level = 1;
    project = next_project;
}

void project_destroy () {
    if (project) {
        safe_free (project->name);
        safe_free (project->NAME);
        safe_free (project->cxx_suffix);
        safe_free (project->depend);
        list_destroy_all (project->language, safe_free);
        list_destroy_all (project->manual, safe_free);
	safe_free (project);
    }
}

void project_end () {
    if (list_is_empty (project->language)) {
        error ("no languages specified\n");
    }
}

void project_add_language () {

    /* check if it is a known format */
    if (strcmp (yytext, "c") && strcmp (yytext, "c++")) {
        error ("unsupported language: \"%s\"\n", yytext);
    }

    /* insert the language */
    list_append (project->language, safe_strdup (yytext));
}

void project_add_manual () {

    /* check if it is a known format */
    if (strcmp (yytext, "texinfo") && strcmp (yytext, "man") &&
        strcmp (yytext, "doc")) {
        error ("unsupported manual type: \"%s\"\n", yytext);
    }

    /* insert the manual type */
    list_append (project->manual, safe_strdup (yytext));
}

void project_set_cxx_suffix () {
    if (project->cxx_suffix) {
        parse_error ("c++ suffix specified twice\n");
    }
    project->cxx_suffix = safe_strdup (yytext);
}

void project_set_depend () {
    if (project->depend) {
        parse_error ("dependency mode specified twice\n");
    }
    if (strcmp (yytext, "smart") && strcmp (yytext, "traditional")) {
        parse_error ("unknown dependency mode\n");
    }
    project->depend = safe_strdup (yytext);
}

void project_set_version () {
    char *next;

    /* parse the high number */
    project->version_major = strtol (yytext, &next, 10);
    if ((project->version_major < 0) || (*next != '.')) {
        error ("invalid version\n");
    }

    /* parse the low number */
    project->version_minor = strtol (next+1, &next, 10);
    if ((project->version_minor < 0) || (*next != '.')) {
        error ("invalid version\n");
    }

    /* parse the patchlevel */
    project->patch_level = strtol (next+1, &next, 10);
    if ((project->patch_level < 0) || (*next != '\0')) {
        error ("invalid version\n");
    }
}

void program_add_near () {
    list_append (next_program->near, safe_strdup (yytext));
}

void program_create () {
    next_program = (PROGRAM*) safe_malloc (sizeof (LIBRARY));
    next_program->name = safe_strdup (yytext);
    next_program->NAME = xX (yytext);
    next_program->alternative = list_create ();
    next_program->near = list_create ();
    next_program->file = 0;
    next_program->replace = 0;
}

void program_destroy (void *ptr) {
    PROGRAM *prog = (PROGRAM*) ptr;
    safe_free (prog->name);
    safe_free (prog->NAME);
    list_destroy_all (prog->near, safe_free);
    list_unlink_all (prog->alternative);
    safe_free (prog->replace);
    safe_free (prog->file);
    safe_free (prog);
}

void program_destroy_all () {
    list_destroy_all (program, program_destroy);
}

void program_end () {
    if (!next_program->file) {
        parse_error ("no program file specified\n");
    }
    list_append (program, next_program);
}

void program_set_file () {
    if (next_program->file) {
        parse_error ("file already specified\n");
    }
    next_program->file = safe_strdup (yytext);
}

void program_set_replace () {
    if (next_program->replace) {
        parse_error ("replace program already specified\n");
    }
    next_program->replace = safe_strdup (yytext);
}

void read_definition () {
    yyin = safe_fopen (fargv[0], "rt");
    yyparse ();
}

void run_autoconf () {
    if (autoconf) {
        system ("cd config && autoconf");
    }
}

void set_target (const char *dir, const char *name, int mode) {
    struct stat statbuf;
    char *filename, *tmp;

    /* create the directory */
    safe_mkdir (dir, mode | 0111);

    /* close the old output file */
    if (mk) {
        fclose (mk);
    }

    /* concatenate dir and filename */
    filename = safe_strcat (dir, "/", name, (char*)0);

    /* make a backup of the old file */
    if (backup) {
	printf ("stat_size: %d\n", sizeof (struct stat));
        fflush (stdout);
        if (stat (filename, &statbuf) != -1) {
	    tmp = safe_strcat (filename, ".bat", (char*)0);
    	    if (rename (filename, tmp) == -1) {
    	        error ("can't make a backup of \"%s\"\n", filename);
    	    }
    	    safe_free (tmp);
        }
    }

    /* open the new file */
    mk = safe_fopen ((char*)filename, "wt");
    chmod (filename, mode);

    safe_free (filename);
}

void shut_down () {
    if (mk) {
        fclose (mk);
    }
    project_destroy ();
    library_destroy_all ();
    program_destroy_all ();
    module_destroy_all ();
    application_destroy_all ();
    list_unlink_all (library_order);
    list_unlink_all (module_order);
}

/* check the dependencies of the applications */
void sort_applications () {
    struct LIST_ITERATOR *iterator, *iterator2;
    APPLICATION *app;
    char *like, *need;

    iterator = list_iterator_create ();
    iterator2 = list_iterator_create ();

    list_iterator_reset (iterator, application);
    while ((app = list_get_next (iterator)) != 0) {

        /* check the "like"-dependencies */
	list_iterator_reset (iterator2, app->like);
	while ((like = list_get_next (iterator2)) != 0) {
	    if (!list_search (library, like, module_search) &&
	        !list_search (module, like, module_search)) {
		handle_unknown_dependency (app->name, 0, like);
            }
	}

        /* check the "need"-dependencies */
	list_iterator_reset (iterator2, app->need);
	while ((need = list_get_next (iterator2)) != 0) {
	    if (!list_search (library, need, module_search) &&
	        !list_search (module, need, module_search)) {
		handle_unknown_dependency (app->name, 0, need);
            }
	}
    }
    list_iterator_destroy (iterator);
    list_iterator_destroy (iterator2);
}

void sort_libraries () {
    struct LIST_ITERATOR *iterator, *iterator2;
    LIBRARY *lib, *lib2, *lib3, *x11;
    char *need;
    int left;

    iterator = list_iterator_create ();
    iterator2 = list_iterator_create ();

    /* x11 always depend on net - make sure we we that one */
    x11 = list_search (library, "x11", compare_library_name);
    if (x11) {
        need = list_search (x11->need, "NET", txtcmp);
	if (!need) {
	    list_insert (x11->need, safe_strdup ("NET"));
	}
    }

    /* check the dependencies */
    list_iterator_reset (iterator, library);
    while ((lib = list_get_next (iterator)) != 0) {

	/* divide the "need"-dependencies */
        list_iterator_reset (iterator2, lib->need);
	while ((need = list_get_next (iterator2)) != 0) {
	    if (list_search (library, need, module_search)) {
	        list_append (lib->depend, need);
	    } else {
		handle_unknown_dependency (lib->name, lib->depend, need);
	    }
	}
    }

    /* mark all dependencies as pending */
    list_iterator_reset (iterator, library);
    while ((lib = list_get_next (iterator)) != 0) {
        lib->pending = list_duplicate (lib->depend);
    }

    left = list_get_size (library);
    library_order = list_create ();

    while (left--) {

	/* find a library with no pending dependencies */
        list_iterator_reset (iterator, library);
        while ((lib = list_get_next (iterator)) != 0) {
	    if (!lib->resolved && (list_get_size (lib->pending) == 0)) {
                break;
	    }
	}

	/* circular dependency */
	if (!lib) {
            list_iterator_reset (iterator, library);
            while ((lib = list_get_next (iterator)) != 0) {
	        if (!lib->resolved && (list_get_size (lib->pending) > 0)) {
		    fprintf (stderr, "%s ", lib->name);
		}
	    }
            fprintf (stderr, "\n");
	    error ("circular dependencies in libraries\n");
	}

	/* found a library */
	lib->resolved = 1;
	list_append (library_order, lib);

        /* mark this library as no longer pending */
        list_iterator_reset (iterator, library);
	while ((lib2 = list_get_next (iterator)) != 0) {
	    while ((lib3=list_search (lib2->pending, lib->NAME, txtcmp))!=0) {
	        list_unlink (lib2->pending, lib3);
	    }
	}
    }
    list_iterator_destroy (iterator);
    list_iterator_destroy (iterator2);
}

void sort_modules () {
    struct LIST_ITERATOR *iterator, *iterator2;
    char *need, *like;
    MODULE *mod, *mod2, *mod3;
    int left;

    iterator = list_iterator_create ();
    iterator2 = list_iterator_create ();

    /* split the module-dependencies into proper groups */
    list_iterator_reset (iterator, module);
    while ((mod = list_get_next (iterator)) != 0) {

	/* divide the "need"-dependencies */
        list_iterator_reset (iterator2, mod->need);
	while ((need = list_get_next (iterator2)) != 0) {
	    if (list_search (module, need, module_search)) {
	        list_append (mod->depend, need);
	    } else if (!list_search (program, need, module_search) &&
	               !list_search (library, need, module_search)) {
		handle_unknown_dependency (mod->name, 0, need);
	    }
	}

	/* divide the "like"-dependencies */
        list_iterator_reset (iterator2, mod->like);
	while ((like = list_get_next (iterator2)) != 0) {
	    if (list_search (module, like, module_search)) {
	        list_append (mod->depend, like);
	    } else if (!list_search (program, like, module_search) &&
	               !list_search (library, like, module_search)) {
		handle_unknown_dependency (mod->name, 0, like);
	    }
	}
    }

    left = list_get_size (module);
    module_order = list_create ();

    /* mark all dependencies as pending */
    list_iterator_reset (iterator, module);
    while ((mod = list_get_next (iterator)) != 0) {
        mod->pending = list_duplicate (mod->depend);
    }

    while (left--) {

	/* find a module with no pending dependencies */
        list_iterator_reset (iterator, module);
        while ((mod = list_get_next (iterator)) != 0) {
	    if (!mod->resolved && (list_get_size (mod->pending) == 0)) {
                break;
	    }
	}

	/* circular dependency */
	if (!mod) {
            list_iterator_reset (iterator, module);
            while ((mod = list_get_next (iterator)) != 0) {
	        if (!mod->resolved && (list_get_size (mod->pending) > 0)) {
		    fprintf (stderr, "%s ", mod->name);
		}
	    }
            fprintf (stderr, "\n");
	    error ("circular dependencies in modules\n");
	}
        debug_printf ("resolved dependencies of %s\n", mod->name);

	/* found a module */
	mod->resolved = 1;
	list_append (module_order, mod);

        /* mark this module as no longer pending */
        list_iterator_reset (iterator, module);
	while ((mod2 = list_get_next (iterator)) != 0) {
	    while ((mod3=list_search (mod2->pending, mod->NAME, txtcmp))!=0) {
	        list_unlink (mod2->pending, mod3);
	    }
	}
    }
    list_iterator_destroy (iterator);
    list_iterator_destroy (iterator2);
}

void switch_directory () {
    if (!project) {
        error ("no project specified\n");
    }
    safe_mkdir (project->name, 0755);
    chdir (project->name);
}

/* check if "name" is a special library */
int token_is_special (char *name) {
    int i;

    for (i=0; i<SPECIALS; i++) {
        if (!strcmp (name, special[i].name)) {
	    return 1;
	}
    }
    return 0;
}

/* create the file for the project specific checks */
void touch_project_m4 () {
    int len = 6+1+strlen(project->name)+1+2+1;
    char *buffer = (char*) safe_malloc (len);
    struct stat sstat;

    sprintf (buffer, "config/%s.m4", project->name);
    if (stat (buffer, &sstat) == -1) {
        fclose (safe_fopen (buffer, "w"));
    }
    safe_free (buffer);
}

void translate (const char *key) {
    if ((*key=='C') && !strcmp (key, "CXXSUFFIX")) {
        fprintf (mk, "%s", project->cxx_suffix ? project->cxx_suffix : "cc"); 
	return;
    }
    if ((*key=='P') && !strcmp (key, "PROJECT")) {
        fprintf (mk, "%s", project->NAME); 
	return;
    }
    if ((*key=='p') && !strcmp (key, "project")) {
        fprintf (mk, "%s", project->name); 
	return;
    }
    if ((*key=='V') && !strcmp (key, "VERSIONMAJOR")) {
        fprintf (mk, "%d", project->version_major); 
	return;
    }
    if ((*key=='V') && !strcmp (key, "VERSIONMINOR")) {
        fprintf (mk, "%d", project->version_minor); 
	return;
    }
    if ((*key=='P') && !strcmp (key, "PATCHLEVEL")) {
        fprintf (mk, "%d", project->patch_level); 
	return;
    }

    /* list the directory-names of the libraries */
    if ((*key=='L') && !strcmp (key, "LIBRARYDIRS")) {
        struct LIST_ITERATOR *iterator;
	MODULE *mod;

        iterator = list_iterator_create ();
        list_iterator_reset (iterator, module);
        while ((mod = list_get_next (iterator)) != 0) {
	    fprintf (mk, "lib%s ", mod->alias ? mod->alias : mod->name);
	}
	list_iterator_destroy (iterator);
	return;
    }

    /* list the directory-names of the applications */
    if ((*key=='A') && !strcmp (key, "APPLICATIONDIRS")) {
        struct LIST_ITERATOR *iterator;
	APPLICATION *app;

        iterator = list_iterator_create ();
        list_iterator_reset (iterator, application);
        while ((app = list_get_next (iterator)) != 0) {
	    fprintf (mk, "%s ", app->name);
	}
	list_iterator_destroy (iterator);
	return;
    }
    fprintf (mk, "%s", key);
}

void write_haves () {
    struct LIST_ITERATOR *iterator, *iterator2;
    LIBRARY *lib;
    PROGRAM *prog;
    MODULE *mod, *mod2;
    char *next;
    int conditions;

    iterator = list_iterator_create ();
    iterator2 = list_iterator_create ();

    list_iterator_reset (iterator, library);
    while ((lib = list_get_next (iterator)) != 0) {

        if (!lib->special) {
            fprintf (mk, "%s_LIBS =", lib->NAME);
            list_iterator_reset (iterator2, lib->lib);
            while ((next = list_get_next (iterator2)) != 0) {
                fprintf (mk, " -l%s", next);
            }
            fprintf (mk, "\n");
        }
    }
    fprintf (mk, "\n");

    /* write alternatives for librares */
    list_iterator_reset (iterator, library);
    while ((lib = list_get_next (iterator)) != 0) {
        if (!lib->special && !list_is_empty (lib->alternative)) {
            list_iterator_reset (iterator2, lib->alternative);
	    while ((next = list_get_next (iterator2)) != 0) {
		fprintf (mk, "ifeq ($(HAVE_%s),NO)\n", lib->NAME);
	        fprintf (mk, "  ifeq ($(HAVE_%s),YES)\n", next);
	        fprintf (mk, "    override HAVE_%s = $(HAVE_%s)\n",
		    lib->NAME, next);
	        fprintf (mk, "    override %s_ALTERNATIVE = -DHAVE_%s\n",
		    lib->NAME, next);
	        fprintf (mk, "    override %s_HOME = $(%s_HOME)\n",
		    lib->NAME, next);
	        fprintf (mk, "    override %s_LIBS = $(%s_LIBS)\n",
		    lib->NAME, next);
	        fprintf (mk, "  endif\n");
	        fprintf (mk, "endif\n");
	        fprintf (mk, "\n");
	    }
	}
    }

    /* write alternatives for programs */
    list_iterator_reset (iterator, program);
    while ((prog = list_get_next (iterator)) != 0) {
        if (!list_is_empty (prog->alternative)) {
	    list_iterator_reset (iterator2, prog->alternative);
	    while ((next = list_get_next (iterator2)) != 0) {
		fprintf (mk, "ifeq ($(HAVE_%s),NO)\n", prog->NAME);
	        fprintf (mk, "  ifeq ($(HAVE_%s),YES)\n", next);
	        fprintf (mk, "    override HAVE_%s = $(HAVE_%s)\n",
		    prog->NAME, next);
	        fprintf (mk, "    override %s_ALTERNATIVE = -DHAVE_%s\n",
		    prog->NAME, next);
	        fprintf (mk, "    override %s_HOME = $(%s_HOME)\n",
		    prog->NAME, next);
	        fprintf (mk, "  endif\n");
	        fprintf (mk, "endif\n");
	        fprintf (mk, "\n");
	    }
	}
    }

    list_iterator_reset (iterator, module_order);
    while ((mod = list_get_next (iterator)) != 0) {
	conditions = 0;
        list_iterator_reset (iterator2, mod->need);
	while ((next = list_get_next (iterator2)) != 0) {
	    mod2 = list_search (module, next, module_search);
	    if (mod2) {
	        if (mod2->failsafe) {
		    continue;
		}
	    }
	    conditions++;
	    fprintf (mk, "ifeq ($(HAVE_%s),YES)\n", next);
	}
	if (conditions) {
	    fprintf (mk, "HAVE_%s = YES\n", mod->NAME);
	    while (conditions--) {
                fprintf (mk, "endif\n");
	    }
            fprintf (mk, "\n");
	} else {
	    mod->failsafe = 1;
	}
    }
    list_iterator_destroy (iterator);
    list_iterator_destroy (iterator2);
}

void write_x11_libs () {
    struct LIST_ITERATOR *iterator;
    LIBRARY *lib;
    char *next;

    iterator = list_iterator_create ();
    lib = list_search (library, "x11", compare_library_name);
    if (list_is_empty (lib->lib)) {
        fprintf (mk, " -lXext -lX11");
    } else {
        list_iterator_reset (iterator, lib->lib);
	while ((next = list_get_next (iterator)) != 0) {
	    fprintf (mk, " -l%s", next);
	}
    }
    list_iterator_destroy (iterator);
}

void write_library_needs () {
    struct LIST_ITERATOR *iterator, *iterator2;
    struct LIST *tmp;
    LIBRARY *lib;
    char *next;

    tmp = list_flip (library_order);
    iterator = list_iterator_create ();
    iterator2 = list_iterator_create ();

    list_iterator_reset (iterator, tmp);
    while ((lib = list_get_next (iterator)) != 0) {
        if (lib->replace) {
	    continue;
	}
	fprintf (mk, "ifeq ($(HAVE_%s),YES)\n", lib->NAME);
	fprintf (mk, "  ifneq (,$(findstring %s,$(LIKES)))\n", lib->NAME);
	fprintf (mk, "    ifeq (,$(findstring %s,$(NEEDS)))\n", lib->NAME);
	fprintf (mk, "      NEEDS += %s\n", lib->NAME);
	fprintf (mk, "    endif\n");
	fprintf (mk, "  endif\n");
	fprintf (mk, "endif\n");
	fprintf (mk, "\n");

        fprintf (mk, "ifneq (,$(findstring %s,$(NEEDS)))\n", lib->NAME);

	/* check availability */
	fprintf (mk, "ifneq ($(HAVE_%s),YES)\n", lib->NAME);
	fprintf (mk, "ifneq ($(FAILURE),YES)\n");
	fprintf (mk, "$(TARGET):\n");
	fprintf (mk, "\t@echo \"%s not available\"\n", lib->name);
	fprintf (mk, "\t@false\n");
	fprintf (mk, "FAILURE=YES\n");
	fprintf (mk, "endif\n");
	fprintf (mk, "endif\n");

        /* list all needs */
	list_iterator_reset (iterator2, lib->need);
	while ((next = list_get_next (iterator2)) != 0) {
	    fprintf (mk, "ifeq (,$(findstring %s,$(NEEDS)))\n", next);
	    fprintf (mk, "  NEEDS += %s\n", next);
	    fprintf (mk, "endif\n");
	}

	fprintf (mk, "EXTERNAL_FLAGS += ");
	if (!lib->special && !list_is_empty (lib->alternative)) {
	    fprintf (mk, "$(%s_ALTERNATIVE) ", lib->NAME);
	}
	fprintf (mk, "-DHAVE_%s\n", lib->NAME);

	if (1) {
	    fprintf (mk, "ifneq ($(%s_HOME),/usr)\n", lib->NAME);
	}
	fprintf (mk, "ifeq (,$(findstring $(%s_HOME)/include, $(EXTERNAL_INCLUDES)))\n",
	    lib->NAME);
	fprintf (mk, "  EXTERNAL_INCLUDES += -I$(%s_HOME)/include\n", lib->NAME);
	fprintf (mk, "  EXTERNAL_LIBS += -L$(%s_HOME)/lib\n", lib->NAME);
	fprintf (mk, "endif\n");
	if (1) {
	    fprintf (mk, "endif\n");
	}
	fprintf (mk, "EXTERNAL_LIBS += $(%s_LIBS)\n", lib->NAME);
	fprintf (mk, "endif\n");
	fprintf (mk, "\n");
    }

    list_unlink_all (tmp);
    list_iterator_destroy (iterator);
    list_iterator_destroy (iterator2);
}

void write_module_needs () {
    struct LIST_ITERATOR *iterator, *iterator2;
    struct LIST *tmp;
    MODULE *mod;
    char *next;

    tmp = list_flip (module_order);
    iterator = list_iterator_create ();
    iterator2 = list_iterator_create ();

    fprintf (mk, "%s_LIBS += -L$(LIB_DIR)\n\n", project->NAME);

    list_iterator_reset (iterator, tmp);
    while ((mod = list_get_next (iterator)) != 0) {

	if (!mod->failsafe) {
	     fprintf (mk, "ifeq ($(HAVE_%s),YES)\n", mod->NAME);
	}
	fprintf (mk, "ifneq (,$(findstring %s,$(LIKES)))\n", mod->NAME);
	fprintf (mk, "  ifeq (,$(findstring %s,$(NEEDS)))\n", mod->NAME);
	fprintf (mk, "    NEEDS += %s\n", mod->NAME);
	fprintf (mk, "  endif\n");
	fprintf (mk, "endif\n");
	if (!mod->failsafe) {
	    fprintf (mk, "endif\n");
	}
	fprintf (mk, "\n");

        fprintf (mk, "ifneq (,$(findstring %s,$(NEEDS)))\n", mod->NAME);

	/* check for availability */
	if (!mod->failsafe) {
	    fprintf (mk, "ifneq ($(HAVE_%s),YES)\n", mod->NAME);
	    fprintf (mk, "ifneq ($(FAILURE),YES)\n");
  	    fprintf (mk, "$(TARGET):\n");
	    fprintf (mk, "\t@echo \"%s not available\"\n", mod->name);
	    fprintf (mk, "\t@false\n");
	    fprintf (mk, "FAILURE=YES\n");
	    fprintf (mk, "endif\n");
	    fprintf (mk, "endif\n");
	}

        /* list all needs */
	list_iterator_reset (iterator2, mod->need);
	while ((next = list_get_next (iterator2)) != 0) {
	    fprintf (mk, "ifeq (,$(findstring %s,$(NEEDS)))\n", next);
	    fprintf (mk, "  NEEDS += %s\n", next);
	    fprintf (mk, "endif\n");
	}

	/* list the wishes */
	list_iterator_reset (iterator2, mod->like);
	while ((next = list_get_next (iterator2)) != 0) {
	    fprintf (mk, "ifeq (,$(findstring %s,$(LIKES)))\n", next);
	    fprintf (mk, "  LIKES += %s\n", next);
	    fprintf (mk, "endif\n");
	}

        fprintf (mk, "%s_FLAGS += -DHAVE_%s\n", project->NAME, mod->NAME);
        fprintf (mk, "%s_LIBS += -l%s\n", project->NAME, mod->name);
	fprintf (mk, "endif\n");
	fprintf (mk, "\n");
    }

    list_unlink_all (tmp);
    list_iterator_destroy (iterator);
    list_iterator_destroy (iterator2);
}

void write_template_config () {
    struct LIST_ITERATOR *iterator;
    LIBRARY *lib;
    PROGRAM *prog;
    char *NAME;
    int i;

    iterator = list_iterator_create ();

    set_target ("config", "template.dat", 0644);

    /* where to put all the things */
    fprintf (mk, "PREFIX = @prefix@\n");
    fprintf (mk, "\n");

    /* write the c-compiler section */
    if (list_search (project->language, "c", txtcmp)) {
        fprintf (mk, "CC = @CC@\n");
        fprintf (mk, "GCC = @GCC@\n");
        fprintf (mk, "CFLAGS = @CFLAGS@\n");
        fprintf (mk, "CC_COMPAT_FLAGS = @CC_COMPAT_FLAGS@\n");
        fprintf (mk, "CC_HAS_M = @CC_HAS_M@\n");
	fprintf (mk, "GCC_USES_GLD = @GCC_USES_GLD@\n");
	fprintf (mk, "@CONFIG_CC@\n");
	fprintf (mk, "\n");
    }

    /* write the c++-compiler section */
    if (list_search (project->language, "c++", txtcmp)) {
        fprintf (mk, "CXX = @CXX@\n");
        fprintf (mk, "GXX = @GXX@\n");
        fprintf (mk, "CXXFLAGS = @CXXFLAGS@\n");
        fprintf (mk, "CXX_COMPAT_FLAGS = @CXX_COMPAT_FLAGS@\n");
        fprintf (mk, "CXX_HAS_M = @CXX_HAS_M@\n");
	fprintf (mk, "GXX_USES_GLD = @GXX_USES_GLD@\n");
	fprintf (mk, "@CONFIG_CXX@\n");
	fprintf (mk, "\n");
    }

    if (!list_is_empty (module)) {
        fprintf (mk, "SHARED_LIBS = @SHARED_LIBS@\n");
        fprintf (mk, "@CONFIG_SHARED@\n");
    }
    fprintf (mk, "@CONFIG_MACHINE@\n");
    fprintf (mk, "\n");

    fprintf (mk, "LN_S = @LN_S@\n");
    fprintf (mk, "INSTALL = @INSTALL@\n");
    fprintf (mk, "MAKEDEPEND = @MAKEDEPEND@\n");
    fprintf (mk, "RANLIB = @RANLIB@\n");
    fprintf (mk, "\n");

    /* write the library section section */
    list_iterator_reset (iterator, library);
    while ((lib = list_get_next (iterator)) != 0) {
        if (library_is_special (lib->NAME)) {
	    continue;
	}
        fprintf (mk, "HAVE_%s = @HAVE_%s@\n", lib->NAME, lib->NAME);
        fprintf (mk, "%s_HOME = @%s_HOME@\n", lib->NAME, lib->NAME);
	fprintf (mk, "\n");
    }

    /* write special x11-library stuff */
    if (list_search (library, "x11", compare_library_name)) {
        fprintf (mk, "HAVE_X11 = @HAVE_X11@\n");
        fprintf (mk, "X11_HOME = @X11_HOME@\n");
        fprintf (mk, "X11_LIBS = ");
	write_x11_libs ();
        fprintf (mk, " @X_PRE_LIBS@\n");
        fprintf (mk, "\n");
    }

    /* write stuff for other specials */
    for (i=0; i<SPECIALS; i++) {
        if (special[i].fix) {
	    NAME = special[i].NAME;
            if (list_search (library, special[i].name, compare_library_name)) {
                fprintf (mk, "HAVE_%s = @HAVE_%s@\n", NAME, NAME);
                fprintf (mk, "%s_HOME = @%s_HOME@\n", NAME, NAME);
                fprintf (mk, "%s_LIBS = @%s_LIBS@\n", NAME, NAME);
                fprintf (mk, "\n");
            }
	}
    }
    fprintf (mk, "\n");

    /* write the program section section */
    list_iterator_reset (iterator, program);
    while ((prog = list_get_next (iterator)) != 0) {
        fprintf (mk, "HAVE_%s = @HAVE_%s@\n", prog->NAME, prog->NAME);
	fprintf (mk, "\n");
    }
    list_iterator_destroy (iterator);
}

void write_tests () {
    struct LIST_ITERATOR *iterator, *iterator2;
    PROGRAM *prog;
    LIBRARY *lib;
    char *tmp;
    int nears;
    int space;

    iterator = list_iterator_create ();
    iterator2 = list_iterator_create ();

    list_iterator_reset (iterator, library);
    while ((lib = list_get_next (iterator)) != 0) {

	if (lib->special) {
	    continue;
	}

        fprintf (mk, "AC_FIND_LIB(%s,%s,", lib->NAME, lib->name);

	/* write down an include file for tests */
	if (!lib->include) {
	    error ("no headers specified for %s\n", lib->name);
	}
	fprintf (mk, "%s,", lib->include);

	/* write down a library for tests */
	list_iterator_reset (iterator2, lib->lib);
	tmp = list_get_next (iterator2);
	if (!tmp) {
	    error ("no libraries specified for %s\n", lib->name);
	}
	fprintf (mk, "%s,", tmp);

	/* write down a place where the package usually resides */
	nears=0;
	fprintf (mk, "[");
	list_iterator_reset (iterator2, lib->near);
	while ((tmp = list_get_next (iterator2)) != 0) {
	    if (nears++) {
	        fprintf (mk, " ");
	    }
	    fprintf (mk, "%s", tmp);
	}

	list_iterator_reset (iterator2, lib->need);
	while ((tmp = list_get_next (iterator2)) != 0) {
	    if (nears++) {
	        fprintf (mk, " ");
	    }
	    fprintf (mk, "$%s_HOME", tmp);
	}
	fprintf (mk, "]");

	/* calculate the space between the columns in --help */
	space = 12-strlen (lib->name);
	fprintf (mk, ", [");
	while (space-- > 0) {
	    fputc (' ', mk);
	}
	fprintf (mk, "])\n");
    }
    fprintf (mk, "\n");

    list_iterator_reset (iterator, program);
    while ((prog = list_get_next (iterator)) != 0) {
        fprintf (mk, "AC_FIND_APP(%s,%s,%s, [",
	    prog->NAME, prog->name, prog->file);
        space = 12 - strlen (prog->name);
        while (space-- > 0) {
            fputc (' ', mk);
        }
        fprintf (mk, "])\n");
    }
    fprintf (mk, "\n");

    fprintf (mk, "cp template.dat config.dat.in\n");
    fprintf (mk, "AC_OUTPUT(config.dat)\n");
    fprintf (mk, "rm config.dat.in\n");

    list_iterator_destroy (iterator);
    list_iterator_destroy (iterator2);
}

char *xX (const char *lower) {
    char *upper, *next;
    int len;
    
    len = strlen (lower);
    upper = (char*) safe_malloc (len+1);
    next = upper;
    while (*lower) {
        *next = toupper (*lower);
	next++;
	lower++;
    }
    *next = '\0';
    return upper;
}

char *Xx (const char *upper) {
    char *lower, *next;
    int len;
    
    len = strlen (upper);
    lower = (char*) safe_malloc (len+1);
    next = lower;
    while (*upper) {
        *next = tolower (*upper);
	next++;
	upper++;
    }
    *next = '\0';
    return lower;
}

void yyerror (const char *err_msg) {
    parse_error (err_msg);
}

int yywrap () {
    return 1;
}

int main (int argc, char **argv) {
    init ();
    get_args (argc, argv);
    init_data ();

    read_definition ();
    switch_directory ();
    check_names ();
    find_alternatives ();
    sort_modules ();
    sort_applications ();
    sort_libraries ();

    /* create the make subdirectory */
    set_target ("rules", "needs.mk", 0644);

    write_haves ();
    write_module_needs ();
    write_library_needs ();

    compose ("rules", "app.mk", 0644);
    compose ("rules", "clean.mk", 0644);
    compose ("rules", "compile.mk", 0644);
    compose ("rules", "depend.mk", 0644);
    compose ("rules", "doc.mk", 0644);
    compose ("rules", "init.mk", 0644);
    if (!list_is_empty (module)) {
        compose ("rules", "lib.mk", 0644);
    }
    compose ("rules", "subdirs.mk", 0644);

    /* create the scripts subdirectory */
    compose ("rules", "clean.sh", 0755);
    compose ("rules", "collect.sh", 0755);

    /* copy the config and machine directory */
    copy_directory ("machine");
    write_template_config ();

    /* create the configure files */
    compose (".", "GNUmakefile", 0644);
    compose (".", "configure", 0755);
    compose ("config", "install-sh", 0644);
    compose ("config", "standard.m4", 0644);
    touch_project_m4 ();
    compose ("config", "configure.in", 0644);
    write_tests ();

    /* copy the config directory */
    create_src_dirs ();

    /* run autoconf to produce the configure script */
    run_autoconf ();

    return EXIT_SUCCESS;
}
