/*
 * Copyright (C) 1997 Tobias Gloth (gloth@unknown.westfalen.de)
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>

#include "xtoto/xtoto.h"

int debug=0;
int verbose=0;

void init () {
    atexit (shut_down);
#ifdef DEBUG_MALLOC
    if (getenv ("DEBUG"))
        debug = 1;
#endif
}

void verror (const char *format, va_list args) {
    vfprintf (stderr, format, args);
    exit (EXIT_FAILURE);
}

void error (const char *format, ...) {
    va_list args;
    va_start (args, format);
    verror (format, args);
}

void show_help () {
    fprintf (stderr, HELP);
    exit (EXIT_FAILURE);
}

void show_license () {
    fprintf (stdout, LICENSE);
    exit (EXIT_SUCCESS);
}

void show_version () {
    fprintf (stdout, "version %s\n", VERSION);
    exit (EXIT_SUCCESS);
}

/* read input till eof or term */
char *safe_input (FILE *file, int term) {
    char *buffer=NULL;
    int ch, ptr=0, size=0;

    while (1) {

        /* resize the buffer when it is full */
        if (ptr == size) {
            if (!size)
                size = 256;
            else
                size *= 2;
            buffer = (char*) safe_realloc (buffer, size);
        }

        ch = fgetc (file);

        /* check for end of input */
        if ((ch == term) || (ch == EOF)) {
	    if ((ch == EOF) && (ptr == 0)) {
	        safe_free (buffer);
                return 0;
	    }
            buffer[ptr++] = '\0';
            buffer = (char*) safe_realloc (buffer, ptr);
            break;
        }
        buffer[ptr++] = ch;
    }
    /* return the buffer */
    return buffer;
}

/* reads a file, even from non-seekable streams */
char *safe_load (const char *name, int trailing_zero) {
    char *buffer=NULL;
    FILE *file;
    int read=0, got;
    int size=0;

    file = safe_fopen (name, "rb");
    while (1) {
        if (read == size) {
            /* resize the buffer if it is full */
            if (!size) {
                size = 1024;
                buffer = (char*) safe_malloc (size);
            } else {
                size *= 2;
                buffer = (char*) safe_realloc (buffer, size);
            }

            got = fread (buffer+read, 1, size-read, file);
            if (got < size-read)
               break;
            read = size;
        }
    }
    fclose (file);
    read += got;
    buffer = (char*) safe_realloc (buffer, read+trailing_zero);
    if (trailing_zero)
        buffer[read] = '\0';
    return buffer;
}

FILE *safe_fopen (const char *name, const char *mode) {
    FILE *file = fopen (name, mode);
    if (!file)
        error ("can't open \"%s\"\n", name);
    return file;
}

int safe_fread (void *p, size_t size, size_t count, FILE *file) {
    int blocks = fread (p, size, count, file);
    if (blocks != count)
        error ("can't read from file\n", NULL);
    return blocks;
}

int safe_fwrite (void *p, size_t size, size_t count, FILE *file) {
    int blocks = fwrite (p, size, count, file);
    if (blocks != count)
        error ("can't write to file\n", NULL);
    return blocks;
}

void safe_mkdir (const char *name, int mode) {
    struct stat statbuf;

    if (stat (name, &statbuf) == -1) {
        if (mkdir (name, mode) == -1) {
            error ("can't create directory \"%s\"\n", name);
        }
    }
}

void *safe_malloc (size_t size) {
    void *ptr = malloc (size);
    if (!ptr)
        error ("out of memory\n", NULL);
#ifdef DEBUG_MALLOC
    if (debug)
        fprintf (stderr, "A: %p\n", ptr);
#endif
    return ptr;
}

void *safe_realloc (void* ptr, size_t size) {
    void *ptr2;
    ptr2 = realloc (ptr, size);
    if (!ptr2) {
        error ("out of memory\n");
    }
#ifdef DEBUG_MALLOC
    if (debug && (ptr != ptr2)) {
        fprintf (stderr, "A: %p\n", ptr2);
        if (ptr) {
            fprintf (stderr, "F: %p\n", ptr);
        }
    }
#endif
    return ptr2;
}

char *safe_strcat (const char *first, ...) {
    int len[1024];
    const char *start[1024];
    char *next, *buffer;
    va_list args;
    int i, size, all_len=0;

    start[0] = first;
    all_len = len[0] = strlen(first);
    va_start (args, first);
    size = 1;

    while (1) {

	/* get the next string */
        start[size] = va_arg (args, const char*);
        if (!start[size]) {
	    break;
	}

	/* store the data */
	len[size] = strlen (start[size]);
	all_len += len[size];
        if (++size == 1024) {
	    error ("strcat called with silly args\n");
	}
    }

    /* allocate memory */
    next = buffer = (char*) safe_malloc (all_len+1);

    /* copy all strings */
    for (i=0; i<size; i++) {
        strcpy (next, start[i]);
	next += len[i];
    }
    va_end (args);
    return buffer;
}

char *safe_strdup (const char *ptr) {
    char* ptr2 = malloc (strlen (ptr));
    if (!ptr2)
        error ("out of memory\n", NULL);
#ifdef DEBUG_MALLOC
    if (debug)
        fprintf (stderr, "A: %p\n", ptr2);
#endif
    strcpy (ptr2, ptr);
    return ptr2;
}

char *safe_strdupu (char *ptr) {
    int length = strlen (ptr);
    char* ptr2 = (char*) malloc (length+1);
    char* dummy;
    if (!ptr2)
        error ("out of memory\n", NULL);
#ifdef DEBUG_MALLOC
    if (debug)
        fprintf (stderr, "A: %p\n", ptr2);
#endif

    /* string quoted at all? */
    if (*(ptr++) != '\"')
        return ptr2;

    /* resolve escape-sequences */
    dummy = ptr2;
    while (*ptr) {
        if (*ptr == '"')
            break;
        else if (*ptr == '\\') {
            switch (ptr[1]) {
            case 'n':
                *dummy = '\n';
                break;
            case 't':
                *dummy = '\t';
                break;
            case '"':
                *dummy = '\"';
                break;
            case '\\':
                *dummy = '\\';
                break;
            default:
                *dummy = ptr[1];
            }
            dummy++;
            ptr += 2;
        } else {
            *(dummy++) = *(ptr++);
        }
    }
    *dummy = '\0';
    return ptr2;
}

void safe_free (void *ptr) {
    free (ptr);
#ifdef DEBUG_MALLOC
    if (debug && (ptr != NULL))
        fprintf (stderr, "F: %p\n", ptr);
#endif
}

/* print only in verbose mode */
void verbose_printf (const char *format, ...) {
    va_list args;

    if (!verbose) {
        return;
    }

    va_start (args, format);
    vprintf (format, args);
    fflush (stdout);
    va_end (args);
}

/* print only in debug mode */
void debug_printf (const char *format, ...) {
    va_list args;

    if (!debug) {
        return;
    }

    va_start (args, format);
    vprintf (format, args);
    fflush (stdout);
    va_end (args);
}
