/*
 * Copyright (C) 1997 Tobias Gloth (gloth@unknown.westfalen.de)
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "xtoto/list.h"
#include "xtoto/xtoto.h"

typedef struct LIST_NODE {
    void *data;
    struct LIST_NODE *next;
} LIST_NODE;

typedef struct LIST {
    LIST_NODE *start;
    LIST_NODE *end;
} LIST;

typedef struct LIST_ITERATOR {
    LIST_NODE *next;
} LIST_ITERATOR;

void list_append (LIST *list, void *data) {
    LIST_NODE *new;

    new = (LIST_NODE*) safe_malloc (sizeof (LIST_NODE));
    new->next = 0;
    new->data = data;
    if (list->end) {
        list->end->next = new;
        list->end = new;
    } else {
        list->start = new;
        list->end = new;
    }
}

LIST *list_create () {
    LIST *list = (LIST*) safe_malloc (sizeof (LIST));
    list->start = 0;
    list->end = 0;
    return list;
}

void list_destroy (LIST *list, void *data, void (*crush) (void*)) {
    crush (data);
    list_unlink (list, data);
}

void list_destroy_all (LIST *list, void (*crush) (void *)) {
    LIST_NODE *this, *next;
    if (!list) {
        return;
    }
    for (this=list->start; this; this=next) {
        next = this->next;
        crush (this->data);
        safe_free (this);
    }
    safe_free (list);
}

LIST *list_duplicate (LIST *list) {
    LIST_NODE *this, **target, *tmp;
    LIST *new;

    new = list_create ();
    target = &new->start;
    for (this=list->start; this; this=this->next) {
        tmp = (LIST_NODE*) safe_malloc (sizeof (LIST_NODE));
	*target = tmp;
	tmp->data = this->data;
	target = &tmp->next;
    }
    *target = 0;
    return new;
}

LIST *list_flip (LIST *list) {
    LIST_NODE *this, *first;
    LIST *new;

    new = list_create ();
    for (this=list->start; this; this=this->next) {
        first = (LIST_NODE*) safe_malloc (sizeof (LIST_NODE));
	first->next = new->start;
	first->data = this->data;
	new->start = first;
    }
    return new;
}

int list_get_size (LIST *list) {
    LIST_NODE *this;
    int count = 0;

    for (this=list->start; this; this=this->next) {
        count++;
    }
    return count;
}

int list_has (LIST *list, const void *data) {
    LIST_NODE *this;
    for (this=list->start; this; this=this->next) {
        if (this->data == data)
            return 1;
    }
    return 0;
}

void list_insert (LIST *list, void *data) {
    LIST_NODE *new = (LIST_NODE*) safe_malloc (sizeof (LIST_NODE));
    new->next = list->start;
    new->data = data;
    list->start = new;
    if (!list->end) {
        list->end = new;
    }
}

int list_is_empty (LIST *list) {
    return (list->start == 0);
}

void list_iterate (LIST *list, void (*func)(void *, void *), void *misc) {
    LIST_NODE *this;
    for (this=list->start; this; this=this->next) {
        func (this->data, misc);
    }
}

void *list_search (LIST *list, const void *data, int (*comp) (const void *, const void *)) {
    LIST_NODE *this;
    for (this=list->start; this; this=this->next) {
        if (!comp (this->data, data))
            return this->data;
    }
    return (void*)0;
}

void list_unlink (LIST *list, const void *data) {
    LIST_NODE *this, *last=0, **ptr;
    for (this=list->start,ptr=&list->start; this; this=*ptr) {
        if (this->data == data) {
            *ptr = this->next;
            safe_free (this);
	    if (!ptr) {
	        list->end = last;
	    }
        } else {
	    last = this;
            ptr = &this->next;
	}
    }
}

void list_unlink_all (LIST *list) {
    LIST_NODE *this, *next;
    if (!list) {
        return;
    }
    for (this=list->start; this; this=next) {
        next = this->next;
        safe_free (this);
        this = next;
    }
    safe_free (list);
}

LIST_ITERATOR *list_iterator_create () {
    return (LIST_ITERATOR*) safe_malloc (sizeof (LIST_ITERATOR));
}

void list_iterator_destroy (LIST_ITERATOR *iterator) {
    safe_free (iterator);
}

void *list_get_next (LIST_ITERATOR *iterator) {
    void *result;

    if (iterator->next) {
        result = iterator->next->data;
        iterator->next = iterator->next->next;
    } else {
        result = 0;
    }
    return result;
}

void list_iterator_reset (LIST_ITERATOR *iterator, LIST *list) {
    iterator->next = list->start;
}

