/*
 * Copyright (C) 1997 Tobias Gloth (gloth@unknown.westfalen.de)
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "xtoto/hash.h"
#include "xtoto/listx.h"
#include "xtoto/xtoto.h"

typedef struct HASH {
    int lists;
    struct LISTX **list;
    int (*delta) (const void *);
} HASH;

HASH *hash_create (int lists, int (*delta)(const void*)) {
    HASH *hash;
    int i;

    hash = (HASH*) safe_malloc (sizeof (HASH));
    hash->lists = lists;
    hash->list = (struct LISTX **) safe_malloc (lists * sizeof (struct LISTX*));
    for (i=0; i<lists; i++)
        hash->list[i] = (struct LISTX*)0;
    hash->delta = delta;
    return hash;
}

void hash_destroy (HASH *hash, void *data, void (*crush) (void *)) {
    int idx = hash->delta (data);
    hash->list[idx] = listx_destroy (hash->list[idx], data, crush);
}

void hash_destroy_all (HASH *hash, void (*crush) (void *)) {
    int idx;
    for (idx=0; idx<hash->lists; idx++)
        listx_destroy_all (hash->list[idx], crush);
    safe_free (hash->list);
    safe_free (hash);
}

int hash_has (HASH *hash, const void *data) {
    int idx = hash->delta (data);
    return listx_has (hash->list[idx], data);
}

void hash_insert (HASH *hash, void *data) {
    int idx = hash->delta (data);
    hash->list[idx] = listx_insert (hash->list[idx], data);
}

void hash_iterate (HASH *hash, void (*func) (void *, void *), void *misc) {
    int idx;
    for (idx=0; idx<hash->lists; idx++) {
        listx_iterate (hash->list[idx], func, misc);
    }
}

void *hash_search (HASH *hash, const void *data, int (*comp) (const void *, const void *)) {
    int idx = hash->delta (data);
    return listx_search (hash->list[idx], data, comp);
}

void *hash_search_all (HASH *hash, const void *data, int (*comp) (const void *, const void *)) {
    void *match;
    int idx;
    for (idx=0; idx<hash->lists; idx++)
        if ((match = listx_search (hash->list[idx], data, comp)) != (void*)0)
            return match;
    return (void*)0;
}

void hash_unlink (HASH *hash, const void *data) {
    int idx = hash->delta (data);
    hash->list[idx] = listx_unlink (hash->list[idx], data);
}

void hash_unlink_all (HASH *hash) {
    int idx;
    for (idx=0; idx<hash->lists; idx++) {
        listx_unlink_all (hash->list[idx]);
        hash->list[idx] = (struct LISTX*)0;
    }
}
