/*
 * Copyright (C) 1997 Tobias Gloth (gloth@unknown.westfalen.de)
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "xtoto/avltree.h"
#include "xtoto/xtoto.h"

typedef struct AVLTREE_NODE {
    struct AVLTREE_NODE *left;
    struct AVLTREE_NODE *right;
    int left_height;
    int right_height;
    void *data;
} AVLTREE_NODE;

typedef struct AVLTREE {
    AVLTREE_NODE *root;
    int (*comp) (const void *, const void *);
} AVLTREE;

#undef MAX
#define MAX(X,Y) ((X)>(Y)?(X):(Y))

#define AVLTREE_LEFT -1
#define AVLTREE_RIGHT 1

static void node_destroy_all (AVLTREE_NODE *, void (*crush)(void *));
static void node_iterate (AVLTREE_NODE *, int order, void (*func)(void *, void *), void *misc);
static AVLTREE_NODE *node_search (AVLTREE_NODE *node, const void *data, int (*comp)(const void *, const void *));
static void node_unlink_all (AVLTREE_NODE *);

static AVLTREE_NODE *rebalance (AVLTREE_NODE ***path, int *way, int i);
static int node_height (AVLTREE_NODE *);
static AVLTREE_NODE *node_rotate_left (AVLTREE_NODE *);
static AVLTREE_NODE *node_rotate_right (AVLTREE_NODE *);

AVLTREE *avltree_create (int (*comp) (const void *, const void *)) {
    AVLTREE *avltree = (AVLTREE*) safe_malloc (sizeof (AVLTREE));
    avltree->root = NULL;
    avltree->comp = comp;
    return avltree;
}

int avltree_destroy (AVLTREE *avltree, void *data, void (*crush)(void *)) {
    int result = avltree_unlink (avltree, data);
    crush (data);
    return result;
}

void avltree_destroy_all (AVLTREE *avltree, void (*crush)(void *)) {
    if (!avltree) {
        return;
    }
    node_destroy_all (avltree->root, crush);
    safe_free (avltree);
}

/* find the maximum element */
void *avltree_get_maximum (AVLTREE *avltree) {
    AVLTREE_NODE *node = avltree->root;

    /* go as far to the right as possible */
    if (node) {
        while (node->right) {
	    node = node->right;
	}
	return node->data;
    }

    /* empty tree */
    return 0;
}

/* find the minimum element */
void *avltree_get_minimum (AVLTREE *avltree) {
    AVLTREE_NODE *node = avltree->root;

    /* go as far to the left as possible */
    if (node) {
        while (node->left) {
	    node = node->left;
	}
	return node->data;
    }

    /* empty tree */
    return 0;
}

int avltree_has (AVLTREE *avltree, const void *data) {
    AVLTREE_NODE *node = avltree->root;
    while (node) {
        if (node->data == data)
            return 1;
        if (avltree->comp (data, node->data) <= 0)
            node = node->left;
        else
            node = node->right;
    }
    return 0;
}

void avltree_insert (AVLTREE *avltree, void *data) {
    AVLTREE_NODE **path[1000];
    int son[1000];
    AVLTREE_NODE **last = &avltree->root;
    AVLTREE_NODE *node = avltree->root;
    int i=0;

    /* find and record the path to the insert-point */
    while (node) {
	path[i] = last;
        if (avltree->comp (data, node->data) <= 0) {
	    son[i] = AVLTREE_LEFT;
            last = &node->left;
        } else {
	    son[i] = AVLTREE_RIGHT;
            last = &node->right;
	}
        node = *last;
	i++;
    }

    /* insert the data as a leaf */
    node = (AVLTREE_NODE*) safe_malloc (sizeof (AVLTREE_NODE));
    node->left = NULL;
    node->left_height = 0;
    node->right = NULL;
    node->right_height = 0;
    node->data = data;
    *last = node;

    if (i) {
        avltree->root = rebalance (path, son, i);
    }
}

/* update the heights and rebalance the tree */
AVLTREE_NODE *rebalance (AVLTREE_NODE ***path, int *son, int i) {
    AVLTREE_NODE *node;

    while (i--) {

        node = *path[i];

        /* recalculate the heights */
	if (son[i] == AVLTREE_LEFT) {
	    node->left_height = node_height (node->left);
        } else {
	    node->right_height = node_height (node->right);
	}

        /* rebalance the tree if needed */
        if (abs (node->left_height - node->right_height) > 1) {
            if (node->left_height > node->right_height) {
	        if (node->left->left_height < node->left->right_height) {
	            node->left = node_rotate_left (node->left);
		}
                *path[i] = node_rotate_right (node);
            } else {
                if (node->right->right_height < node->right->left_height) {
	            node->right = node_rotate_right (node->right);
		}
                *path[i] = node_rotate_left (node);
            }
        }
    }

    return *path[0];
}

void avltree_iterate (AVLTREE *avltree, int order, void (*func) (void*, void*), void *misc) {
    node_iterate (avltree->root, order, func, misc);
}

/* find the predecessor of of an element */
void *avltree_get_predecessor (AVLTREE *avltree, const void *data) {
    AVLTREE_NODE *path[1000];
    int son[1000];
    AVLTREE_NODE *node = avltree->root;
    int i=0;

    /* find the node itself */
    while (node) {
	path[i] = node;
	if (node->data == data)
            break;
	if (avltree->comp (data, node->data) <= 0) {
	    son[i] = AVLTREE_LEFT;
	    node = node->left;
	} else {
	    son[i] = AVLTREE_RIGHT;
	    node = node->right;
	}
	i++;
    }

    /* found the node at all? */
    if (!node) {
        return 0;
    }

    /* if there is a left subtree, take the maximum of that */
    if (node->left) {
        node = node->left;
	while (node->right) {
	    node = node->right;
	}
	return node->data;
    }

    /* trace back to a node where the right son was chosen */
    while (--i >= 0) {
        if (son[i] == AVLTREE_RIGHT) {
	    break;
	}
    }

    /* this means we started at the minimum */
    if (i == -1) {
        return 0;
    }

    return path[i]->data;
}

void *avltree_search (AVLTREE *avltree, const void *data) {
    AVLTREE_NODE *node = node_search (avltree->root, data, avltree->comp);
    if (node) {
        return node->data;
    } else {
        return 0;
    }
}

void *avltree_search2 (AVLTREE *avltree, const void *data, int (*comp) (const void *, const void *)) {
    AVLTREE_NODE *node = node_search (avltree->root, data, comp);
    if (node) {
        return node->data;
    } else {
        return 0;
    }
}

/* find the successor of of an element */
void *avltree_get_successor (AVLTREE *avltree, const void *data) {
    AVLTREE_NODE *path[1000];
    int son[1000];
    AVLTREE_NODE *node = avltree->root;
    int i=0;

    /* find the node itself */
    while (node) {
	path[i] = node;
	if (node->data == data)
            break;
	if (avltree->comp (data, node->data) <= 0) {
	    son[i] = AVLTREE_LEFT;
	    node = node->left;
	} else {
	    son[i] = AVLTREE_RIGHT;
	    node = node->right;
	}
	i++;
    }

    /* found the node at all? */
    if (!node) {
        return 0;
    }

    /* if there is a rigth subtree, take the minimum of that */
    if (node->right) {
        node = node->right;
	while (node->left) {
	    node = node->left;
	}
	return node->data;
    }

    /* trace back to a node where the left son was chosen */
    while (--i >= 0) {
        if (son[i] == AVLTREE_LEFT) {
	    break;
	}
    }

    /* this means we started at the maximum */
    if (i == -1) {
        return 0;
    }

    return path[i]->data;
}

void avltree_swap (AVLTREE *avltree, const void *p1, const void *p2) {
    AVLTREE_NODE *node1, *node2;

    node1 = node_search (avltree->root, p1, avltree->comp);
    node2 = node_search (avltree->root, p2, avltree->comp);

    if (node1 && node2) {
        node1->data = (void*) p2;
        node2->data = (void*) p1;
    }
}

int avltree_unlink (AVLTREE *avltree, const void *data) {
    AVLTREE_NODE **path[1000];
    int son[1000];
    AVLTREE_NODE **last = &avltree->root;
    AVLTREE_NODE *node = avltree->root;
    AVLTREE_NODE *next;
    int i=0;

    /* find and record the path to the insert-point */
    while (node) {
	path[i] = last;
	if (node->data == data)
            break;
	if (avltree->comp (data, node->data) <= 0) {
	    son[i] = AVLTREE_LEFT;
	    last = &node->left;
	} else {
	    son[i] = AVLTREE_RIGHT;
	    last = &node->right;
	}
	node = *last;
	i++;
    }

    /* found the data at all? */
    if (!node) {
	return 0;
    }

    /* in a "local list" just skip the node */
    if (!node->right) {
	*last = node->left;
        safe_free (node);
    } else {

	/* find the next higher element */
	son[i++] = AVLTREE_RIGHT;
	next = node->right;
	last = &node->right;
	while (next->left) {
	    path[i] = last;
	    son[i] = AVLTREE_LEFT;
	    i++;
	    last = &next->left;
	    next = next->left;
	}

	/* exchange the elements, relink the tree */
	*last = next->right;
	node->data = next->data;
	safe_free (next);
    }

    /* rebalance the tree */
    avltree->root = rebalance (path, son, i);
    return 1;
}

void avltree_unlink_all (AVLTREE *avltree) {
    node_unlink_all (avltree->root);
    safe_free (avltree);
}

void node_destroy_all (AVLTREE_NODE *node, void (*crush)(void *)) {
    if (node) {
	node_destroy_all (node->left, crush);
	node_destroy_all (node->right, crush);
	crush (node->data);
	safe_free (node);
    }
}

int node_height (AVLTREE_NODE *node) {
    if (node)
        return MAX (node->left_height, node->right_height) + 1;
    return 0;
}

void node_iterate (AVLTREE_NODE *node, int order, void (*func)(void *, void *), void *misc) {
    if (node) {
	if (order == AVLTREE_PRE_ORDER)
	    func (node->data, misc);
        node_iterate (node->left, order, func, misc);
	if (order == AVLTREE_IN_ORDER)
	    func (node->data, misc);
        node_iterate (node->right, order, func, misc);
	if (order == AVLTREE_POST_ORDER)
	    func (node->data, misc);
    }
}

AVLTREE_NODE *node_rotate_left (AVLTREE_NODE *node) {
    AVLTREE_NODE *root = node->right;
    AVLTREE_NODE *tmp = root->left;

    node->right = tmp;
    node->right_height = node_height (node->right);
    root->left = node;
    root->left_height = node_height (node);
    return root;
}

AVLTREE_NODE *node_rotate_right (AVLTREE_NODE *node) {
    AVLTREE_NODE *root = node->left;
    AVLTREE_NODE *tmp = root->right;
    node->left = tmp;
    node->left_height = node_height (node->left);
    root->right = node;
    root->right_height = node_height (node);
    return root;
}

AVLTREE_NODE *node_search (AVLTREE_NODE *node, const void *data, int (*comp)(const void *, const void *)) {
    int where;

    while (node) {
        where = comp (data, node->data);
        if (where < 0)
            node = node->left;
        else if (where > 0)
            node = node->right;
        else
            return node;
    }
    return 0;
}

void node_unlink_all (AVLTREE_NODE *node) {
    if (node) {
	node_unlink_all (node->left);
	node_unlink_all (node->right);
	safe_free (node);
    }
}
