/*
 * Copyright (C) 1997 Tobias Gloth (gloth@unknown.westfalen.de)
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdarg.h>
#include <string.h>
#include <unistd.h>

#include "xtoto/args.h"
#include "xtoto/hash.h"
#include "xtoto/xtoto.h"

static struct HASH *hash;

typedef struct {
    ARG_TYPE type;
    char *name;
    void *data;
    void *next;
} ARGUMENT;

int fargc;
char **fargv;

static int comp_args (const void *p1, const void *p2) {
    return !strcmp (((ARGUMENT*)p1)->name, ((ARGUMENT*)p2)->name);
}

static int delta (const void *data) {
    char *name = ((ARGUMENT*)data)->name;
    int sum=0;
    while (*name)
        sum += *(name++);
    return sum % 17;
}

static ARGUMENT *make_arg (ARG_TYPE type, char *name, void *data, void *next) {
    ARGUMENT *arg = (ARGUMENT*) safe_malloc (sizeof (ARGUMENT));
    arg->type = type;
    arg->name = name;
    arg->data = data;
    arg->next = next;
    return arg;
}
    
static ARGUMENT *insert_arg (ARG_TYPE type, char *name, void *data, void *next) {
    ARGUMENT *arg = make_arg (type, name, data, next);
    hash_insert (hash, arg);
    return arg;
}

#define X(type) ((type)(match->data))

static void make_command_table (int flags, va_list varargs) {
    ARGUMENT *arg=0, *first=0;
    int choices=0, type, done=0;
    char *short_name, *long_name;
    void *data;

    hash = hash_create (17, delta);

    if (flags & ARG_HELP) {
        insert_arg (ARG_COMMAND, "--help", (void*) show_help, NULL);
        insert_arg (ARG_COMMAND, "-?", (void*) show_help, NULL);
    }
  
    if (flags & ARG_VERSION) {
        insert_arg (ARG_COMMAND, "--version", (void*) show_version, NULL);
        insert_arg (ARG_COMMAND, "-V", (void*) show_version, NULL);
    }

    if (flags & ARG_LICENSE) {
        insert_arg (ARG_COMMAND, "--license", (void*) show_license, NULL);
        insert_arg (ARG_COMMAND, "-L", (void*) show_license, NULL);
    }

    if (flags & ARG_DEBUG) {
        insert_arg (ARG_SWITCH, "--debug", &debug, NULL);
        insert_arg (ARG_SWITCH, "-d", &debug, NULL);
    }

    if (flags & ARG_VERBOSE) {
        insert_arg (ARG_SWITCH, "--verbose", &verbose, NULL);
        insert_arg (ARG_SWITCH, "-v", &verbose, NULL);
    }

    while (!done) {
	type = va_arg (varargs, ARG_TYPE);
        switch (type) {
            case ARG_END:
               if (choices) {
                   first->next=arg;
                   choices=0;
               } else
                   done=1;
               break;
           case ARG_CHOICE:
               choices++;
           default:
               short_name=va_arg(varargs, char*);
	       long_name=va_arg(varargs, char*);
	       data=va_arg(varargs, void*);
               if (short_name)
                   arg = insert_arg (type, short_name, data, choices?arg:NULL);
               if ((choices == 1) && short_name)
                   first = arg;
               if (long_name)
                   arg = insert_arg (type, long_name, data, choices?arg:NULL);
               if ((choices == 1) && !short_name)
                   first = arg;
        }
    }
    va_end (varargs);
}

static int parse_arg (char *name, char *ext_value) {
    ARGUMENT *arg, *match, *this;
    char cut_char, *value, *rest;
    int len;

    /* strip possible numeric values */
    len = strlen (name);
    while (len && strchr ("=.+-0123456789", name[len-1]))
        len--;

    /* separate option and value */
    cut_char = name[len];
    name[len] = '\0';

    /* search the option */
    arg = make_arg (ARG_UNKNOWN, name, NULL, NULL);
    match = hash_search (hash, arg, comp_args);
    safe_free (arg);
    name[len] = cut_char;

    if (!match)
        return 0;

    value = name+len;

    switch (match->type) {
        case ARG_SWITCH:
            if (!value[0])
                *X(int*) = 1;
            else if (value[1] || ((value[0]!='+') && (value[0]!='-')))
                error ("not a switch: \"%s\"\n", value);
            else
                *X(int*) = (value[0] == '+');
	    return 1;

        case ARG_ISWITCH:
            if (!value[0])
                *X(int*) = 0;
            else if (value[1] || (value[0]!='+' && value[0]!='-'))
                error ("not a switch: \"%s\"\n", value);
            else
                *X(int*) = (value[0] == '-');
            return 1;

        case ARG_CHOICE:
            if (value[0])
                error ("illegal option: \"%s\"\n", value);
            this = match->next;
            while (this != match) {
                *(int*)this->data = 0;
                this = this->next;
            }
            *X(int*) = 0;
	    return 1;

        case ARG_COMMAND:
	    if (value[0]) {
                error ("\"%s\" needs no argument\n", name);
	    }
            X(void(*)())();
	    return 1;

        case ARG_COMMAND_ARG:
	    if (value[0]) {
                X(void(*)(char*))(value);
	    } else if (ext_value) {
                X(void(*)(char*))(ext_value);
	    } else {
                error ("missing argument for \"%s\"\n", name);
	    }
	    return 1 + !*value;

        case ARG_FLOAT:
            if (value[0]) {
                *X(double*) = strtod (value, &rest);
                if (*rest)
                    error ("missing number for \"%s\"\n", value);
            } else if (ext_value) {
                *X(double*) = strtod (ext_value, &rest);
                if (*rest)
                    error ("missing number for \"%s\"\n", name);
            } else
                error ("missing number for \"%s\"\n", name);
	    return 1 + !*value;
      
        case ARG_NNINT:
        case ARG_PINT:
        case ARG_INT:
            if (value[0]) {
                *X(int*) = strtol (value, &rest, 10);
                if (*rest)
                    error ("integer missing for \"%s\"\n", value);
            } else if (ext_value) {
                *X(int*) = strtol (ext_value, &rest, 10);
                if (*rest)
                    error ("integer missing for \"%s\"\n", name);
            } else
                error ("missing integer for \"%s\"\n", name);
            if ((match->type == ARG_PINT) && (*X(int*) <= 0))
                error ("not positive: \"%s\"\n", value);
            if ((match->type == ARG_NNINT) && (*X(int*) < 0))
                error ("negative: \"%s\"\n", value);
	    return 1 + !*value;
      
        case ARG_STRING:
            if (!ext_value)
                error ("missing argument for: \"%s\"\n", name);
            *X(char**) = ext_value;
	    return 2;

        default:
	    break;
    }
    return 0;
}

static int parse_multi_arg (char *name) {
    char broken[3] = "-?";
    int i, len;

    len = strlen (name);
    for (i=1; i<len; i++) {
        broken[1] = name[i];
	if (!parse_arg (broken, (char*)0))
	    return 0;
    }
    return 1;
}

void parse_command_line (int argc, char **argv, int flags, ...) {
    va_list varargs;
    int i, count;

    va_start (varargs, flags);
    make_command_table (flags, varargs);

    for (i=1; i<argc; i += count) {

	/* options must begin with a - */
	if (argv[i][0] != '-')
	    break;

	/* use -- to terminate option-list */
	if (!strcmp (argv[i], "--")) {
	    i++;
	    break;
	}

        /* parse the option */
        count = parse_arg (argv[i], (i<argc-1) ? argv[i+1] : (char*)0);
	if (!count) {
	    if (!parse_multi_arg (argv[i])) {
	        error ("unknown option \"%s\"\n", argv[i]);
            } else {
	        count = 1;
            }
	}
    }

    hash_destroy_all (hash, safe_free);
    fargc=argc-i;
    fargv=argv+i;
}
