/*
 *
 * $Id: udp_daemon.c,v 1.2.4.41 2000/03/16 17:13:19 ajp Exp $
 *
 * Andrew Pitman
 *
 * pvmsync, a distributed synchronization server:  udp daemon code.
 *
 * Server that accepts connections (requests) for shared
 * data or synchronization mechanisms continuously and
 * changes some internal state according to the request.
 *
 * Copyright (C) 1999, 2000 Andrew J. Pitman
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <sys/un.h>
#include <unistd.h>

#include <sys/stat.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <asm/ioctls.h>

#include <string.h>
#include <stdio.h>
#include <stdlib.h>

#ifndef __USE_UNIX98
  #define __USE_UNIX98
#endif
#include <pthread.h>
#include <semaphore.h>

/* We need the name of our runfile in config.h */
#ifndef __FILE_NEEDS_UDP_RUNFILE
#define __FILE_NEEDS_UDP_RUNFILE
#endif

#ifndef __CONFIG_H
#include <config.h>
#endif

#ifndef __UDP_DAEMON_H
#include <udp_daemon.h>
#endif

#ifndef __UDP_SHANDLER_H
#include <udp_shandler.h>
#endif

#ifndef __LOGGING_H
#include <logging.h>
#endif

/* Object types */
#include <include/pvm_gen.h>


#ifndef lint
static char copyright[] =
"Copyright (c) 1999 Andrew J. Pitman.\nThis daemon is free software and comes with NO WARRANTY.\n";
#undef vers
static char vers[] = "1.2.4beta1";
static char *rcsid = "$Id: udp_daemon.c,v 1.2.4.41 2000/03/16 17:13:19 ajp Exp $"; 
#endif


/* Filename after RUNDIR has been resolved */
char log_filename[MAX_FILEPATH];

/* Socket to communicate with clients */
int udp_client_sock;

/* Array of file descriptors and addresses */
struct udp_respond udp_response_table[32];

/* Protect access to the udp_response_table */
pthread_mutex_t udp_response_mutex;
pthread_cond_t udp_response_cv;
int udp_response_sentinel;

/* Socket to communicate with pvmsyncd locally */
int unix_server_sock;

/* Protect access to the unix_server_sock */
pthread_mutex_t unix_server_mutex;

/* We'll need this when we reconnect */
struct sockaddr_un unix_addr;

/* File descriptor for run/log file */
extern int log_fd;

/* Protect access to the logfile fd */
extern pthread_mutex_t *wrlog_mutex;

/* What we were called as */
char *progname;

/* Print usage information and exit*/
void usage (char *program_name)
{
  fprintf (stderr, "Invalid program argument\n");
  fprintf (stderr, "usage: %s [portnum]\n", program_name);
  exit (1);
}

/* Write PID to pidfile */
/* 0 on success, -1 on fail */
int wr_pidfile (char *run_dir)
{
int pid_fd;
ssize_t bytes;
char pid_path[MAX_FILEPATH],
     wr_buff[32];

  snprintf (pid_path, MAX_FILEPATH, "%s/%s", run_dir, udp_pid_file);
  pid_fd = open (pid_path, (O_WRONLY | O_CREAT | O_TRUNC), 0644);
  if (pid_fd < 0) {
    fprintf(stderr,
            "%s: wr_pidfile (open): %s\n",
            progname,
            strerror(errno));
    return -1; 
  }

  snprintf (wr_buff, 32, "%d", (int) getpid ());
  /* Just to be sure.... */
  wr_buff[31] = '\0';

  if ((bytes = write (pid_fd, wr_buff, strlen (wr_buff))) == -1) {
    fprintf(stderr,
            "%s: wr_pidfile (write): %s\n",
            progname,
            strerror(errno));
    (void) close (pid_fd);
    return -1;
  }

  /* Close fd */
  (void) close (pid_fd);

return 0;
}

/* Set up struct sockaddr_un */
/* 0 on success, -1 on fail */
int setup_sockaddr_un (struct sockaddr_un *unix_addr)
{
char *rd = NULL, *ipc_path = NULL;

  #ifdef __HONOR_ENV_RUNDIR
    rd = getenv("RUNDIR");
    if (rd == NULL) {
      ipc_path = (char *) malloc (strlen ((const char *) rundir) + 17);
      if (!ipc_path) {
        fprintf (stderr,
                 "%s: setup_sockaddr_un: can't allocate memory\n",
                 progname);
        return -1;
      }
      (void) snprintf (ipc_path, strlen (rundir) + 17, "%s/.pvmsync/.pvm-1",
                      (const char *) rundir);
    } else {
      ipc_path = (char *) malloc (strlen (rd) + 17);
      if (!ipc_path) {
        fprintf (stderr,
                 "%s: setup_sockaddr_un: can't allocate memory\n",
                 progname);
        return -1;
      }
      snprintf(ipc_path, strlen (rd) + 17, "%s/.pvmsync/.pvm-1", rd);
    }
  #else
    ipc_path = (char *) malloc (strlen ((const char *) rundir) + 17);
    if (!ipc_path) {
      fprintf (stderr,
               "%s: setup_sockaddr_un: can't allocate memory\n",
               progname);
      return -1;
    }
    (void) snprintf (ipc_path, strlen (rundir) + 17, "%s/.pvmsync/.pvm-1",
                    (const char *) rundir);
  #endif

  (void) memset (unix_addr, 0, sizeof (struct sockaddr_un));

  unix_addr->sun_family = AF_UNIX;
  (void) memcpy (unix_addr->sun_path, ipc_path, strlen (ipc_path));

  /* Free ipc_path....  Only the other daemon needs to know
     the name of our socket (to unlink it).  We can safely
     forget it */
  free (ipc_path);

return 0;
}

int main(int argc, char *argv[])
{
int status, pid, index;
int *curr_index = NULL;
char *mesg = NULL;
char errstring[64];
char *run_dir;
char *rd = NULL;
int addrlen, portnum;
struct sockaddr_in inet_addr;
struct sigaction *act_term, *act_shup;
struct stat *stat_buff;

/* Attributes object to create threads detached */
pthread_attr_t attrib;

/* Dummy thread identifier */
pthread_t thread_id;

  /* Set what we were called as */
  progname = argv[0];

  if (argc > 2)
    usage (progname);

  if (argc == 2)
    portnum = atoi (argv[1]);
  else
    portnum = UDP_SERVER_PORT;

  pid = fork();
  if (pid) {
    printf("%s", copyright);
    printf("%s (%s): server starting at port %d\n",
                    progname, (char *) vers, portnum);
    exit(0);
  }

  /* Struct sigaction for SIGTERM */
  act_term = (struct sigaction *) malloc (sizeof (struct sigaction));
  if (act_term == NULL) {
          fprintf(stderr, "%s: malloc: couldn't allocate memory\n", progname);
          exit(1);
  }
  (void) memset (act_term, 0, sizeof (struct sigaction));

  /* Struct sigaction for SIGHUP */
  act_shup = (struct sigaction *) malloc (sizeof (struct sigaction));
  if (act_shup == NULL) {
          fprintf(stderr, "%s: malloc: couldn't allocate memory\n", progname);
          exit(1);
  }
  (void) memset (act_shup, 0, sizeof (struct sigaction));

  /* Set the appropriate signal handlers */
  act_term->sa_handler = term_handler;
  act_shup->sa_handler = shup_handler;
  if ((sigaction (SIGTERM, act_term, NULL)) ||
       (sigaction (SIGHUP, act_shup, NULL))) {
    (void) snprintf (errstring, 64, "%s: sigaction", progname);
    perror(errstring);
    exit(1);
  }
  free(act_term);
  free(act_shup);

  (void) memset (errstring, 0, 64);

  #ifdef __SET_CONCURRENCY_LEVEL
    /* This may or may not have any effect, depending on the system */
    status = pthread_setconcurrency (PTHREAD_CONCURRENCY);
    if (status)
      fprintf(stderr, "warning: pthread_setconcurrency: %s\n",
                                                strerror (status));
  #endif /* __SET_CONCURRENCY_LEVEL */

  status = pthread_attr_init (&attrib);
  if (status) {
    fprintf(stderr, "%s: pthread_attr_init: %s\n", progname, strerror(status));
    exit(1);
  }
  status = pthread_attr_setdetachstate (&attrib, PTHREAD_CREATE_DETACHED);
  if (status) {
    fprintf(stderr, "%s: pthread_attr_setdetachstate: %s\n", progname,
            strerror(status));
    exit (1);
  }

  if (setup_sockaddr_un (&unix_addr))
    exit (1);

  /* See if our UNIX domain socket is there */
  /* If not, barf and exit. */
  stat_buff = (struct stat *) malloc (sizeof (struct stat));
  status = stat (unix_addr.sun_path, stat_buff);
  if (status) {
    (void) snprintf (errstring, 64, "%s: stat", progname);
    perror (errstring);
    exit (1);
  }
  if (!(stat_buff->st_mode & S_IFSOCK)) {
    fprintf (stderr, "%s: socket does not exist\n", progname);
    fprintf (stderr, "Are you sure you have pvmsyncd running?\n");
    exit (1);
  }

  (void) memset (&inet_addr, 0, sizeof (struct sockaddr_in));

  /* Listen on port portnum */
  inet_addr.sin_family = AF_INET;
  inet_addr.sin_port = htons (portnum);
  inet_addr.sin_addr.s_addr = INADDR_ANY; /* default address for client */

  udp_client_sock = socket (AF_INET, SOCK_DGRAM, 0);
  if (udp_client_sock == -1) {
    (void) snprintf (errstring, 64, "%s: socket", progname);
    perror (errstring);
    exit (1);
  }

  status = bind (udp_client_sock, &inet_addr, sizeof (struct sockaddr_in));
  if (status) {
    (void) snprintf (errstring, 64, "%s: bind", progname);
    perror (errstring);
    exit (1);
  }

  /* Set up our mutex to protect the table during creation/
     destruction of udp_response_table entries. */
  status = pthread_mutex_init (&udp_response_mutex, NULL);
  if (status) {
    fprintf (stderr, "%s: pthread_mutex_init: %s\n", progname,
                                           strerror (status));
    exit (1);
  }

  /* Initialize condition variable we wait on if all 32 table
     entries have been handed out. */
  status = pthread_cond_init (&udp_response_cv, NULL);
  if (status) {
    fprintf (stderr, "%s: pthread_cond_init: %s\n", progname,
                                          strerror (status));
    exit (1);
  }

  for (index = 0; index < 32; index++) {
    /* Mark these as invalid */
    udp_response_table[index].sock = -1;
    udp_response_table[index].inet_addr = NULL;
    (void) memset (&udp_response_table[index].packet, 0,
                         sizeof (struct udp_app_dgram));
  }
  udp_response_sentinel = 0;

  index = 0;

/* Open run/log file */
/* For later calls to report_log */
  run_dir = (char *) malloc (MAX_FILEPATH);

    #ifdef __HONOR_ENV_RUNDIR
        if (rd == NULL) {
          (void) snprintf (run_dir, MAX_FILEPATH, "%s/.pvmsync",
                           (const char *) rundir);
        } else {
          (void) snprintf (run_dir, MAX_FILEPATH, "%s/.pvmsync", rd);
        }
    #else
        (void) snprintf (run_dir, MAX_FILEPATH, "%s/.pvmsync",
                         (const char *) rundir);
    #endif

        status = stat (run_dir, stat_buff);
        if (status) {
          snprintf(errstring, 64, "%s: stat", progname);
          perror(errstring);
          exit(1);
        } else {
          if (!(stat_buff->st_mode & S_IFDIR)) {
            fprintf(stderr, "%s: %s exists and is not a directory\n",
                                                   progname, run_dir);
            exit (1);
          }
        }

  #ifndef __LOG_SYSLOG

        (void) snprintf (log_filename, MAX_FILEPATH, "%s/%s", run_dir,
                                                       udp_run_file);

        log_fd = open (log_filename, (O_CREAT | O_APPEND | O_WRONLY), 0644);
        if (log_fd == -1) {
                snprintf(errstring, 64, "%s: open", progname);
                perror(errstring);
                exit(1);
        }

        /* init mutex to protect file writes for run/log file */
        wrlog_mutex = (pthread_mutex_t *) malloc (sizeof (pthread_mutex_t));
        if (wrlog_mutex == NULL) {
          fprintf(stderr, "%s: malloc: couldn't allocate memory\n", progname);
          exit(1);
        }
        status = pthread_mutex_init(wrlog_mutex, NULL);
        if (status) {
          fprintf(stderr, "%s: pthread_mutex_init: %s\n", progname,
                                strerror(status));
          exit(1);
        }

  #else

        #ifdef DEBUG
            openlog ("pvmsync", LOG_CONS | LOG_PID, LOG_USER);
        #endif

  #endif

  /* Write PID to pidfile */
  if (wr_pidfile (run_dir))
    exit (1);

  /* We're done with run_dir */
  free (run_dir);
  /* We're done with stat */
  free (stat_buff);

  mesg = (char *) malloc (128);
  if (mesg == NULL) {
    report_log ("pvmobjd: unable to allocate memory\n");
    exit (1);
  }
  (void) snprintf (mesg, 128, "%s (%s): server started, listening on port %d\n",
                   progname, (char *) vers, portnum);
  report_log (mesg);
  free (mesg);

  /* Detach process from tty, since we're a daemon */
  ioctl(0, TIOCNOTTY, NULL);

  /* Place ourself in our own process group */
  if (setpgid ((pid_t) 0, (pid_t) 0)) {
    (void) snprintf (errstring, 64, "%s: setpgid", progname);
    perror (errstring);
    exit (1);
  }

  while (1) {
    /* Here, we loop and do recvfrom's, then pass the index of
       the buffer and the struct sockaddr_in to a child thread
       that responds if the local server has the given object
       (or the load regardless). */

    /* Update and allocate udp_response_table[index] */
    if (pthread_mutex_lock (&udp_response_mutex)) {
      (void) pthread_mutex_lock (wrlog_mutex);
             report_log ("pvmobjd: unable to lock internal mutex\n");
      (void) pthread_mutex_unlock (wrlog_mutex);
      exit (1);
    }
    /* It won't be invalid anymore, according to our tests */
    udp_response_table[index].sock--; 
    /* Allocate memory for the struct sockaddr_in */
    /* The child thread will free this */
    udp_response_table[index].inet_addr =
                    (struct sockaddr_in *) malloc (sizeof (struct sockaddr_in));
    if (udp_response_table[index].inet_addr == NULL) {
      (void) pthread_mutex_lock (wrlog_mutex);
             report_log ("pvmobjd: unable to allocate memory\n");
      (void) pthread_mutex_unlock (wrlog_mutex);
      exit (1);
    }
    udp_response_sentinel++;
    if (pthread_mutex_unlock (&udp_response_mutex)) {
      (void) pthread_mutex_lock (wrlog_mutex);
             report_log ("pvmobjd: unable to unlock internal mutex\n");
      (void) pthread_mutex_unlock (wrlog_mutex);
      exit (1);
    }

    /* Receive a packet and store it in the appropriate
       struct udp_app_dgram */
    status = recvfrom (udp_client_sock,
                      (void *) &udp_response_table[index].packet,
                       sizeof (struct udp_app_dgram), 0,
                       udp_response_table[index].inet_addr, &addrlen);

    if ((status == -1) && (errno != EINTR)) {
      if (pthread_mutex_lock (&udp_response_mutex)) {
        (void) pthread_mutex_lock (wrlog_mutex);
               report_log ("pvmobjd: unable to lock internal mutex\n");
        (void) pthread_mutex_unlock (wrlog_mutex);
        exit (1);
      }
      udp_response_table[index].sock = -1;
      if (udp_response_table[index].inet_addr != NULL)
        free (udp_response_table[index].inet_addr);
      if (pthread_mutex_unlock (&udp_response_mutex)) {
        (void) pthread_mutex_lock (wrlog_mutex);
               report_log ("pvmobjd: unable to unlock internal mutex\n");
        (void) pthread_mutex_unlock (wrlog_mutex);
        exit (1);
      }
    } else {
      /* Pass the index to a child thread that will query the tcp
         server and respond if appropriate. */
      curr_index = (int *) malloc (sizeof (int));
      *curr_index = index;
      status = pthread_create (&thread_id, &attrib, proc_query, curr_index);
      if (status) {
        (void) pthread_mutex_lock (wrlog_mutex);
               report_log ("pvmobjd: cannot create new thread\n");
        (void) pthread_mutex_unlock (wrlog_mutex);
        exit (1);
      }
    }

    /* We're searching for an unused index, lock. */
    if (pthread_mutex_lock (&udp_response_mutex)) {
      (void) pthread_mutex_lock (wrlog_mutex);
             report_log ("pvmobjd: unable to lock internal mutex\n");
      (void) pthread_mutex_unlock (wrlog_mutex);
      exit (1);
    }

    /* If we don't have any available, block. */
    if (udp_response_sentinel == 32) {
      status = pthread_cond_wait (&udp_response_cv, &udp_response_mutex);
      if (status) {
        (void) pthread_mutex_lock (wrlog_mutex);
               report_log ("pvmobjd: error on condition variable wait\n");
        (void) pthread_mutex_unlock (wrlog_mutex);
        exit (1);
      }
    }

    do {
      if (index >= 31)
        index = -1;
    } while (udp_response_table[++index].sock != -1);

    if (pthread_mutex_unlock (&udp_response_mutex)) {
      (void) pthread_mutex_lock (wrlog_mutex);
             report_log ("pvmobjd: unable to unlock internal mutex\n");
      (void) pthread_mutex_unlock (wrlog_mutex);
      exit (1);
    }

  };

return 0;
}

/* Function to process the query received by the udp daemon 
   (look up with tcp server), then respond with a datagram 
   if appropriate. */
/* We're running in a newly created thread. */
void *proc_query (void *index)
{
int status, *localindex;
ssize_t numbytes;
struct sockaddr_in local;
char localbuf[32], errstring[64];

  localindex = (int *) index;

  /* Check to see that the magic #, version, etc. are OK */
  if (!vrfy_udp_request (&udp_response_table[*localindex].packet))
    cleanup (index); /* Drop it */

  (void) memset (localbuf,  0, 32);
  (void) memset (errstring, 0, 64);

  if (pthread_mutex_lock (&unix_server_mutex))
    cleanup (index);

  unix_server_sock = socket (AF_UNIX, SOCK_STREAM, 0);
  if (unix_server_sock == -1) {
    (void) pthread_mutex_lock (wrlog_mutex);
    (void) snprintf (errstring, 64, "%s: socket: %s\n", progname,
                                                 strerror (errno));
           report_log (errstring);
    (void) pthread_mutex_unlock (wrlog_mutex);
    exit (1);
  }

  if (connect (unix_server_sock, &unix_addr, sizeof (struct sockaddr_un))) {
    (void) pthread_mutex_lock (wrlog_mutex);
    (void) snprintf (errstring, 64, "%s: connect: %s\n", progname,
                                                 strerror (errno));
           report_log (errstring);
    (void) pthread_mutex_unlock (wrlog_mutex);
    if ((errno != EISCONN) && (errno != ECONNREFUSED) && (errno != EALREADY)) {
      (void) raise (SIGINT);
    } else {
      (void) close (unix_server_sock);
      if (pthread_mutex_unlock (&unix_server_mutex))
        cleanup (index);
    }
    (void) pthread_exit (NULL);
  }

  numbytes = write (unix_server_sock,
                    &udp_response_table[*localindex].packet,
                    (size_t) udp_pcktlen
                        (&udp_response_table[*localindex].packet) * 4);
  if (numbytes == -1)
    cleanup (index);

  numbytes = read (unix_server_sock, localbuf, 32);
  if (numbytes == -1)
    cleanup (index);

  close (unix_server_sock);

  if (pthread_mutex_unlock (&unix_server_mutex))
    cleanup (index);

  /* According to what localbuf contains and the request that was
     made, alter the packet just right and send it back */
  if ((localbuf[0] >= '0') && (localbuf[0] <= 9)) {

    /* Don't reply if we're unwilling to host another object */
    if (atoi (localbuf) >= MAX_OBJS)
      cleanup (index);

    udp_response_table[*localindex].sock = socket (AF_INET, SOCK_DGRAM, 0);
    if (udp_response_table[*localindex].sock == -1)
      cleanup (index);

    (void) memset (&local, 0, sizeof (struct sockaddr_in));
    local.sin_family = AF_INET;
    local.sin_port = 0;  /* Any port the system hands us is alright */
    local.sin_addr.s_addr = INADDR_ANY; /* default address for this host */

    status = bind (udp_response_table[*localindex].sock, &local,
             sizeof (struct sockaddr_in));
    if (status)
      cleanup (index);

    udp_response_table[*localindex].packet.length = (char) 2;
    udp_response_table[*localindex].packet.dgram_type = UDP_REPLY;
    udp_response_table[*localindex].packet.num_arg = htons (atoi (localbuf));

    status = sendto(udp_response_table[*localindex].sock,
                   (void *) &udp_response_table[*localindex].packet,
                   16, 0, udp_response_table[*localindex].inet_addr,
                   sizeof (struct sockaddr_in));
    if (status)
      cleanup (index);

  } else if (!strncmp (localbuf, "YES:", 4)) {

    udp_response_table[*localindex].sock = socket (AF_INET, SOCK_DGRAM, 0);
    if (udp_response_table[*localindex].sock == -1)
      cleanup (index);

    (void) memset (&local, 0, sizeof (struct sockaddr_in));
    local.sin_family = AF_INET;
    local.sin_port = 0;  /* Any port the system hands us is alright */
    local.sin_addr.s_addr = INADDR_ANY; /* default address for this host */

    status = bind (udp_response_table[*localindex].sock, &local,
             sizeof (struct sockaddr_in));
    if (status)
      cleanup (index);

    /* Here, we must craft the return packet to send back to the client */
    /* We will only send as much of this data as is needed, in complete
       words */
    udp_response_table[*localindex].packet.length = (char) 2;
    udp_response_table[*localindex].packet.dgram_type = UDP_REPLY;
    udp_response_table[*localindex].packet.num_arg = 0;

    status = sendto(udp_response_table[*localindex].sock,
                   (void *) &udp_response_table[*localindex].packet,
                    16, 0, udp_response_table[*localindex].inet_addr,
                    sizeof (struct sockaddr_in));
    if (status < 0)
      cleanup (index);
  }

  /* Update and deallocate udp_response_table[localindex] */
  if (pthread_mutex_lock (&udp_response_mutex)) {
    (void) raise (SIGINT);
    (void) pthread_exit (NULL);
  }

  udp_response_table[*localindex].sock = -1;
  if (udp_response_table[*localindex].inet_addr != NULL)
    free (udp_response_table[*localindex].inet_addr);
  (void) memset (&udp_response_table[*localindex].packet, 0,
                             sizeof (struct udp_app_dgram));
  if (udp_response_sentinel == 32) {
    if (pthread_cond_signal (&udp_response_cv)) {
      (void) raise (SIGINT);
      (void) pthread_exit (NULL);
    }
  }
  udp_response_sentinel--;

  if (pthread_mutex_unlock (&udp_response_mutex)) {
    (void) raise (SIGINT);
    (void) pthread_exit (NULL);
  }

  if (index != NULL)
    free (index);

  (void) pthread_exit (NULL);

return NULL; /* NOTREACHED */
}

/* free stuff, mark our fd as invalid, and exit (pthread_exit) */
inline void cleanup (void *index)
{
int *localindex;

  localindex = (int *) index;

  /* Update and deallocate udp_response_table[localindex] */
  if (pthread_mutex_lock (&udp_response_mutex)) {
    (void) raise (SIGINT);
    (void) pthread_exit (NULL);
  }

  /* Zealously destroy all elements of
     struct udp_response_table[*localindex] */
  udp_response_table[*localindex].sock = -1;
  if (udp_response_table[*localindex].inet_addr != NULL)
    free (udp_response_table[*localindex].inet_addr);
  (void) memset (&udp_response_table[*localindex].packet,
                 0, sizeof (struct udp_app_dgram));
  if (udp_response_sentinel == 32) {
    if (pthread_cond_signal (&udp_response_cv)) {
      (void) raise (SIGINT);
      (void) pthread_exit (NULL);
    }
  }
  udp_response_sentinel--;
  
  if (pthread_mutex_unlock (&udp_response_mutex)) {
    (void) raise (SIGINT);
    (void) pthread_exit (NULL);
  }

  if (index != NULL)
    free (index);

  (void) pthread_exit (NULL);

}


