/*
 *
 * $Id: tcp_daemon.h,v 1.2.4.41 2000/03/16 17:13:11 ajp Exp $
 *
 * Andrew Pitman
 *
 * pvmsync, a distributed synchronization server:  main daemon decls.
 *
 * Server that accepts connections (requests) for shared
 * data or synchronization mechanisms continuously and
 * changes some internal state according to the request.
 *
 * Copyright (C) 1999, 2000 Andrew J. Pitman
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


#ifndef __TCP_DAEMON_H
#define __TCP_DAEMON_H


#ifndef __CONFIG_H
  /* SYMBOL_LENGTH, STALE_OBJECT_TIMEOUT */
#include <config.h>
#endif

#ifndef __GARBAGE_H
  /* garbage_collect () */
#include <garbage.h>
#endif


#ifndef __USE_UNIX98
  #define __USE_UNIX98
#endif
#include <pthread.h>
#include <semaphore.h>


/* va_arg, etc. */
#include <stdarg.h>


/* Structure for our internal "pool" of mutexes */

typedef struct mutex_pool_tag {
  pthread_mutex_t mutex;            /* Internal mutex to protect
                                       access. */
  pthread_cond_t mutex_cv;          /* Condition variable to wait
				       on if mutex is already locked */
  char mutex_symbol[SYMBOL_LENGTH]; /* mutex name */
  int  valid_mutex;                 /* mutex is valid */
  int mutex_locked;                 /* 0 if unlocked, pseudo-
				       random number if locked */
  unsigned int mutex_lockaddr;      /* IP of host that locked
				       this mutex */
  unsigned int waiters;             /* Processes waiting on mutex */
  time_t access_time;               /* Last access */
} mutex_pool_t;

typedef struct sem_pool_tag {
  char sem_symbol[SYMBOL_LENGTH]; /* Semaphore symbol name */
  int valid_semaphore;            /* Is semaphore valid? */
  sem_t semaphore;                /* The semaphore */
  time_t access_time;             /* Last access */
} sem_pool_t;

/* A pool of pvm_int64_t's */
typedef struct integer_pool_tag {
  pvm_int64_t data;      /* payload */
  int valid;             /* is this element valid? */
  char symbol[SYMBOL_LENGTH]; /* Integer variable's name */
  pthread_mutex_t mutex; /* Protect data updates */
                         /* This does not insure */
                         /* atomicity between updates. */
  time_t access_time;    /* Last access */
} integer_pool_t;

/* A pool of pvm_float64_t's */
typedef struct float_pool_tag {
  pvm_float64_t data;    /* payload */
  short int valid;       /* is this element valid? */
  char symbol[SYMBOL_LENGTH]; /* Floating pt. variable's name */
  pthread_mutex_t mutex; /* Protect data updates */
                         /* This does not insure */
                         /* atomicity between updates. */
  time_t access_time;    /* Last access */
} float_pool_t;

/* A pool of pointers to heap-dynamic memory blocks */
typedef struct memptr_pool_tag {
  void *block;           /* Our pointer */
  int numbytes;          /* How big our block of memory is */
  short int valid;       /* is this element valid? */
  char symbol[SYMBOL_LENGTH]; /* Memory block's name */
  pthread_mutex_t mutex; /* Protect data updates */
                         /* This does not insure */
                         /* atomicity between updates. */
  time_t access_time;    /* Last access */
} memptr_pool_t;

/* Pool of condition variables */
typedef struct cond_pool_tag {
  char cond_symbol[SYMBOL_LENGTH]; /* Condition variable symbol name */
  int valid_cond;                  /* Is condvar valid? */
  pthread_cond_t cond;             /* Our condition variable */
  pthread_mutex_t mutex;           /* Associated mutex */
  time_t access_time;              /* Last access */
} cond_pool_t;


/* Structure for each hash table index */
struct ht_ent {
  pthread_rwlock_t ent_rwlock;
  void *head;
  unsigned int num_subent;
};

/* Sub-entry structures. */
struct ht_subent_mutex {
  mutex_pool_t pvm_mutex;
  struct ht_subent_mutex *next;
  struct ht_subent_mutex *prev;
};

struct ht_subent_cv {
  cond_pool_t pvm_cv;
  struct ht_subent_cv *next;
  struct ht_subent_cv *prev;
};

struct ht_subent_sem {
  sem_pool_t pvm_sem;
  struct ht_subent_sem *next;
  struct ht_subent_sem *prev;
};


/* It probably wouldn't be a bad idea for these routines to return
   a positive (or negative) error code instead of -1 and setting
   errno.  Since the server is multithreaded, it's possible for
   error handling tasks to collide with each other. */

/* Print usage information and exit*/
void usage __P ((char *program_name)) __attribute__ ((noreturn));

/* Return the proper index into the hash table
   given a symbol name */
inline int hash_index __P ((char * obj_name));

/* Return zero if symbol name is legal,
   negative error code if not */
inline int legal_sym __P ((char * obj_name));

/* Return zero if we don't have symbol,
   positive nonzero if we do, negative
   error code on error */
/* Better than setting errno and returning -1...
   safer for multiple threads to return the error
   code than set a global var... */
int search_ht __P ((char * obj_name, pvm_t obj_type));

/* Insert an entry into the hash table
   Return 0 on success, negative error
   code on failure */
int insert_ht __P ((char * obj_name, pvm_t obj_type, ...));

/* Find an entry by name in the appropriate hash
   table for obj_type, and return a pointer to
   it's subentry.  NULL on failure. */
void * find_ht __P ((char * obj_name, pvm_t obj_type));

/* Remove an entry from the hash table
   Return 0 on success, -1 on failure */
int remove_ht __P ((char * obj_name, pvm_t obj_type));

/* Unlock rwlock or die */
inline void unlock_r_die __P ((pthread_rwlock_t * rwlock));

/* Log message to runfile and die */
/* log_mesg should end with '\n'. */
void die_with_mesg __P ((char *log_mesg)) __attribute__ ((noreturn));

/* Send reply to client using tcp...  Reply is of the
   form:  SUCCESS:nilval or FAIL:errcode, where errcode
   is a nonnegative integer. */
inline void tcp_send_n_exit __P ((int success, int code, int *index));


#endif /* __TCP_DAEMON_H */

