/*
 *
 * $Id: tcp_daemon.c,v 1.2.4.41 2000/03/16 17:13:08 ajp Exp $
 *
 * Andrew Pitman
 *
 * pvmsync, a distributed synchronization server:  main daemon code.
 *
 * Server that accepts connections (requests) for shared
 * data or synchronization mechanisms continuously and
 * changes some internal state according to the request.
 *
 * Copyright (C) 1999, 2000 Andrew J. Pitman
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <sys/un.h>
#include <unistd.h>

#include <sys/stat.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <asm/ioctls.h>

#include <string.h>
#include <stdio.h>
#include <stdlib.h>

#include <time.h>

/* For RW Locks */
#ifndef __USE_UNIX98
  #define __USE_UNIX98
#endif
#include <pthread.h>
#include <semaphore.h>

#include <syslog.h>

#ifndef __CONFIG_H
#include <config.h>
#endif

#ifndef __ACCEPT_LOCAL_H
#include <accept_local.h>
#endif

#ifndef __TCP_DAEMON_H
#include <tcp_daemon.h>
#endif

#ifndef __FLTPT_H
#include <fltpt.h>
#endif

#ifndef __INITIALIZE_H
#include <initialize.h>
#endif

#ifndef __INTEGER_H
#include <integer.h>
#endif

#ifndef __LOGGING_H
#include <logging.h>
#endif

#ifndef __MEMORY_H
#include <memory.h>
#endif

#ifndef __PROC_REQ_H
#include <proc_req.h>
#endif

#ifndef __TCP_SHANDLER_H
#include <tcp_shandler.h>
#endif

/* Cmdline options */
#include <popt.h>


#ifndef lint
static char copyright[] =
"Copyright (c) 1999 Andrew J. Pitman.\nThis daemon is free software and comes with NO WARRANTY.\n";
#undef vers
static char vers[] = "1.2.4beta1";
static char *rcsid = "$Id: tcp_daemon.c,v 1.2.4.41 2000/03/16 17:13:08 ajp Exp $";
#endif


/* Return string on success */
extern const char *nilval;

int req_sock;    /* Request socket */
int ipc_fd;      /* Socket to communicate with the UDP daemon */
extern int  log_fd;     /* Log file */
char log_filename[MAX_FILEPATH];  /* Filename after TMPDIR has been resolved */
int accept_fds[CONCURRENT_CONNECTIONS]; /* An array of fd's to keep
                                           track of */
unsigned int accept_addrs[CONCURRENT_CONNECTIONS]; /* Client addresses */
char temp_buf[CONCURRENT_CONNECTIONS][BUF_LENGTH];
int  next_fd;    /* The next fd returned by accept will go here. */
int  *curr_fd;   /* Pointer to index of current fd */
char *hostname;  /* The local host name */
char *ipc_path;  /* Path to UNIX domain socket */
char *progname;  /* The name of this program */

/* Attributes for inititalization of each */
/* entry in the hash table's associated */
/* read-write lock */
pthread_rwlockattr_t *mutex_init_lock_attr;

/* Internal "pool" of mutexes (mutexi?) */
/* Hash table indexed according to first character */
struct ht_ent mutex_pool[NUM_HASH_ENTS];

/* Keep a counter of the number of objects we have of this type */
/* This is protected by the rwlock and is only updated on mutex */
/* creation and destruction. */
unsigned int num_mutexes;

/* Protect access to variable num_mutexes */
pthread_rwlock_t mutex_counter_lock;

/* Protect initialization and destruction of individual semaphores */
/* by locking a read lock to search, and a write lock to create    */
/* or destroy a semaphore from the "pool".                         */
pthread_rwlock_t sem_init_lock;
pthread_rwlockattr_t *sem_init_lock_attr;

/* Internal "pool" of semaphores */
struct ht_ent sem_pool[NUM_HASH_ENTS];

/* Keep a counter of the number of objects we have of this type */
/* This is protected by the rwlock and is only updated on sem   */
/* creation and destruction. */
unsigned int num_sems;

/* Protect access to global variable num_sems */
pthread_rwlock_t sem_counter_lock;

/* Protect access to the integer pool with a rwlock */
pthread_rwlock_t integer_init_lock;
pthread_rwlockattr_t *integer_init_lock_attr;

/* Internal "pool" of pvm_int64_t's */
integer_pool_t integer_pool[CONCURRENT_INSTANCES];

/* Keep a counter of the number of objects we have of this type */
unsigned int num_integers;

/* Protect access to the float pool with a rwlock */
pthread_rwlock_t float_init_lock;
pthread_rwlockattr_t *float_init_lock_attr;

/* Internal "pool" of pvm_float64_t. */
float_pool_t float_pool[CONCURRENT_INSTANCES]; /* float pool! */

/* Keep a counter of the number of objects we have of this type */
unsigned int num_floating;

/* Protect access to the memptr pool with a rwlock */
pthread_rwlock_t memptr_init_lock;
pthread_rwlockattr_t *memptr_init_lock_attr;

/* Internal "pool" of pointers to heap-dynamic data. */
memptr_pool_t memptr_pool[CONCURRENT_INSTANCES];

/* Keep a counter of the number of objects we have of this type */
unsigned int num_memblocks;

/* Attributes for inititalization of each */
/* entry in the hash table's associated */
/* read-write lock */
pthread_rwlockattr_t *condvar_init_lock_attr;

/* Internal "pool" of condition variables */
/* Hash table indexed according to first character */
struct ht_ent cond_pool[NUM_HASH_ENTS];

/* Keep a counter of the number of objects we have of this type */
/* This is protected by the rwlock and is only updated on condvar */
/* creation and destruction. */
unsigned int num_conds;

/* Protect access to variable num_conds */
pthread_rwlock_t cond_counter_lock;

/* Protect updates to the log file */
pthread_mutex_t *wrlog_mutex;

/* Print usage information and exit*/
void usage (char *program_name)
{
  fprintf (stderr, "usage: %s [-T | -t timeout] [portnum]\n", program_name);
  exit (1);
}

int main(int argc, char *argv[])
{
struct sockaddr_in daemon_sin, client_sin;  /* sockaddr struct for creating
					       daemon socket, and returned
					       by accept containing client
					       info, respectively.        */
struct hostent *server;
struct  sigaction  *act_shup, *act_term;
int status, sin_addr_length, portnum;
unsigned int secs = STALE_OBJECT_TIMEOUT;
pid_t pid;
char errstring[64], mesg_buf[128];

/* options holds table of possible arguments, args is
   the context for the command-line set.              */
struct poptOption options[] = { { "timeout",   't', POPT_ARG_INT, &secs, 1 },
                                { "notimeout", 'T', POPT_ARG_NONE, NULL, 2 },
                                { "help",      'h', POPT_ARG_NONE, NULL, 3 },
                                { NULL,        0,   0,             NULL, 0 } };
poptContext args;
int poptretn = 0, tflag = 1, Tflag = 0;
int have_extra_opts = 0;
char *new_argv; /* New argv after options have been removed */

/* Attributes object to create threads detached */
pthread_attr_t attrib;

/* Dummy thread identifier */
pthread_t thread_id;

  /* What were we run as?? */
  progname = strdup(argv[0]);

  /* Get poptContext args so we can update flags below. */
  args = (poptContext) poptGetContext ("pvmsync", argc, argv, options, 0);

  /* Finally, we update the flags. */
  do {
    poptretn = poptGetNextOpt((poptContext) args);

      switch (poptretn)  {

        case 1:   tflag = 1;
                  Tflag = 0;
          break;
        case 2:   Tflag = 1;
                  tflag = 0;
          break;
        case 3:   usage(progname);
                  exit(1);
      };
       
  } while (poptretn > 0);

  /* Handle errors returned by poptGetNextOpt() */
  switch (poptretn) {
    case POPT_ERROR_NOARG:
    case POPT_ERROR_BADOPT:
    case POPT_ERROR_BADNUMBER:
    case POPT_ERROR_OVERFLOW:
      fprintf (stderr, "Invalid program argument\n");
      usage(progname);
      exit(1);
    case POPT_ERROR_OPTSTOODEEP:
      (void) fflush (stdout);
      fprintf(stderr, "%s: options nested too deeply!\n", progname);
      (void) fflush (stderr);
      exit(1);
    case POPT_ERROR_BADQUOTE:
      (void) fflush (stdout);
      fprintf(stderr, "%s: Unmatched quote.\n", progname);
      (void) fflush (stderr);
      exit(1);
  };

  /* And save any leftovers (we should only have a port number left). */
  new_argv = (char *) poptGetArgs ((poptContext) args);
  if (new_argv) {
      have_extra_opts = 1;
      argv = (char **) strdup (new_argv);
  }
  /* new argv now contains ONLY extra, non-option type stuff. */
  /* argv[0] is now the first nonoption cmdline arg. */
  /* We must free the dynamically allocated stuff in Context args,
     now that we're done with argv (a pointer to within it).       */
  poptFreeContext((poptContext) args);

  if (have_extra_opts)
    portnum = atoi (argv[0]);
  else
    portnum = TCP_SERVER_PORT;

  /* Zero out errstring */
  memset(errstring, 0, 64);

  pid = fork();
  if (pid) {
    printf("%s", copyright);
    printf("%s (%s): server starting at port %d\n",
                    progname, (char *) vers, portnum);
    exit(0);
  }
	
  /* Struct sigaction for SIGTERM */
  act_term = (struct sigaction *) malloc (sizeof (struct sigaction));
  if (act_term == NULL) {
	  fprintf(stderr, "%s: malloc: couldn't allocate memory\n", progname);
	  exit(1);
  }
  memset (act_term, 0, sizeof (struct sigaction));

  /* Struct sigaction for SIGHUP */
  act_shup = (struct sigaction *) malloc (sizeof (struct sigaction));
  if (act_shup == NULL) {
	  fprintf(stderr, "%s: malloc: couldn't allocate memory\n", progname);
	  exit(1);
  }
  memset (act_shup, 0, sizeof (struct sigaction));

  /* Set the appropriate signal handlers */
  act_term->sa_handler = term_handler;
  act_shup->sa_handler = shup_handler;
  act_shup->sa_flags = SA_RESTART; /* Restart slow syscalls */
  if ((sigaction (SIGTERM, act_term, NULL)) ||
       (sigaction (SIGHUP, act_shup, NULL))) {
	snprintf(errstring, 64, "%s: sigaction", progname);
	perror(errstring);
        exit(1);
  }
  free(act_term);
  free(act_shup);

#ifdef __SET_CONCURRENCY_LEVEL
  /* This may or may not have any effect, depending on the system */
  status = pthread_setconcurrency (PTHREAD_CONCURRENCY);
  if (status)
	fprintf(stderr, "warning: pthread_setconcurrency: %s\n",
						strerror (status));
#endif /* __SET_CONCURRENCY_LEVEL */

  /* Initialize rundir for UNIX domain socket and
     runfiles (if used) */
  if (init_rundir ())
    exit (1);

  /* Initialize log/run file */
  if (init_log ())
    exit (1);

  /* Write PID file */
  if (wr_pid ())
    exit (1);

  /* Lets try it out */
  /* Lock the mutex protecting logfile updates, and
     print a message that we're starting up...  */
  (void) pthread_mutex_lock (wrlog_mutex);
  snprintf(mesg_buf, 128, "pvmsyncd (%s): server starting up\n",
           (char *) vers);
  report_log (mesg_buf);

  /* General initialization */
  /* Call funcs to init pools of different resources, init
     random number generator, open UNIX socket */
  /* 0 on success, -1 on failure */
  if (gen_init ())
    exit (1);

  /* Prep attribs for thread creation */
  /* OK if these fail, just report error and continue */
  status = pthread_attr_init (&attrib);
  if (status) {
	snprintf(errstring, 64, "%s: pthread_attr_init", progname);
	fprintf(stderr, "%s: %s\n", errstring, strerror(status));
  }
  status = pthread_attr_setdetachstate (&attrib, PTHREAD_CREATE_DETACHED);
  if (status) {
	snprintf(errstring, 64, "%s: pthread_attr_setdetachstate", progname);
	fprintf(stderr, "%s: %s\n", errstring, strerror(status));
  }

#ifdef __DO_GARBAGE_COLLECTION

  if (!Tflag) {
    status = pthread_create (&thread_id, &attrib, garbage_collect, &secs);
    if (status) {
      snprintf(errstring, 64, "%s: pthread_create: %s\n", progname,
                                                    strerror (status));
      (void) pthread_mutex_lock (wrlog_mutex);
             report_log (errstring);
      (void) pthread_mutex_unlock (wrlog_mutex);
      exit(1);
    }
  }

#endif /* __DO_GARBAGE_COLLECTION */

  memset (&daemon_sin, 0, sizeof (struct sockaddr_in));

  req_sock = socket (AF_INET, SOCK_STREAM, 0);
  if (req_sock == -1) {
	snprintf(errstring, 64, "%s: socket", progname);
	perror(errstring);
	exit(1);
  }

  hostname = (char *) malloc (HOSTNAMELEN);
  if (hostname == NULL) {
	fprintf(stderr, "%s: malloc: couldn't allocate memory\n", progname);
	exit(1);
  }	

  status = gethostname(hostname, HOSTNAMELEN);
  if (status) {
	snprintf(errstring, 64, "%s: gethostname", progname);
	perror(errstring);
	exit(1);
  }

  server = gethostbyname (hostname);
  if (server == NULL) {
	snprintf(errstring, 64, "%s: gethostbyname", progname);
	perror(errstring);
	exit(1);
  }

  daemon_sin.sin_family = AF_INET;
  daemon_sin.sin_port = htons(portnum);
  /* We want to listen on any interface, so we'll leave this commented */
  /* memcpy(&daemon_sin.sin_addr, server->h_addr, server->h_length);   */

  status = bind (req_sock, &daemon_sin, sizeof (struct sockaddr_in));
  if (status) {
	snprintf(errstring, 64, "%s: bind", progname);
	perror(errstring);
	exit(1);
  }

  status = listen (req_sock, SERVER_BACK_LOG);
  if (status) {
	snprintf(errstring, 64, "%s: listen", progname);
	perror(errstring);
	exit(1);
  }

  status = listen (req_sock, SERVER_BACK_LOG);
  if (status) {
	snprintf(errstring, 64, "%s: listen", progname);
	perror(errstring);
	exit(1);
  }

  snprintf(mesg_buf, 128, "%s: listening on port %d\n", progname, portnum);
  report_log (mesg_buf);
  (void) pthread_mutex_unlock (wrlog_mutex);

  for (next_fd = 0; next_fd < CONCURRENT_CONNECTIONS; next_fd++) {
	/* Mark all fd's as invalid */
	accept_fds[next_fd] = -1;
	/* Set client addr's to all zeros */
	accept_addrs[next_fd] = 0;
  }
  /* It might be a good idea if we start at index 0 ;) */
  next_fd = 0;

  /* Detach process from tty, since we're a daemon */
  ioctl(0, TIOCNOTTY, NULL);

  /* Place ourself in our own process group */
  if (setpgid ((pid_t) 0, (pid_t) 0)) {
    (void) snprintf (errstring, 64, "%s: setpgid", progname);
    perror (errstring);
    exit (1);
  }

  while (1) {
	/* maintain an array of fd's that accept will use one of,
					and start a new thread.     */
	memset (&client_sin, 0, sizeof (struct sockaddr_in));
	sin_addr_length = sizeof (struct sockaddr_in);
	accept_fds[next_fd] = accept (req_sock, &client_sin, &sin_addr_length);
	accept_addrs[next_fd] = client_sin.sin_addr.s_addr;
  #ifdef DEBUG
	snprintf(mesg_buf, 128, "%s: connection from %s\n", progname,
				inet_ntoa (client_sin.sin_addr));
	(void) pthread_mutex_lock (wrlog_mutex);
	report_log (mesg_buf);
	(void) pthread_mutex_unlock (wrlog_mutex);
  #endif /* DEBUG */
	curr_fd = (int *) malloc (sizeof (int));
	*curr_fd = next_fd;
	status = pthread_create (&thread_id, &attrib, proc_req, curr_fd);
	if (status) {
	  (void) pthread_mutex_lock (wrlog_mutex);
	  snprintf(errstring, 64, "%s: pthread_create: %s\n", progname,
                                                      strerror (status));
	         report_log (errstring);
	  (void) pthread_mutex_unlock (wrlog_mutex);
	}
	while (accept_fds[++next_fd] != -1) {
	    if (next_fd >= CONCURRENT_CONNECTIONS)
		next_fd = -1;
	};
  };

return 0;
}


/* Return the proper index into the hash table
   given a symbol name */
inline int hash_index (char *obj_name)
{
  if ((obj_name[0] >= 'a') && (obj_name[0] <= 'z')) {
    return (((int) obj_name[0]) - 65);
  } else if (obj_name[0] == '_') {
    return 26;
  } else {
    return (((int) obj_name[0]) - 70);
  }
}


/* Return zero if symbol name is legal, 
   negative error code if not */
inline int legal_sym (char *obj_name)
{
int count, length;

  length = strlen (obj_name);

  /* First, check to see if this symbol name is valid */
  if ((obj_name[0] < 'A') ||
     ((obj_name[0] > 'Z') && (obj_name[0] < '_')) ||
     ((obj_name[0] > '_') && (obj_name[0] < 'a')) ||
     (obj_name[0] > 'z')) {
    return -EINVAL;
  }
  for (count = 1; count < length-1; count++) {
    if ((obj_name[count] < '0') ||
       ((obj_name[count] > '9') && (obj_name[count] < 'A')) ||
       ((obj_name[count] > 'Z') && (obj_name[count] < '_')) ||
       ((obj_name[count] > '_') && (obj_name[count] < 'a')) ||
       (obj_name[count] > 'z')) {
      return -EINVAL;
    }
  }

return 0;
}

/* Return zero if we don't have symbol,
   positive nonzero if we do, negative
   error code on error */
/* Better than setting errno and returning -1...
   safer for multiple threads to return the error
   code than set a global var... */
int search_ht (char *obj_name, pvm_t obj_type)
{
int retnval = 0, index;
struct ht_ent *generic_pool;
struct ht_subent_mutex *hsm_curr;
struct ht_subent_cv *hsc_curr;
struct ht_subent_sem *hss_curr;

  /* Find the index of the linked list of subentries,
     and set generic_pool to point to it. */
  index = hash_index (obj_name);
  if (obj_type == __PVM_MUTEX) {
    generic_pool = (struct ht_ent *) &mutex_pool[index];
  } else if (obj_type == __PVM_SEM) {
    generic_pool = (struct ht_ent *) &sem_pool[index];
  } else if (obj_type == __PVM_CV) {
    generic_pool = (struct ht_ent *) &cond_pool[index];
 } else if (obj_type == __PVM_SD_INT) {
    generic_pool = (struct ht_ent *) &integer_pool[index];
  } else if (obj_type == __PVM_SD_FLOAT) {
    generic_pool = (struct ht_ent *) &float_pool[index];
  } else if (obj_type == __PVM_SD_MEM) {
    generic_pool = (struct ht_ent *) &memptr_pool[index];
  } else {
    return -EINVAL;
  }

  /* Now, traverse the LL until we find the symbol or
     reach the end of the list. */

  if (obj_type == __PVM_MUTEX) {
    hsm_curr = generic_pool->head;
    while ((hsm_curr) && (!retnval)) {
      /* We're not actually operating on the mutex,
         so we don't need to lock the individual
         mutex for access */
      if (!strncmp (hsm_curr->pvm_mutex.mutex_symbol,
          obj_name,
          strlen (obj_name)))
        retnval++;  /* Houston, we have the mutex */
      else
        hsm_curr = hsm_curr->next;
    };
  } else if (obj_type == __PVM_SEM) {
    hss_curr = generic_pool->head;
    while ((hss_curr) && (!retnval)) {
      /* We're not actually operating on the semaphore,
         so we don't need to lock the individual
         mutex for access */
      if (!strncmp (hss_curr->pvm_sem.sem_symbol,
          obj_name,
          strlen (obj_name)))
        retnval++;  /* Houston, we have the semaphore */
      else
        hss_curr = hss_curr->next;
    };
  } else if (obj_type == __PVM_CV) {
    hsc_curr = generic_pool->head;
    while ((hsc_curr) && (!retnval)) {
      /* We're not actually operating on the condvar,
         so we don't need to lock the individual
         mutex for access */
      if (!strncmp (hsc_curr->pvm_cv.cond_symbol,
          obj_name,
          strlen (obj_name)))
        retnval++;  /* Houston, we have the condvar */
      else
        hsc_curr = hsc_curr->next;
    };
  }

return retnval;
}

/* Insert an entry into the hash table
   Return 0 on success, negative error
   code on failure */
int insert_ht (char * obj_name, pvm_t obj_type, ...)
{
int status, retnval = 0, index;
struct ht_ent *generic_pool;
struct ht_subent_mutex *hsm_curr, *hsm_new;
struct ht_subent_cv *hsc_curr, *hsc_new;
struct ht_subent_sem *hss_curr, *hss_new;

/* stdarg variable argument handling */
va_list args;

  /* Find the index of the linked list of subentries,
     and set generic_pool to point to it. */
  index = hash_index (obj_name);
  if (obj_type == __PVM_MUTEX) {
    generic_pool = (struct ht_ent *) &mutex_pool[index];
  } else if (obj_type == __PVM_SEM) {
    generic_pool = (struct ht_ent *) &sem_pool[index];
  } else if (obj_type == __PVM_CV) {
    generic_pool = (struct ht_ent *) &cond_pool[index];
  } else {
    return -EINVAL;
  }

  /* Now, traverse the LL until we find the
     end of the list, then add an entry. */
  /* This function should be called with the lock on the rwlock held */

  switch (obj_type) {
  case __PVM_MUTEX:
    /* Create new entry and fill it in */
    hsm_new =
      (struct ht_subent_mutex *) malloc (sizeof (struct ht_subent_mutex));
    if (!hsm_new) {
      return -ENOMEM;
    }
    if ((status = pthread_mutex_init (&hsm_new->pvm_mutex.mutex, NULL))) {
      return -status;
    }
    if ((status = pthread_cond_init (&hsm_new->pvm_mutex.mutex_cv, NULL))) {
      (void) pthread_mutex_destroy (&hsm_new->pvm_mutex.mutex);
      return -status;
    }
    (void) strncpy (hsm_new->pvm_mutex.mutex_symbol,
                    obj_name,
                    PVM_MAX_SYMLEN);

    hsm_new->pvm_mutex.valid_mutex = VALID;
    hsm_new->pvm_mutex.mutex_locked = UNLOCKED;
    hsm_new->pvm_mutex.waiters = 0;

    hsm_curr = generic_pool->head;
    if (!hsm_curr) {
      /* Empty list, create the beginning */
      hsm_new->next = NULL;
      hsm_new->prev = NULL;
      generic_pool->head = hsm_new;
    } else {
      while (hsm_curr->next)
        hsm_curr = hsm_curr->next;
      /* We now have the end of the list */
      /* Attach hsm_new to end of list */
      hsm_new->next = NULL;
      hsm_new->prev = hsm_curr;
      hsm_curr->next = hsm_new;
    }

    /* Update access time */
    hsm_new->pvm_mutex.access_time = time (NULL);
  break;

  case __PVM_SEM:
    /* Create new entry and fill it in */
    hss_new =
      (struct ht_subent_sem *) malloc (sizeof (struct ht_subent_sem));
    if (!hss_new) {
      return -ENOSPC;
    }
    va_start (args, obj_type);
    if ((status = sem_init (&hss_new->pvm_sem.semaphore, 0,
                            va_arg (args, unsigned int)))) {
      va_end (args);
      return -errno;
    }
    va_end (args);
    (void) strncpy (hss_new->pvm_sem.sem_symbol,
                    obj_name,
                    PVM_MAX_SYMLEN);

    hss_new->pvm_sem.valid_semaphore = VALID;

    hss_curr = generic_pool->head;
    if (!hss_curr) {
      /* Empty list, create the beginning */
      hss_new->next = NULL;
      hss_new->prev = NULL;
      generic_pool->head = hss_new;
    } else {
      while (hss_curr->next)
        hss_curr = hss_curr->next;
      /* We now have the end of the list */
      /* Attach hsm_new to end of list */
      hss_new->next = NULL;
      hss_new->prev = hss_curr;
      hss_curr->next = hss_new;
    }

    /* Update access time */
    hss_new->pvm_sem.access_time = time (NULL);
  break;

  case __PVM_CV:
    /* Create new entry and fill it in */
    hsc_new =
      (struct ht_subent_cv *) malloc (sizeof (struct ht_subent_cv));
    if (!hsc_new) {
      return -ENOMEM;
    }
    if ((status = pthread_cond_init (&hsc_new->pvm_cv.cond, NULL))) {
      return -status;
    }
    if ((status = pthread_mutex_init (&hsc_new->pvm_cv.mutex, NULL))) {
      (void) pthread_cond_destroy (&hsc_new->pvm_cv.cond);
      return -status;
    }
    (void) strncpy (hsc_new->pvm_cv.cond_symbol,
                    obj_name,
                    PVM_MAX_SYMLEN);

    hsc_new->pvm_cv.valid_cond = VALID;

    hsc_curr = generic_pool->head;
    if (!hsc_curr) {
      /* Empty list, create the beginning */
      hsc_new->next = NULL;
      hsc_new->prev = NULL;
      generic_pool->head = hsc_new;
    } else {
      while (hsc_curr->next)
        hsc_curr = hsc_curr->next;
      /* We now have the end of the list */
      /* Attach hsm_new to end of list */
      hsc_new->next = NULL;
      hsc_new->prev = hsc_curr;
      hsc_curr->next = hsc_new;
    }

    /* Update access time */
    hsc_new->pvm_cv.access_time = time (NULL);
  };

return retnval;
}

/* Find an entry by name in the appropriate hash
   table for obj_type, and return a pointer to
   it's subentry.  NULL on failure. */
void *find_ht (char *obj_name, pvm_t obj_type)
{
int index;
void *retnval = NULL;
struct ht_ent *generic_pool;
struct ht_subent_mutex *hsm_curr;
struct ht_subent_cv *hsc_curr;
struct ht_subent_sem *hss_curr;

  /* Find the index of the linked list of subentries,
     and set generic_pool to point to it. */
  index = hash_index (obj_name);
  if (obj_type == __PVM_MUTEX) {
    generic_pool = (struct ht_ent *) &mutex_pool[index];
  } else if (obj_type == __PVM_SEM) {
    generic_pool = (struct ht_ent *) &sem_pool[index];
  } else if (obj_type == __PVM_CV) {
    generic_pool = (struct ht_ent *) &cond_pool[index];
  } else {
    return NULL;
  }

  /* Now, traverse the LL until we find the symbol or
     reach the end of the list. */

  if (obj_type == __PVM_MUTEX) {
    hsm_curr = generic_pool->head;
    while ((hsm_curr) && (!retnval)) {
      /* We're not actually operating on the mutex,
         so we don't need to lock the individual
         mutex for access */
      if (!strncmp (hsm_curr->pvm_mutex.mutex_symbol,
          obj_name, strlen (obj_name)))
        retnval = (void *) hsm_curr;  /* Houston, we have the mutex */
      else
        hsm_curr = hsm_curr->next;
    };
  } else if (obj_type == __PVM_SEM) {
    hss_curr = generic_pool->head;
    while ((hss_curr) && (!retnval)) {
      /* We're not actually operating on the semaphore,
         so we don't need to lock the individual
         mutex for access */
      if (!strncmp (hss_curr->pvm_sem.sem_symbol,
          obj_name, strlen (obj_name)))
        retnval = (void *) hss_curr;  /* Houston, we have the semaphore */
      else
        hss_curr = hss_curr->next;
    };
  } else if (obj_type == __PVM_CV) {
    hsc_curr = generic_pool->head;
    while ((hsc_curr) && (!retnval)) {
      /* We're not actually operating on the condvar,
         so we don't need to lock the individual
         mutex for access */
      if (!strncmp (hsc_curr->pvm_cv.cond_symbol,
          obj_name, strlen (obj_name)))
        retnval = (void *) hsc_curr;  /* Houston, we have the condvar */
      else
        hsc_curr = hsc_curr->next;
    };
  }

return retnval;
}

/* Remove an entry by name from the appropriate hash
   table for obj_type. */
int remove_ht (char *obj_name, pvm_t obj_type)
{
int index;
int retnval = -1;
struct ht_ent *generic_pool;
struct ht_subent_mutex *hsm_prev, *hsm_curr;
struct ht_subent_cv *hsc_prev, *hsc_curr;
struct ht_subent_sem *hss_prev, *hss_curr;

  /* Find the index of the linked list of subentries,
     and set generic_pool to point to it. */
  index = hash_index (obj_name);
  if (obj_type == __PVM_MUTEX) {
    generic_pool = (struct ht_ent *) &mutex_pool[index];
  } else if (obj_type == __PVM_SEM) {
    generic_pool = (struct ht_ent *) &sem_pool[index];
  } else if (obj_type == __PVM_CV) {
    generic_pool = (struct ht_ent *) &cond_pool[index];
  } else {
    return -EINVAL;
  }

  /* Now, traverse the LL until we find the symbol or
     reach the end of the list. */

  switch (obj_type) {
  case __PVM_MUTEX:
    hsm_prev = hsm_curr = generic_pool->head;
    while ((hsm_curr) && (retnval)) {
      /* We're not actually operating on the mutex,
         so we don't need to lock the individual
         mutex for access */
      if (!strncmp (hsm_curr->pvm_mutex.mutex_symbol,
          obj_name, strlen (obj_name))) {
        if (hsm_prev == hsm_curr) {
          /* Beginning of list */
          generic_pool->head = hsm_curr->next;
          if (hsm_curr->next) {
            /* More than one element in list */
            hsm_curr->next->prev = NULL;
          }
        } else {
          /* Not the beginning of list */
          hsm_prev->next = hsm_curr->next;
          if (hsm_curr->next) {
            /* We're not at the end */
            hsm_curr->next->prev = hsm_prev;
          }
        }
        free (hsm_curr);
        retnval++;
      } else {
        hsm_prev = hsm_curr;
        hsm_curr = hsm_curr->next;
      }
    };
  break;

  case __PVM_SEM:
    hss_prev = hss_curr = generic_pool->head;
    while ((hss_curr) && (retnval)) {
      /* We're not actually operating on the semaphore,
         so we don't need to lock the individual
         mutex for access */
      if (!strncmp (hss_curr->pvm_sem.sem_symbol,
          obj_name, strlen (obj_name))) {
        if (hss_prev == hss_curr) {
          /* Beginning of list */
          generic_pool->head = hss_curr->next;
          if (hss_curr->next) {
            /* More than one element in list */
            hss_curr->next->prev = NULL;
          }
        } else {
          /* Not the beginning of list */
          hss_prev->next = hss_curr->next;
          if (hss_curr->next) {
            /* We're not at the end */
            hss_curr->next->prev = hss_prev;
          }
        }
        free (hss_curr);
        retnval++;
      } else {
        hss_prev = hss_curr;
        hss_curr = hss_curr->next;
      }
    };
  break;

  case __PVM_CV:
    hsc_prev = hsc_curr = generic_pool->head;
    while ((hsc_curr) && (retnval)) {
      /* We're not actually operating on the condition
         variable, so we don't need to lock the individual
         mutex for access */
      if (!strncmp (hsc_curr->pvm_cv.cond_symbol,
          obj_name, strlen (obj_name))) {
        if (hsc_prev == hsc_curr) {
          /* Beginning of list */
          generic_pool->head = hsc_curr->next;
          if (hsc_curr->next) {
            /* More than one element in list */
            hsc_curr->next->prev = NULL;
          }
        } else {
          /* Not the beginning of list */
          hsc_prev->next = hsc_curr->next;
          if (hsc_curr->next) {
            /* We're not at the end */
            hsc_curr->next->prev = hsc_prev;
          }
        }
        free (hsc_curr);
        retnval++;
      } else {
        hsc_prev = hsc_curr;
        hsc_curr = hsc_curr->next;
      }
    };
  };

return retnval;
}

/* Unlock rwlock or die */
inline void unlock_r_die (pthread_rwlock_t *rwlock)
{
  if (pthread_rwlock_unlock (rwlock)) {
    (void) pthread_mutex_lock (wrlog_mutex);
           report_log ("pvmsyncd: couldn't unlock internal rwlock\n");
    (void) pthread_mutex_unlock (wrlog_mutex);
    (void) raise (SIGINT);
    (void) pthread_exit (NULL);
  }
}

/* Log message to runfile and die */
/* log_mesg should end with '\n'. */
void die_with_mesg (char *log_mesg)
{
  (void) pthread_mutex_lock (wrlog_mutex);
         report_log (log_mesg);
  (void) pthread_mutex_unlock (wrlog_mutex);

  /* Commit suicide */
  (void) raise (SIGINT);
  (void) pthread_exit (NULL);
}

/* Send reply to client using tcp...  Reply is of the
   form:  SUCCESS:nilval or FAIL:errcode, where errcode
   is a nonnegative integer. */
inline void tcp_send_n_exit (int success, int code, int *index)
{

  if (success) {
    if (code) {
      /* Don't ignore code */
      (void) snprintf (temp_buf[*index], BUF_LENGTH, "SUCCESS:%d", code);
    } else {
      /* Ignore code */
      (void) snprintf (temp_buf[*index], BUF_LENGTH, "SUCCESS:%s", nilval);
    }
  } else {
    (void) snprintf (temp_buf[*index], BUF_LENGTH, "FAIL:%d", code);
  }

  (void) write (accept_fds[*index],
                temp_buf[*index],
                strlen (temp_buf[*index]));

  /* Close the socket and invalidate the file descriptor */
  close (accept_fds[*index]);
  accept_fds[*index] = -1;
  free (index); /* AKA location in proc_req */
  /* Die */
  (void) pthread_exit (NULL);

}


