/*
 *
 * $Id: proc_req.c,v 1.2.4.41 2000/03/16 17:12:59 ajp Exp $
 *
 * Andrew Pitman
 *
 * pvmsync, a distributed synchronization server:  process
 * incoming requests.
 *
 * Server that accepts connections (requests) for shared
 * data or synchronization mechanisms continuously and
 * changes some internal state according to the request.
 *
 * Copyright (C) 1999, 2000 Andrew J. Pitman
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


#ifndef __LOGGING_H
   /* report_log() */
#include <logging.h>
#endif

/* For RW Locks */
#ifndef __USE_UNIX98
  #define __USE_UNIX98
#endif
#include <pthread.h>
#include <semaphore.h>

#ifndef __TCP_DAEMON_H
   /* mutex_pool_t */
#include <tcp_daemon.h>
#endif

/* Semaphore operations return -1 and
   set errno appropriately */
#include <errno.h>

/* raise() */
#include <signal.h>

/* strlen */
#include <string.h>

/* __symbol_exists AKA symbol_exists */
#include <lib/lib_internals.h>
#include <tcp_exists.h>

#ifndef __CONFIG_H
   /* CONCURRENT_CONNECTIONS, etc... */
#include <config.h>
#endif

#ifndef __PROC_REQ_H
#include <proc_req.h>
#endif

/* Object types according to udp server */
#include <include/pvm_gen.h>


extern int accept_fds[CONCURRENT_CONNECTIONS]; /* An array of fd's to keep
                                           track of */
extern unsigned int accept_addrs[CONCURRENT_CONNECTIONS]; /* Client addresses */
extern char temp_buf[CONCURRENT_CONNECTIONS][BUF_LENGTH]; /* Temp buffer for
                                           reads from client.  Also reused
                                           to write data back to client.   */

/* Attributes for inititalization of each */
/* entry in the hash table's associated */
/* read-write lock */
extern pthread_rwlockattr_t *mutex_init_lock_attr;

/* Internal "pool" of mutexes (mutexi?) */
/* Hash table indexed according to first character */
extern struct ht_ent mutex_pool[NUM_HASH_ENTS];

/* Keep a counter of the number of objects we have of this type */
/* This is protected by the rwlock and is only updated on mutex */
/* creation and destruction. */
extern unsigned int num_mutexes;

/* Protect access to variable num_mutexes */
extern pthread_rwlock_t mutex_counter_lock;

/* Attributes for inititalization of each */
/* entry in the hash table's associated */
/* read-write lock */
extern pthread_rwlockattr_t *sem_init_lock_attr;

/* Internal "pool" of semaphores */
extern struct ht_ent sem_pool[NUM_HASH_ENTS];

/* Keep a counter of the number of objects we have of this type */
/* This is protected by the rwlock and is only updated on sem   */
/* creation and destruction. */
extern unsigned int num_sems;

/* Protect access to variable num_sems */
extern pthread_rwlock_t sem_counter_lock;

/* Protect access to the integer pool with a rwlock */
extern pthread_rwlock_t integer_init_lock;

/* Internal "pool" of pvm_int64_t's */
extern integer_pool_t integer_pool[CONCURRENT_INSTANCES];

/* Keep a counter of the number of objects we have of this type */
extern unsigned int num_integers;

/* Protect access to the float pool with a rwlock */
extern pthread_rwlock_t float_init_lock;

/* Internal "pool" of pvm_float64_t's. */
extern float_pool_t float_pool[CONCURRENT_INSTANCES];

/* Keep a counter of the number of objects we have of this type */
extern unsigned int num_floating;

/* Protect access to the memptr pool with a rwlock */
extern pthread_rwlock_t memptr_init_lock;
extern pthread_rwlockattr_t *memptr_init_lock_attr;

/* Internal "pool" of pointers to heap-dynamic data. */
extern memptr_pool_t memptr_pool[CONCURRENT_INSTANCES];

/* Keep a counter of the number of objects we have of this type */
extern unsigned int num_memblocks;

/* Attributes for inititalization of each */
/* entry in the hash table's associated */
/* read-write lock */
extern pthread_rwlockattr_t *condvar_init_lock_attr;

/* Internal "pool" of condition variables */
/* Hash table indexed according to first character */
extern struct ht_ent cond_pool[NUM_HASH_ENTS];

/* Keep a counter of the number of objects we have of this type */
/* This is protected by the rwlock and is only updated on condvar */
/* creation and destruction. */
extern unsigned int num_conds;

/* Protect access to variable num_conds */
pthread_rwlock_t condvar_counter_lock;


/* Protect our random seed */
pthread_mutex_t rand_seed_mutex = PTHREAD_MUTEX_INITIALIZER;

/* Maintain the value of our random seed across calls */
int random_seed;

/* fd for our log file */
extern int log_fd;

/* Protect access to write on log_fd */
extern pthread_mutex_t *wrlog_mutex;

/* When we don't have a numeric or string value
   to return, return this.                      */
const char *nilval = "nil";


/* Process requests to create, destroy and operate on
   mutexes, semaphores, and shared data objects.  */
void *proc_req(void *index)
{
/* Stack-dynamic variables should be safe */
/* It is OK if there are multiple instances
   of this function running (in different threads). */
int *location; /* Index */
int numbytes;
int status = 0, sizeval; /* Size of mem block */
unsigned int semval; /* Value to initialize semaphore to */
time_t secs; /* Length of time for condvar wait timeout */

pvm_int64_t integer_value;
pvm_float64_t float_value;
void *memptr = NULL;

/* Requests are of form:
   REQ:symbol[:arg]  */
char *symbol,
     *cl_arg1,
     *cl_arg2,
     *cl_temp;
int cl_val;

  /* We're using the reentrant
     version of strtok,  strtok_r */
  char **lasts = (char **) malloc (2048);

  /* The buffer we are assigned is in temp_buf[],
     get the index into it.... */
  location = (int *) index;
  memset(temp_buf[*location], 0, BUF_LENGTH);

  /* For strtok_r */
  if (!lasts) {
    /* Close the socket and invalidate the file descriptor */
    close (accept_fds[*location]);
    accept_fds[*location] = -1;
    free (location);
    (void) pthread_exit (NULL);
  }

  numbytes = read(accept_fds[*location], temp_buf[*location], BUF_LENGTH);
  if (numbytes < 0) {
    (void) pthread_mutex_lock (wrlog_mutex);
           report_log ("pvmsyncd: error reading from client\n");
    (void) pthread_mutex_unlock (wrlog_mutex);

    /* Close the socket and invalidate the file descriptor */
    close (accept_fds[*location]);
    accept_fds[*location] = -1;
    free (location);
    free (lasts);
  (void) pthread_exit (NULL);
  }

  /* Get the symbol (and if it's there, the argument to go with it) */
  symbol = strtok_r (temp_buf[*location]+4, ":", lasts);
  /* Catch NULL returns from strtok, so we don't segfault */
  if (symbol == NULL) {
    symbol = "";
  }
  cl_arg1 = strtok_r (NULL, ":", lasts);
  /* Catch NULL returns from strtok, so we don't segfault */
  if (cl_arg1 == NULL) {
    cl_arg1 = "";
  }
  cl_arg2 = strtok_r (NULL, ":", lasts);


  /* Create Mutex */
  if (!(strncasecmp (temp_buf[*location], "CMT:", 4))) {

    status = mutex_create (symbol, numbytes-3);

    free (lasts);
    if (status) {
      tcp_send_n_exit (FALSE, -status, location);
    } else {
      tcp_send_n_exit (TRUE, 0, location);
    }


  /* Destroy Mutex */
  } else if (!(strncasecmp (temp_buf[*location], "DMT:", 4))) {

    status = mutex_dest (symbol, numbytes-3);

    free (lasts);
    if (status) {
      tcp_send_n_exit (FALSE, -status, location);
    } else {
      tcp_send_n_exit (TRUE, 0, location);
    }


  /* Lock Mutex */
  } else if (!(strncasecmp (temp_buf[*location], "LCK:", 4))) {

    status = mutex_lock (symbol, numbytes-3, accept_addrs[*location]);

    free (lasts);
    if (status < 0) {
      tcp_send_n_exit (FALSE, -status, location);
    } else {
      tcp_send_n_exit (TRUE, status, location);
    }


  /* Trylock Mutex */
  } else if (!(strncasecmp (temp_buf[*location], "TRL:", 4))) {

    status = mutex_trylock (symbol, numbytes-3, accept_addrs[*location]);

    free (lasts);
    if (status < 0) {
      tcp_send_n_exit (FALSE, -status, location);
    } else {
      tcp_send_n_exit (TRUE, status, location);
    }


  /* Unlock Mutex */
  } else if (!(strncasecmp (temp_buf[*location], "UNL:", 4))) {

    if (!strlen (cl_arg1)) {
      free (lasts);
      tcp_send_n_exit (FALSE, EINVAL, location);
    } else {
      cl_val = atoi (cl_arg1);

      status =
	mutex_unlock (symbol, strlen (symbol), accept_addrs[*location], cl_val);

      free (lasts);
      if (status) {
        tcp_send_n_exit (FALSE, -status, location);
      } else {
        tcp_send_n_exit (TRUE, 0, location);
      }
    }


  /* Create Semaphore */
  } else if (!(strncasecmp (temp_buf[*location], "CSM:", 4))) {

    if (!strlen (cl_arg1)) {
      free (lasts);
      tcp_send_n_exit (FALSE, EINVAL, location);
    } else {
      semval = (unsigned int) atoi (cl_arg1);
      status = sem_create (symbol, strlen (symbol), semval);

      free (lasts);
      if (status) {
        tcp_send_n_exit (FALSE, -status, location);
      } else {
        tcp_send_n_exit (TRUE, 0, location);
      }
    }


  /* Destroy Semaphore */
  } else if (!(strncasecmp (temp_buf[*location], "DSM:", 4))) {

    status = sem_dest (symbol, numbytes-3);

    free (lasts);
    if (status) {
      tcp_send_n_exit (FALSE, -status, location);
    } else {
      tcp_send_n_exit (TRUE, 0, location);
    }


  /* Wait on semaphore */
  } else if (!(strncasecmp (temp_buf[*location], "PSM:", 4))) {

    /* Blocking wait */
    status = sem_p (symbol, numbytes-3, FALSE);

    free (lasts);
    if (status) {
      tcp_send_n_exit (FALSE, -status, location);
    } else {
      tcp_send_n_exit (TRUE, 0, location);
    }


  /* Trywait on semaphore */
  } else if (!(strncasecmp (temp_buf[*location], "TPS:", 4))) {

    /* Nonblocking wait on semaphore */
    status = sem_p (symbol, numbytes-3, TRUE);

    free (lasts);
    if (status) {
      tcp_send_n_exit (FALSE, -status, location);
    } else {
      tcp_send_n_exit (TRUE, 0, location);
    }


  /* Post semaphore */
  } else if (!(strncasecmp (temp_buf[*location], "VSM:", 4))) {

    status = sem_v (symbol, numbytes-3);

    free (lasts);
    if (status) {
      tcp_send_n_exit (FALSE, -status, location);
    } else {
      tcp_send_n_exit (TRUE, 0, location);
    }


  /* Create shared data */
  } else if (!(strncasecmp (temp_buf[*location], "CSD:", 4))) {

    /* With global data, the symbol is second */
    /* Switch, so that symbol is indeed the name and
       cl_arg1 is the type */
    cl_temp = cl_arg1;
    cl_arg1 = symbol;
    symbol = cl_temp;

    /* In case we don't match at all */
    status = -1;
    errno = EINVAL; /* Invalid Argument */

    if (!(strncasecmp (cl_arg1, "INTEGER", 7))) {
      /* Integer variable */

      status = integer_create (symbol, strlen (symbol));

    } else if (!(strncasecmp (cl_arg1, "FLOAT", 5))) {
      /* Floating pt. variable */

      status = float_create (symbol, strlen (symbol));

    } else if (!(strncasecmp (cl_arg1, "MEM", 3))) {
      /* Block of `shared memory' */

      if (cl_arg2 == NULL)
        sizeval = 0;
      else
        sizeval = atoi (cl_arg2);

      status = mem_create (symbol, strlen (symbol), sizeval);
    }

    free (lasts);
    if (status == -1) {
      tcp_send_n_exit (FALSE, errno, location);
    } else {
      tcp_send_n_exit (TRUE, 0, location);
    }


  /* Set shared data */
  } else if (!(strncasecmp (temp_buf[*location], "SSD:", 4))) {

    /* With global data, the symbol is second */
    /* Switch, so that symbol is indeed the name and
       cl_arg1 is the type */
    cl_temp = cl_arg1;
    cl_arg1 = symbol;
    symbol = cl_temp;

    if (!(strncasecmp (cl_arg1, "INTEGER", 7))) {
      /* Integer variable */

      /* Catch NULL returns from strtok, so we don't segfault */
      if (cl_arg2 == NULL) {
        free (lasts);
        tcp_send_n_exit (FALSE, EINVAL, location);
      } else {
        (void) sscanf (cl_arg2, "%Ld", &integer_value);
        status = integer_set (symbol, strlen (symbol), integer_value);
        free (lasts);
        if (status) {
          tcp_send_n_exit (FALSE, errno, location);
        } else {
          tcp_send_n_exit (TRUE, 0, location);
        }
      }

    } else if (!(strncasecmp (cl_arg1, "FLOAT", 5))) {
      /* Floating pt. variable */

      /* Catch NULL returns from strtok, so we don't segfault */
      if (cl_arg2 == NULL) {
        free (lasts);
        tcp_send_n_exit (FALSE, EINVAL, location);
      } else {
        status = float_set (symbol, strlen (symbol), atof (cl_arg2));
        free (lasts);
        if (status) {
          tcp_send_n_exit (FALSE, errno, location);
        } else {
          tcp_send_n_exit (TRUE, 0, location);
        }
      }

    } else if (!(strncasecmp (cl_arg1, "MEM", 3))) {
      /* Block of `shared memory' */

      /* Catch NULL returns from strtok, so we don't segfault */
      if (cl_arg2 == NULL)
        cl_arg2 = "";

      sizeval = atoi (cl_arg2);
      if (sizeval > 0)
        memptr = (void *) malloc (sizeval);
      if (memptr == NULL) {
        free (lasts);
        /* We'll read their data, but we can't do anything with it. */
        (void) read (accept_fds[*location], temp_buf[*location], 32);
        /* Report that we can't allocate the memory */
        tcp_send_n_exit (FALSE, ENOMEM, location);
      } else {
        numbytes = read (accept_fds[*location], memptr, sizeval);
        free (lasts);
        if (numbytes <= 0) {
          tcp_send_n_exit (FALSE, EIO, location);
        } else {

          status = mem_set (symbol, strlen(symbol), memptr, sizeval);
          /* Done with memptr */
          if (memptr != NULL)
            free (memptr);

          free (lasts);
          if (status == -1) {
            tcp_send_n_exit (FALSE, errno, location);
          } else {
            tcp_send_n_exit (TRUE, 0, location);
          }
        }
      }
    }


  } else if (!(strncasecmp (temp_buf[*location], "GSD:", 4))) {


    /* With global data, the symbol is second */
    /* Switch, so that symbol is indeed the name and
       cl_arg1 is the type */
    cl_temp = cl_arg1;
    cl_arg1 = symbol;
    symbol = cl_temp;

    if (!(strncasecmp (cl_arg1, "INTEGER", 7))) {
      /* Integer variable */

      integer_value = integer_get (symbol, strlen (symbol));
      free (lasts);
      if ((integer_value == -1) && (errno)) {
        tcp_send_n_exit (FALSE, errno, location);
      } else {
        tcp_send_n_exit (TRUE, 0, location);
      }

    } else if (!(strncasecmp (cl_arg1, "FLOAT", 5))) {
      /* Float variable */

      float_value = float_get (symbol, strlen (symbol));
      free (lasts);
      if ((float_value == (float) -1) && (errno)) {
        tcp_send_n_exit (FALSE, errno, location);
      } else {
        tcp_send_n_exit (TRUE, 0, location);
      }

    } else if (!(strncasecmp (cl_arg1, "MEM", 3))) {
      /* Memory block */

      memptr = mem_get (symbol, strlen (symbol), &sizeval);
      if (memptr == NULL) {
        free (lasts);
        tcp_send_n_exit (FALSE, errno, location);
      } else {

        (void) snprintf (temp_buf[*location], BUF_LENGTH,
              "SUCCESS:%d", sizeval);
        (void) write (accept_fds[*location], temp_buf[*location],
              strlen (temp_buf[*location]));
        (void) write (accept_fds[*location], memptr, sizeval);

        /* Close the socket and invalidate the file descriptor */
        close (accept_fds[*location]);
        accept_fds[*location] = -1;
        free (location);
        free (lasts);

      /* Die */
      (void) pthread_exit (NULL);
      }
    }


  /* Destroy shared data */
  } else if (!(strncasecmp (temp_buf[*location], "DSD:", 4))) {

    /* With global data, the symbol is second */
    /* Switch, so that symbol is indeed the name and
       cl_arg1 is the type */
    cl_temp = cl_arg1;
    cl_arg1 = symbol;
    symbol = cl_temp;

    if (!(strncasecmp (cl_arg1, "INTEGER", 7))) {
      /* Destroy... destroy... must destroy
         integer variable.... Will Robinson. */

      status = integer_dest (symbol, strlen (symbol));
      free (lasts);
      if (status) {
        tcp_send_n_exit (FALSE, errno, location);
      } else {
        tcp_send_n_exit (TRUE, 0, location);
      }

    } else if (!(strncasecmp (cl_arg1, "FLOAT", 5))) {
      /* Destroy float variable */

      status = float_dest (symbol, strlen (symbol));
      free (lasts);
      if (status) {
        tcp_send_n_exit (FALSE, errno, location);
      } else {
        tcp_send_n_exit (TRUE, 0, location);
      }

    } else if (!(strncasecmp (cl_arg1, "MEM", 3))) {
      /* Destroy memory block */

      status = mem_dest (symbol, strlen (symbol));
      free (lasts);
      if (status) {
        tcp_send_n_exit (FALSE, errno, location);
      } else {
        tcp_send_n_exit (TRUE, 0, location);
      }
    }


  /* Create condition variable */
  } else if (!(strncasecmp (temp_buf[*location], "CCV:", 4))) {

    status = cond_create (symbol, numbytes-3);

    free (lasts);
    if (status) {
      tcp_send_n_exit (FALSE, -status, location);
    } else {
      tcp_send_n_exit (TRUE, 0, location);
    }


  /* Destroy condition variable */
  } else if (!(strncasecmp (temp_buf[*location], "DCV:", 4))) {

    status = cond_dest (symbol, numbytes-3);

    free (lasts);
    if (status) {
      tcp_send_n_exit (FALSE, -status, location);
    } else {
      tcp_send_n_exit (TRUE, 0, location);
    }


  /* Broadcast condition varaible */
  } else if (!(strncasecmp (temp_buf[*location], "BCV:", 4))) {

    status = cv_sig (symbol, numbytes-3, TRUE);

    free (lasts);
    if (status) {
      tcp_send_n_exit (FALSE, -status, location);
    } else {
      tcp_send_n_exit (TRUE, 0, location);
    }


  /* Signal condition varaible */
  } else if (!(strncasecmp (temp_buf[*location], "SCV:", 4))) {

    status = cv_sig (symbol, numbytes-3, FALSE);

    free (lasts);
    if (status) {
      tcp_send_n_exit (FALSE, -status, location);
    } else {
      tcp_send_n_exit (TRUE, 0, location);
    }


  /* Wait on condition varaible */
  } else if (!(strncasecmp (temp_buf[*location], "WCV:", 4))) {

    status = cond_timedwait (symbol, numbytes-3, (time_t) 0);

    free (lasts);
    if (status) {
      tcp_send_n_exit (FALSE, -status, location);
    } else {
      tcp_send_n_exit (TRUE, 0, location);
    }


  /* Wait on condition variable */
  } else if (!(strncasecmp (temp_buf[*location], "TWC:", 4))) {

    if (!strlen (cl_arg1)) {
      secs = (time_t) 0;
    } else {
      secs = time (NULL);
      secs += (time_t) atoi (cl_arg1);
    }

    status = cond_timedwait (symbol, strlen (symbol), (time_t) secs);

    free (lasts);
    if (status) {
      tcp_send_n_exit (FALSE, -status, location);
    } else {
      tcp_send_n_exit (TRUE, 0, location);
    }


  /* IO error */
  } else {
    free (lasts);
    tcp_send_n_exit (FALSE, EIO, location);
  }

/*NOTREACHED*/
return NULL;
}


/* Create a mutex */
/* EAGAIN, ENOMEM, EPERM, EBUSY, EINVAL (nonstandard) */
int mutex_create (char *mutex_name, int length)
{
int status, index;
int retnval = 0;

#ifdef DEBUG
char debug_mesg[128];
#endif

  /* First, check to see if this symbol name is valid, if we
     do indeed have a string */
  if ((!length) || (legal_sym (mutex_name) == -EINVAL))
    return -EINVAL;

  /* Get the index into the hash table */
  index = hash_index (mutex_name);

  /* Lock rwlock for writing */
  /* We don't need to lock the individual mutexes here,
     because write locks are exclusive. */
  if (pthread_rwlock_wrlock (&mutex_pool[index].ent_rwlock)) {
    return -EAGAIN;
  }

  /* Find an existing mutex with the same name, if it's there. */
  /* Iterate through the sub-entries (linked list) and check to
     see if we would be creating a duplicate. */
  if ((status = search_ht (mutex_name, __PVM_MUTEX)) < 0) {
    retnval = -EAGAIN;
  } else if (status > 0) {
    /* We've found a duplicate */
    retnval = -EBUSY;
  }
  if (retnval) {
    unlock_r_die (&mutex_pool[index].ent_rwlock);
    return retnval;
  }
  /* We don't have a dup locally, continue.... */

  /* At this point, the local daemon doesn't have a dup, check
     with others on the cluster */
  /* We'll check to see if anyone else has this symbol
     before we continue. */
  if (symbol_exists (__PVM_MUTEX, mutex_name)) {
    unlock_r_die (&mutex_pool[index].ent_rwlock);
    return -EBUSY;
  }

  /* Place mutex into hash table */
  if (insert_ht (mutex_name, __PVM_MUTEX))
    retnval = -EAGAIN;

  if (pthread_rwlock_wrlock (&mutex_counter_lock))
    die_with_mesg ("pvmsyncd: unable to lock rwlock\n");
  num_mutexes++;
  unlock_r_die (&mutex_counter_lock);

  unlock_r_die (&mutex_pool[index].ent_rwlock);

#ifdef DEBUG
  if (!retnval) {
    (void) pthread_mutex_lock (wrlog_mutex);
    (void) snprintf (debug_mesg, 128, "pvmsyncd: object %s created\n",
                                                        mutex_name);
           report_log (debug_mesg);
    (void) pthread_mutex_unlock (wrlog_mutex);
  }
#endif

return retnval;
}

/* Destroy mutex */
/* EAGAIN, EBUSY, EINVAL */
int mutex_dest (char *mutex_name, int length)
{
int status, index;
struct ht_subent_mutex *rem;

  /* No funny business, make sure the symbol name is valid
     so we won't get an out-of-bounds error for the array */
  if ((!length) || (legal_sym (mutex_name)))
    return -EINVAL;

  /* Get the index into the hash table */
  index = hash_index (mutex_name);

  /* Lock rwlock for writing */
  /* We don't need to lock the individual mutexes here,
     because write locks are exclusive. */
  if (pthread_rwlock_wrlock (&mutex_pool[index].ent_rwlock)) {
    return -EAGAIN;
  }

  /* First find the mutex */
  if (!(status = search_ht (mutex_name, __PVM_MUTEX))) {
    unlock_r_die (&mutex_pool[index].ent_rwlock);
    return -EINVAL;
  } else if (status < 0) {
    unlock_r_die (&mutex_pool[index].ent_rwlock);
    return -EAGAIN;
  }

  /* Then, if it's not in use (locked) remove it from ht */
  /* First, make sure mutex is not currently locked */
  /* Here, get a ptr to the ht_subent_mutex, check if it's
     locked.  If it is, unlock rwlock and return error
     (EBUSY). */
  rem = (struct ht_subent_mutex *) find_ht (mutex_name, __PVM_MUTEX);
  if (!rem) {
    unlock_r_die (&mutex_pool[index].ent_rwlock);
    return -EAGAIN;
  }

  if (rem->pvm_mutex.mutex_locked) {
    unlock_r_die (&mutex_pool[index].ent_rwlock);
    return -EBUSY;
  }

  /* Mutex is not locked, remove it */
  if (remove_ht (mutex_name, __PVM_MUTEX)) {
    unlock_r_die (&mutex_pool[index].ent_rwlock);
    return -EAGAIN;
  }
  /* At this point, mutex is removed from hash table */
  /* Decrement counter */
  if (pthread_rwlock_wrlock (&mutex_counter_lock))
    die_with_mesg ("pvmsyncd: unable to lock rwlock\n");
  num_mutexes--;
  unlock_r_die (&mutex_counter_lock);

  /* Unlock corresponding rwlock for ht index */
  unlock_r_die (&mutex_pool[index].ent_rwlock);

return 0;
}

/* Lock mutex */
/* EAGAIN, EINVAL, EDEADLK */
int mutex_lock (char *mutex_name, int length, unsigned int addr)
{
int index;
struct ht_subent_mutex *curr = NULL;

  /* No funny business, make sure the symbol name is valid
     so we won't get an out-of-bounds error for the array */
  if ((!length) || (legal_sym (mutex_name)))
    return -EINVAL;

  /* Get the index into the hash table */
  index = hash_index (mutex_name);

  /* Lock for reading */
  if (pthread_rwlock_rdlock (&mutex_pool[index].ent_rwlock)) {
    return -EAGAIN;
  }

  /* Find the mutex in the hash table */
  curr = (struct ht_subent_mutex *) find_ht (mutex_name, __PVM_MUTEX);
  if (!curr) {
    /* mutex_name does not exist, bad request */
    unlock_r_die (&mutex_pool[index].ent_rwlock);
    return -EINVAL;
  }

  /* Now, lock the internal mutex for this entry */
  /* Shouldn't fail ;) */
  if (pthread_mutex_lock (&curr->pvm_mutex.mutex))
    die_with_mesg ("pvmsyncd: couldn't lock internal mutex\n");

  /* Make sure mutex isn't already locked */
  if (curr->pvm_mutex.mutex_locked) {

    /* If mutex is already locked, check to make sure we
       don't hold the lock. */
    if (addr == curr->pvm_mutex.mutex_lockaddr) {

      /* Return -EDEADLK if so... the dumbass tried to lock it twice. */
      if (pthread_mutex_unlock (&curr->pvm_mutex.mutex))
        die_with_mesg ("pvmsyncd: couldn't unlock internal mutex\n");

      /* Release readlock */
      unlock_r_die (&mutex_pool[index].ent_rwlock);
      return -EDEADLK;

    } else {

      /* Otherwise, we'll wait on mutex_cv */
      curr->pvm_mutex.waiters++;
      if (pthread_cond_wait (&curr->pvm_mutex.mutex_cv,
                             &curr->pvm_mutex.mutex)) {
        /* We wan't to log an error and exit */
        die_with_mesg ("pvmsyncd: error in condvar wait\n");
      }

      (void) pthread_mutex_lock (&rand_seed_mutex);
             /* All random numbers returned will be positive, to
                differentiate errors. */
      do {
             curr->pvm_mutex.mutex_locked = abs (rand_r (&random_seed));
      } while (!curr->pvm_mutex.mutex_locked);
      (void) pthread_mutex_unlock (&rand_seed_mutex);

      curr->pvm_mutex.mutex_lockaddr = addr;
      curr->pvm_mutex.access_time = time (NULL);
      if (pthread_mutex_unlock (&curr->pvm_mutex.mutex)) {
        die_with_mesg ("pvmsyncd: couldn't unlock internal mutex\n");
      }
    }

  } else {

    /* Not already locked */
    /* Just lock it. */
    (void) pthread_mutex_lock (&rand_seed_mutex);
           /* All random numbers returned will be positive, to
              differentiate errors. */
           curr->pvm_mutex.mutex_locked = abs (rand_r (&random_seed));
    (void) pthread_mutex_unlock (&rand_seed_mutex);

    curr->pvm_mutex.mutex_lockaddr = addr;
    curr->pvm_mutex.access_time = time (NULL);
    if (pthread_mutex_unlock (&curr->pvm_mutex.mutex)) {
      die_with_mesg ("pvmsyncd: couldn't unlock internal mutex\n");
    }
  }

  /* Release readlock before returning */
  unlock_r_die (&mutex_pool[index].ent_rwlock);

/* Return our randomly generated integer */
return curr->pvm_mutex.mutex_locked;
}

/* Trylock mutex */
/* EAGAIN, EINVAL, EBUSY, EDEADLK */
int mutex_trylock (char *mutex_name, int length, unsigned int addr)
{
int index;
struct ht_subent_mutex *curr = NULL;

  /* No funny business, make sure the symbol name is valid
     so we won't get an out-of-bounds error for the array */
  if ((!length) || (legal_sym (mutex_name)))
    return -EINVAL;

  /* Get the index into the hash table */
  /* Index will be OK, since we've already checked
     the validity of the symbol name */
  index = hash_index (mutex_name);

  /* Lock for reading */
  if (pthread_rwlock_rdlock (&mutex_pool[index].ent_rwlock)) {
    return -EAGAIN;
  }

  /* Find the mutex in the hash table */
  curr = (struct ht_subent_mutex *) find_ht (mutex_name, __PVM_MUTEX);
  if (!curr) {
    /* mutex_name does not exist, bad request */
    unlock_r_die (&mutex_pool[index].ent_rwlock);
    return -EINVAL;
  }

  /* Now, lock the internal mutex for this entry */
  /* Shouldn't fail ;) */
  if (pthread_mutex_lock (&curr->pvm_mutex.mutex))
    die_with_mesg ("pvmsyncd: couldn't lock internal mutex\n");

  /* Make sure mutex isn't already locked */
  if (curr->pvm_mutex.mutex_locked) {

    /* If mutex is already locked, check to make sure we
       don't hold the lock. */
    if (addr == curr->pvm_mutex.mutex_lockaddr) {

      /* Return -EDEADLK if so... the dumbass tried to lock it twice. */
      if (pthread_mutex_unlock (&curr->pvm_mutex.mutex))
        die_with_mesg ("pvmsyncd: couldn't unlock internal mutex\n");

      /* Release readlock */
      unlock_r_die (&mutex_pool[index].ent_rwlock);
      return -EDEADLK;

    } else {

      /* Otherwise, we'll return -EBUSY */
      if (pthread_mutex_unlock (&curr->pvm_mutex.mutex))
        die_with_mesg ("pvmsyncd: couldn't unlock internal mutex\n");

      /* Release readlock */
      unlock_r_die (&mutex_pool[index].ent_rwlock);
      return -EBUSY;

    }

  } else {

    /* Not already locked */
    /* Just lock it. */
    (void) pthread_mutex_lock (&rand_seed_mutex);
           /* All random numbers returned will be positive, to
              differentiate errors. */
    do {
           curr->pvm_mutex.mutex_locked = abs (rand_r (&random_seed));
    } while (!curr->pvm_mutex.mutex_locked);
    (void) pthread_mutex_unlock (&rand_seed_mutex);

    curr->pvm_mutex.mutex_lockaddr = addr;
    curr->pvm_mutex.access_time = time (NULL);
    if (pthread_mutex_unlock (&curr->pvm_mutex.mutex)) {
      die_with_mesg ("pvmsyncd: couldn't unlock internal mutex\n");
    }
  }

  /* Release readlock before returning */
  unlock_r_die (&mutex_pool[index].ent_rwlock);

/* Return our randomly generated integer */
return curr->pvm_mutex.mutex_locked;
}

/* Unlock mutex */
/* EAGAIN, EINVAL, EPERM */
int mutex_unlock (char *mutex_name, int length, unsigned int addr,
						int sentry_val)
{
int index;
int retnval = 0;
struct ht_subent_mutex *curr = NULL;

  /* No funny business, make sure the symbol name is valid
     so we won't get an out-of-bounds error for the array */
  if ((!length) || (legal_sym (mutex_name)))
    return -EINVAL;

  /* Get the index into the hash table */
  /* Index will be OK, since we've already checked
     the validity of the symbol name */
  index = hash_index (mutex_name);

  /* Lock for reading */
  if (pthread_rwlock_rdlock (&mutex_pool[index].ent_rwlock)) {
    return -EAGAIN;
  }

  /* Find the mutex in the hash table */
  curr = (struct ht_subent_mutex *) find_ht (mutex_name, __PVM_MUTEX);
  if (!curr) {
    /* mutex_name does not exist, bad request */
    unlock_r_die (&mutex_pool[index].ent_rwlock);
    return -EINVAL;
  }

  /* Now, lock the internal mutex for this entry */
  /* Shouldn't fail ;) */
  if (pthread_mutex_lock (&curr->pvm_mutex.mutex))
    die_with_mesg ("pvmsyncd: couldn't lock internal mutex\n");

  if ((sentry_val == curr->pvm_mutex.mutex_locked) &&
              (addr == curr->pvm_mutex.mutex_lockaddr)) {
    curr->pvm_mutex.mutex_locked = UNLOCKED;
    curr->pvm_mutex.mutex_lockaddr = (unsigned int) 0;
    /* Time of last access */
    curr->pvm_mutex.access_time = time (NULL);
    if (curr->pvm_mutex.waiters) {
      if (pthread_cond_signal(&curr->pvm_mutex.mutex_cv)) {
        /* We wan't to log an error and exit */
        die_with_mesg ("pvmsyncd: error in condvar signal\n");
      }
      curr->pvm_mutex.waiters--;
    }
  } else {
    /* Imposter!!  You don't own the mutex */
    retnval = -EPERM;
  }

  /* Unlock the internal mutex for this entry */
  /* Shouldn't fail ;) */
  if (pthread_mutex_unlock (&curr->pvm_mutex.mutex))
    die_with_mesg ("pvmsyncd: couldn't lock internal mutex\n");

  /* Unlock rwlock */
  unlock_r_die (&mutex_pool[index].ent_rwlock);

return retnval;
}

/* Create a semaphore */
/* Return 0 on success, negative error code on failure */
/* EAGAIN, EINVAL, ENOSPC */
int sem_create (char *sem_name, int length, unsigned int sem_val)
{
int status, index;
int retnval = 0;

#ifdef DEBUG
char debug_mesg[128];
#endif

  /* First, check to see if this symbol name is valid,
     if we actually have a string */
  if ((!length) || (legal_sym (sem_name) == -EINVAL))
    return -EINVAL;

  /* Get the index into the hash table */
  index = hash_index (sem_name);

  /* Lock rwlock for writing */
  /* We don't need to lock the individual mutexes here,
     because write locks are exclusive. */
  if (pthread_rwlock_wrlock (&sem_pool[index].ent_rwlock)) {
    return -EAGAIN;
  }

  /* Find an existing semaphore wth the same name, if it's there. */
  /* Iterate through the sub-entries (linked list) and check to
     see if we would be creating a duplicate. */
  if ((status = search_ht (sem_name, __PVM_SEM)) < 0) {
    retnval = -EAGAIN;
  } else if (status > 0) {
    /* We've found a duplicate */
    /* Not sure if this is the right thing to do...
       POSIX semaphores doesn't return EBUSY if there
       is a duplicate. */
    retnval = -EINVAL;
  }
  if (retnval) {
    unlock_r_die (&sem_pool[index].ent_rwlock);
    return retnval;
  }
  /* We don't have a dup locally, continue.... */

  /* At this point, the local daemon doesn't have a dup, check
     with others on the cluster */
  /* We'll check to see if anyone else has this symbol
     before we continue. */
  if (symbol_exists (__PVM_SEM, sem_name)) {
    /* Again, I'm not sure if this is the right thing
       to do...  POSIX semaphores doesn't return EBUSY
       if there is a duplicate. */
    unlock_r_die (&sem_pool[index].ent_rwlock);
    return -EINVAL;
  }

  /* Place semaphore into hash table */
  if (insert_ht (sem_name, __PVM_SEM, sem_val))
    retnval = -EAGAIN;

  if (pthread_rwlock_wrlock (&sem_counter_lock))
    die_with_mesg ("pvmsyncd: unable to lock rwlock\n");
  num_sems++;
  unlock_r_die (&sem_counter_lock);

  unlock_r_die (&sem_pool[index].ent_rwlock);

#ifdef DEBUG
  if (!retnval) {
    (void) pthread_mutex_lock (wrlog_mutex);
    (void) snprintf (debug_mesg, 128, "pvmsyncd: object %s created\n",
                                                            sem_name);
           report_log (debug_mesg);
    (void) pthread_mutex_unlock (wrlog_mutex);
  }
#endif

return retnval;
}

/* Destroy a semaphore */
/* Return 0 on success, negative error code on failure */
/* EAGAIN, EINVAL */
int sem_dest (char *sem_name, int length)
{
int status, index;

  /* No funny business, make sure the symbol name is valid
     so we won't get an out-of-bounds error for the array */
  if ((!length) || (legal_sym (sem_name)))
    return -EINVAL;

  /* Get the index into the hash table */
  index = hash_index (sem_name);

  /* Lock rwlock for writing */
  /* We don't need to lock the individual mutexes here,
     because write locks are exclusive. */
  if (pthread_rwlock_wrlock (&sem_pool[index].ent_rwlock)) {
    return -EAGAIN;
  }

  /* First find the semaphore */
  if (!(status = search_ht (sem_name, __PVM_SEM))) {
    unlock_r_die (&sem_pool[index].ent_rwlock);
    return -EINVAL;
  } else if (status < 0) {
    unlock_r_die (&sem_pool[index].ent_rwlock);
    return -EAGAIN;
  }

  if (remove_ht (sem_name, __PVM_SEM)) {
    unlock_r_die (&sem_pool[index].ent_rwlock);
    return -EAGAIN;
  }
  /* At this point, semaphore is removed from hash table */
  /* Decrement counter */
  if (pthread_rwlock_wrlock (&sem_counter_lock))
    die_with_mesg ("pvmsyncd: unable to lock rwlock\n");
  num_sems--;
  unlock_r_die (&sem_counter_lock);

  /* Unlock corresponding rwlock for ht index */
  unlock_r_die (&sem_pool[index].ent_rwlock);

return 0;
}

/* Wait on semaphore, nonblocking if try is nonzero */
/* Return 0 on success, negative error code on failure */
/* EAGAIN, EINVAL */
int sem_p (char *sem_name, int length, int try)
{
int index;
struct ht_subent_sem *curr = NULL;

  /* No funny business, make sure the symbol name is valid
     so we won't get an out-of-bounds error for the array */
  if ((!length) || (legal_sym (sem_name)))
    return -EINVAL;

  /* Get the index into the hash table */
  index = hash_index (sem_name);

  /* Lock for reading */
  if (pthread_rwlock_rdlock (&sem_pool[index].ent_rwlock)) {
    return -EAGAIN;
  }

  /* Find the semaphore in the hash table */
  curr = (struct ht_subent_sem *) find_ht (sem_name, __PVM_SEM);
  if (!curr) {
    /* sem_name does not exist, bad request */
    unlock_r_die (&sem_pool[index].ent_rwlock);
    return -EINVAL;
  }

  if (try) {
    if (sem_trywait (&curr->pvm_sem.semaphore)) {
      /* curr->pvm_sem.semaphore is probably invalid,
         return an error... although this should never
         happen ;) */
      unlock_r_die (&sem_pool[index].ent_rwlock);
      return -errno;
    }
  } else {
    if (sem_wait (&curr->pvm_sem.semaphore)) {
      /* curr->pvm_sem.semaphore is probably invalid,
         return an error... although this should never
         happen ;) */
      unlock_r_die (&sem_pool[index].ent_rwlock);
      return -errno;
    }
  }

  /* Update access time */
  curr->pvm_sem.access_time = time (NULL);

  /* Release readlock before returning */
  unlock_r_die (&sem_pool[index].ent_rwlock);

/* Return success */
return 0;
}

/* Post semaphore */
/* Return 0 on success, negative error code on failure */
/* EAGAIN, EINVAL */
int sem_v (char *sem_name, int length)
{
int index;
struct ht_subent_sem *curr = NULL;

  /* No funny business, make sure the symbol name is valid
     so we won't get an out-of-bounds error for the array */
  if ((!length) || (legal_sym (sem_name)))
    return -EINVAL;

  /* Get the index into the hash table */
  index = hash_index (sem_name);

  /* Lock for reading */
  if (pthread_rwlock_rdlock (&sem_pool[index].ent_rwlock)) {
    return -EAGAIN;
  }

  /* Find the semaphore in the hash table */
  curr = (struct ht_subent_sem *) find_ht (sem_name, __PVM_SEM);
  if (!curr) {
    /* sem_name does not exist, bad request */
    unlock_r_die (&sem_pool[index].ent_rwlock);
    return -EINVAL;
  }

  if (sem_post (&curr->pvm_sem.semaphore)) {
    /* curr->pvm_sem.semaphore is probably invalid,
      return an error... although this should never
      happen ;) */
    unlock_r_die (&sem_pool[index].ent_rwlock);
    /* Could also be ENOSYS, theoretically */
    return -errno;
  }

  /* Update access time */
  curr->pvm_sem.access_time = time (NULL);

  /* Release readlock before returning */
  unlock_r_die (&sem_pool[index].ent_rwlock);

/* Return success */
return 0;
}

/* Create shared int variable */
int integer_create (char *int_name, int length)
{
int count, done = 0, retnval = -1;

#ifdef DEBUG
char debug_mesg[128];
#endif

  /* First, check to see if this symbol name is valid */
  if ((int_name[0] < 'A') ||
     ((int_name[0] > 'Z') && (int_name[0] < '_')) ||
     ((int_name[0] > '_') && (int_name[0] < 'a')) ||
     (int_name[0] > 'z')) {
    errno = EINVAL;
    return -1;
  }
  for (count = 1; count < length-1; count++) {
    if ((int_name[count] < '0') ||
       ((int_name[count] > '9') && (int_name[count] < 'A')) ||
       ((int_name[count] > 'Z') && (int_name[count] < '_')) ||
       ((int_name[count] > '_') && (int_name[count] < 'a')) ||
       (int_name[count] > 'z')) {
      errno = EINVAL;
      return -1;
    }
  }

  if (pthread_rwlock_wrlock (&integer_init_lock)) {
    errno = EAGAIN;
    return -1;
  }

  if (num_integers == CONCURRENT_INSTANCES) {
    /* All integers have been handed out */
    if (pthread_rwlock_unlock (&integer_init_lock)) {
      (void) pthread_mutex_lock (wrlog_mutex);
             report_log ("pvmsyncd: couldn't unlock internal rwlock\n");
      (void) pthread_mutex_unlock (wrlog_mutex);
      (void) raise (SIGINT);
      (void) pthread_exit (NULL);
    }
    errno = EAGAIN;
    return -1;
  }

  /* Find duplicates */
  for (count = 0; count < CONCURRENT_INSTANCES; count++) {
    if ((integer_pool[count].valid) && (!strcmp (integer_pool[count].symbol,
                                                       int_name))) {
      done = 1;
      break;
    }
  }

  if (done) {
    /* There's already one with the same name. */
    if (pthread_rwlock_unlock (&integer_init_lock)) {
      (void) pthread_mutex_lock (wrlog_mutex);
             report_log ("pvmsyncd: couldn't unlock internal rwlock\n");
      (void) pthread_mutex_unlock (wrlog_mutex);
      (void) raise (SIGINT);
    (void) pthread_exit (NULL);
    }
    errno = EBUSY;
    return -1;
  } else {

    /* Now, we'll check to see if anyone else has this symbol
       before we continue. */
    if (symbol_exists (__PVM_SD_INT, int_name)) {
      /* There's already one with the same name. */
      if (pthread_rwlock_unlock (&integer_init_lock)) {
        (void) pthread_mutex_lock (wrlog_mutex);
               report_log ("pvmsyncd: couldn't unlock internal rwlock\n");
        (void) pthread_mutex_unlock (wrlog_mutex);
        (void) raise (SIGINT);
      (void) pthread_exit (NULL);
      }
      errno = EBUSY;
      return -1;
    }

    done = 0;
    for (count = 0; count < CONCURRENT_INSTANCES; count++) {
      if (!integer_pool[count].valid) {
        done = 1;
        break;
      }
    }
    if (!done) {
      errno = ENOSPC;
      retnval = -1;
    } else {
      /* Set the symbol name for our shared data */
      (void) strncpy (integer_pool[count].symbol, int_name, PVM_MAX_SYMLEN);
      integer_pool[count].valid = VALID;
      retnval = count;
      num_integers++;
      /* Update the time of last access */
      integer_pool[count].access_time = time (NULL);
    }
  }

  if (pthread_rwlock_unlock (&integer_init_lock)) {
    (void) pthread_mutex_lock (wrlog_mutex);
           report_log ("pvmsyncd: couldn't unlock internal rwlock\n");
    (void) pthread_mutex_unlock (wrlog_mutex);
    (void) raise (SIGINT);
  (void) pthread_exit (NULL);
  }

#ifdef DEBUG
  if (retnval >= 0) {
    (void) pthread_mutex_lock (wrlog_mutex);
    (void) snprintf (debug_mesg, 128, "pvmsyncd: object %s created\n",
                                                            int_name);
           report_log (debug_mesg);
    (void) pthread_mutex_unlock (wrlog_mutex);
  }
#endif

return retnval;
}

/* Create shared floating pt. variable */
int float_create (char *float_name, int length)
{
int count, done = 0, retnval = -1;

#ifdef DEBUG
char debug_mesg[128];
#endif

  /* First, check to see if this symbol name is valid */
  if ((float_name[0] < 'A') ||
     ((float_name[0] > 'Z') && (float_name[0] < '_')) ||
     ((float_name[0] > '_') && (float_name[0] < 'a')) ||
     (float_name[0] > 'z')) {
    errno = EINVAL;
    return -1;
  }
  for (count = 1; count < length-1; count++) {
    if ((float_name[count] < '0') ||
       ((float_name[count] > '9') && (float_name[count] < 'A')) ||
       ((float_name[count] > 'Z') && (float_name[count] < '_')) ||
       ((float_name[count] > '_') && (float_name[count] < 'a')) ||
       (float_name[count] > 'z')) {
      errno = EINVAL;
      return -1;
    }
  }

  if (pthread_rwlock_wrlock (&float_init_lock)) {
    errno = EAGAIN;
    return -1;
  }

  if (num_floating == CONCURRENT_INSTANCES) {
    /* All floats have been handed out */
    if (pthread_rwlock_unlock (&float_init_lock)) {
      (void) pthread_mutex_lock (wrlog_mutex);
             report_log ("pvmsyncd: couldn't unlock internal rwlock\n");
      (void) pthread_mutex_unlock (wrlog_mutex);
      (void) raise (SIGINT);
      (void) pthread_exit (NULL);
    }
    errno = EAGAIN;
    return -1;
  }

  /* Find duplicates */
  for (count = 0; count < CONCURRENT_INSTANCES; count++) {
    if ((float_pool[count].valid) && (!strcmp (float_pool[count].symbol,
                                                       float_name))) {
      done = 1;
      break;
    }
  }

  if (done) {
    /* There's already one with the same name. */
    if (pthread_rwlock_unlock (&float_init_lock)) {
      (void) pthread_mutex_lock (wrlog_mutex);
             report_log ("pvmsyncd: couldn't unlock internal rwlock\n");
      (void) pthread_mutex_unlock (wrlog_mutex);
      (void) raise (SIGINT);
    (void) pthread_exit (NULL);
    }
    errno = EBUSY;
    return -1;
  } else {

    /* Now, we'll check to see if anyone else has this symbol
       before we continue. */
    if (symbol_exists (__PVM_SD_FLOAT, float_name)) {
      /* There's already one with the same name. */
      if (pthread_rwlock_unlock (&float_init_lock)) {
        (void) pthread_mutex_lock (wrlog_mutex);
               report_log ("pvmsyncd: couldn't unlock internal rwlock\n");
        (void) pthread_mutex_unlock (wrlog_mutex);
        (void) raise (SIGINT);
      (void) pthread_exit (NULL);
      }
      errno = EBUSY;
      return -1;
    }

    done = 0;
    for (count = 0; count < CONCURRENT_INSTANCES; count++) {
      if (!float_pool[count].valid) {
        done = 1;
        break;
      }
    }
    if (!done) {
      errno = ENOSPC;
      retnval = -1;
    } else {
      /* Set the symbol name for our shared data */
      (void) strncpy (float_pool[count].symbol, float_name, PVM_MAX_SYMLEN);
      float_pool[count].valid = VALID;
      retnval = count;
      num_floating++;
      /* Update the time of last access */
      float_pool[count].access_time = time (NULL);
    }
  }

  if (pthread_rwlock_unlock (&float_init_lock)) {
    (void) pthread_mutex_lock (wrlog_mutex);
           report_log ("pvmsyncd: couldn't unlock internal rwlock\n");
    (void) pthread_mutex_unlock (wrlog_mutex);
    (void) raise (SIGINT);
  (void) pthread_exit (NULL);
  }

#ifdef DEBUG
  if (retnval >= 0) {
    (void) pthread_mutex_lock (wrlog_mutex);
    (void) snprintf (debug_mesg, 128, "pvmsyncd: object %s created\n",
                                                          float_name);
           report_log (debug_mesg);
    (void) pthread_mutex_unlock (wrlog_mutex);
  }
#endif

return retnval;
}

/* Create shared memory block */
int mem_create (char *block_name, int length, int size)
{
int count, done = 0, retnval = -1;

#ifdef DEBUG
char debug_mesg[128];
#endif

  if ((size <= 0) || (length <= 0)) {
    errno = EINVAL;
    return -1;
  }

  /* First, check to see if this symbol name is valid */
  if ((block_name[0] < 'A') ||
     ((block_name[0] > 'Z') && (block_name[0] < '_')) ||
     ((block_name[0] > '_') && (block_name[0] < 'a')) ||
     (block_name[0] > 'z')) {
    errno = EINVAL;
    return -1;
  }
  for (count = 1; count < length-1; count++) {
    if ((block_name[count] < '0') ||
       ((block_name[count] > '9') && (block_name[count] < 'A')) ||
       ((block_name[count] > 'Z') && (block_name[count] < '_')) ||
       ((block_name[count] > '_') && (block_name[count] < 'a')) ||
       (block_name[count] > 'z')) {
      errno = EINVAL;
      return -1;
    }
  }

  if (pthread_rwlock_wrlock (&memptr_init_lock)) {
    errno = EAGAIN;
    return -1;
  }

  if (num_memblocks == CONCURRENT_INSTANCES) {
    /* All memptrs have been handed out */
    if (pthread_rwlock_unlock (&memptr_init_lock)) {
      (void) pthread_mutex_lock (wrlog_mutex);
             report_log ("pvmsyncd: couldn't unlock internal rwlock\n");
      (void) pthread_mutex_unlock (wrlog_mutex);
      (void) raise (SIGINT);
      (void) pthread_exit (NULL);
    }
    errno = EAGAIN;
    return -1;
  }

  /* Find duplicates */
  for (count = 0; count < CONCURRENT_INSTANCES; count++) {
    if ((memptr_pool[count].valid) &&
       (!strcmp (memptr_pool[count].symbol, block_name))) {
      done = 1;
      break;
    }
  }

  if (done) {
    /* There's already one with the same name. */
    errno = EBUSY;
    retnval = -1;
  } else {

    /* Now, we'll check to see if anyone else has this symbol
       before we continue. */
    if (symbol_exists (__PVM_SD_MEM, block_name)) {
      if (pthread_rwlock_unlock (&memptr_init_lock)) {
        (void) pthread_mutex_lock (wrlog_mutex);
               report_log ("pvmsyncd: couldn't unlock internal rwlock\n");
        (void) pthread_mutex_unlock (wrlog_mutex);
        (void) raise (SIGINT);
      (void) pthread_exit (NULL);
      }
      errno = EBUSY;
      return -1;
    }

    done = 0;
    for (count = 0; count < CONCURRENT_INSTANCES; count++) {
      if (!memptr_pool[count].valid) {
        done = 1;
        break;
      }
    }
  }

  if (!done) {
    errno = ENOSPC;
    retnval = -1;
  } else {
    /* Allocate memory and store a ptr to it. */
    memptr_pool[count].numbytes = size;
    memptr_pool[count].block = (void *) malloc (memptr_pool[count].numbytes);
    if (memptr_pool[count].block == NULL) {
      memptr_pool[count].numbytes = 0;
      errno = ENOMEM;
      retnval = -1;
    } else {
      /* Set the symbol name for our shared data */
      (void) strncpy (memptr_pool[count].symbol, block_name, PVM_MAX_SYMLEN);
      memptr_pool[count].valid = VALID;
      retnval = count;
      num_memblocks++;
      /* Update the time of last access */
      memptr_pool[count].access_time = time (NULL);
    }
  }

  if (pthread_rwlock_unlock (&memptr_init_lock)) {
    (void) pthread_mutex_lock (wrlog_mutex);
           report_log ("pvmsyncd: couldn't unlock internal rwlock\n");
    (void) pthread_mutex_unlock (wrlog_mutex);
    (void) raise (SIGINT);
  (void) pthread_exit (NULL);
  }

#ifdef DEBUG
  if (retnval >= 0) {
    (void) pthread_mutex_lock (wrlog_mutex);
    (void) snprintf (debug_mesg, 128, "pvmsyncd: object %s created\n",
                                                          block_name);
           report_log (debug_mesg);
    (void) pthread_mutex_unlock (wrlog_mutex);
  }
#endif

return retnval;
}

/* Set the value of a shared integer */
int integer_set (char *int_name, int length, pvm_int64_t value)
{
int count, done = 0, retnval = 0;

  if (length <= 0) {
    errno = EINVAL;
    return -1;
  }

  if (pthread_rwlock_rdlock (&integer_init_lock)) {
    errno = EAGAIN;
    return -1;
  }

  if (((char) int_name[0] >= '0') && ((char) int_name[0] <= '9')) {
    /* We have an index, not an integer name */
    count = atoi (int_name);
    if ((count >= 0) && (count < CONCURRENT_INSTANCES)) {
      if (integer_pool[count].valid) {
        /* We've found it */
        done = 1;
      }
    }

  } else {

    for (count = 0; count < CONCURRENT_INSTANCES; count++) {
      if ((integer_pool[count].valid) &&
         (!strcmp (integer_pool[count].symbol, int_name))) {
           done = 1;
           break;
      }
    }
  }

  if (!done) {
    /* There isn't one with that name/index. */
    errno = EINVAL;
    retnval = -1;
  } else {
    /* Set the value of our shared data */
    if (pthread_mutex_lock (&integer_pool[count].mutex)) {
      (void) pthread_mutex_lock (wrlog_mutex);
             report_log ("pvmsyncd: couldn't lock internal mutex\n");
      (void) pthread_mutex_unlock (wrlog_mutex);
      (void) raise (SIGINT);
      (void) pthread_exit (NULL);
    }

    integer_pool[count].data = value;
    /* Update the time of last access */
    integer_pool[count].access_time = time (NULL);

    if (pthread_mutex_unlock (&integer_pool[count].mutex)) {
      (void) pthread_mutex_lock (wrlog_mutex);
             report_log ("pvmsyncd: couldn't unlock internal mutex\n");
      (void) pthread_mutex_unlock (wrlog_mutex);
      (void) raise (SIGINT);
      (void) pthread_exit (NULL);
    }
  }

  if (pthread_rwlock_unlock (&integer_init_lock)) {
    (void) pthread_mutex_lock (wrlog_mutex);
           report_log ("pvmsyncd: couldn't unlock internal rwlock\n");
    (void) pthread_mutex_unlock (wrlog_mutex);
    (void) raise (SIGINT);
    (void) pthread_exit (NULL);
  }

return retnval;
}

/* Set the value of a shared float */
int float_set (char *float_name, int length, pvm_float64_t value)
{
int count, done = 0, retnval = 0;

  if (length <= 0) {
    errno = EINVAL;
    return -1;
  }

  if (pthread_rwlock_rdlock (&float_init_lock)) {
    errno = EAGAIN;
    return -1;
  }

  if (((char) float_name[0] >= '0') && ((char) float_name[0] <= '9')) {
    /* We have an index, not a float name */
    count = atoi (float_name);
    if ((count >= 0) && (count < CONCURRENT_INSTANCES)) {
      if (float_pool[count].valid) {
        /* We've found it */
        done = 1;
      }
    }

  } else {

    for (count = 0; count < CONCURRENT_INSTANCES; count++) {
      if ((float_pool[count].valid) &&
         (!strcmp (float_pool[count].symbol, float_name))) {
           done = 1;
           break;
      }
    }
  }

  if (!done) {
    /* There isn't one with that name/index. */
    errno = EINVAL;
    retnval = -1;
  } else {
    /* Set the value of our shared data */
    if (pthread_mutex_lock (&float_pool[count].mutex)) {
      (void) pthread_mutex_lock (wrlog_mutex);
             report_log ("pvmsyncd: couldn't lock internal mutex\n");
      (void) pthread_mutex_unlock (wrlog_mutex);
      (void) raise (SIGINT);
      (void) pthread_exit (NULL);
    }

    float_pool[count].data = value;
    /* Update the time of last access */
    float_pool[count].access_time = time (NULL);

    if (pthread_mutex_unlock (&float_pool[count].mutex)) {
      (void) pthread_mutex_lock (wrlog_mutex);
             report_log ("pvmsyncd: couldn't unlock internal mutex\n");
      (void) pthread_mutex_unlock (wrlog_mutex);
      (void) raise (SIGINT);
      (void) pthread_exit (NULL);
    }
  }

  if (pthread_rwlock_unlock (&float_init_lock)) {
    (void) pthread_mutex_lock (wrlog_mutex);
           report_log ("pvmsyncd: couldn't unlock internal rwlock\n");
    (void) pthread_mutex_unlock (wrlog_mutex);
    (void) raise (SIGINT);
    (void) pthread_exit (NULL);
  }

return retnval;
}

/* Set the value of a shared memory block */
int mem_set (char *block_name, int length, void *ptr, int size)
{
int count, done = 0, retnval;

  if (length <= 0) {
    errno = EINVAL;
    return -1;
  }

  if (pthread_rwlock_rdlock (&memptr_init_lock)) {
    errno = EAGAIN;
    return -1;
  }

  retnval = size;

  if (((char) block_name[0] >= '0') && ((char) block_name[0] <= '9')) {
    /* We have an index, not a block name */
    count = atoi (block_name);
    if ((count >= 0) && (count < CONCURRENT_INSTANCES)) {
      if (memptr_pool[count].valid) {
        /* We've found it */
        done = 1;
      }
    }

  } else {

    for (count = 0; count < CONCURRENT_INSTANCES; count++) {
      if ((memptr_pool[count].valid) &&
         (!strcmp (memptr_pool[count].symbol, block_name))) {
           done = 1;
           break;
      }
    }
  }

  if (!done) {
    /* There isn't one with that name/index. */
    errno = EINVAL;
    retnval = -1;
  } else {
    /* Set the value of our shared data */
    if (pthread_mutex_lock (&memptr_pool[count].mutex)) {
      (void) pthread_mutex_lock (wrlog_mutex);
             report_log ("pvmsyncd: couldn't lock internal mutex\n");
      (void) pthread_mutex_unlock (wrlog_mutex);
      (void) raise (SIGINT);
      (void) pthread_exit (NULL);
    }
    if (size > memptr_pool[count].numbytes) {
      errno = ENOBUFS;
      retnval = -1;
    } else {
      (void) memcpy (memptr_pool[count].block, ptr, size);
      /* Update the time of last access */
      memptr_pool[count].access_time = time (NULL);
    }
    if (pthread_mutex_unlock (&memptr_pool[count].mutex)) {
      (void) pthread_mutex_lock (wrlog_mutex);
             report_log ("pvmsyncd: couldn't unlock internal mutex\n");
      (void) pthread_mutex_unlock (wrlog_mutex);
      (void) raise (SIGINT);
      (void) pthread_exit (NULL);
    }
  }

  if (pthread_rwlock_unlock (&memptr_init_lock)) {
    (void) pthread_mutex_lock (wrlog_mutex);
           report_log ("pvmsyncd: couldn't unlock internal rwlock\n");
    (void) pthread_mutex_unlock (wrlog_mutex);
    (void) raise (SIGINT);
    (void) pthread_exit (NULL);
  }

return retnval;
}

/* Get a shared integer */
pvm_int64_t integer_get (char *int_name, int length)
{
int count, done = 0;
pvm_int64_t retnval = (pvm_int64_t) -1;

  if (length <= 0) {
    errno = EINVAL;
    return -1;
  }

  if (pthread_rwlock_rdlock (&integer_init_lock)) {
    errno = EAGAIN;
    return -1;
  }

  if (((char) int_name[0] >= '0') && ((char) int_name[0] <= '9')) {
    /* We have an index, not an int name */
    count = atoi (int_name);
    if ((count >= 0) && (count < CONCURRENT_INSTANCES)) {
      if (integer_pool[count].valid) {
        /* We've found it */
        done = 1;
      }
    }

  } else {

    for (count = 0; count < CONCURRENT_INSTANCES; count++) {
      if ((integer_pool[count].valid) &&
         (!strcmp (integer_pool[count].symbol, int_name))) {
           done = 1;
           break;
      }
    }
  }

  if (!done) {
    /* There isn't one with that name/index. */
    errno = EINVAL;
    retnval = -1;
  } else {
    /* Get the value of our shared data */
    if (pthread_mutex_lock (&integer_pool[count].mutex)) {
      (void) pthread_mutex_lock (wrlog_mutex);
             report_log ("pvmsyncd: couldn't lock internal mutex\n");
      (void) pthread_mutex_unlock (wrlog_mutex);
      (void) raise (SIGINT);
      (void) pthread_exit (NULL);
    }
    errno = 0;
    retnval = integer_pool[count].data;
    /* Update the time of last access */
    integer_pool[count].access_time = time (NULL);
    if (pthread_mutex_unlock (&integer_pool[count].mutex)) {
      (void) pthread_mutex_lock (wrlog_mutex);
             report_log ("pvmsyncd: couldn't unlock internal mutex\n");
      (void) pthread_mutex_unlock (wrlog_mutex);
      (void) raise (SIGINT);
      (void) pthread_exit (NULL);
    }
  }

  if (pthread_rwlock_unlock (&integer_init_lock)) {
    (void) pthread_mutex_lock (wrlog_mutex);
           report_log ("pvmsyncd: couldn't unlock internal rwlock\n");
    (void) pthread_mutex_unlock (wrlog_mutex);
    (void) raise (SIGINT);
    (void) pthread_exit (NULL);
  }

return retnval;
}

/* Get a shared integer */
pvm_float64_t float_get (char *float_name, int length)
{
int count, done = 0;
pvm_float64_t retnval = -1;

  if (length <= 0) {
    errno = EINVAL;
    return (pvm_float64_t) -1;
  }

  if (pthread_rwlock_rdlock (&float_init_lock)) {
    errno = EAGAIN;
    return (pvm_float64_t) -1;
  }

  if (((char) float_name[0] >= '0') && ((char) float_name[0] <= '9')) {
    /* We have an index, not a float name */
    count = atoi (float_name);
    if ((count >= 0) && (count < CONCURRENT_INSTANCES)) {
      if (float_pool[count].valid) {
        /* We've found it */
        done = 1;
      }
    }

  } else {

    for (count = 0; count < CONCURRENT_INSTANCES; count++) {
      if ((float_pool[count].valid) &&
         (!strcmp (float_pool[count].symbol, float_name))) {
           done = 1;
           break;
      }
    }
  }

  if (!done) {
    /* There isn't one with that name/index. */
    errno = EINVAL;
    retnval = (pvm_float64_t) -1;
  } else {
    /* Get the value of our shared data */
    if (pthread_mutex_lock (&float_pool[count].mutex)) {
      (void) pthread_mutex_lock (wrlog_mutex);
             report_log ("pvmsyncd: couldn't lock internal mutex\n");
      (void) pthread_mutex_unlock (wrlog_mutex);
      (void) raise (SIGINT);
      (void) pthread_exit (NULL);
    }
    errno = 0;
    retnval = float_pool[count].data;
    /* Update the time of last access */
    float_pool[count].access_time = time (NULL);
    if (pthread_mutex_unlock (&float_pool[count].mutex)) {
      (void) pthread_mutex_lock (wrlog_mutex);
             report_log ("pvmsyncd: couldn't unlock internal mutex\n");
      (void) pthread_mutex_unlock (wrlog_mutex);
      (void) raise (SIGINT);
      (void) pthread_exit (NULL);
    }
  }

  if (pthread_rwlock_unlock (&float_init_lock)) {
    (void) pthread_mutex_lock (wrlog_mutex);
           report_log ("pvmsyncd: couldn't unlock internal rwlock\n");
    (void) pthread_mutex_unlock (wrlog_mutex);
    (void) raise (SIGINT);
    (void) pthread_exit (NULL);
  }

return retnval;
}

/* Get the contents of memory block */
void * mem_get (char *block_name, int length, int *size)
{
int count, done = 0;
void * retnval = NULL;

  if (length <= 0) {
    errno = EINVAL;
    return retnval;
  }

  if (pthread_rwlock_rdlock (&memptr_init_lock)) {
    errno = EAGAIN;
    return retnval;
  }

  if (((char) block_name[0] >= '0') && ((char) block_name[0] <= '9')) {
    /* We have an index, not a block name */
    count = atoi (block_name);
    if ((count >= 0) && (count < CONCURRENT_INSTANCES)) {
      if (memptr_pool[count].valid) {
        /* We've found it */
        done = 1;
      }
    }

  } else {

    for (count = 0; count < CONCURRENT_INSTANCES; count++) {
      if ((memptr_pool[count].valid) &&
         (!strcmp (memptr_pool[count].symbol, block_name))) {
           done = 1;
           break;
      }
    }
  }

  if (!done) {
    /* There isn't one with that name/index. */
    errno = EINVAL;
  } else {
    /* Get the value of our shared data */
    if (pthread_mutex_lock (&memptr_pool[count].mutex)) {
      (void) pthread_mutex_lock (wrlog_mutex);
             report_log ("pvmsyncd: couldn't lock internal mutex\n");
      (void) pthread_mutex_unlock (wrlog_mutex);
      (void) raise (SIGINT);
      (void) pthread_exit (NULL);
    }
    retnval = memptr_pool[count].block;
    *size = memptr_pool[count].numbytes;
    /* Update the time of last access */
    memptr_pool[count].access_time = time (NULL);
    if (pthread_mutex_unlock (&memptr_pool[count].mutex)) {
      (void) pthread_mutex_lock (wrlog_mutex);
             report_log ("pvmsyncd: couldn't unlock internal mutex\n");
      (void) pthread_mutex_unlock (wrlog_mutex);
      (void) raise (SIGINT);
      (void) pthread_exit (NULL);
    }
  }

  if (pthread_rwlock_unlock (&memptr_init_lock)) {
    (void) pthread_mutex_lock (wrlog_mutex);
           report_log ("pvmsyncd: couldn't unlock internal rwlock\n");
    (void) pthread_mutex_unlock (wrlog_mutex);
    (void) raise (SIGINT);
    (void) pthread_exit (NULL);
  }

return retnval;
}

/* Destroy integer variable */
int integer_dest (char *int_name, int length)
{
int count, done = 0, retnval = 0;

  if (length <= 0) {
    errno = EINVAL;
    return -1;
  }

  if (pthread_rwlock_wrlock (&integer_init_lock)) {
    errno = EAGAIN;
    return -1;
  }

  if (((char) int_name[0] >= '0') && ((char) int_name[0] <= '9')) {
    /* We have an index, not an int name */
    count = atoi (int_name);
    if ((count >= 0) && (count < CONCURRENT_INSTANCES)) {
      if (integer_pool[count].valid) {
        /* We've found it */
        done = 1;
      }
    }

  } else {

    for (count = 0; count < CONCURRENT_INSTANCES; count++) {
      if ((integer_pool[count].valid) &&
         (!strcmp (integer_pool[count].symbol, int_name))) {
           done = 1;
           break;
      }
    }
  }

  if (!done) {
    /* There isn't one with that name. */
    errno = EINVAL;
    retnval = -1;
  } else {
    /* Destroy our shared data object */
    if (pthread_mutex_lock (&integer_pool[count].mutex)) {
      (void) pthread_mutex_lock (wrlog_mutex);
             report_log ("pvmsyncd: couldn't lock internal mutex\n");
      (void) pthread_mutex_unlock (wrlog_mutex);
      (void) raise (SIGINT);
      (void) pthread_exit (NULL);
    }
    integer_pool[count].data = (pvm_int64_t) 0;
    integer_pool[count].valid = (short int) INVALID;
    (void) memset (integer_pool[count].symbol, 0, SYMBOL_LENGTH);
    integer_pool[count].access_time = 0;
    num_integers--;
    if (pthread_mutex_unlock (&integer_pool[count].mutex)) {
      (void) pthread_mutex_lock (wrlog_mutex);
             report_log ("pvmsyncd: couldn't unlock internal mutex\n");
      (void) pthread_mutex_unlock (wrlog_mutex);
      (void) raise (SIGINT);
      (void) pthread_exit (NULL);
    }
  }

  if (pthread_rwlock_unlock (&integer_init_lock)) {
    (void) pthread_mutex_lock (wrlog_mutex);
           report_log ("pvmsyncd: couldn't unlock internal rwlock\n");
    (void) pthread_mutex_unlock (wrlog_mutex);
    (void) raise (SIGINT);
    (void) pthread_exit (NULL);
  }

return retnval;
}

/* Destroy float variable */
int float_dest (char *float_name, int length)
{
int count, done = 0, retnval = 0;

  if (length <= 0) {
    errno = EINVAL;
    return -1;
  }

  if (pthread_rwlock_wrlock (&float_init_lock)) {
    errno = EAGAIN;
    return -1;
  }

  if (((char) float_name[0] >= '0') && ((char) float_name[0] <= '9')) {
    /* We have an index, not a float name */
    count = atoi (float_name);
    if ((count >= 0) && (count < CONCURRENT_INSTANCES)) {
      if (float_pool[count].valid) {
        /* We've found it */
        done = 1;
      }
    }

  } else {

    for (count = 0; count < CONCURRENT_INSTANCES; count++) {
      if ((float_pool[count].valid) &&
         (!strcmp (float_pool[count].symbol, float_name))) {
           done = 1;
           break;
      }
    }
  }

  if (!done) {
    /* There isn't one with that name. */
    errno = EINVAL;
    retnval = -1;
  } else {
    /* Destroy our shared data object */
    if (pthread_mutex_lock (&float_pool[count].mutex)) {
      (void) pthread_mutex_lock (wrlog_mutex);
             report_log ("pvmsyncd: couldn't lock internal mutex\n");
      (void) pthread_mutex_unlock (wrlog_mutex);
      (void) raise (SIGINT);
      (void) pthread_exit (NULL);
    }
    float_pool[count].data = (pvm_float64_t) 0.0;
    float_pool[count].valid = (short int) INVALID;
    (void) memset (float_pool[count].symbol, 0, SYMBOL_LENGTH);
    float_pool[count].access_time = 0;
    num_floating--;
    if (pthread_mutex_unlock (&float_pool[count].mutex)) {
      (void) pthread_mutex_lock (wrlog_mutex);
             report_log ("pvmsyncd: couldn't unlock internal mutex\n");
      (void) pthread_mutex_unlock (wrlog_mutex);
      (void) raise (SIGINT);
      (void) pthread_exit (NULL);
    }
  }

  if (pthread_rwlock_unlock (&float_init_lock)) {
    (void) pthread_mutex_lock (wrlog_mutex);
           report_log ("pvmsyncd: couldn't unlock internal rwlock\n");
    (void) pthread_mutex_unlock (wrlog_mutex);
    (void) raise (SIGINT);
    (void) pthread_exit (NULL);
  }

return retnval;
}

/* Destroy (free) memory block */
int mem_dest (char *block_name, int length)
{
int count, done = 0, retnval = 0;

  if (length <= 0) {
    errno = EINVAL;
    return -1;
  }

  if (pthread_rwlock_wrlock (&memptr_init_lock)) {
    errno = EAGAIN;
    return -1;
  }

  if (((char) block_name[0] >= '0') && ((char) block_name[0] <= '9')) {
    /* We have an index, not a block name */
    count = atoi (block_name);
    if ((count >= 0) && (count < CONCURRENT_INSTANCES)) {
      if (memptr_pool[count].valid) {
        /* We've found it */
        done = 1;
      }
    }

  } else {

    for (count = 0; count < CONCURRENT_INSTANCES; count++) {
      if ((memptr_pool[count].valid) &&
         (!strcmp (memptr_pool[count].symbol, block_name))) {
           done = 1;
           break;
      }
    }
  }

  if (!done) {
    /* There isn't one with that name. */
    errno = EINVAL;
    retnval = -1;
  } else {
    /* Destroy our shared data object */
    if (pthread_mutex_lock (&memptr_pool[count].mutex)) {
      (void) pthread_mutex_lock (wrlog_mutex);
             report_log ("pvmsyncd: couldn't lock internal mutex\n");
      (void) pthread_mutex_unlock (wrlog_mutex);
      (void) raise (SIGINT);
      (void) pthread_exit (NULL);
    }
    free (memptr_pool[count].block);
    memptr_pool[count].block = NULL;
    memptr_pool[count].numbytes = 0;
    memptr_pool[count].valid = (short int) INVALID;
    (void) memset (memptr_pool[count].symbol, 0, SYMBOL_LENGTH);
    memptr_pool[count].access_time = 0;
    num_memblocks--;
    if (pthread_mutex_unlock (&memptr_pool[count].mutex)) {
      (void) pthread_mutex_lock (wrlog_mutex);
             report_log ("pvmsyncd: couldn't unlock internal mutex\n");
      (void) pthread_mutex_unlock (wrlog_mutex);
      (void) raise (SIGINT);
      (void) pthread_exit (NULL);
    }
  }

  if (pthread_rwlock_unlock (&memptr_init_lock)) {
    (void) pthread_mutex_lock (wrlog_mutex);
           report_log ("pvmsyncd: couldn't unlock internal rwlock\n");
    (void) pthread_mutex_unlock (wrlog_mutex);
    (void) raise (SIGINT);
    (void) pthread_exit (NULL);
  }

return retnval;
}

/* Create a condition variable... */
/* Returns 0 on success, negative error code on error */
/* EAGAIN, ENOMEM, EBUSY, EINVAL (nonstandard usage) */
int cond_create (char *cv_name, int length)
{
int status, index;
int retnval = 0;

#ifdef DEBUG
char debug_mesg[128];
#endif

  /* First, check to see if this symbol name is valid,
     if we really have a string of length > 0 */
  if ((!length) || (legal_sym (cv_name) == -EINVAL))
    return -EINVAL;

  /* Get the index into the hash table */
  index = hash_index (cv_name);

  /* Lock rwlock for writing */
  /* We don't need to lock the individual mutexes here,
     because write locks are exclusive. */
  if (pthread_rwlock_wrlock (&cond_pool[index].ent_rwlock)) {
    return -EAGAIN;
  }

  /* Find an existing condition variable with the same name,
     if it's there. */
  /* Iterate through the sub-entries (linked list) and check to
     see if we would be creating a duplicate. */
  if ((status = search_ht (cv_name, __PVM_CV)) < 0) {
    retnval = -EAGAIN;
  } else if (status > 0) {
    /* We've found a duplicate */
    retnval = -EBUSY;
  }
  if (retnval) {
    unlock_r_die (&cond_pool[index].ent_rwlock);
    return retnval;
  }
  /* We don't have a dup locally, continue.... */

  /* At this point, the local daemon doesn't have a dup, check
     with others on the cluster */
  /* We'll check to see if anyone else has this symbol
     before we continue. */
  if (symbol_exists (__PVM_CV, cv_name)) {
    unlock_r_die (&cond_pool[index].ent_rwlock);
    return -EBUSY;
  }

  /* Place condition variable into hash table */
  if (insert_ht (cv_name, __PVM_CV))
    retnval = -EAGAIN;

  if (pthread_rwlock_wrlock (&condvar_counter_lock))
    die_with_mesg ("pvmsyncd: unable to lock rwlock\n");
  num_conds++;
  unlock_r_die (&condvar_counter_lock);

  unlock_r_die (&cond_pool[index].ent_rwlock);

#ifdef DEBUG
  if (!retnval) {
    (void) pthread_mutex_lock (wrlog_mutex);
    (void) snprintf (debug_mesg, 128, "pvmsyncd: object %s created\n",
                                                        cv_name);
           report_log (debug_mesg);
    (void) pthread_mutex_unlock (wrlog_mutex);
  }
#endif

return retnval;
}

/* Destroy condition variable */
/* EAGAIN, EINVAL */
int cond_dest (char *cv_name, int length)
{
int status, index;

  /* No funny business, make sure the symbol name is valid
     so we won't get an out-of-bounds error for the array */
  if ((!length) || (legal_sym (cv_name)))
    return -EINVAL;

  /* Get the index into the hash table */
  index = hash_index (cv_name);

  /* Lock rwlock for writing */
  /* We don't need to lock the individual mutexes here,
     because write locks are exclusive. */
  if (pthread_rwlock_wrlock (&cond_pool[index].ent_rwlock)) {
    return -EAGAIN;
  }

  /* First find the cond var */
  if (!(status = search_ht (cv_name, __PVM_CV))) {
    unlock_r_die (&cond_pool[index].ent_rwlock);
    return -EINVAL;
  } else if (status < 0) {
    unlock_r_die (&cond_pool[index].ent_rwlock);
    return -EAGAIN;
  }

  if (remove_ht (cv_name, __PVM_CV)) {
    unlock_r_die (&cond_pool[index].ent_rwlock);
    return -EAGAIN;
  }
  /* At this point, condvar is removed from hash table */
  /* Decrement counter */
  if (pthread_rwlock_wrlock (&condvar_counter_lock))
    die_with_mesg ("pvmsyncd: unable to lock rwlock\n");
  num_conds--;
  unlock_r_die (&condvar_counter_lock);

  /* Unlock corresponding rwlock for ht index */
  unlock_r_die (&cond_pool[index].ent_rwlock);

return 0;
}

/* Zero on success, negative error code on failure */
/* EAGAIN, EINVAL */
int cv_sig (char *cv_name, int length, int broadc_flag)
{
int index;
struct ht_subent_cv *curr = NULL;

  /* No funny business, make sure the symbol name is valid
     so we won't get an out-of-bounds error for the array */
  if ((!length) || (legal_sym (cv_name)))
    return -EINVAL;

  /* Get the index into the hash table */
  index = hash_index (cv_name);

  /* Lock for reading */
  if (pthread_rwlock_rdlock (&cond_pool[index].ent_rwlock)) {
    return -EAGAIN;
  }

  /* Find the condition variable in the hash table */
  curr = (struct ht_subent_cv *) find_ht (cv_name, __PVM_CV);
  if (!curr) {
    /* cv_name does not exist, bad request */
    unlock_r_die (&cond_pool[index].ent_rwlock);
    return -EINVAL;
  }

  if (broadc_flag) {
    if (pthread_cond_broadcast (&curr->pvm_cv.cond)) {
      /* curr->pvm_cv.cond is probably invalid, return an error... */
      /* although this should never happen */
      unlock_r_die (&cond_pool[index].ent_rwlock);
      return -EINVAL;
    }
  } else {
    if (pthread_cond_signal (&curr->pvm_cv.cond)) {
      /* curr->pvm_cv.cond is probably invalid, return an error... */
      /* although this should never happen */
      unlock_r_die (&cond_pool[index].ent_rwlock);
      return -EINVAL;
    }
  }

  /* Update access time */
  curr->pvm_cv.access_time = time (NULL);

  /* Release readlock before returning */
  unlock_r_die (&cond_pool[index].ent_rwlock);

/* Return success */
return 0;
}

/* Timed wait on condition variable */
/* If secs is 0, do a blocking wait (wait indefinitely) */
/* Zero on success, negative error code on failure */
/* EAGAIN, EINVAL, ETIMEDOUT */
int cond_timedwait (char *cv_name, int cv_len, time_t secs)
{
int status, index;
struct timespec wait_time;
struct ht_subent_cv *curr = NULL;

  /* No funny business, make sure the symbol name is valid
     so we won't get an out-of-bounds error for the array */
  if ((!cv_len) || (legal_sym (cv_name)))
    return -EINVAL;

  /* Get the index into the hash table */
  index = hash_index (cv_name);

  /* Lock for reading */
  if (pthread_rwlock_rdlock (&cond_pool[index].ent_rwlock)) {
    return -EAGAIN;
  }

  /* Find the condition variable in the hash table */
  curr = (struct ht_subent_cv *) find_ht (cv_name, __PVM_CV);
  if (!curr) {
    /* cv_name does not exist, bad request */
    unlock_r_die (&cond_pool[index].ent_rwlock);
    return -EINVAL;
  }

  if (secs) {
    wait_time.tv_sec = secs;
    wait_time.tv_nsec = 0;
    if (pthread_mutex_lock (&curr->pvm_cv.mutex)) {
        unlock_r_die (&cond_pool[index].ent_rwlock);
        return -EINVAL;
    }
    status = pthread_cond_timedwait (&curr->pvm_cv.cond,
                                     &curr->pvm_cv.mutex,
                                     &wait_time);
    if (status) {
      unlock_r_die (&cond_pool[index].ent_rwlock);
      /* Could be ETIMEDOUT or EINVAL */
      return -status;
    }

    if (pthread_mutex_unlock (&curr->pvm_cv.mutex)) {
      unlock_r_die (&cond_pool[index].ent_rwlock);
      return -EINVAL;
    }
  } else {
    if (pthread_mutex_lock (&curr->pvm_cv.mutex)) {
      unlock_r_die (&cond_pool[index].ent_rwlock);
      return -EINVAL;
    }

    status = pthread_cond_wait (&curr->pvm_cv.cond,
                                &curr->pvm_cv.mutex);
    if (status) {
      unlock_r_die (&cond_pool[index].ent_rwlock);
      /* Could only be EINVAL */
      return -status;
    }

    if (pthread_mutex_unlock (&curr->pvm_cv.mutex)) {
      unlock_r_die (&cond_pool[index].ent_rwlock);
      return -EINVAL;
    }
  }

  /* Update access time */
  curr->pvm_cv.access_time = time (NULL);

  /* Release readlock before returning */
  unlock_r_die (&cond_pool[index].ent_rwlock);

/* Return success */
return 0;
}


