/*
 *
 * $Id: pvm_sem.c,v 1.2.4.41 2000/03/16 17:14:14 ajp Exp $
 *
 * Andrew Pitman              pvm_sem.c
 *
 * pvmsync, a distributed synchronization server: semaphore-specific
 * client routines for pvmsync.
 *
 * Server that accepts connections (requests) for shared
 * data or synchronization mechanisms continuously and
 * changes some internal state according to the request.
 *
 * Copyright (C) 1999, 2000 Andrew J. Pitman
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


#ifndef __PVM_SEM_H
#include <pvm_sem.h>
#endif

#include <stdio.h>
#include <string.h>

#include <unistd.h>
#include <stdlib.h>
#include <time.h>

#ifndef __PVM_GEN_H
#include <pvm_gen.h>
#endif

#ifndef __LIB_INTERNALS_H
#include <lib_internals.h>
#endif

#ifndef __CONFIG_H
#include <config.h>
#endif


/* Sockets to use when contacting server(s) */
int __pvm_client_tcp_sock;
int __pvm_client_udp_sock;


/* These routines follow the convention of returning -1
   and setting errno appropriately. */

/* Initialize a pvm_sem */
/* This pvm_sem will be instantiated on a server, and the
   pvm_sem->sym_addr will be filled in with the appropriate
   server address. */
/* Return 0 on success and -1 failure. */
int pvm_sem_init (pvm_sem *sem, unsigned int value)
{
  return (__pvm_generic_init_sync (__PVM_SEM, (void *) sem, value));
}

/* The given pvm_sem will be destroyed on the server it
   resides on (if possible), or an error returned. */
/* Return 0 on success and -1 on failure. */
int pvm_sem_destroy (pvm_sem *sem)
{
  return (__pvm_generic_destroy_sync (__PVM_SEM, (void *) sem));
}

/* Wait on the semaphore.  sem must be initialized first,
   or if already created by another process, looked up with
   __pvm_getaddr. */
/* Return 0 on success and -1 on failure. */
int pvm_sem_wait (pvm_sem *sem)
{

  /* Blocking wait on semaphore */

return (__pvm_sem_wait (sem, 0));
}

/* Wait on the semaphore.  sem must be initialized first,
   or if already created by another process, looked up with
   __pvm_getaddr.  This is the nonblocking version.... it
   will return with an error if waiting on the semaphore
   would block. */
/* Return 0 on success and -1 on failure. */
int pvm_sem_trywait (pvm_sem *sem)
{

  /* Non-blocking wait on semaphore */

return (__pvm_sem_wait (sem, 1));
}

/* Post the semaphore.  Will return with an error if sem is
   invalid. */
/* Return 0 on success and -1 on failure. */
int pvm_sem_post (pvm_sem *sem)
{
char comm_buff[BUF_LENGTH];

  (void) memset (&comm_buff, 0, BUF_LENGTH);
  (void) snprintf (comm_buff, BUF_LENGTH, "VSM:%s", sem->sym_name);

  /* We already have the server's address in sem->sym_addr */
  if (__pvm_tcp_server_sendreq (comm_buff, strlen (comm_buff), sem->sym_addr)) {
    /* We may have migrated to another node, look it up */
    if (__pvm_getaddr (__PVM_SEM, sem)) {
      errno = EAGAIN;
      return -1;
    }
    /* Try it again */
    if (__pvm_tcp_server_sendreq (comm_buff, strlen (comm_buff), sem->sym_addr))
      return -1; /* errno already set */
  }

/* Default */
return 0;
}


