/*
 *
 * $Id: pvm_sd.c,v 1.2.4.41 2000/03/16 17:14:11 ajp Exp $
 *
 * Andrew Pitman              pvm_sd.c
 *
 * pvmsync, a distributed synchronization server:  shared data
 * specific client routines for pvmsync.
 *
 * Server that accepts connections (requests) for shared
 * data or synchronization mechanisms continuously and
 * changes some internal state according to the request.
 *
 * Copyright (C) 1999, 2000 Andrew J. Pitman
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


#ifndef __PVM_SD_H
#include <pvm_sd.h>
#endif

#include <stdio.h>
#include <string.h>

#include <unistd.h>
#include <stdlib.h>
#include <time.h>

#ifndef __PVM_GEN_H
#include <pvm_gen.h>
#endif

#ifndef __LIB_INTERNALS_H
#include <lib_internals.h>
#endif

#ifndef __CONFIG_H
#include <config.h>
#endif


#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

#include <errno.h>

/* Routine pvm_sd_init can have either 2 or 3 arguments */
#include <stdarg.h>


/* Sockets to use when contacting server(s) */
extern int __pvm_client_tcp_sock;
extern int __pvm_client_udp_sock;


/* These routines follow the convention of returning -1
   and setting errno appropriately. */

/* Initialize a pvm_sd */
/* This pvm_sd will be instantiated on a server, and the
   pvm_sd->sym_addr will be filled in with the appropriate
   server address. */
/* Return 0 on success and -1 failure. */
int pvm_sd_init (pvm_t obj_type, pvm_sd *sd, ...)
{
int status;
size_t size;
va_list args;
char *hostname_fqdn, *hostname;
unsigned int rand_int;

  /* Zero out the symbol's (object's) name */
  (void) memset (sd->sym_name, 0, PVM_MAX_SYMLEN);

  hostname_fqdn = (char *) malloc (MAXHOSTNAMELEN);
  if (!hostname_fqdn) {
      (void) close (__pvm_client_udp_sock);
      errno = ENOMEM;
      return -1;
  }
  (void) memset (hostname_fqdn, 0, MAXHOSTNAMELEN);

  /* Set sd->sym_name according to some rules */
  status = gethostname (hostname_fqdn, MAXHOSTNAMELEN);
  if (status) {
      (void) close (__pvm_client_udp_sock);
      errno = EAGAIN;
      return -1;
  }

  hostname = strtok (hostname_fqdn, ".");
  /* Sanitize */
  __pvm_sanitize_sym (hostname);

  srand ((unsigned int) time (NULL));
  rand_int = (unsigned int) abs (rand ());

  switch (obj_type) {
    case __PVM_SD_INT:
      (void) snprintf (sd->sym_name, PVM_MAX_SYMLEN, "%.8sINT%u", hostname,
                                                                    rand_int);
      hostname = NULL;
      free (hostname_fqdn);
      return (__pvm_int_init (sd));
    case __PVM_SD_FLOAT:
      (void) snprintf (sd->sym_name, PVM_MAX_SYMLEN, "%.8sFLT%u", hostname,
                                                                    rand_int);
      hostname = NULL;
      free (hostname_fqdn);
      return (__pvm_float_init (sd));
    case __PVM_SD_MEM:
      (void) snprintf (sd->sym_name, PVM_MAX_SYMLEN, "%.8sMEM%u", hostname,
                                                                    rand_int);
      hostname = NULL;
      free (hostname_fqdn);
      va_start (args, sd);
      size = va_arg (args, size_t);
      status = __pvm_mem_init (sd, size);
      va_end (args);
      return status;
    default:
      errno = EINVAL;
      return -1;
  };
}

/* The given pvm_sd will be destroyed on the server it
   resides on (if possible), or an error returned. */
/* Return 0 on success and -1 on failure. */
int pvm_sd_destroy (pvm_sd *sd)
{
int status, retnval = 0;
ssize_t comm_len;
char comm_buff[BUF_LENGTH];
char *err_string = NULL;

  /* Set comm_buff to all zeros */
  (void) memset (&comm_buff, 0, BUF_LENGTH);

  /* We already have the server's address in sd->sym_addr */
  /* Connect to it and destroy the mutex */
  __pvm_client_tcp_sock = socket (AF_INET, SOCK_STREAM, 0);
  if (__pvm_client_tcp_sock < 0) {
    errno = EAGAIN;
    return -1;
  }

  status = connect (__pvm_client_tcp_sock, &sd->sym_addr,
                               sizeof (struct sockaddr_in));

  if (status) {
    if (__pvm_getaddr (sd->sym_type, sd)) {
      (void) close (__pvm_client_tcp_sock);
      errno = EAGAIN;
      return -1;
    }

    status = connect (__pvm_client_tcp_sock, &sd->sym_addr,
                               sizeof (struct sockaddr_in));
    if (status) {
      (void) close (__pvm_client_tcp_sock);
      errno = EAGAIN;
      return -1;
    }
  }

  switch (sd->sym_type) {
    case __PVM_SD_INT:
      (void) snprintf (comm_buff, BUF_LENGTH, "DSD:INTEGER:%d",
                                                sd->sym_index);
      break;
    case __PVM_SD_FLOAT:
      (void) snprintf (comm_buff, BUF_LENGTH, "DSD:FLOAT:%d",
                                                sd->sym_index);
      break;
    case __PVM_SD_MEM:
      (void) snprintf (comm_buff, BUF_LENGTH, "DSD:MEM:%d",
                                                sd->sym_index);
      break;
    default:
      (void) close (__pvm_client_tcp_sock);
      errno = EINVAL;
      return -1;
  };

  comm_len = write (__pvm_client_tcp_sock, comm_buff, strlen (comm_buff));
  if (comm_len == -1) {
    (void) close (__pvm_client_tcp_sock);
    errno = EAGAIN;
    return -1;
  }

  (void) memset (comm_buff, 0, BUF_LENGTH);

  comm_len = read (__pvm_client_tcp_sock, comm_buff, BUF_LENGTH);
  if ((!comm_len) || (comm_len < 0)) {
    errno =  EAGAIN;
    retnval = -1;
  } else {
    if (!strncmp (comm_buff, "SUCCESS:", 8)) {
      free (sd->sym_value);
      (void) memset (sd, 0, sizeof (pvm_sd));
    } else if (!strncmp (comm_buff, "FAIL:", 5)) {
      err_string = comm_buff+5;
      errno = atoi (err_string);
      retnval = -1;
    } else {
      errno = EIO;
      retnval = -1;
    }
  }

  /* Close the tcp socket */
  (void) close (__pvm_client_tcp_sock);

return retnval;
}

/* Get the value of a shared data object */
int pvm_sd_get (pvm_sd *sd)
{

  switch (sd->sym_type) {
    case __PVM_SD_INT:
    case __PVM_SD_FLOAT:
      return (__pvm_scalar_get (sd));
    case __PVM_SD_MEM:
      return (__pvm_mem_get (sd));
    default:
      errno = EINVAL;
      return -1;
  };

/* NOTREACHED */
return 0;
}

/* Set the value of a shared data object */
int pvm_sd_set (pvm_sd *sd)
{

  switch (sd->sym_type) {
    case __PVM_SD_INT:
    case __PVM_SD_FLOAT:
      return (__pvm_scalar_set (sd));
    case __PVM_SD_MEM:
      return (__pvm_mem_set (sd));
    default:
      errno = EINVAL;
      return -1;
  };

/* NOTREACHED */
return 0;
}

/* Extract the data value from a pvm_sd */
/* Returns 0 on success, -1 on failure and sets errno */
int pvm_getvalue (pvm_sd *sd, void *val)
{

  if (sd->sym_type & PVM_DATA) {
      val = sd->sym_value;
  } else {
      errno = EINVAL;
      return -1;
  }

return 0;
}

/* Integer-specific get value of object */
pvm_int64_t pvm_getint (pvm_sd *sd)
{
pvm_int64_t *integer_64_ptr;

  if (sd->sym_type != __PVM_SD_INT) {
    errno = EINVAL;
    return (pvm_int64_t) -1;
  }

  integer_64_ptr = (pvm_int64_t *) sd->sym_value;

  /* If we're returning an error, we'll return -1 and
     set errno.  In case the value of *integer_64_ptr
     is -1, we'll zero out errno. */
  errno = 0;

return *integer_64_ptr;
}

/* Float-specific get value of object */
pvm_float64_t pvm_getfloat (pvm_sd *sd)
{
pvm_float64_t *float_64_ptr;

  if (sd->sym_type != __PVM_SD_FLOAT) {
    errno = EINVAL;
    return (pvm_float64_t) -1;
  }

  float_64_ptr = (pvm_float64_t *) sd->sym_value;

  /* If we're returning an error, we'll return -1 and
     set errno.  In case the value of *float_64_ptr
     is -1, we'll zero out errno. */
  errno = 0;

return *float_64_ptr;
}

/* Memory block-specific get value of object */
void * pvm_getmem (pvm_sd *sd)
{
void * memptr = NULL;

  if (sd->sym_type != __PVM_SD_MEM) {
    errno = EINVAL;
    return (void *) NULL;
    /* We'll know that if pvm_getmem returns NULL,
       we can check errno for what happened. */
  }

  memptr = (void *) sd->sym_value;

  if (memptr == NULL)
    errno = EAGAIN;

return memptr;
}

/* Set the data member */
int pvm_setvalue (pvm_sd *sd, ...)
{
va_list args;
pvm_int64_t intval = 0;
pvm_float64_t floatval = 0.0;
void *mem = NULL;

  va_start (args, sd);
  switch (sd->sym_type) {
    case __PVM_SD_INT:
      intval = va_arg (args, pvm_int64_t);
      (void) memcpy (sd->sym_value, (void *) &intval, sizeof (pvm_int64_t));
      break;
    case __PVM_SD_FLOAT:
      floatval = va_arg (args, pvm_float64_t);
      (void) memcpy (sd->sym_value, (void *) &floatval, sizeof (pvm_float64_t));
      break;
    case __PVM_SD_MEM:
      mem = va_arg (args, void *);
      (void) memcpy (sd->sym_value, mem, sd->sym_size);
      break;
    default:
      errno = EINVAL;
      return -1;
  };

return 0;
}


