/*
 *
 * $Id: pvm_mutex.c,v 1.2.4.41 2000/03/16 17:14:08 ajp Exp $
 *
 * Andrew Pitman              pvm_mutex.c
 *
 * pvmsync, a distributed synchronization server:  library code
 *
 * Server that accepts connections (requests) for shared
 * data or synchronization mechanisms continuously and
 * changes some internal state according to the request.
 *
 * Copyright (C) 1999, 2000 Andrew J. Pitman
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


/* extern int errno */
#include <errno.h>

#include <pvm_mutex.h>
#include <pvmerrs.h>

#include <stdio.h>
#include <string.h>

#include <unistd.h>
#include <stdlib.h>
#include <time.h>

#ifndef __PVM_GEN_H
#include <pvm_gen.h>
#endif

#ifndef __LIB_INTERNALS_H
#include <lib_internals.h>
#endif

#ifndef __CONFIG_H
#include <config.h>
#endif


/* Sockets to use when contacting server(s) */
int __pvm_client_tcp_sock;
int __pvm_client_udp_sock;


/* These routines follow after pthreads convention of
   returning the error number instead of returning -1
   and setting errno. */

/* Initialize a pvm_mutex */
/* This pvm_mutex will be instantiated on a server, and the
   pvm_mutex->sym_addr will be filled in with the appropriate
   server address. */
/* Return 0 on success and a positive error code on failure. */
int pvm_mutex_init(pvm_mutex *mutex)
{
int __errno,
    retnval;

  /* Preserve previous state of errno */
  __errno = errno;

  if (__pvm_generic_init_sync (__PVM_MUTEX, (void *) mutex)) {
    retnval = errno;
    errno = __errno;
  } else {
    retnval = 0;
    errno = __errno;
  }

return retnval;
}

/* The given pvm_mutex will be destroyed on the server it
   resides on (if possible), or an error returned. */
int pvm_mutex_destroy(pvm_mutex *mutex)
{
int __errno,
    retnval;

  /* Preserve previous state of errno */
  __errno = errno;

  if (__pvm_generic_destroy_sync (__PVM_MUTEX, (void *) mutex)) {
    retnval = errno;
    errno = __errno;
  } else {
    retnval = 0;
    errno = __errno;
  }

return retnval;
}

/* Lock the mutex.  Mutex must be initialized first, or if
   already created by another process, looked up with
   __pvm_getaddr. */
int pvm_mutex_lock(pvm_mutex *mutex)
{
int __errno, retnval;

  /* Preserve previous state of errno */
  __errno = errno;

  /* Blocking lock on mutex */
  retnval = __pvm_mutex_lock (mutex, 0);

  /* Restore errno */
  errno = __errno;

return retnval;
}

/* Lock the mutex.  Mutex must be initialized first, or if
   already created by another process, looked up with
   __pvm_getaddr.  This is the nonblocking version.... it
   will return with an error if the mutex is already locked
   by another process. */
int pvm_mutex_trylock(pvm_mutex *mutex)
{
int __errno, retnval;

  /* Preserve previous state of errno */
  __errno = errno;

  /* Try to lock the mutex */
  retnval = __pvm_mutex_lock (mutex, 1);

  /* Restore errno */
  errno = __errno;

return retnval;
}

/* Unlock the mutex.  Will return an error if mutex is
   invalid or if it is still locked. */
int pvm_mutex_unlock(pvm_mutex *mutex)
{
int __errno, retn_val = 0;
char req_buff[BUF_LENGTH];

  /* Preserve previous state of errno */
  __errno = errno;

  (void) memset (&req_buff, 0, BUF_LENGTH);

  (void) snprintf (req_buff, BUF_LENGTH, "UNL:%s:%d", mutex->sym_name,
                                                         mutex->sym_lock);

  /* We already have the server's address in mutex->sym_addr */
  if (__pvm_tcp_server_sendreq (req_buff, strlen (req_buff), mutex->sym_addr)) {
    /* We may have migrated to another node, look it up */
    if (__pvm_getaddr (__PVM_MUTEX, mutex)) {
      /* Restore errno */
      errno = __errno;
      return EAGAIN;
    }
    /* Try it again */
    if (__pvm_tcp_server_sendreq (req_buff, strlen (req_buff), mutex->sym_addr))
      retn_val = errno;
  }

  /* Restore errno */
  errno = __errno;

return retn_val;
}


