/*
 *
 * $Id: pvm_gen.c,v 1.2.4.41 2000/03/16 17:14:05 ajp Exp $
 *
 * Andrew Pitman              pvm_gen.c
 *
 * pvmsync, a distributed synchronization server:  library code
 *
 * Server that accepts connections (requests) for shared
 * data or synchronization mechanisms continuously and
 * changes some internal state according to the request.
 *
 * Copyright (C) 1999, 2000 Andrew J. Pitman
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


#include <pvm_gen.h>
#include <pvm_mutex.h>
#include <pvm_cv.h>
#include <pvm_sem.h>
#include <pvm_sd.h>

#include <pvmerrs.h>

#include <config.h>

#include <stdio.h>
#include <string.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>


/* Max number of fd for slect to consider */
#define __MAX_FDS          256


/* Sockets to use when contacting server(s) */
extern int __pvm_client_tcp_sock;
extern int __pvm_client_udp_sock;


/* Bind a symbolic name to the object */
int __pvm_bind (pvm_t type, char *sym_name, void *object)
{
pvm_mutex * mutex = NULL;
pvm_cond * condvar = NULL;
pvm_sem * semaphore = NULL;
pvm_sd * shared_data = NULL;

  switch (type) {
    case __PVM_MUTEX:
      mutex = (pvm_mutex *) object;
      if (mutex == NULL) {
        errno = EINVAL;
        return -1;
      }
      (void) memset (mutex, 0, sizeof (pvm_mutex));
      (void) strncpy (mutex->sym_name, sym_name, 32);
      break;
    case __PVM_CV:
      condvar = (pvm_cond *) object;
      if (condvar == NULL) {
        errno = EINVAL;
        return -1;
      }
      (void) memset (condvar, 0, sizeof (pvm_cond));
      (void) strncpy (condvar->sym_name, sym_name, 32);
      break;
    case __PVM_SEM:
      semaphore = (pvm_sem *) object;
      if (semaphore == NULL) {
        errno = EINVAL;
        return -1;
      }
      (void) memset (semaphore, 0, sizeof (pvm_sem));
      (void) strncpy (semaphore->sym_name, sym_name, 32);
      break;
    case __PVM_SD_INT:
    case __PVM_SD_FLOAT:
    case __PVM_SD_MEM:
      shared_data = (pvm_sd *) object;
      if (shared_data == NULL) {
        errno = EINVAL;
        return -1;
      }
      (void) memset (shared_data, 0, sizeof (pvm_sd));
      shared_data->sym_type = type;
      (void) strncpy (shared_data->sym_name, sym_name, 32);
      break;
    default:
      errno = EINVAL;
      return -1;
  };

/* return success */
return 0;
}

/* Get the address of the server that arg->sym_name resides
   on and return it in arg->sym_addr (struct sockaddr_in).
   This must be done if the object was created by another
   process and we must discover which server has the object
   we are looking for (fill in the sym_addr). */
int __pvm_getaddr(pvm_t type, void *object)
{
int enable, status, __errno;
char comm_buff[BUF_LENGTH];
struct sockaddr_in client_sin, broadcast_sin, server_sin;
int server_sin_length;
struct timeval sel_timeout = { 0, 0 };

/* For select(2) */
fd_set read_fd_set;

/* One of each */
pvm_mutex *mutex = NULL;
pvm_cond * condvar = NULL;
pvm_sem *semaphore = NULL;
pvm_sd *shared_data = NULL;

  /* Set comm_buff to all zeros */
  (void) memset (&comm_buff, 0, BUF_LENGTH);

  /* Set broadcast_sin to all 0's */
  (void) memset (&broadcast_sin, 0, sizeof (struct sockaddr_in));

  /* Set client_sin to all 0's */
  (void) memset (&client_sin, 0, sizeof (struct sockaddr_in));

  /* Set server_sin to all 0's */
  (void) memset (&server_sin, 0, sizeof (struct sockaddr_in));

  /* Set the values for our struct sockaddr_in's */
  client_sin.sin_family = AF_INET; /* we're using the network, duh */
  client_sin.sin_port = 0;         /* any port will do */
  client_sin.sin_addr.s_addr = INADDR_ANY; /* default address for client */

  broadcast_sin.sin_family = AF_INET;
  broadcast_sin.sin_port = htons (UDP_SERVER_PORT); /* as in config.h, udp */
  broadcast_sin.sin_addr.s_addr = INADDR_BROADCAST; /* send to all hosts */

  __pvm_client_udp_sock = socket (AF_INET, SOCK_DGRAM, 0);
  if (__pvm_client_udp_sock < 0)
    return -1; /* extern int errno has been set by socket(2) */

  /* Allow broadcasts on this socket */
  enable = ENABLE;
  status = setsockopt (__pvm_client_udp_sock, SOL_SOCKET, SO_BROADCAST,
                                           &enable, sizeof(int));
  if (status) {
    (void) close (__pvm_client_udp_sock);
    return -1; /* extern int errno has been set by setsockopt(2) */
  }

  /* Bind to a local port that the system hands us */
  status = bind (__pvm_client_udp_sock, &client_sin, sizeof (struct sockaddr_in));
  if (status) {
    __errno = errno; /* Save the error code in the unlikely case
                        that close decides to fail. */
    (void) close (__pvm_client_udp_sock);
    errno = __errno; /* Restore */
    return -1;
  }

  switch (type) {
    case __PVM_MUTEX:
      mutex = (pvm_mutex *) object;
      if (mutex == NULL) {
        (void) close (__pvm_client_udp_sock);
        errno = EINVAL;
        return -1;
      }
      (void) snprintf (comm_buff, BUF_LENGTH, "%d:%s", type, mutex->sym_name);
      break;
    case __PVM_CV:
      condvar = (pvm_cond *) object;
      if (condvar == NULL) {
        (void) close (__pvm_client_udp_sock);
        errno = EINVAL;
        return -1;
      }
      (void) snprintf (comm_buff, BUF_LENGTH, "%d:%s", type, condvar->sym_name);
      break;
    case __PVM_SEM:
      semaphore = (pvm_sem *) object;
      if (semaphore == NULL) {
        (void) close (__pvm_client_udp_sock);
        errno = EINVAL;
        return -1;
      }
      (void) snprintf (comm_buff, BUF_LENGTH, "%d:%s", type,
                                                semaphore->sym_name);
      break;
    case __PVM_SD_INT:
    case __PVM_SD_FLOAT:
    case __PVM_SD_MEM:
      shared_data = (pvm_sd *) object;
      if (shared_data == NULL) {
        (void) close (__pvm_client_udp_sock);
        errno = EINVAL;
        return -1;
      }
      (void) snprintf (comm_buff, BUF_LENGTH, "%d:%s", type,
                                                shared_data->sym_name);
      break;
    default:
      errno = EINVAL;
      return -1;
  };

  status = sendto (__pvm_client_udp_sock, (void *) comm_buff,
    strlen (comm_buff), 0, &broadcast_sin, sizeof (struct sockaddr_in));
  if (status == -1) {
      __errno = errno; /* Save the error code in the unlikely case
                          that close decides to fail. */
      (void) close (__pvm_client_udp_sock);
      errno = __errno; /* Restore */
      return -1;
  }

  /* Set the timeout for select to one second, giving any client that
     already has an object of the same type with the same name time
     to object. */
  /* One second may seem long, but should be acceptable because mutex
     creation should happen before any processing begins, and as a
     result of reader preference for the server's rwlock, we may block
     during creation anyway. Mutex creation may take some time. */
  sel_timeout.tv_sec = 1;
  FD_ZERO (&read_fd_set);
  FD_SET  (__pvm_client_udp_sock, &read_fd_set);
  status = select (__MAX_FDS, &read_fd_set, NULL, NULL, &sel_timeout);
  if (status > 0) {

    server_sin_length = sizeof (struct sockaddr_in);
    status = recvfrom (__pvm_client_udp_sock, (void *) comm_buff,
      BUF_LENGTH, 0, &server_sin, &server_sin_length);
    if (status >= 0) {
      if (!strncmp (comm_buff, "EXISTS", (size_t) status)) {
        (void) close (__pvm_client_udp_sock);
        switch (type) {
          case __PVM_MUTEX:
            (void) memset (&mutex->sym_addr, 0,
                                         sizeof (struct sockaddr_in));
            mutex->sym_addr.sin_family = AF_INET;
            mutex->sym_addr.sin_port = htons (TCP_SERVER_PORT);
            mutex->sym_addr.sin_addr.s_addr = server_sin.sin_addr.s_addr;
            break;
          case __PVM_CV:
            (void) memset (&condvar->sym_addr, 0,
                                         sizeof (struct sockaddr_in));
            condvar->sym_addr.sin_family = AF_INET;
            condvar->sym_addr.sin_port = htons (TCP_SERVER_PORT);
            condvar->sym_addr.sin_addr.s_addr = server_sin.sin_addr.s_addr;
            break;
          case __PVM_SEM:
            (void) memset (&semaphore->sym_addr, 0,
                                         sizeof (struct sockaddr_in));
            semaphore->sym_addr.sin_family = AF_INET;
            semaphore->sym_addr.sin_port = htons (TCP_SERVER_PORT);
            semaphore->sym_addr.sin_addr.s_addr =
                                        server_sin.sin_addr.s_addr;
            break;
          case __PVM_SD_INT:
          case __PVM_SD_FLOAT:
          case __PVM_SD_MEM:
            (void) memset (&shared_data->sym_addr, 0,
                                         sizeof (struct sockaddr_in));
            shared_data->sym_addr.sin_family = AF_INET;
            shared_data->sym_addr.sin_port = htons (TCP_SERVER_PORT);
            shared_data->sym_addr.sin_addr.s_addr =
                                        server_sin.sin_addr.s_addr;
            break;
          default:
            errno = EINVAL;
            return -1;
        };
        return 0;
      } else {
        errno = EIO;
        return -1;
      }
    } else if ((status == -1) && (errno != EINTR)) {
      __errno = errno; /* Save the error code in the unlikely case
                          that close decides to fail. */
      (void) close (__pvm_client_udp_sock);
      errno = __errno; /* Restore */
      return -1;
    }
  } else if (status == -1) {
    __errno = errno; /* Save the error code in the unlikely case
                        that close decides to fail. */
    (void) close (__pvm_client_udp_sock);
    errno = __errno; /* Restore */
    return -1;
  }

/* We didn't get a response */
(void) close (__pvm_client_udp_sock);
errno = EINVAL;
return -1;
}


