/*
 *
 * $Id: pvm_cv.c,v 1.2.4.41 2000/03/16 17:14:01 ajp Exp $
 *
 * Andrew Pitman              pvm_cv.c
 *
 * pvmsync, a distributed synchronization server: condition
 * variable specific client routines for pvmsync.
 *
 * Server that accepts connections (requests) for shared
 * data or synchronization mechanisms continuously and
 * changes some internal state according to the request.
 *
 * Copyright (C) 1999, 2000 Andrew J. Pitman
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


#ifndef __PVM_CV_H
#include <pvm_cv.h>
#endif

#include <stdio.h>
#include <string.h>

#include <unistd.h>
#include <stdlib.h>
#include <time.h>

#ifndef __PVM_GEN_H
#include <pvm_gen.h>
#endif

#ifndef __LIB_INTERNALS_H
#include <lib_internals.h>
#endif

#ifndef __CONFIG_H
#include <config.h>
#endif


/* Sockets to use when contacting server(s) */
int __pvm_client_tcp_sock;
int __pvm_client_udp_sock;


/* These routines follow the convention of returning a positive
   error code instead of returning -1 and setting errno
   appropriately. */

/* Initialize a pvm_cond */
/* This pvm_cond will be instantiated on a server, and the
   pvm_cond->sym_addr will be filled in with the appropriate
   server address. */
/* Returns 0 on success, a positive error code on failure. */
int pvm_cond_init (pvm_cond *cv)
{
int __errno,
    retnval;

  /* Preserve previous state of errno */
  __errno = errno;

  if (__pvm_generic_init_sync (__PVM_CV, (void *) cv)) {
    retnval = errno;
    errno = __errno;
  } else {
    retnval = 0;
    errno = __errno;
  }

return retnval;
}

/* The given pvm_cond will be destroyed on the server it
   resides on (if possible), or an error returned. */
/* Returns 0 on success, a positive error code on failure. */
int pvm_cond_destroy (pvm_cond *cv)
{
int __errno,
    retnval;

  /* Preserve previous state of errno */
  __errno = errno;

  if (__pvm_generic_destroy_sync (__PVM_CV, (void *) cv)) {
    retnval = errno;
    errno = __errno;
  } else {
    retnval = 0;
    errno = __errno;
  }

return retnval;
}

/* Wait on the condvar.  cv must be initialized first,
   or if already created by another process, looked up with
   __pvm_getaddr. */
/* Returns 0 on success, a positive error code on failure. */
int pvm_cond_wait (pvm_cond *cv, pvm_mutex *mutex)
{
int __errno, retnval;

  /* Preserve previous state of errno */
  __errno = errno;

  retnval = pvm_mutex_unlock (mutex);
  if (retnval) {
    /* Restore errno */
    errno = __errno;
    return retnval;
  }

  /* Blocking lock on condvar */
  retnval = __pvm_cond_timedwait (cv, 0);
  if (retnval) {
    /* Restore errno */
    errno = __errno;
    return retnval;
  }

  retnval = __pvm_mutex_lock (mutex, 0);

  /* Restore errno */
  errno = __errno;

return retnval;
}

/* Wait on the condvar.  cv must be initialized first,
   or if already created by another process, looked up with
   __pvm_getaddr.  This is the timed version.... it will
   return with ETIMEDOUT if waiting on the condition varaible
   timed out. */
/* Returns 0 on success, a positive error code on failure. */
int pvm_cond_timedwait (pvm_cond *cv, pvm_mutex *mutex,
                                               unsigned int secs)
{
int __errno, retnval;

  /* Preserve previous state of errno */
  __errno = errno;

  retnval = pvm_mutex_unlock (mutex);
  if (retnval) {
    /* Restore errno */
    errno = __errno;
    return retnval;
  }
  
  /* Blocking lock on mutex */
  retnval = __pvm_cond_timedwait (cv, secs);
  if (retnval) {
    /* Restore errno */
    errno = __errno;
    return retnval;
  }

  retnval = __pvm_mutex_lock (mutex, 0);

  /* Restore errno */
  errno = __errno;

return retnval;
}

/* Signal the condition variable.  Awakens only one waiter. */
/* Returns 0 on success, a positive error code on failure. */
int pvm_cond_signal (pvm_cond *cv)
{
int __errno, retnval = 0;

  /* Preserve previous state of errno */
  __errno = errno;

  retnval = __pvm_cond_signal (cv, 0);

  /* Restore errno */
  errno = __errno;

return retnval;
}

/* Broadcast the condition variable.  Awakens all waiters. */
/* Returns 0 on success, a positive error code on failure. */
int pvm_cond_broadcast (pvm_cond *cv)
{
int __errno, retnval = 0;

  /* Preserve previous state of errno */
  __errno = errno;

  retnval = __pvm_cond_signal (cv, 0);

  /* Restore errno */
  errno = __errno;

return retnval;
}


