/*
 *
 * $Id: lib_internals.c,v 1.2.4.41 2000/03/16 17:13:53 ajp Exp $
 *
 * Andrew Pitman              lib_internals.c
 *
 * pvmsync, a distributed synchronization server:  internal
 * functions for the pvmsync client library.
 *
 * Server that accepts connections (requests) for shared
 * data or synchronization mechanisms continuously and
 * changes some internal state according to the request.
 *
 * Copyright (C) 1999, 2000 Andrew J. Pitman
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


#ifndef __LIB_INTERNALS_H
#include <lib_internals.h>
#endif

#ifndef __PVM_MUTEX_H
#include <pvm_mutex.h>
#endif

#ifndef __PVM_CV_H
#include <pvm_cv.h>
#endif

#ifndef __PVM_SEM_H
#include <pvm_sem.h>
#endif

#ifndef __PVM_SD_H
#include <pvm_sd.h>
#endif

#include <config.h>
/* struct udp_app_dgram */
#include <udp_daemon.h>

#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#include <stdarg.h>

/* rint (3) */
#include <math.h>


/* Address for the loopback interface */
struct in_addr * __pvm_loopback_in_addr;


/* Sockets to use when contacting server(s) */
extern int __pvm_client_tcp_sock;
extern int __pvm_client_udp_sock;


/* Keep track of a sequence number for transactions */
char *__pvm_seq;


/* Find out if obj_symbol, of type obj_type exists on the cluster */
int __pvm_symbol_exists (pvm_t obj_type, char *obj_symbol)
{
int enable, status, __errno;
int send_count;  /* Loop counter */
struct sockaddr_in client_sin, broadcast_sin;
struct timeval sel_timeout = { 0, 0 };

/* Datagram to send */
struct udp_app_dgram request,
                     reply;

/* For select(2) */
fd_set read_fd_set;

  /* Zero out dgrams request, reply */
  (void) memset (&request, 0, sizeof (struct udp_app_dgram));
  (void) memset (&reply,   0, sizeof (struct udp_app_dgram));

  /* Set broadcast_sin to all 0's */
  (void) memset (&broadcast_sin, 0, sizeof (struct sockaddr_in));

  /* Set client_sin to all 0's */
  (void) memset (&client_sin, 0, sizeof (struct sockaddr_in));

  /* Set the values for our struct sockaddr_in's */
  client_sin.sin_family = AF_INET;
  client_sin.sin_port = 0;                 /* any port will do */
  client_sin.sin_addr.s_addr = INADDR_ANY; /* default address for client */

  broadcast_sin.sin_family = AF_INET;
  broadcast_sin.sin_port = htons (UDP_SERVER_PORT); /* as in config.h, udp */
  broadcast_sin.sin_addr.s_addr = INADDR_BROADCAST; /* send to all hosts */

  __pvm_client_udp_sock = socket (AF_INET, SOCK_DGRAM, 0);
  if (__pvm_client_udp_sock < 0)
    return -1; /* extern int errno has been set by socket(2) */

  /* Allow broadcasts on this socket */
  enable = ENABLE;
  status = setsockopt (__pvm_client_udp_sock, SOL_SOCKET, SO_BROADCAST,
                                           &enable, sizeof(int));
  if (status) {
    (void) close (__pvm_client_udp_sock);
    return -1; /* extern int errno has been set by setsockopt(2) */
  }

  /* Bind to a local port that the system hands us */
  status = bind (__pvm_client_udp_sock, &client_sin,
                          sizeof (struct sockaddr_in));
  if (status) {
    __errno = errno; /* Save the error code in the unlikely case
                        that close decides to fail. */
    (void) close (__pvm_client_udp_sock);
    errno = __errno; /* Restore */
    return -1;
  }

  /* Here, we should craft our datagram according to the
     format in udp_daemon.h (udp_app_dgram) */
  request.magic    = UDP_MAGIC_NO;
  request.version  = PROTOCOL_VERS;
  request.obj_type = (char) obj_type;
  if (strlen (obj_symbol) % 4)
    request.length = (char) (2 + strlen (obj_symbol) / 4 + 1);
  else
    request.length = (char) (2 + strlen (obj_symbol) / 4);
  request.dgram_type = UDP_REQUEST;

  if (!__pvm_seq) {
    __pvm_seq = (char *) malloc (sizeof (char));
    if (!__pvm_seq) {
      /* Bad */
      errno = ENOMEM;
      return -1;
    }
    *__pvm_seq = 0;
  }

  /* Wrap around (don't overflow) */
  /* Might as well make them all positive */
  if (*__pvm_seq < 127) {
    request.seq_num = ++(*__pvm_seq);
  } else {
    request.seq_num = *__pvm_seq = 0;
  }
  /* Numeric argument is unused */

  (void) strncpy (request.dp, obj_symbol, PVM_MAX_SYMLEN);

  /* Loop, sending a datagram then select()ing for a timeout of
     2000 usecs (2 msec).  Repeat a handful of times before giving
     up or until a reply is received. */
  for (send_count = 0; send_count < 5; send_count++) {

    status = sendto (__pvm_client_udp_sock, (void *) &request,
                     request.length * 4, 0, &broadcast_sin,
                     sizeof (struct sockaddr_in));
    if (status == -1) {
      __errno = errno; /* Save the error code in the unlikely case
                          that close decides to fail. */
      (void) close (__pvm_client_udp_sock);
      errno = __errno; /* Restore */
      return -1;
    }

    /* Timeouts for select */
    /* Set here because this gets updated by select */
    sel_timeout.tv_sec = 0;
    sel_timeout.tv_usec = 2000; /* Rough estimate of the time it should
                                   take to get a response */
    /* Set up the fd set for select */
    /* Set here because this gets updated by select */
    FD_ZERO (&read_fd_set);
    FD_SET  (__pvm_client_udp_sock, &read_fd_set);
    status = select (__MAX_FDS, &read_fd_set, NULL, NULL, &sel_timeout);
    if (status > 0) {

      status = recvfrom (__pvm_client_udp_sock, (void *) &reply,
                         sizeof (struct udp_app_dgram), 0, NULL, NULL);
      if (status >= 0) {
        if ((vrfy_udp_reply (&reply)) &&
            (udp_objtype (&reply) == (char) obj_type) &&
            (udp_seqnum (&reply) == *__pvm_seq)) {
          /* Object exists */
          (void) close (__pvm_client_udp_sock);
          return 1;
        }
      } else if ((status == -1) && (errno != EINTR)) {
        __errno = errno; /* Save the error code in the unlikely case
                            that close decides to fail. */
        (void) close (__pvm_client_udp_sock);
        errno = __errno; /* Restore */
        return -1;
      }
    } else if (status == -1) {
      __errno = errno; /* Save the error code in the unlikely case
                          that close decides to fail. */
      (void) close (__pvm_client_udp_sock);
      errno = __errno; /* Restore */
      return -1;
    }
  } /* for */

/* We didn't get a response */
(void) close (__pvm_client_udp_sock);
return 0;
}

/* Find a suitable server to hold obj_type and fill in s_addr */
/* Returns -1 on error, zero otherwise */
int __pvm_find_available (pvm_t obj_type, struct sockaddr_in *s_addr,
                     size_t *s_addr_len)
{
int j, enable, status, __errno;
struct udp_app_dgram req_dgram;
struct sockaddr_in client_sin, broadcast_sin;
/* Number of reads pending on socket */
int num_reads;
struct timeval sel_timeout = { 0, 0 }, /* Select blocks this long */
               sav_timeout = { 0, 0 }, /* Saved timeval */
               cum_tv = { 0, 0 };      /* Keep track of how long
                                          we've been waiting for a
                                          response. */

struct __pvm_host_cand *head = NULL,         /* LL of candidates */
                   *curr_pckt;           /* The current one we're
                                            working on. */

/* For select(2) */
fd_set read_fd_set;

  /* Set datagram to all zeros */
  (void) memset (&req_dgram, 0, sizeof (struct udp_app_dgram));

  /* Set broadcast_sin to all 0's */
  (void) memset (&broadcast_sin, 0, sizeof (struct sockaddr_in));

  /* Set client_sin to all 0's */
  (void) memset (&client_sin, 0, sizeof (struct sockaddr_in));

  /* Set the values for our struct sockaddr_in's */
  client_sin.sin_family = AF_INET;
  client_sin.sin_port = 0;                 /* any port will do */
  client_sin.sin_addr.s_addr = INADDR_ANY; /* default address for client */

  broadcast_sin.sin_family = AF_INET;
  broadcast_sin.sin_port = htons (UDP_SERVER_PORT); /* as in config.h, udp */
  broadcast_sin.sin_addr.s_addr = INADDR_BROADCAST; /* send to all hosts */

  __pvm_client_udp_sock = socket (AF_INET, SOCK_DGRAM, 0);
  if (__pvm_client_udp_sock < 0)
    return -1; /* extern int errno has been set by socket(2) */

  /* Allow broadcasts on this socket */
  enable = ENABLE;
  status = setsockopt (__pvm_client_udp_sock, SOL_SOCKET, SO_BROADCAST,
                                           &enable, sizeof(int));
  if (status) {
    (void) close (__pvm_client_udp_sock);
    return -1; /* extern int errno has been set by setsockopt(2) */
  }

  /* Bind to a local port that the system hands us */
  status = bind (__pvm_client_udp_sock, &client_sin, sizeof (struct sockaddr_in));
  if (status) {
    __errno = errno; /* Save the error code in the unlikely case
                        that close decides to fail. */
    (void) close (__pvm_client_udp_sock);
    errno = __errno; /* Restore */
    return -1;
  }

  /* Craft request packet */
  /* Here, we should craft our datagram according to the
     format in udp_daemon.h (udp_app_dgram) */
  req_dgram.magic  = UDP_MAGIC_NO;
  req_dgram.version = PROTOCOL_VERS;
  req_dgram.obj_type = (char) __PVM_GET_LD;
  req_dgram.length = (char) 2;
  req_dgram.dgram_type = UDP_REQUEST;

  if (!__pvm_seq) {
    __pvm_seq = (char *) malloc (sizeof (char));
    if (!__pvm_seq) {
      /* Bad */
      errno = ENOMEM;
      return -1;
    }
    *__pvm_seq = 0;
  }

  /* Wrap around (don't overflow) */
  /* Might as well make them all positive */
  if (*__pvm_seq < 127) {
    req_dgram.seq_num = ++(*__pvm_seq);
  } else {
    req_dgram.seq_num = *__pvm_seq = 0;
  }
  /* Numeric argument is unused */

  /* Our packet is ready, now send it.  We'll loop
     and wait (exponential backoff) until we've waited
     too long.  In that case, return an error. */
  /* Exponential backoff in this case means squaring
     the previous timeout instead of just doubling it.
     Needless to say, this diverges (and reaches
     the limit for timeout values) quicker. */
  do {

    /* Send it */
    status = sendto (__pvm_client_udp_sock, (void *) &req_dgram,
                       8, 0, &broadcast_sin, sizeof (struct sockaddr_in));
    if (status == -1) {
      __errno = errno; /* Save the error code in the unlikely case
                          that close decides to fail. */
      (void) close (__pvm_client_udp_sock);
      errno = __errno; /* Restore */
      return -1;
    }

    /* Set timeout */
    __pvm_calc_timeout (&sav_timeout);
    /* Copy it to sel_timeout, which select will use (and update). */
    (void) memcpy (&sel_timeout, &sav_timeout,
                                   (size_t) sizeof (struct timeval));

    FD_ZERO (&read_fd_set);
    FD_SET  (__pvm_client_udp_sock, &read_fd_set);
    num_reads = select (__MAX_FDS, &read_fd_set, NULL, NULL, &sel_timeout);

    if (!num_reads) {
      __pvm_update_tv_sum (sav_timeout, &cum_tv);
      /* We'll just round to the nearest second */
      if ((cum_tv.tv_sec + (int) rint (cum_tv.tv_usec / 1000000.00))
                                             >= __MAX_CUM_TIMEOUT) {
        (void) close (__pvm_client_udp_sock);
        errno = ETIMEDOUT;
        return -1;
      }
    }

  } while (!num_reads);
  /* We've gotten a response, loop num_reads times, reading
     packets and entering them onto a singly-linked (unordered)
     list for processing if they're valid */

  for (j = 0; j < num_reads; j++) {

    curr_pckt = (struct __pvm_host_cand *)
                 malloc (sizeof (struct __pvm_host_cand));
    if (!curr_pckt) {
      errno = ENOMEM;
      return -1;
    }
    (void) memset (curr_pckt, 0, sizeof (struct __pvm_host_cand));

    status = recvfrom (__pvm_client_udp_sock, (void *) &curr_pckt->dgram,
                         sizeof (struct udp_app_dgram), 0, &curr_pckt->s_addr,
                         &curr_pckt->s_addr_len);

    /* Check the format of the incoming packet
       to see if the responding host has committed
       an atrocity. */
    if ((!vrfy_udp_reply (&curr_pckt->dgram)) ||
        (udp_objtype (&curr_pckt->dgram) != (char) __PVM_GET_LD) ||
        (udp_seqnum (&curr_pckt->dgram) != *__pvm_seq)) {

      /* Discard packet */
      free (curr_pckt);

    } else {

      /* Enter onto list */
      __pvm_add_to_list (curr_pckt, (void *) head);

    }
  }

  /* Find the best candidate server */
  curr_pckt = __pvm_find_best ((void *) head);
  if (!curr_pckt) {
    __pvm_destroy_list ((void *) head);
    errno = EAGAIN;
    return -1;
  }

  /* Copy into s_addr and s_addr_len */
  (void) memcpy (s_addr, &curr_pckt->s_addr, curr_pckt->s_addr_len);
  (void) memcpy (s_addr_len, &curr_pckt->s_addr_len, sizeof (size_t));

  /* Safe to destroy LL now */
  __pvm_destroy_list ((void *) head);

  /* We're done with the udp socket, close it */
  (void) close (__pvm_client_udp_sock);

return 0;
}

/* Create a symbol on the local server, if possible.  Return -1 if
   object cannot be created on the localhost. */
int __pvm_create_local (pvm_t obj_type, char * obj_name, ...)
{
int status;
char req_buff[BUF_LENGTH];
struct sockaddr_in local_sin;

/* Variable argument handling */
va_list args;
unsigned int arg_value;

  /* Set local_sin to all 0's */
  (void) memset (&local_sin, 0, sizeof (struct sockaddr_in));

  /* Set the value of __pvm_loopback_in_addr->sin_addr if
     not already initialized. */
  if (__pvm_loopback_in_addr == NULL) {
    __pvm_loopback_in_addr = (struct in_addr *)
                              malloc (sizeof (struct in_addr));
    if (__pvm_loopback_in_addr == NULL) {
      errno = ENOMEM;
      return -1;
    }
    status = inet_aton (__LOOPBACK_ADDR, __pvm_loopback_in_addr);
    if (!status) {
      (void) close (__pvm_client_tcp_sock);
      errno = EAGAIN;
      return -1;
    }
  }

  /* Fill in the family, address, and port for the localhost */
  local_sin.sin_family = AF_INET;
  local_sin.sin_addr.s_addr = __pvm_loopback_in_addr->s_addr;
  local_sin.sin_port = htons ((unsigned short int) TCP_SERVER_PORT);

  switch (obj_type) {
    case __PVM_MUTEX:
        (void) snprintf (req_buff, BUF_LENGTH, "CMT:%s", obj_name);
      break;
    case __PVM_CV:
        (void) snprintf (req_buff, BUF_LENGTH, "CCV:%s", obj_name);
      break;
    case __PVM_SEM:
        /* We'll possibly have one parameter beyond obj_name */
        va_start (args, obj_name);
        arg_value = va_arg (args, unsigned int);

        (void) snprintf (req_buff, BUF_LENGTH, "CSM:%s:%u", obj_name,
                                                          arg_value);
        va_end (args);
      break;
    case __PVM_SD_INT:
        (void) snprintf (req_buff, BUF_LENGTH, "CSD:INTEGER:%s", obj_name);
      break;
    case __PVM_SD_FLOAT:
        (void) snprintf (req_buff, BUF_LENGTH, "CSD:FLOAT:%s", obj_name);
      break;
    case __PVM_SD_MEM:
        /* We should have one parameter beyond obj_name */
        va_start (args, obj_name);
        arg_value = va_arg (args, unsigned int);

        (void) snprintf (req_buff, BUF_LENGTH, "CSD:MEM:%s:%u", obj_name,
                                                              arg_value);
        va_end (args);
      break;
    default:
        errno = EINVAL;
        return -1;
  };

/* Do the transaction */
return (__pvm_tcp_server_sendreq (req_buff,
        (size_t) strlen (req_buff),
        local_sin));
}

/* Lock/Trylock mutex */
/* Blocking lock if try is 0, nonblocking if try is 1 */
int __pvm_mutex_lock (pvm_mutex * mutex, int try)
{
char req_buff[BUF_LENGTH];

  (void) memset (&req_buff, 0, BUF_LENGTH);

  /* try indicates nonblocking or not (blocking) */
  if (!try)
    (void) snprintf (req_buff, BUF_LENGTH, "LCK:%s", mutex->sym_name);
  else
    (void) snprintf (req_buff, BUF_LENGTH, "TRL:%s", mutex->sym_name);

  /* We already have the server's address in mutex->sym_addr */
  if (__pvm_tcp_server_sendreq (req_buff, strlen (req_buff), mutex->sym_addr)) {
    /* We may have migrated to another node, look it up */
    if (__pvm_getaddr (__PVM_MUTEX, mutex)) {
      return EAGAIN;
    }
    /* Try it again */
    if (__pvm_tcp_server_sendreq (req_buff, strlen (req_buff), mutex->sym_addr))
      return errno;
  }

return 0;
}

/* Wait/Trywait semaphore */
/* Blocking wait if try is 0, nonblocking if try is 1 */
int __pvm_sem_wait (pvm_sem * sem, int try)
{
char req_buff[BUF_LENGTH];

  (void) memset (&req_buff, 0, BUF_LENGTH);

  /* Flag try indicates nonblocking or not (blocking) */
  if (!try)
    (void) snprintf (req_buff, BUF_LENGTH, "PSM:%s", sem->sym_name);
  else
    (void) snprintf (req_buff, BUF_LENGTH, "TPS:%s", sem->sym_name);

  /* We already have the server's address in sem->sym_addr */
  if (__pvm_tcp_server_sendreq (req_buff, strlen (req_buff), sem->sym_addr)) {
    /* We may have migrated to another node, look it up */
    if (__pvm_getaddr (__PVM_SEM, sem)) {
      errno = EAGAIN;
      return -1;
    }
    if (__pvm_tcp_server_sendreq (req_buff, strlen (req_buff), sem->sym_addr))
      /* errno is already set appropriately from the server's response */
      return -1;
  }

return 0;
}


/* Init shared objects of type int, float, and mem, respectively */

/* Initialize an integer object */
int __pvm_int_init (pvm_sd *sd)
{
int status, num_left;
int retnval = 0;
ssize_t comm_len;
char comm_buff[BUF_LENGTH];
char *retn_string = NULL;
struct sockaddr_in server_sin;
int server_sin_length;

  /* Set comm_buff to all zeros */
  (void) memset (&comm_buff, 0, BUF_LENGTH);

  /* Set server_sin to all 0's */
  (void) memset (&server_sin, 0, sizeof (struct sockaddr_in));

  status = __pvm_symbol_exists ((pvm_t) __PVM_SD_INT, sd->sym_name);
  if (status == -1) {
    if ((errno == ENOBUFS) || (errno == ENOMEM)) {
      (void) close (__pvm_client_udp_sock);
      errno = ENOMEM;
      return -1;
    } else {
      (void) close (__pvm_client_udp_sock);
      errno = EAGAIN;
      return -1;
    }
  } else if (status) {
      /* We'd be creating a duplicate, complain */
      (void) close (__pvm_client_udp_sock);
      errno = EBUSY;
      return -1;
  }

  /* If we're here, we didn't get a positive match above
     (we timed out) */
  (void) close (__pvm_client_udp_sock);

  /* First, try to create the integer locally.  Then, if
     this fails, find an available server. */

  status = __pvm_create_local ((pvm_t) __PVM_SD_INT, sd->sym_name);
  if (status != -1) {
    sd->sym_index = errno; /* kludge!!! */
    sd->sym_type = __PVM_SD_INT;
    status = inet_aton (__LOOPBACK_ADDR, &sd->sym_addr.sin_addr);
    if (!status) {
      sd->sym_index = -1;
      sd->sym_type = 0;
      (void) close (__pvm_client_udp_sock);
      errno = EAGAIN;
      return -1;
    }
    sd->sym_value = (void *) malloc (sizeof (long int));
    if (sd->sym_value == NULL) {
      sd->sym_index = -1;
      sd->sym_addr.sin_addr.s_addr = (unsigned int) 0;
      errno = ENOMEM;
      return -1;
    }
    sd->sym_addr.sin_family = AF_INET;
    sd->sym_addr.sin_port = htons (TCP_SERVER_PORT);
    sd->sym_size = (size_t) sizeof (long int);

    /* I know this is kludgy, but for now it's necessary */
    usleep ((unsigned long) 50);
    (void) __pvm_scalar_get (sd);
    /* Indicate success */
    return 0;
  } else {
    return -1;
  }

  /* No room on the local server, or connection refused */

  num_left = __pvm_find_available ((pvm_t) __PVM_SD_INT, &server_sin,
                                   &server_sin_length);
  if (num_left <= 0) {
    errno = EAGAIN;
    return -1;
  }

  /* We should have found a willing server at this point. */
  /* Connect to it and create an integer */
  __pvm_client_tcp_sock = socket (AF_INET, SOCK_STREAM, 0);
  if (__pvm_client_tcp_sock < 0) {
    if ((errno == ENFILE) || (errno == ENOBUFS) || (errno == ENOMEM)) {
      errno = ENOMEM;
      return -1;
    } else if ((errno == EACCES) || (errno == EPROTONOSUPPORT) ||
                                                    (errno == EINVAL)) {
      errno = EPERM;
      return -1;
    } else {
      errno = EAGAIN;
      return -1;
    }
  }

  /* We have the server's IP already in server_sin */
  server_sin.sin_port = htons (TCP_SERVER_PORT);

  status = connect (__pvm_client_tcp_sock, &server_sin, server_sin_length);
  if (status) {
    (void) close (__pvm_client_tcp_sock);
    errno = EAGAIN;
    return -1;
  }

  (void) snprintf (comm_buff, BUF_LENGTH, "CSD:INTEGER:%s", sd->sym_name);
  comm_len = write (__pvm_client_tcp_sock, comm_buff, strlen (comm_buff));
  if (comm_len == -1) {
    (void) close (__pvm_client_tcp_sock);
    errno = EAGAIN;
    return -1;
  }

  (void) memset (comm_buff, 0, BUF_LENGTH);

  comm_len = read (__pvm_client_tcp_sock, comm_buff, BUF_LENGTH);
  if ((!comm_len) || (comm_len < 0)) {
    errno = EAGAIN;
    retnval = -1;
  } else {
    if (!strncmp (comm_buff, "SUCCESS:", 8)) {
      memcpy (&sd->sym_addr, &server_sin, sizeof (struct sockaddr_in));
      retn_string = comm_buff+8;
      sd->sym_index = atoi (retn_string);
      sd->sym_size = sizeof (long int);
      sd->sym_type = __PVM_SD_INT;
    } else if (!strncmp (comm_buff, "FAIL:", 5)) {
      retn_string = comm_buff+5;
      errno = atoi (retn_string);
      retnval = -1;
    } else {
      errno = EIO;
      retnval = -1;
    }
  }

  /* Close the tcp socket */
  (void) close (__pvm_client_tcp_sock);

  /* I know this is kludgy, but for now it's necessary */
  usleep ((unsigned long) 50);
  (void) __pvm_scalar_get (sd);

return retnval;
}

/* Initialize a float object */
int __pvm_float_init (pvm_sd *sd)
{
int status, num_left;
int retnval = 0;
ssize_t comm_len;
char comm_buff[BUF_LENGTH];
char *retn_string = NULL;
struct sockaddr_in server_sin;
int server_sin_length;

  /* Set comm_buff to all zeros */
  (void) memset (&comm_buff, 0, BUF_LENGTH);

  /* Set server_sin to all 0's */
  (void) memset (&server_sin, 0, sizeof (struct sockaddr_in));

  status = __pvm_symbol_exists ((pvm_t) __PVM_SD_FLOAT, sd->sym_name);
  if (status == -1) {
    if ((errno == ENOBUFS) || (errno == ENOMEM)) {
      (void) close (__pvm_client_udp_sock);
      errno = ENOMEM;
      return -1;
    } else {
      (void) close (__pvm_client_udp_sock);
      errno = EAGAIN;
      return -1;
    }
  } else if (status) {
      /* We'd be creating a duplicate, complain */
      (void) close (__pvm_client_udp_sock);
      errno = EBUSY;
      return -1;
  }

  /* If we're here, we didn't get a positive match above
     (we timed out) */
  (void) close (__pvm_client_udp_sock);

  /* First, try to create the float locally.  Then, if
     this fails, find an available server. */

  status = __pvm_create_local ((pvm_t) __PVM_SD_FLOAT, sd->sym_name);
  if (status != -1) {
    sd->sym_index = errno; /* kludge!!! */
    sd->sym_type = __PVM_SD_FLOAT;
    status = inet_aton (__LOOPBACK_ADDR, &sd->sym_addr.sin_addr);
    if (!status) {
      sd->sym_index = -1;
      sd->sym_type = 0;
      (void) close (__pvm_client_udp_sock);
      errno = EAGAIN;
      return -1;
    }
    sd->sym_value = (void *) malloc (sizeof (pvm_float64_t));
    if (sd->sym_value == NULL) {
      sd->sym_index = -1;
      sd->sym_addr.sin_addr.s_addr = (unsigned int) 0;
      errno = ENOMEM;
      return -1;
    }
    sd->sym_addr.sin_family = AF_INET;
    sd->sym_addr.sin_port = htons (TCP_SERVER_PORT);
    sd->sym_size = (size_t) sizeof (pvm_float64_t);

    /* I know this is kludgy, but for now it's necessary */
    usleep ((unsigned long) 50);
    (void) __pvm_scalar_get (sd);
    /* Indicate success */
    return 0;
  } else {
    return -1;
  }

  /* No room on the local server, or connection refused */

  num_left = __pvm_find_available ((pvm_t) __PVM_SD_FLOAT, &server_sin,
                                   &server_sin_length);
  if (num_left <= 0) {
    errno = EAGAIN;
    return -1;
  }

  /* We should have found a willing server at this point. */
  /* Connect to it and create an integer */
  __pvm_client_tcp_sock = socket (AF_INET, SOCK_STREAM, 0);
  if (__pvm_client_tcp_sock < 0) {
    if ((errno == ENFILE) || (errno == ENOBUFS) || (errno == ENOMEM)) {
      errno = ENOMEM;
      return -1;
    } else if ((errno == EACCES) || (errno == EPROTONOSUPPORT) ||
                                                    (errno == EINVAL)) {
      errno = EPERM;
      return -1;
    } else {
      errno = EAGAIN;
      return -1;
    }
  }

  /* We have the server's IP already in server_sin */
  server_sin.sin_port = htons (TCP_SERVER_PORT);

  status = connect (__pvm_client_tcp_sock, &server_sin, server_sin_length);
  if (status) {
    (void) close (__pvm_client_tcp_sock);
    errno = EAGAIN;
    return -1;
  }

  (void) snprintf (comm_buff, BUF_LENGTH, "CSD:FLOAT:%s", sd->sym_name);
  comm_len = write (__pvm_client_tcp_sock, comm_buff, strlen (comm_buff));
  if (comm_len == -1) {
    (void) close (__pvm_client_tcp_sock);
    errno = EAGAIN;
    return -1;
  }

  (void) memset (comm_buff, 0, BUF_LENGTH);

  comm_len = read (__pvm_client_tcp_sock, comm_buff, BUF_LENGTH);
  if ((!comm_len) || (comm_len < 0)) {
    errno = EAGAIN;
    retnval = -1;
  } else {
    if (!strncmp (comm_buff, "SUCCESS:", 8)) {
      memcpy (&sd->sym_addr, &server_sin, sizeof (struct sockaddr_in));
      retn_string = comm_buff+8;
      sd->sym_index = atoi (retn_string);
      sd->sym_size = sizeof (pvm_float64_t);
      sd->sym_type = __PVM_SD_FLOAT;
    } else if (!strncmp (comm_buff, "FAIL:", 5)) {
      retn_string = comm_buff+5;
      errno = atoi (retn_string);
      retnval = -1;
    } else {
      errno = EIO;
      retnval = -1;
    }
  }

  /* Close the tcp socket */
  (void) close (__pvm_client_tcp_sock);

  /* I know this is kludgy, but for now it's necessary */
  usleep ((unsigned long) 50);
  (void) __pvm_scalar_get (sd);

return retnval;
}

/* This one takes the size of the block to create */
/* Initialize a memory block object */
int __pvm_mem_init (pvm_sd *sd, size_t size)
{
int status, num_left;
int retnval = 0;
ssize_t comm_len;
char comm_buff[BUF_LENGTH];
char *retn_string = NULL;
struct sockaddr_in server_sin;
int server_sin_length;

  /* Set comm_buff to all zeros */
  (void) memset (&comm_buff, 0, BUF_LENGTH);

  /* Set server_sin to all 0's */
  (void) memset (&server_sin, 0, sizeof (struct sockaddr_in));

  status = __pvm_symbol_exists ((pvm_t) __PVM_SD_MEM, sd->sym_name);
  if (status == -1) {
    if ((errno == ENOBUFS) || (errno == ENOMEM)) {
      (void) close (__pvm_client_udp_sock);
      errno = ENOMEM;
      return -1;
    } else {
      (void) close (__pvm_client_udp_sock);
      errno = EAGAIN;
      return -1;
    }
  } else if (status) {
      /* We'd be creating a duplicate, complain */
      (void) close (__pvm_client_udp_sock);
      errno = EBUSY;
      return -1;
  }

  /* If we're here, we didn't get a positive match above
     (we timed out) */
  (void) close (__pvm_client_udp_sock);

  /* First, try to create the memory block locally.  Then, if
     this fails, find an available server. */

  status = __pvm_create_local ((pvm_t) __PVM_SD_MEM, sd->sym_name,
                                           (unsigned int) size);
  if (status != -1) {
    sd->sym_index = errno; /* kludge!!! */
    sd->sym_type = __PVM_SD_MEM;
    status = inet_aton (__LOOPBACK_ADDR, &sd->sym_addr.sin_addr);
    if (!status) {
      sd->sym_index = -1;
      sd->sym_type = 0;
      (void) close (__pvm_client_udp_sock);
      errno = EAGAIN;
      return -1;
    }
    sd->sym_value = (void *) malloc (size);
    if (sd->sym_value == NULL) {
      sd->sym_index = -1;
      sd->sym_addr.sin_addr.s_addr = 0;
      (void) close (__pvm_client_udp_sock);
      errno = EAGAIN;
      return -1;
    }
    sd->sym_addr.sin_family = AF_INET;
    sd->sym_addr.sin_port = htons (TCP_SERVER_PORT);
    sd->sym_size = size;

    /* I know this is kludgy, but for now it's necessary */
    usleep ((unsigned long) 50);
    (void) __pvm_mem_get (sd);
    /* Indicate success */
    return 0;
  } else {
    return -1;
  }

  /* No room on the local server, or connection refused */

  num_left = __pvm_find_available ((pvm_t) __PVM_SD_MEM, &server_sin,
                              &server_sin_length);
  if (num_left <= 0) {
    errno = EAGAIN;
    return -1;
  }

  /* We should have found a willing server at this point. */
  /* Connect to it and create an integer */
  __pvm_client_tcp_sock = socket (AF_INET, SOCK_STREAM, 0);
  if (__pvm_client_tcp_sock < 0) {
    if ((errno == ENFILE) || (errno == ENOBUFS) || (errno == ENOMEM)) {
      errno = ENOMEM;
      return -1;
    } else if ((errno == EACCES) || (errno == EPROTONOSUPPORT) ||
                                                    (errno == EINVAL)) {
      errno = EPERM;
      return -1;
    } else {
      errno = EAGAIN;
      return -1;
    }
  }

  /* We have the server's IP already in server_sin */
  server_sin.sin_port = htons (TCP_SERVER_PORT);

  status = connect (__pvm_client_tcp_sock, &server_sin, server_sin_length);
  if (status) {
    (void) close (__pvm_client_tcp_sock);
    errno = EAGAIN;
    return -1;
  }

  (void) snprintf (comm_buff, BUF_LENGTH, "CSD:MEM:%s:%u", sd->sym_name,
                                                     (unsigned int) size);
  comm_len = write (__pvm_client_tcp_sock, comm_buff, strlen (comm_buff));
  if (comm_len == -1) {
    (void) close (__pvm_client_tcp_sock);
    errno = EAGAIN;
    return -1;
  }

  (void) memset (comm_buff, 0, BUF_LENGTH);

  comm_len = read (__pvm_client_tcp_sock, comm_buff, BUF_LENGTH);
  if ((!comm_len) || (comm_len < 0)) {
    errno = EAGAIN;
    retnval = -1;
  } else {
    if (!strncmp (comm_buff, "SUCCESS:", 8)) {
      sd->sym_value = (void *) malloc (size);
      if (sd->sym_value == NULL) {
        errno = ENOMEM;
        retnval = -1;
      }
      memcpy (&sd->sym_addr, &server_sin, sizeof (struct sockaddr_in));
      retn_string = comm_buff+8;
      sd->sym_index = atoi (retn_string);
      sd->sym_size = size;
    } else if (!strncmp (comm_buff, "FAIL:", 5)) {
      retn_string = comm_buff+5;
      errno = atoi (retn_string);
      retnval = -1;
    } else {
      errno = EIO;
      retnval = -1;
    }
  }

  /* Close the tcp socket */
  (void) close (__pvm_client_tcp_sock);

  /* I know this is kludgy, but for now it's necessary */
  usleep ((unsigned long) 50);
  (void) __pvm_mem_get (sd);

return retnval;
}

/* Get the value of the appropriate shared object and fill in
   sd->sym_value to length sd->sym_size */
/* Get the value of an integer or float object */
int __pvm_scalar_get (pvm_sd *sd)
{
int status, retnval = 0;
ssize_t comm_len;
char comm_buff[BUF_LENGTH];
char *retn_string = NULL;

/* sd->sym_value will be cast to one of these */
pvm_int64_t *intptr;
pvm_float64_t *floatptr;

  /* Initialize the sym_value pointer if it hasn't been already */
  if (sd->sym_value == NULL) {
    switch (sd->sym_type) {
      case __PVM_SD_INT:
        sd->sym_size = sizeof (pvm_int64_t);
        sd->sym_value = (void *) malloc (sd->sym_size);
        if (sd->sym_value == NULL) {
          (void) close (__pvm_client_tcp_sock);
          errno = ENOMEM;
          return -1;
        }
        break;
      case __PVM_SD_FLOAT:
        sd->sym_size = sizeof (pvm_float64_t);
        sd->sym_value = (void *) malloc (sd->sym_size);
        if (sd->sym_value == NULL) {
          (void) close (__pvm_client_tcp_sock);
          errno = ENOMEM;
          return -1;
        }
        break;
      default:
        (void) close (__pvm_client_tcp_sock);
        errno = EINVAL;
        return -1;
    };
  }

  /* Set comm_buff to all zeros */
  (void) memset (&comm_buff, 0, BUF_LENGTH);

  /* We already have the server's address in sd->sym_addr */
  /* Connect to it and get its value */
  __pvm_client_tcp_sock = socket (AF_INET, SOCK_STREAM, 0);
  if (__pvm_client_tcp_sock < 0) {
    errno = EAGAIN;
    return -1;
  }

  status = connect (__pvm_client_tcp_sock, &sd->sym_addr,
                               sizeof (struct sockaddr_in));

  if (status) {
    if (__pvm_getaddr (sd->sym_type, sd)) {
      (void) close (__pvm_client_tcp_sock);
      errno = EAGAIN;
      return -1;
    }

    status = connect (__pvm_client_tcp_sock, &sd->sym_addr,
                               sizeof (struct sockaddr_in));
    if (status) {
      (void) close (__pvm_client_tcp_sock);
      errno = EAGAIN;
      return -1;
    }
  }

  switch (sd->sym_type) {
    case __PVM_SD_INT:
      (void) snprintf (comm_buff, BUF_LENGTH, "GSD:INTEGER:%d",
                                                sd->sym_index);
      break;
    case __PVM_SD_FLOAT:
      (void) snprintf (comm_buff, BUF_LENGTH, "GSD:FLOAT:%d",
                                                sd->sym_index);
      break;
    default:
      (void) close (__pvm_client_tcp_sock);
      errno = EINVAL;
      return -1;
  };

  comm_len = write (__pvm_client_tcp_sock, comm_buff, strlen (comm_buff));
  if (comm_len == -1) {
    (void) close (__pvm_client_tcp_sock);
    errno = EAGAIN;
    return -1;
  }

  (void) memset (comm_buff, 0, BUF_LENGTH);

  comm_len = read (__pvm_client_tcp_sock, comm_buff, BUF_LENGTH);
  if ((!comm_len) || (comm_len < 0)) {
    errno =  EAGAIN;
    retnval = -1;
  } else {
    if (!strncmp (comm_buff, "SUCCESS:", 8)) {
      retn_string = comm_buff+8;
      switch (sd->sym_type) {
        case __PVM_SD_INT:
          intptr = (pvm_int64_t *) sd->sym_value;
          *intptr = strtol (retn_string, (char **) NULL, 10);
          break;
        case __PVM_SD_FLOAT:
          floatptr = (pvm_float64_t *) sd->sym_value;
          *floatptr = strtod (retn_string, (char **) NULL);
          break;
        default:
          (void) close (__pvm_client_tcp_sock);
          errno = EINVAL;
          return -1;
      };
    } else if (!strncmp (comm_buff, "FAIL:", 5)) {
      retn_string = comm_buff+5;
      errno = atoi (retn_string);
      retnval = -1;
    } else {
      errno = EIO;
      retnval = -1;
    }
  }

  /* Close the tcp socket */
  (void) close (__pvm_client_tcp_sock);

return retnval;
}

/* Get the value of a mem object */
int __pvm_mem_get (pvm_sd *sd)
{
int status, retnval;
ssize_t comm_len, blck_len;
char comm_buff[BUF_LENGTH];
char *retn_string = NULL;

  /* Check to make sure we have a pvm_sd of type __PVM_SD_MEM */
  if (sd->sym_type != __PVM_SD_MEM) {
    errno = EINVAL;
    return -1;
  }

  /* Set comm_buff to all zeros */
  (void) memset (&comm_buff, 0, BUF_LENGTH);

  /* We already have the server's address in sd->sym_addr */
  /* Connect to it and set its value */
  __pvm_client_tcp_sock = socket (AF_INET, SOCK_STREAM, 0);
  if (__pvm_client_tcp_sock < 0) {
    errno = EAGAIN;
    return -1;
  }

  status = connect (__pvm_client_tcp_sock, &sd->sym_addr,
                               sizeof (struct sockaddr_in));

  if (status) {
    if (__pvm_getaddr (__PVM_SD_MEM, sd)) {
      (void) close (__pvm_client_tcp_sock);
      errno = EAGAIN;
      return -1;
    }

    status = connect (__pvm_client_tcp_sock, &sd->sym_addr,
                               sizeof (struct sockaddr_in));
    if (status) {
      (void) close (__pvm_client_tcp_sock);
      errno = EAGAIN;
      return -1;
    }
  }

  (void) snprintf (comm_buff, BUF_LENGTH, "GSD:MEM:%d", sd->sym_index);

  comm_len = write (__pvm_client_tcp_sock, comm_buff, strlen (comm_buff));
  if (comm_len == -1) {
    (void) close (__pvm_client_tcp_sock);
    errno = EAGAIN;
    return -1;
  }

  (void) memset (comm_buff, 0, BUF_LENGTH);

  comm_len = read (__pvm_client_tcp_sock, comm_buff, BUF_LENGTH);
  if ((!comm_len) || (comm_len < 0)) {
    errno =  EAGAIN;
    retnval = -1;
  } else {
    if (!strncmp (comm_buff, "SUCCESS:", 8)) {
      retn_string = comm_buff+8;
      blck_len = atoi (retn_string);
      comm_len = read (__pvm_client_tcp_sock, sd->sym_value, blck_len);
      if ((!comm_len) || (comm_len < 0)) {
        errno =  EAGAIN;
        retnval = -1;
      } else {
        retnval = 0;
      }
    } else if (!strncmp (comm_buff, "FAIL:", 5)) {
      retn_string = comm_buff+5;
      errno = atoi (retn_string);
      retnval = -1;
    } else {
      errno = EIO;
      retnval = -1;
    }
  }

  /* Close the tcp socket */
  (void) close (__pvm_client_tcp_sock);

return retnval;
}

/* Set the value of the appropriate shared object */ 
/* Set the value of an integer or float object */
int __pvm_scalar_set (pvm_sd *sd)
{
int status, retnval;
ssize_t comm_len;
char comm_buff[BUF_LENGTH];
char *err_string = NULL;

/* sd->sym_value will be cast to one of these */
pvm_int64_t *intptr;
pvm_float64_t *floatptr;

  /* Set comm_buff to all zeros */
  (void) memset (&comm_buff, 0, BUF_LENGTH);

  /* We already have the server's address in sd->sym_addr */
  /* Connect to it and set its value */
  __pvm_client_tcp_sock = socket (AF_INET, SOCK_STREAM, 0);
  if (__pvm_client_tcp_sock < 0) {
    errno = EAGAIN;
    return -1;
  }

  status = connect (__pvm_client_tcp_sock, &sd->sym_addr,
                               sizeof (struct sockaddr_in));

  if (status) {
    if (__pvm_getaddr (sd->sym_type, sd)) {
      (void) close (__pvm_client_tcp_sock);
      errno = EAGAIN;
      return -1;
    }

    status = connect (__pvm_client_tcp_sock, &sd->sym_addr,
                               sizeof (struct sockaddr_in));
    if (status) {
      (void) close (__pvm_client_tcp_sock);
      errno = EAGAIN;
      return -1;
    }
  }

  switch (sd->sym_type) {
    case __PVM_SD_INT:
      intptr = (pvm_int64_t *) sd->sym_value;
      (void) snprintf (comm_buff, BUF_LENGTH, "SSD:INTEGER:%d:%Ld",
                                           sd->sym_index, *intptr);
      break;
    case __PVM_SD_FLOAT:
      floatptr = (pvm_float64_t *) sd->sym_value;
      (void) snprintf (comm_buff, BUF_LENGTH, "SSD:FLOAT:%d:%f",
                                      sd->sym_index, *floatptr);
      break;
    default:
      (void) close (__pvm_client_tcp_sock);
      errno = EINVAL;
      return -1;
  };

  comm_len = write (__pvm_client_tcp_sock, comm_buff, strlen (comm_buff));
  if (comm_len == -1) {
    (void) close (__pvm_client_tcp_sock);
    errno = EAGAIN;
    return -1;
  }

  (void) memset (comm_buff, 0, BUF_LENGTH);

  comm_len = read (__pvm_client_tcp_sock, comm_buff, BUF_LENGTH);
  if ((!comm_len) || (comm_len < 0)) {
    errno =  EAGAIN;
    retnval = -1;
  } else {
    if (!strncmp (comm_buff, "SUCCESS:", 8)) {
      /* We succeeded in setting its value */
      retnval = 0;
    } else if (!strncmp (comm_buff, "FAIL:", 5)) {
      err_string = comm_buff+5;
      errno = atoi (err_string);
      retnval = -1;
    } else {
      errno = EIO;
      retnval = -1;
    }
  }

  /* Close the tcp socket */
  (void) close (__pvm_client_tcp_sock);

return retnval;
}

/* Set the value of a mem object */
int __pvm_mem_set (pvm_sd *sd)
{
int status, retnval;
ssize_t comm_len;
char comm_buff[BUF_LENGTH];
char *err_string = NULL;

  /* Check to make sure we have a pvm_sd of type __PVM_SD_MEM */
  if (sd->sym_type != __PVM_SD_MEM) {
    errno = EINVAL;
    return -1;
  }

  /* Set comm_buff to all zeros */
  (void) memset (&comm_buff, 0, BUF_LENGTH);

  /* We already have the server's address in sd->sym_addr */
  /* Connect to it and set its value */
  __pvm_client_tcp_sock = socket (AF_INET, SOCK_STREAM, 0);
  if (__pvm_client_tcp_sock < 0) {
    errno = EAGAIN;
    return -1;
  }

  status = connect (__pvm_client_tcp_sock, &sd->sym_addr,
                               sizeof (struct sockaddr_in));

  if (status) {
    if (__pvm_getaddr (__PVM_SD_MEM, sd)) {
      (void) close (__pvm_client_tcp_sock);
      errno = EAGAIN;
      return -1;
    }

    status = connect (__pvm_client_tcp_sock, &sd->sym_addr,
                               sizeof (struct sockaddr_in));
    if (status) {
      (void) close (__pvm_client_tcp_sock);
      errno = EAGAIN;
      return -1;
    }
  }

  (void) snprintf (comm_buff, BUF_LENGTH, "SSD:MEM:%d:%d", sd->sym_index,
                                                             sd->sym_size);

  comm_len = write (__pvm_client_tcp_sock, comm_buff, strlen (comm_buff));
  if (comm_len == -1) {
    (void) close (__pvm_client_tcp_sock);
    errno = EAGAIN;
    return -1;
  }

  (void) memset (comm_buff, 0, BUF_LENGTH);

  comm_len = write (__pvm_client_tcp_sock, sd->sym_value, sd->sym_size);
  if (comm_len == -1) {
    (void) close (__pvm_client_tcp_sock);
    errno = EAGAIN;
    return -1;
  }

  comm_len = read (__pvm_client_tcp_sock, comm_buff, BUF_LENGTH);
  if ((!comm_len) || (comm_len < 0)) {
    errno =  EAGAIN;
    retnval = -1;
  } else {
    if (!strncmp (comm_buff, "SUCCESS:", 8)) {
      retnval = 0;
    } else if (!strncmp (comm_buff, "FAIL:", 5)) {
      err_string = comm_buff+5;
      errno = atoi (err_string);
      retnval = -1;
    } else {
      errno = EIO;
      retnval = -1;
    }
  }

  /* Close the tcp socket */
  (void) close (__pvm_client_tcp_sock);

return retnval;
}

/* Wait on condition variable cv until timer (unsigned int secs)
   goes off, or indefinitely if secs == 0 */
int __pvm_cond_timedwait (pvm_cond *cv, unsigned int secs)
{
char req_buff[BUF_LENGTH];

  (void) memset (&req_buff, 0, BUF_LENGTH);

  if (secs) {
    (void) snprintf (req_buff, BUF_LENGTH, "TWC:%s:%u", cv->sym_name, secs);
  } else {
    (void) snprintf (req_buff, BUF_LENGTH, "WCV:%s", cv->sym_name);
  }

  /* We already have the server's address in cv->sym_addr */
  if (__pvm_tcp_server_sendreq (req_buff, strlen (req_buff), cv->sym_addr)) {
    /* We may have migrated to another node, look it up */
    if (__pvm_getaddr (__PVM_CV, cv)) {
      return EAGAIN;
    }
    if (__pvm_tcp_server_sendreq (req_buff, strlen (req_buff), cv->sym_addr))
      return errno;
  }

return 0;
}

/* Signal condition varaible */
/* Broadcast if int broadcast is nonzero. */
int __pvm_cond_signal (pvm_cond *cv, int broadcast)
{
char req_buff[BUF_LENGTH];

  (void) memset (&req_buff, 0, BUF_LENGTH);

  if (broadcast) {
    (void) snprintf (req_buff, BUF_LENGTH, "BCV:%s", cv->sym_name);
  } else {
    (void) snprintf (req_buff, BUF_LENGTH, "SCV:%s", cv->sym_name);
  }

  /* We already have the server's address in cv->sym_addr */
  if (__pvm_tcp_server_sendreq (req_buff, strlen (req_buff), cv->sym_addr)) {
    /* We may have migrated to another node, look it up */
    if (__pvm_getaddr (__PVM_CV, cv)) {
      return EAGAIN;
    }
    if (__pvm_tcp_server_sendreq (req_buff, strlen (req_buff), cv->sym_addr))
      return errno;
  }

return 0;
}


/* Add contents of struct timeval itv to sum stv.... */
/* Keep a running tally of time given in struct timeval's */
inline void __pvm_update_tv_sum (struct timeval itv, struct timeval *stv)
{

  /* No test for overflow */
  stv->tv_sec += itv.tv_sec;

  /* Add two usec elements.  If the sum is greater than 1,000,000,
     increment sec element and new usec is sum % 1,000,000 */
  stv->tv_usec += itv.tv_usec;
  if (stv->tv_usec > 1000000) {
    stv->tv_sec++;
    stv->tv_usec %= 1000000;
  }

}


/* Add element to LL list */
void __pvm_add_to_list (struct __pvm_host_cand *element, void *list)
{
struct __pvm_host_cand *curr = NULL;

  if (!list) {
    list = (void *) element;
    curr = (struct __pvm_host_cand *) list;
  } else {
    curr = (struct __pvm_host_cand *) list;
    while (curr->next)
      curr = curr->next;
    /* Tack onto end */
    curr->next = element;
    /* Advance, one more time */
    curr = curr->next;
  }

  /* We're at the end */
  curr->next = NULL;

}

/* Destroy the whole thing */
void __pvm_destroy_list (void *list)
{
struct __pvm_host_cand *curr = NULL, *prev = NULL;

  curr = (struct __pvm_host_cand *) list;
  while (curr) {
    prev = curr;
    curr = curr->next;
    free (prev);
  };

}

/* Return a pointer to the entry for the best candidate,
   given list.... NULL on failure.  Values must be
   extracted from element before __pvm_destroy_list is called!!!! */
struct __pvm_host_cand *  __pvm_find_best (void *list)
{
/* Keep track of the current element as well as the
   lowest entry we've found so far. */
struct __pvm_host_cand *curr = NULL, *lowest = NULL;
int16_t low_val = 0;

  curr = (struct __pvm_host_cand *) list;
  lowest = curr;
  /* Packets are already checked for sanity by __pvm_find_available
     before we get them. */
  if (curr) {
    low_val = ntohs (curr->dgram.num_arg);
    curr = curr->next;
  }

  while (curr) {
    if (ntohs (curr->dgram.num_arg) < low_val) {
      lowest = curr;
      low_val = ntohs (curr->dgram.num_arg);
      curr = curr->next;  /* Advance ptr */
    }
  };

/* May be NULL if list has no elements */
return lowest;
}

/* Calculate square */
inline int __pvm_sqr (int val)
{
  return (int) val*val;
}

/* Calculate the next timeout value.  Increases exponenitally.
   Starts at 2 msec, then increases to 2 seconds. */
/* timeval passed initially should = { 0, 0 } */
inline void __pvm_calc_timeout (struct timeval *tv)
{
int msec = 0;

  if ((!tv->tv_sec) && (!tv->tv_usec)) {
    tv->tv_usec = 2000;
  } else if (!tv->tv_sec) {
    msec = tv->tv_usec / 1000;  /* Convert to milliseconds */
    msec = __pvm_sqr (msec);        /* Increase to */
    tv->tv_usec = msec * 1000;  /* Convert back */
    if (tv->tv_usec > 1000000) {
      tv->tv_sec = 1;
      tv->tv_usec = 0;
    }
  } else {
    if (tv->tv_sec == 1)
      tv->tv_sec++;
  }
  /* tv->tv_sec == 2 remains unchanged */

}

/* Sanitize part (or all) of a variable name */
inline void __pvm_sanitize_sym (char * varname)
{
int k, vstr_len;

  vstr_len = strlen (varname);
  /* Sanitize */
  for (k = 0; k < vstr_len; k++) {
    if ((varname[k] < '0') ||
       ((varname[k] > '9') && (varname[k] < 'A')) ||
       ((varname[k] > 'Z') && (varname[k] < '_')) ||
       ((varname[k] > '_') && (varname[k] < 'a')) ||
       (varname[k] > 'z')) {
         varname[k] = '_';
    }
  }
}

/* Generate first character of symbol name, since objects
   are placed in hash tables indexed by 1st char.  We want
   to mix things up a bit.... */
inline char __pvm_gen_initchar ()
{
char initial;    /* Initial character to return */
int gen_ok = 0;  /* Character generated OK? */

  do {
    initial = (char) (abs (rand ()) % 58) + 65;
    switch (initial) {
      case 'A' ... 'Z':
        gen_ok++;
        break;
      case '_':
        gen_ok++;
        break;
      case 'a' ... 'z':
        gen_ok++;
        break;
    };
  } while (!gen_ok);

return initial;
}

/* Generic init function for synchronization primitives */
/* Return 0 on success, -1 on failure (set errno) */
int __pvm_generic_init_sync (pvm_t obj_type, void *obj_ptr, ...)
{
/* Variable argument handling */
va_list args;
unsigned int sem_ctr_init = 0;

int status;
int __errno;

char initial; /* Randomly generated */
char comm_buff[BUF_LENGTH];
char *hostname_fqdn, *hostname;
struct sockaddr_in server_sin;
int server_sin_length;

unsigned int rand_int;

  /* Set server_sin to all 0's */
  (void) memset (&server_sin, 0, sizeof (struct sockaddr_in));

  /* Zero out the symbol's (object's) name */
  switch (obj_type) {
    case __PVM_MUTEX:
      (void) memset (((pvm_mutex *) obj_ptr)->sym_name, 0, PVM_MAX_SYMLEN);
    break;
    case __PVM_SEM:
      (void) memset (((pvm_sem *) obj_ptr)->sym_name, 0, PVM_MAX_SYMLEN);
      /* Do the stdarg stuff while we're at it */
      va_start (args, obj_ptr);
      sem_ctr_init = va_arg (args, unsigned int);
      va_end (args);
    break;
    case __PVM_CV:
      (void) memset (((pvm_cond *) obj_ptr)->sym_name, 0, PVM_MAX_SYMLEN);
    break;
    default:
      /* Error */
      errno = EINVAL;
      return -1;
  };

  /* Set sym_name according to some rules */
  /* First, generate the initial character */
  initial = __pvm_gen_initchar ();

  hostname_fqdn = (char *) malloc (MAXHOSTNAMELEN);
  if (!hostname_fqdn) {
      (void) close (__pvm_client_udp_sock);
      errno = ENOMEM;
      return -1;
  }
  (void) memset (hostname_fqdn, 0, MAXHOSTNAMELEN);

  status = gethostname (hostname_fqdn, MAXHOSTNAMELEN);
  if (status) {
      (void) close (__pvm_client_udp_sock);
      errno = EAGAIN;
      return -1;
  }

  hostname = strtok (hostname_fqdn, ".");
  /* Sanitize */
  __pvm_sanitize_sym (hostname);

  /* This is not foolproof, there is a possibility (albeit slight)
     that another symbol of the same type with the same name could
     get created, as long as it happens in the same second. */
  srand ((unsigned int) time (NULL));
  rand_int = (unsigned int) abs (rand ());

  /* Adding the time of day to the string probably wouldn't add any
     uniqueness if the pseudorandom number generator is really lame. */
  switch (obj_type) {
    case __PVM_MUTEX:
      (void) snprintf (((pvm_mutex *) obj_ptr)->sym_name,
                       PVM_MAX_SYMLEN, "%c%.8sMUT%u", initial,
                       hostname, rand_int);
    break;
    case __PVM_SEM:
      (void) snprintf (((pvm_sem *) obj_ptr)->sym_name,
                       PVM_MAX_SYMLEN, "%c%.8sSEM%u", initial,
                       hostname, rand_int);
    break;
    case __PVM_CV:
      (void) snprintf (((pvm_cond *) obj_ptr)->sym_name,
                       PVM_MAX_SYMLEN, "%c%.8sCND%u", initial,
                       hostname, rand_int);
    break;
  };
  free (hostname_fqdn);

  /* Find out if symbol already exists */
  switch (obj_type) {
    case __PVM_MUTEX:
      status = __pvm_symbol_exists ((pvm_t) __PVM_MUTEX,
                                ((pvm_mutex *) obj_ptr)->sym_name);
    break;
    case __PVM_SEM:
      status = __pvm_symbol_exists ((pvm_t) __PVM_SEM,
                                ((pvm_sem *) obj_ptr)->sym_name);
    break;
    case __PVM_CV:
      status = __pvm_symbol_exists ((pvm_t) __PVM_CV,
                                ((pvm_cond *) obj_ptr)->sym_name);
    break;
  };

  /* Check status of last operation */
  if (status == -1) {
    if ((errno == ENOBUFS) || (errno == ENOMEM)) {
      (void) close (__pvm_client_udp_sock);
      errno = ENOMEM;
      return  -1;
    } else {
      (void) close (__pvm_client_udp_sock);
      errno = EAGAIN;
      return -1;
    }
  } else if (status) {
    /* We'd be creating a duplicate, complain */
    (void) close (__pvm_client_udp_sock);
    errno = EBUSY;
    return -1;
  }

  /* If we're here, we didn't get a positive match above
     (we timed out) */

  /* First, try to create the object locally.  Then, if
     this fails, find an available server. */

  switch (obj_type) {
    case __PVM_MUTEX:
      status = __pvm_create_local ((pvm_t) __PVM_MUTEX,
                                   ((pvm_mutex *) obj_ptr)->sym_name);
    break;
    case __PVM_SEM:
      status = __pvm_create_local ((pvm_t) __PVM_SEM,
                                   ((pvm_sem *) obj_ptr)->sym_name,
                                   sem_ctr_init);
    break;
    case __PVM_CV:
      status = __pvm_create_local ((pvm_t) __PVM_CV,
                                   ((pvm_cond *) obj_ptr)->sym_name);
    break;
  };

  if (status != -1) {
    switch (obj_type) {
      case __PVM_MUTEX:
        status = inet_aton (__LOOPBACK_ADDR,
                            &((pvm_mutex *) obj_ptr)->sym_addr.sin_addr);
      break;
      case __PVM_SEM:
        status = inet_aton (__LOOPBACK_ADDR,
                            &((pvm_sem *) obj_ptr)->sym_addr.sin_addr);
      break;
      case __PVM_CV:
        status = inet_aton (__LOOPBACK_ADDR,
                            &((pvm_cond *) obj_ptr)->sym_addr.sin_addr);
      break;
    };

    /* I know this is kludgy, but for now it's necessary */
    usleep ((unsigned long) 50);

    if (!status) {
      (void) close (__pvm_client_udp_sock);
      errno = EAGAIN;
      return -1;
    }

    switch (obj_type) {
      case __PVM_MUTEX:
        ((pvm_mutex *) obj_ptr)->sym_addr.sin_family = AF_INET;
        ((pvm_mutex *) obj_ptr)->sym_addr.sin_port = htons (TCP_SERVER_PORT);
      break;
      case __PVM_SEM:
        ((pvm_sem *) obj_ptr)->sym_addr.sin_family = AF_INET;
        ((pvm_sem *) obj_ptr)->sym_addr.sin_port = htons (TCP_SERVER_PORT);
      break;
      case __PVM_CV:
        ((pvm_cond *) obj_ptr)->sym_addr.sin_family = AF_INET;
        ((pvm_cond *) obj_ptr)->sym_addr.sin_port = htons (TCP_SERVER_PORT);
      break;
    };

    /* Indicate success */
    return 0;

  }

  /* No room on the local server, or connection refused */

  switch (obj_type) {
    case __PVM_MUTEX:
      status = __pvm_find_available ((pvm_t) __PVM_MUTEX,
                                     &server_sin,
                                     &server_sin_length);
    break;
    case __PVM_SEM:
      status = __pvm_find_available ((pvm_t) __PVM_SEM,
                                     &server_sin,
                                     &server_sin_length);
    break;
    case __PVM_CV:
      status = __pvm_find_available ((pvm_t) __PVM_CV,
                                     &server_sin,
                                     &server_sin_length);
    break;
  };

  if (status) {
    errno = EAGAIN; /* Maybe we should decide what to return
    return -1;         according to what errno was */
  }

  /* We should have found a willing server at this point. */

  /* We have the server's IP already in server_sin */
  server_sin.sin_port = htons (TCP_SERVER_PORT);

  /* Set comm_buff to all zeros */
  (void) memset (&comm_buff, 0, BUF_LENGTH);

  /* Here's where the real object-specific code goes.... */
  switch (obj_type) {
    case __PVM_MUTEX:
      (void) snprintf (comm_buff, BUF_LENGTH, "CMT:%s",
                      ((pvm_mutex *) obj_ptr)->sym_name);
    break;
    case __PVM_SEM:
      (void) snprintf (comm_buff, BUF_LENGTH, "CSM:%s:%u",
                      ((pvm_sem *) obj_ptr)->sym_name,
                      sem_ctr_init);
    break;
    case __PVM_CV:
      (void) snprintf (comm_buff, BUF_LENGTH, "CCV:%s",
                      ((pvm_cond *) obj_ptr)->sym_name);
    break;
  };

  /* Connect to node that volunteered and create the object */
  if (__pvm_tcp_server_sendreq (comm_buff, strlen (comm_buff), server_sin)) {

    /* errno already set */
    return -1;

  } else {

      switch (obj_type) {
        case __PVM_MUTEX:
          memcpy (&((pvm_mutex *) obj_ptr)->sym_addr,
                  &server_sin, sizeof (struct sockaddr_in));
        break;
        case __PVM_SEM:
          memcpy (&((pvm_sem *) obj_ptr)->sym_addr,
                  &server_sin, sizeof (struct sockaddr_in));
        break;
        case __PVM_CV:
          memcpy (&((pvm_cond *) obj_ptr)->sym_addr,
                  &server_sin, sizeof (struct sockaddr_in));
        break;
      };

    }

  /* I know this is kludgy, but for now it's necessary */
  usleep ((unsigned long) 50);

  /* Close the tcp socket */
  __errno = errno;
  (void) close (__pvm_client_tcp_sock);
  /* Restore errno */
  errno = __errno;

/* Default is 0 */
return 0;
}

/* Generic destroy function for synchronization primitives */
/* Return 0 on success, -1 on failure (set errno) */
/* The given object of type obj_type will be destroyed on
   the server it resides on (if possible), or an error
   returned. */
int __pvm_generic_destroy_sync (pvm_t obj_type, void *obj_ptr)
{
char comm_buff[BUF_LENGTH],
     obj_name[SYMBOL_LENGTH];
struct sockaddr_in sav_in_addr;

  (void) memset (&comm_buff, 0, BUF_LENGTH);

  switch (obj_type) {
    case __PVM_MUTEX:
      (void) strncpy (obj_name,
                      ((pvm_mutex *) obj_ptr)->sym_name,
                      PVM_MAX_SYMLEN);
      (void) memcpy (&sav_in_addr,
                     &((pvm_mutex *) obj_ptr)->sym_addr,
                     sizeof (struct sockaddr_in));
      (void) snprintf (comm_buff, BUF_LENGTH, "DMT:%s", obj_name);
    break;
    case __PVM_SEM:
      (void) strncpy (obj_name,
                      ((pvm_sem *) obj_ptr)->sym_name,
                      PVM_MAX_SYMLEN);
      (void) memcpy (&sav_in_addr,
                     &((pvm_sem *) obj_ptr)->sym_addr,
                     sizeof (struct sockaddr_in));
      (void) snprintf (comm_buff, BUF_LENGTH, "DSM:%s", obj_name);
    break;
    case __PVM_CV:
      (void) strncpy (obj_name,
                      ((pvm_cond *) obj_ptr)->sym_name,
                      PVM_MAX_SYMLEN);
      (void) memcpy (&sav_in_addr,
                     &((pvm_cond *) obj_ptr)->sym_addr,
                     sizeof (struct sockaddr_in));
      (void) snprintf (comm_buff, BUF_LENGTH, "DCV:%s", obj_name);
    break;
    default:
      errno = EINVAL;
      return -1;
  };

  /* We already have the server's address in obj_ptr->sym_addr */
  /* Connect to it and destroy the object */
  if (__pvm_tcp_server_sendreq (comm_buff, strlen (comm_buff), sav_in_addr)) {

    /* We may have migrated to another node, look it up */
    if (__pvm_getaddr (obj_type, obj_ptr)) {
      errno = EAGAIN;
      return -1;
    }

    switch (obj_type) {
      case __PVM_MUTEX:
        (void) memcpy (&sav_in_addr,
                       &((pvm_mutex *) obj_ptr)->sym_addr,
                       sizeof (struct sockaddr_in));
      break;
      case __PVM_SEM:
        (void) memcpy (&sav_in_addr,
                       &((pvm_sem *) obj_ptr)->sym_addr,
                       sizeof (struct sockaddr_in));
      break;
      case __PVM_CV:
        (void) memcpy (&sav_in_addr,
                       &((pvm_cond *) obj_ptr)->sym_addr,
                       sizeof (struct sockaddr_in));
      break;
      default:
        errno = EINVAL;
        return -1;
    };

    if (__pvm_tcp_server_sendreq (comm_buff, strlen (comm_buff), sav_in_addr))
      /* errno is already set appropriately from the server's response */
      return -1;
  }

  switch (obj_type) {
    case __PVM_MUTEX:
      (void) memset (obj_ptr, 0, sizeof (pvm_mutex));
    break;
    case __PVM_SEM:
      (void) memset (obj_ptr, 0, sizeof (pvm_sem));
    break;
    case __PVM_CV:
      (void) memset (obj_ptr, 0, sizeof (pvm_cond));
    break;
  };

return 0;
}

/* Send contents of buffer to tcp server given by address send_addr
   (which is presumably in the correct format) and parse the return
   value out from the reply.  This routine handles setting up the
   connection. */
/* Return 0 on success, -1 on failure (set errno) */
int __pvm_tcp_server_sendreq (char *buff, size_t bufflen,
                          struct sockaddr_in send_addr)
{
/* Buffer for read () */
ssize_t repl_len;
char repl_buff[BUF_LENGTH];

/* Our return value */
int retn_val = -1;

/* Value returned by node */
char *retn_code_string = NULL;

/* Saved errno */
int __errno;

  __pvm_client_tcp_sock = socket (AF_INET, SOCK_STREAM, 0);
  if (__pvm_client_tcp_sock < 0) {
    if ((errno == ENFILE) || (errno == ENOBUFS) || (errno == ENOMEM)) {
      errno = ENOMEM;
      return -1;
    } else if ((errno == EACCES) || (errno == EPROTONOSUPPORT) ||
                                                    (errno == EINVAL)) {
      errno = EPERM;
      return -1;
    } else {
      errno = EAGAIN;
      return -1;
    }
  }

  if (connect (__pvm_client_tcp_sock, &send_addr, sizeof (struct sockaddr_in))) {
    (void) close (__pvm_client_tcp_sock);
    errno = ESRCH;
    return -1;
  }

  repl_len = write (__pvm_client_tcp_sock, buff, bufflen);
  if (repl_len == -1) {
    (void) close (__pvm_client_tcp_sock);
    errno = EAGAIN;
    return -1;
  }

  /* Clear repl_buff */
  (void) memset (repl_buff, 0, BUF_LENGTH);

  repl_len = read (__pvm_client_tcp_sock, repl_buff, BUF_LENGTH);
  if ((!repl_len) || (repl_len < 0)) {
    errno =  EAGAIN;
  } else {
    if (!strncmp (repl_buff, "SUCCESS:", 8)) {
        retn_val = 0;
        /* *** */
        /* Quick kludge fix for sd */
        if (strncmp (repl_buff+8, "nilval", (size_t) 6))
          errno = atoi (repl_buff+8);
        /* *** */
    } else if (!strncmp (repl_buff, "FAIL:", 5)) {
      retn_code_string = repl_buff+5;
      errno = atoi (retn_code_string);
    } else {
      errno = EIO;
    }
  }

  /* Save value of errno before close */
  __errno = errno;
  (void) close (__pvm_client_tcp_sock);
  /* Restore */
  errno = __errno;

return retn_val;
}


