/*
 *
 * $Id: initialize.c,v 1.2.4.41 2000/03/16 17:12:41 ajp Exp $
 *
 * Andrew Pitman
 *
 * pvmsync, a distributed synchronization server:  initialization
 * routines.
 *
 * Server that accepts connections (requests) for shared
 * data or synchronization mechanisms continuously and
 * changes some internal state according to the request.
 *
 * Copyright (C) 1999, 2000 Andrew J. Pitman
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


#ifndef __CONFIG_H
  /* We need to get at the runfile name */
  #define __FILE_NEEDS_TCP_RUNFILE
#include <config.h>
#endif

#ifndef __INITIALIZE_H
#include <initialize.h>
#endif

#ifndef __INTEGER_H
#include <integer.h>
#endif

#ifndef __FLTPT_H
#include <fltpt.h>
#endif

#ifndef __MEMORY_H
#include <memory.h>
#endif

#include <stdio.h>
#include <string.h>

#include <stdlib.h>

/* stat (2) */
#include <sys/stat.h>
#include <unistd.h>

/* open (2) */
#include <sys/types.h>
#include <fcntl.h>

/* socket (2) */
#include <sys/socket.h>

/* struct sockaddr_un */
#include <sys/un.h>

/* extern int errno */
#include <errno.h>

/* For RW Locks */
#ifndef __USE_UNIX98
  #define __USE_UNIX98
#endif
#include <pthread.h>
#include <semaphore.h>

#ifndef __TCP_DAEMON_H
   /* mutex_pool_t */
#include <tcp_daemon.h>
#endif

#ifndef __LOGGING_H
  #include <logging.h>
#endif

#ifndef __ACCEPT_LOCAL_H
#include <accept_local.h>
#endif

/* What we were called as */
extern char *progname;

/* Attributes for inititalization of each */
/* entry in the hash table's associated */
/* read-write lock */
extern pthread_rwlockattr_t *mutex_init_lock_attr;

/* Internal "pool" of mutexes (mutexi?) */
/* Hash table indexed according to first character */
extern struct ht_ent mutex_pool[NUM_HASH_ENTS];

/* Keep a counter of the number of objects we have of this type */
/* This is protected by the rwlock and is only updated on mutex */
/* creation and destruction. */
extern unsigned int num_mutexes;

/* Protect access to variable num_mutexes */
extern pthread_rwlock_t mutex_counter_lock;

/* Protect initialization and destruction of individual semaphores */
/* by locking a read lock to search, and a write lock to create    */
/* or destroy a semaphore from the "pool".                         */
extern pthread_rwlock_t sem_init_lock;
extern pthread_rwlockattr_t *sem_init_lock_attr;

/* Internal "pool" of semaphores */
extern struct ht_ent sem_pool[NUM_HASH_ENTS];

/* Keep a counter of the number of objects we have of this type */
/* This is protected by the rwlock and is only updated on sem   */
/* creation and destruction. */
extern unsigned int num_sems;

/* Protect access to variable num_sems */
extern pthread_rwlock_t sem_counter_lock;

/* Attributes for inititalization of each */
/* entry in the hash table's associated */
/* read-write lock */
extern pthread_rwlockattr_t *condvar_init_lock_attr;

/* Internal "pool" of condition variables */
/* Hash table indexed according to first character */
extern struct ht_ent cond_pool[NUM_HASH_ENTS];

/* Keep a counter of the number of objects we have of this type */
/* This is protected by the rwlock and is only updated on condvar */
/* creation and destruction. */
extern unsigned int num_conds;

/* Protect access to variable num_conds */
pthread_rwlock_t cond_counter_lock;

/* Where we put runfiles and our socket */
char *runloc;

extern int  log_fd;     /* Log/run file */
extern char log_filename[MAX_FILEPATH];  /* Filename after RUNDIR
                                            has been resolved */
/* Protect updates to the log file */
pthread_mutex_t *wrlog_mutex;

extern char *ipc_path;  /* Path to UNIX domain socket */
int ipc_fd;      /* Socket to communicate with the UDP daemon */


/* Init rundir... 0 on success, -1 on failure */
int init_rundir ()
{
int status;
char errstring[64];
char *rd;

/* Stat run directory */
struct stat *rundir_buf;

  runloc = (char *) malloc (MAX_FILEPATH);

  #ifdef __HONOR_ENV_RUNDIR
    rd = getenv("RUNDIR");
    if (rd == NULL) {
      (void) snprintf (runloc, MAX_FILEPATH, "%s/.pvmsync",
                      (const char *) rundir);
    } else {
      (void) snprintf (runloc, MAX_FILEPATH, "%s/.pvmsync", rd);
    }
  #else
      (void) snprintf (runloc, MAX_FILEPATH, "%s/.pvmsync",
                      (const char *) rundir);
  #endif

  /* Allocate space for the structure */
  rundir_buf = (struct stat *) malloc (sizeof (struct stat));

  status = stat (runloc, rundir_buf);
  if ((status) && (errno == ENOENT)) {
    status = mkdir (runloc, 0777);
    if (status) {
      snprintf(errstring, 64, "%s: mkdir", progname);
      perror(errstring);
      free (runloc);
      free (rundir_buf);
      return -1;
    }
  } else if (status) {
    snprintf(errstring, 64, "%s: stat", progname);
    perror(errstring);
    free (runloc);
    free (rundir_buf);
    return -1;
  } else {
    if (!(rundir_buf->st_mode & S_IFDIR)) {
      fprintf(stderr, "%s: %s exists and is not a directory\n",
                                                   progname, runloc);
      free (runloc);
      free (rundir_buf);
      return -1;
    }
  }

  /* We're done with rundir_buf */
  free (rundir_buf);

return 0;
}

/* Open syslog or runfile */
/* 0 on success, -1 on failure */
int init_log ()
{
int status;
char errstring[64];

  #ifndef __LOG_SYSLOG

    (void) snprintf (log_filename, MAX_FILEPATH, "%s/%s", runloc,
                                                       tcp_run_file);

    log_fd = open (log_filename, (O_CREAT | O_APPEND | O_WRONLY), 0644);
    if (log_fd == -1) {
      snprintf(errstring, 64, "%s: open", progname);
      perror(errstring);
      free (runloc);
      return -1;
    }

    /* init mutex to protect file writes for log file */
    wrlog_mutex = (pthread_mutex_t *) malloc (sizeof (pthread_mutex_t));
    if (wrlog_mutex == NULL) {
      fprintf(stderr, "%s: malloc: couldn't allocate memory\n", progname);
      free (runloc);
      return -1;
    }
    status = pthread_mutex_init(wrlog_mutex, NULL);
    if (status) {
      fprintf(stderr, "%s: pthread_mutex_init: %s\n", progname,
                                              strerror(status));
      free (runloc);
      return -1;
    }

  #else

    #ifdef DEBUG
      openlog ("pvmsync", LOG_CONS | LOG_PID, LOG_USER);
    #endif

  #endif

return 0;
}

/* Write to PID file */
/* 0 on success, -1 on failure */
int wr_pid ()
{
int pid_fd;
ssize_t bytes;
char pid_path[MAX_FILEPATH],
     wr_buff[32];

  snprintf (pid_path, MAX_FILEPATH, "%s/%s", runloc, tcp_pid_file);
  pid_fd = open (pid_path, (O_WRONLY | O_CREAT | O_TRUNC), 0644);
  if (pid_fd < 0) {
    fprintf(stderr, "%s: wr_pid (open): %s\n", progname, strerror(errno));
    free (runloc);
    return -1;
  }

  snprintf (wr_buff, 32, "%d", (int) getpid ());
  /* Just to be sure.... */
  wr_buff[31] = '\0';

  if ((bytes = write (pid_fd, wr_buff, strlen (wr_buff))) == -1) {
    fprintf(stderr, "%s: wr_pid (write): %s\n", progname, strerror(errno));
    (void) close (pid_fd);
    free (runloc);
    return -1;
  }

  /* Close fd */
  (void) close (pid_fd);

return 0;
}

/* Call funcs to init pools of different resources, init
   random number generator, open UNIX socket */
/* 0 on success, -1 on failure */
int gen_init ()
{
int status;
struct sockaddr_un local_sun;               /* for communications with
                                               UDP daemon.                */
char errstring[64];

/* Attributes object to create threads detached */
pthread_attr_t attrib;

/* Dummy thread identifier */
pthread_t thread_id;

  report_log ("pvmsyncd: now initializing internal mutexes\n");

  status = init_mutex_pool (NUM_HASH_ENTS);
  if (status) {
    report_log ("pvmsyncd: initialization of internal mutexes failed\n");
    (void) pthread_mutex_unlock (wrlog_mutex);
    free (runloc);
    return -1;
  }
  report_log ("pvmsyncd: initialization of internal mutexes completed\n");

  report_log ("pvmsyncd: now initializing internal semaphores\n");

  /* Initialize semaphores' synchronization data */
  status = init_sem_pool (NUM_HASH_ENTS);
  if (status) {
    report_log ("pvmsyncd: initialization of internal semaphores failed\n");
    (void) pthread_mutex_unlock (wrlog_mutex);
    free (runloc);
    return -1;
  }
  report_log ("pvmsyncd: initialization of internal semaphores completed\n");

  report_log ("pvmsyncd: now initializing internal condition variables\n");

  status = init_cond_pool (NUM_HASH_ENTS);
  if (status) {
    report_log ("pvmsyncd: initialization of internal condvars failed\n");
    (void) pthread_mutex_unlock (wrlog_mutex);
    free (runloc);
    return -1;
  }
  report_log ("pvmsyncd: initialization of internal condvars completed\n");

  report_log ("pvmsyncd: now initializing integer pool\n");

  /* Initialize integers' synchronization data */
  status = init_integer_array (CONCURRENT_INSTANCES);
  if (status) {
    report_log ("pvmsyncd: initialization of internal integer pool failed\n");
    (void) pthread_mutex_unlock (wrlog_mutex);
    free (runloc);
    return -1;
  }
  report_log ("pvmsyncd: initialization of internal integer pool completed\n");

  report_log ("pvmsyncd: now initializing float pool\n");

  /* Initialize floats' synchronization data */
  status = init_float_array (CONCURRENT_INSTANCES);
  if (status) {
    report_log ("pvmsyncd: initialization of internal float pool failed\n");
    (void) pthread_mutex_unlock (wrlog_mutex);
    free (runloc);
    return -1;
  }
  report_log ("pvmsyncd: initialization of internal float pool completed\n");

  report_log ("pvmsyncd: now initializing memory pool\n");

  /* Initialize memptr's synchronization data */
  status = init_memptr_array (CONCURRENT_INSTANCES);
  if (status) {
    report_log ("pvmsyncd: initialization of internal memptr pool failed\n");
    (void) pthread_mutex_unlock (wrlog_mutex);
    free (runloc);
    return -1;
  }
  report_log ("pvmsyncd: initialization of internal memory pool completed\n");


  /* Seed pseudo-random number generator */
  report_log ("pvmsyncd: seeding pseudo-random number generator\n");
  srand (time (NULL));

  /* Here, we create a UNIX domain socket with address
    ipc_path, with fd ipc_fd.                         */

  ipc_path = (char *) malloc (strlen (runloc) + 8);
  snprintf(ipc_path, strlen (runloc) + 8, "%s/.pvm-1", runloc);

  memset (&local_sun, 0, sizeof (struct sockaddr_un));

  ipc_fd = socket (AF_UNIX, SOCK_STREAM, 0);
  if (ipc_fd == -1) {
    snprintf(errstring, 64, "%s: socket", progname);
    perror(errstring);
    free (ipc_path);
    return -1;
  }

  local_sun.sun_family = AF_UNIX;
  memcpy(local_sun.sun_path, ipc_path,
             (strlen (ipc_path) > 108 ? 108 : strlen (ipc_path)));
  /* Don't free ipc_path because SIGTERM handler needs this
     (to unlink the file when we're finished) */

  status = bind (ipc_fd, &local_sun, sizeof (struct sockaddr_un));
  if (status) {
    snprintf(errstring, 64, "%s: bind", progname);
    perror(errstring);
    free (ipc_path);
    return -1;
  }

  status = listen (ipc_fd, SERVER_BACK_LOG);
  if (status) {
    snprintf(errstring, 64, "%s: listen", progname);
    perror(errstring);
    free (ipc_path);
    return -1;
  }
  /* We'll accept and hand off fd's to read/write to/from
     in another thread.                                  */

  /* OK if these fail, just report error and continue */
  status = pthread_attr_init (&attrib);
  if (status) {
    snprintf(errstring, 64, "%s: pthread_attr_init", progname);
    fprintf(stderr, "%s: %s\n", errstring, strerror(status));
  }
  status = pthread_attr_setdetachstate (&attrib, PTHREAD_CREATE_DETACHED);
  if (status) {
    snprintf(errstring, 64, "%s: pthread_attr_setdetachstate", progname);
    fprintf(stderr, "%s: %s\n", errstring, strerror(status));
  }

  status = pthread_create (&thread_id, &attrib, accept_local, &ipc_fd);
  if (status) {
    snprintf(errstring, 64, "%s: pthread_create: %s\n", progname,
                                                  strerror (status));
    (void) pthread_mutex_lock (wrlog_mutex);
           report_log (errstring);
    (void) pthread_mutex_unlock (wrlog_mutex);
    free (ipc_path);
    return -1;
  }

  /* We're (finally) done with runloc */
  free (runloc);

return 0;
}

/* Initialize hash table of mutexes */
/* Returns 0 on success and -1 on error */
int init_mutex_pool (int num_ents)
{
int j, status;

  if (num_ents != NUM_HASH_ENTS) {
    fprintf(stderr, "%s: init_mutex_pool: invalid index\n", progname);
    return -1;
  }

  mutex_init_lock_attr = (pthread_rwlockattr_t *) malloc
                         (sizeof (pthread_rwlockattr_t));

  status = pthread_rwlockattr_init (mutex_init_lock_attr);
  if (status) {
    fprintf (stderr,
            "%s: init_mutex_pool: error initializing rw lock attribs\n",
            progname);
    return -1;
  }

  status = pthread_rwlockattr_setkind_np (mutex_init_lock_attr,
           PTHREAD_RWLOCK_PREFER_READER_NP);
  if (status) {
    fprintf (stderr,
            "%s: init_mutex_pool: error setting rw lock attribs\n",
            progname);
    return -1;
  }

  /* Init rwlock to protect num_mutexes */
  /* We'll 'borrow' the attributes we just made up */
  if (pthread_rwlock_init (&mutex_counter_lock, mutex_init_lock_attr)) {
    fprintf(stderr, "%s: init_mutex_pool: rwlock init failed\n", progname);
    return -1;
  }

  /* Init num_mutexes */
  num_mutexes = 0;

  for (j = 0; j < num_ents; j++) {

    status = pthread_rwlock_init (&mutex_pool[j].ent_rwlock,
             mutex_init_lock_attr);
    if (status) {
      fprintf (stderr,
              "%s: init_mutex_pool: error initializing read-write lock\n",
              progname);
      return -1;
    }

    mutex_pool[j].head = NULL;
    mutex_pool[j].num_subent = 0;

  }

  free (mutex_init_lock_attr);

return 0;
}


/* Initialize our pool of semaphores'
   corresponding info/ sync mechanisms. */
int init_sem_pool (int num_ents)
{
int j, status;

  if (num_ents != NUM_HASH_ENTS) {
    fprintf(stderr, "%s: init_sem_pool: invalid index\n", progname);
    return -1;
  }

  sem_init_lock_attr = (pthread_rwlockattr_t *) malloc
                       (sizeof (pthread_rwlockattr_t));

  status = pthread_rwlockattr_init (sem_init_lock_attr);
  if (status) {
    fprintf (stderr,
            "%s: init_sem_pool: error initializing rw lock attribs\n",
            progname);
    return -1;
  }

  status = pthread_rwlockattr_setkind_np (sem_init_lock_attr,
           PTHREAD_RWLOCK_PREFER_READER_NP);
  if (status) {
    fprintf (stderr,
            "%s: init_sem_pool: error setting rw lock attribs\n",
            progname);
    return -1;
  }

  /* Init rwlock to protect num_sems */
  /* We'll 'borrow' the attributes we just made up */
  if (pthread_rwlock_init (&sem_counter_lock, sem_init_lock_attr)) {
    fprintf(stderr, "%s: init_sem_pool: rwlock init failed\n", progname);
    return -1;
  }

  /* Init num_sems */
  num_sems = 0;

  for (j = 0; j < num_ents; j++) {

    status = pthread_rwlock_init (&sem_pool[j].ent_rwlock,
             sem_init_lock_attr);
    if (status) {
      fprintf (stderr,
              "%s: init_sem_pool: error initializing read-write lock\n",
              progname);
      return -1;
    }

    sem_pool[j].head = NULL;
    sem_pool[j].num_subent = 0;

  }

  free (sem_init_lock_attr);

return 0;
}


/* Initialize our pool of condvars'
   corresponding info/ sync mechanisms. */
int init_cond_pool (int num_ents)
{
int j, status;

  if (num_ents != NUM_HASH_ENTS) {
    fprintf(stderr, "%s: init_cond_pool: invalid index\n", progname);
    return -1;
  }

  condvar_init_lock_attr = (pthread_rwlockattr_t *) malloc
                           (sizeof (pthread_rwlockattr_t));

  status = pthread_rwlockattr_init (condvar_init_lock_attr);
  if (status) {
    fprintf (stderr,
            "%s: init_cond_pool: error initializing rw lock attribs\n",
            progname);
    return -1;
  }

  status = pthread_rwlockattr_setkind_np (condvar_init_lock_attr,
           PTHREAD_RWLOCK_PREFER_READER_NP);
  if (status) {
    fprintf (stderr,
            "%s: init_cond_pool: error setting rw lock attribs\n",
            progname);
    return -1;
  }

  /* Init rwlock to protect num_conds */
  /* We'll 'borrow' the attributes we just made up */
  if (pthread_rwlock_init (&cond_counter_lock, condvar_init_lock_attr)) {
    fprintf(stderr, "%s: init_cond_pool: rwlock init failed\n", progname);
    return -1;
  }

  /* Init num_mutexes */
  num_conds = 0;

  for (j = 0; j < num_ents; j++) {

    status = pthread_rwlock_init (&cond_pool[j].ent_rwlock,
             condvar_init_lock_attr);
    if (status) {
      fprintf (stderr,
              "%s: init_cond_pool: error initializing read-write lock\n",
              progname);
      return -1;
    }

    cond_pool[j].head = NULL;
    cond_pool[j].num_subent = 0;

  }

  free (condvar_init_lock_attr);

return 0;
}


