/*
 *
 * $Id: pvm_sd.h,v 1.2.4.41 2000/03/16 17:13:36 ajp Exp $
 *
 * Andrew Pitman              pvm_sd.h
 *
 * pvmsync, a distributed synchronization server:  declarations
 * for shared data specific client routines for pvmsync.
 *
 * Server that accepts connections (requests) for shared
 * data or synchronization mechanisms continuously and
 * changes some internal state according to the request.
 *
 * Copyright (C) 1999, 2000 Andrew J. Pitman
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


#ifndef __PVM_SD_H
#define __PVM_SD_H


#include <pvm_gen.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

#include <errno.h>


#if (!defined __HAVE_PVM_FLOAT64_T)
  /* 64 bit floating point */
  typedef double pvm_float64_t;
  #define __HAVE_PVM_FLOAT64_T
#endif /* !defined __HAVE_PVM_FLOAT64_T */

#if (!defined __HAVE_PVM_INT64_T)
  /* 64 bit integer */
  typedef int64_t pvm_int64_t;
  #define __HAVE_PVM_INT64_T
#endif /* !defined __HAVE_PVM_INT64_T */

/* A pointer to this type must be passed to the
   pvm_sd_get, pvm_sd_set, and pvm_sd_destroy.
   The struct sockaddr_in member must be filled
   in first by one of: pvm_sd_init or pvm_getaddr. */

typedef struct pvm_sd_t {
        int sym_type;                /* One of: __PVM_SD_INT, __PVM_SD_FLOAT,
                                        or __PVM_SD_MEM */
	char sym_name[PVM_MAX_SYMLEN]; /* Name of shared data symbol */
	struct sockaddr_in sym_addr; /* Address of server it resides on */
        int sym_index;               /* Index of symbol in pool */
        void *sym_value;             /* Pointer to cached object value */
                                     /* Allocated and filled in
                                                      by pvm_sd_init */
                                     /* Always use pvm_sd_get if you want the
                                        newest value of this object, as
                                        somebody else may have changed it. */
        size_t sym_size;             /* How many bytes long is this thing??
                                        will be filled in by pvm_sd_init to
                                        (probably) 4 bytes for __PVM_SD_INT
                                        and 8 bytes for __PVM_SD_FLOAT
                                        (integer and float types, depends on
                                        architecture) and the length of the
                                        memory block (in bytes) for
                                        __PVM_SD_MEM */
} pvm_sd;


__BEGIN_DECLS

/* These routines follow the convention of returning -1
   and setting errno appropriately. */

/* Initialize a pvm_sd */
/* This pvm_sd will be instantiated on a server, and
   __sd->sym_addr will be filled in with the appropriate
   server address. */
/* Return 0 on success and -1 failure. */
extern int pvm_sd_init __P ((pvm_t __obj_type, pvm_sd *__sd, ...));

#define pvm_int_init(__sd) pvm_sd_init (__PVM_SD_INT, __sd)

#define pvm_float_init(__sd) pvm_sd_init (__PVM_SD_FLOAT, __sd)

#define pvm_mem_init(__sd, __size) \
                       pvm_sd_init (__PVM_SD_MEM, __sd, __size)

/* The given pvm_sd will be destroyed on the server it
   resides on (if possible), or an error returned. */
/* Return 0 on success and -1 on failure. */
extern int pvm_sd_destroy __P ((pvm_sd *__sd));

#define pvm_int_destroy      pvm_sd_destroy

#define pvm_float_destroy    pvm_sd_destroy

#define pvm_mem_destroy      pvm_sd_destroy

/* Get the value of a shared data object */
extern int pvm_sd_get __P ((pvm_sd *__sd));

/* Set the value of a shared data object */
extern int pvm_sd_set __P ((pvm_sd *__sd));

/* Extract the data value from a pvm_sd */
/* Returns 0 on success, -1 on failure and sets errno */

/* Generic get value of object */
/* __val will point to the data member of __sd */
extern int pvm_getvalue __P ((pvm_sd *__sd, void *__val));

/* Integer-specific get value of object */
/* Return the pvm_int64_t value stored in the data member of __sd */
extern pvm_int64_t pvm_getint __P ((pvm_sd *__sd));

/* Float-specific get value of object */
/* Return the pvm_float64_t value stored in the data member of __sd */
extern pvm_float64_t pvm_getfloat __P ((pvm_sd *__sd));

/* Memory block-specific get value of object */
/* Return a pointer to the block of memory (data member of __sd) */
extern void * pvm_getmem __P ((pvm_sd *__sd));

/* Set the data member */
/* Returns 0 on success, -1 on failure and sets errno */

/* Generic set value of object */
/* For integer and float sets, the 2nd param is passed by value */
extern int pvm_setvalue __P ((pvm_sd *__sd, ...));

/* Integer-specific set value of object */
/* 2nd param is passed by value */
#define pvm_setint   pvm_setvalue

/* Float-specific set value of object */
/* 2nd param is passed by value */
#define pvm_setfloat pvm_setvalue

/* Memory block-specific set value of object */
/* 2nd param is passed by reference */
#define pvm_setmem   pvm_setvalue


__END_DECLS


#endif /* __PVM_SD_H */


