/*
 *
 * $Id: pvm_gen.h,v 1.2.4.41 2000/03/16 17:13:31 ajp Exp $
 *
 * Andrew Pitman              pvm_gen.h
 *
 * pvmsync, a distributed synchronization server:  declarations
 * for general-purpose pvmsync client routines.
 *
 * Server that accepts connections (requests) for shared
 * data or synchronization mechanisms continuously and
 * changes some internal state according to the request.
 *
 * Copyright (C) 1999, 2000 Andrew J. Pitman
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 * 
 * You should have received a copy of the GNU Library General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


#ifndef __PVM_GEN_H
#define __PVM_GEN_H


#include <pvm_gen.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>


/* Type of pvm object */
typedef int pvm_t;

#define PVM_MUTEX             ((pvm_t) 1)
#define PVM_SEMAPHORE         ((pvm_t) 2)
#define PVM_CONDVAR           ((pvm_t) 4)
#define PVM_DATA              ((pvm_t) 8)

/* Object types according to udp server */
/* These are bitmasks and can be used with bitwise
   operations / comparisons */
#define __PVM_MUTEX           ((pvm_t) 1)
#define __PVM_SEM             ((pvm_t) 2)
#define __PVM_CV              ((pvm_t) 4)
#define __PVM_SD_INT          ((pvm_t) 24)
#define __PVM_SD_FLOAT        ((pvm_t) 40)
#define __PVM_SD_MEM          ((pvm_t) 72)
#define __PVM_GET_LD          ((pvm_t) 128)

/* These types are passed to pvm_getaddr,
   provided for convenience */
#define MUTEX                 __PVM_MUTEX
#define SEMAPHORE             __PVM_SEM
#define CONDVAR               __PVM_CV
#define INTEGER               __PVM_SD_INT
#define FLOAT                 __PVM_SD_FLOAT
#define MEMORY                __PVM_SD_MEM


/* Maximum length for all pvmsync symbols */
#define PVM_MAX_SYMLEN        255


/* Keep track of information on pvmsync tasks */
typedef struct pvm_task_type {
    int task_id;
    int detach_state;
} pvm_task_t;

/* Attribues for creation of tasks */
typedef struct pvm_task_attr_type {
    int detach_state;
} pvm_task_attr_t;


__BEGIN_DECLS


/* Keep an in_addr with the address of the loopback interface */
/* Set by some of the internal functions with inet_aton(3) */
extern struct in_addr * __loopback_in_addr;

/* Support routines */

/* Bind a symbolic name to the object */
extern int __pvm_bind __P ((pvm_t __type, char *__sym_name, void *__object));

#define __pvm_mutex_bind(__name, __obj) \
        __pvm_bind (__PVM_MUTEX, __name, __obj)

#define __pvm_sem_bind(__name, __obj) \
        __pvm_bind (__PVM_SEM, __name, __obj)

#define __pvm_cond_bind(__name, __obj) \
        __pvm_bind (__PVM_CV, __name, __obj)

#define __pvm_int_bind(__name, __obj) \
        __pvm_bind (__PVM_SD_INT, __name, __obj)

#define __pvm_float_bind(__name, __obj) \
        __pvm_bind (__PVM_SD_FLOAT, __name, __obj)

#define __pvm_mem_bind(__name, __obj) \
        __pvm_bind (__PVM_SD_MEM, __name, __obj)

/* Get the address of the server that __object->sym_name resides
   on and return it in __object->sym_addr (struct sockaddr_in).
   This must be done if the object was created by another
   process and we must discover which server has the object
   we are looking for (fill in the sym_addr). */
/* For shared data, after calling this it is necessary to
   call pvm_sd_get to initialize the pointer to the actual
   data value within the object before operating on it. */
extern int __pvm_getaddr __P ((pvm_t __type, void *__object));

#define __pvm_mutex_getaddr(__obj) __pvm_getaddr (__PVM_MUTEX, __obj)

#define __pvm_sem_getaddr(__obj)   __pvm_getaddr (__PVM_SEM, __obj)

#define __pvm_cond_getaddr(__obj)  __pvm_getaddr (__PVM_CV, __obj)

#define __pvm_int_getaddr(__obj)   __pvm_getaddr (__PVM_SD_INT, __obj)

#define __pvm_float_getaddr(__obj) __pvm_getaddr (__PVM_SD_FLOAT, __obj)

#define __pvm_mem_getaddr(__obj)   __pvm_getaddr (__PVM_SD_MEM, __obj)


/* The following routines are not yet implemented */
extern int pvm_task_create __P ((pvm_task_t *__task, pvm_task_attr_t *__attr,
                            void * (*__start_routine)(void *), void *__arg));

extern pvm_task_t pvm_task_self __P ((void));

extern int pvm_task_equal __P ((pvm_task_t __task1, pvm_task_t __task2));

void pvm_task_exit __P ((void *__retval)) __attribute__ ((noreturn));

extern int pvm_task_attr_init __P ((pvm_task_attr_t *__attr));

extern int pvm_task_attr_destroy __P ((pvm_task_attr_t *__attr));

extern int pvm_task_attr_setdetachstate __P ((pvm_task_attr_t *__attr,
                                              int *__detachstate));

extern int pvm_task_attr_getdetachstate __P ((const pvm_task_attr_t *__attr,
                                              int *__detachstate));

extern int pvm_task_join __P ((pvm_task_t __task, void **__thread_return));

extern int pvm_task_detach __P ((pvm_task_t __task));

extern int pvm_task_cancel __P ((pvm_task_t __task));


__END_DECLS


#endif /* __PVM_GEN_H */


