/*
 *
 * $Id: pvm_cv.h,v 1.2.4.41 2000/03/16 17:13:29 ajp Exp $
 *
 * Andrew Pitman              pvm_cv.h
 *
 * pvmsync, a distributed synchronization server:  declarations
 * for condition variable specific client routines for pvmsync.
 *
 * Server that accepts connections (requests) for shared
 * data or synchronization mechanisms continuously and
 * changes some internal state according to the request.
 *
 * Copyright (C) 1999, 2000 Andrew J. Pitman
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


#ifndef __PVM_CV_H
#define __PVM_CV_H

#include <pvm_gen.h>

/* Condition variable waits also involve locking
   a mutex, much the same as with pthreads
   condition varaibles. */
#include <pvm_mutex.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

#include <errno.h>


/* A pointer to this type must be passed to the
   pvm_cond_wait, pvm_cond_timedwait, pvm_cond_signal,
   pvm_cond_broadcast, and pvm_cond_destroy.  The
   struct sockaddr_in member must be filled in first
   by one of: pvm_cond_init, pvm_getaddr or pvm_cond_getaddr. */

typedef struct pvm_cond_t {
	char sym_name[PVM_MAX_SYMLEN]; /* Name of condvar symbol */
	struct sockaddr_in sym_addr; /* Address of server it resides on */
} pvm_cond;


__BEGIN_DECLS

/* These routines follow the convention of returning a positive
   error code instead of returning -1 and setting errno
   appropriately. */

/* Initialize a pvm_cond */
/* This pvm_cond will be instantiated on a server, and
   __cv->sym_addr will be filled in with the appropriate
   server address. */
/* Returns 0 on success, a positive error code on failure. */
extern int pvm_cond_init __P ((pvm_cond *__cv));

/* The given pvm_cond will be destroyed on the server it
   resides on (if possible), or an error returned. */
/* Returns 0 on success, a positive error code on failure. */
extern int pvm_cond_destroy __P ((pvm_cond *__cv));

/* Wait on the condvar.  __cv must be initialized first,
   or if already created by another process, looked up with
   pvm_getaddr. */
/* Returns 0 on success, a positive error code on failure. */
extern int pvm_cond_wait __P ((pvm_cond *__cv, pvm_mutex *__mutex));

/* Wait on the condvar.  __cv must be initialized first,
   or if already created by another process, looked up with
   pvm_getaddr.  This is the timed version.... it will
   return with ETIMEDOUT if waiting on the condition variable
   timed out. */
/* Returns 0 on success, a positive error code on failure. */
extern int pvm_cond_timedwait __P ((pvm_cond *__cv, pvm_mutex *__mutex,
                                               unsigned int __secs));

/* Signal the condition variable.  Awakens only one waiter. */
/* Returns 0 on success, a positive error code on failure. */
extern int pvm_cond_signal __P ((pvm_cond *__cv));

/* Broadcast the condition variable.  Awakens all waiters. */
/* Returns 0 on success, a positive error code on failure. */
extern int pvm_cond_broadcast __P ((pvm_cond *__cv));

__END_DECLS


#endif /* __PVM_CV_H */

