/*
 *
 * $Id: garbage.c,v 1.2.4.41 2000/03/16 17:12:36 ajp Exp $
 *
 * Andrew Pitman
 *
 * pvmsync, a distributed synchronization server:
 * Garbage Collection
 *
 * Server that accepts connections (requests) continuously
 * and changes some internal state according to the request.
 *
 * Copyright (C) 1999, 2000 Andrew J. Pitman
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


#ifdef __DO_GARBAGE_COLLECTION


#ifndef __CONFIG_H
#include <config.h>
#endif

#include <signal.h>
#include <string.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>

/* extern int errno */
#include <errno.h>

/* Standard error codes for pvmsync */
#ifndef __PVMERRS_H
#include <include/pvmerrs.h>
#endif

#ifndef __LOGGING_H
#include <logging.h>
#endif

/* For RW Locks */
#ifndef __USE_UNIX98
  #define __USE_UNIX98
#endif
#include <pthread.h>
#include <semaphore.h>

#ifndef __TCP_DAEMON_H
#include <tcp_daemon.h>
#endif

#ifndef __GARBAGE_H
#include <garbage.h>
#endif


/* What we were called as */
extern char *progname;

/* Internal "pool" of mutex_pool.mutex (mutexi?) */
/* Hash table indexed according to first character */
extern struct ht_ent mutex_pool[NUM_HASH_ENTS];

/* Keep a counter of the number of objects we have of this type */
/* This is protected by the rwlock and is only updated on mutex */
/* creation and destruction. */
extern unsigned int num_mutexes;

/* Protect access to variable num_mutexes */
extern pthread_rwlock_t mutex_counter_lock;

/* Internal "pool" of semaphores */
extern struct ht_ent sem_pool[NUM_HASH_ENTS];

/* Keep a counter of the number of objects we have of this type */
/* This is protected by the rwlock and is only updated on sem   */
/* creation and destruction. */
extern unsigned int num_sems;

/* Protect access to variable num_sems */
extern pthread_rwlock_t sem_counter_lock;

/* Protect access to the integer pool with a rwlock */
extern pthread_rwlock_t integer_init_lock;

/* Internal "pool" of pvm_int64_t's */
extern integer_pool_t integer_pool[CONCURRENT_INSTANCES];

/* Keep a counter of the number of objects we have of this type */
extern unsigned int num_integers;

/* Protect access to the float pool with a rwlock */
extern pthread_rwlock_t float_init_lock;

/* Internal "pool" of pvm_float64_t's. */
extern float_pool_t float_pool[CONCURRENT_INSTANCES];

/* Keep a counter of the number of objects we have of this type */
extern unsigned int num_floating;

/* Protect access to the memptr pool with a rwlock */
extern pthread_rwlock_t memptr_init_lock;
extern pthread_rwlockattr_t *memptr_init_lock_attr;

/* Internal "pool" of pointers to heap-dynamic data. */
extern memptr_pool_t memptr_pool[CONCURRENT_INSTANCES];

/* Keep a counter of the number of objects we have of this type */
extern unsigned int num_memblocks;

/* Internal "pool" of condition variables */
/* Hash table indexed according to first character */
extern struct ht_ent cond_pool[NUM_HASH_ENTS];

/* Keep a counter of the number of objects we have of this type */
/* This is protected by the rwlock and is only updated on condvar */
/* creation and destruction. */
extern unsigned int num_conds;

/* Protect access to variable num_conds */
pthread_rwlock_t cond_counter_lock;


/* Serialize access to run file */
extern pthread_mutex_t *wrlog_mutex;


/* Wake up every *secs and take out the
   trash. */
void *garbage_collect (void *secs)
{
int j;
time_t now;
unsigned int *seconds;

/* Save a copy of an object's name */
char obj_name[SYMBOL_LENGTH];

/* Current subentries we are dealing with */
struct ht_subent_mutex *mut_curr = NULL;
struct ht_subent_sem *sem_curr = NULL;
struct ht_subent_cv *cv_curr = NULL;

  seconds = (unsigned int *) secs;

  /* Detach process from tty, since we're a daemon */
  ioctl(0, TIOCNOTTY, NULL);

  while (1) {

    /* Only wake up on standard intervals */
    (void) sleep (*seconds);

    /* Roughly the current time */
    now = time (NULL);

    /* Lock rwlock for writing */
    /* We don't need to lock the individual mutexes here,
       because write locks are exclusive. */
    for (j = 0; j < NUM_HASH_ENTS; j++) {
      if (pthread_rwlock_wrlock (&mutex_pool[j].ent_rwlock))
        die_with_mesg ("pvmsyncd: couldn't lock internal rwlock\n");
      if (mutex_pool[j].num_subent) {
        mut_curr = (struct ht_subent_mutex *) mutex_pool[j].head;
        while (mut_curr) {
          if (mut_curr->pvm_mutex.access_time <= now-(*seconds)) {
            /* Remove this thing regardless of it's state */
            /* First, save obj_name */
            (void) strncpy (obj_name, mut_curr->pvm_mutex.mutex_symbol,
                                                  PVM_MAX_SYMLEN);

            /* Next, advance mut_curr */
            mut_curr = mut_curr->next;

            /* Then, ask for it to be removed */
            if (remove_ht (obj_name, __PVM_MUTEX))
              die_with_mesg ("pvmsyncd: hash table is inconsistent\n");

            /* At this point, mutex is removed from hash table */
            /* Decrement counter */
            if (pthread_rwlock_wrlock (&mutex_counter_lock))
              die_with_mesg ("pvmsyncd: unable to lock rwlock\n");
            num_mutexes--;
            unlock_r_die (&mutex_counter_lock);
          }
        };
      }
      unlock_r_die (&mutex_pool[j].ent_rwlock);
    }

    /* Roughly the current time */
    now = time (NULL);

    /* Lock rwlock for writing */
    /* We don't need to lock the individual mutexes here,
       because write locks are exclusive. */
    for (j = 0; j < NUM_HASH_ENTS; j++) {
      if (pthread_rwlock_wrlock (&sem_pool[j].ent_rwlock))
        die_with_mesg ("pvmsyncd: couldn't lock internal rwlock\n");
      if (sem_pool[j].num_subent) {
        sem_curr = (struct ht_subent_sem *) sem_pool[j].head;
        while (sem_curr) {
          if (sem_curr->pvm_sem.access_time <= now-(*seconds)) {
            /* Remove this thing regardless of it's state */
            /* First, save obj_name */
            (void) strncpy (obj_name, sem_curr->pvm_sem.sem_symbol,
                                                   PVM_MAX_SYMLEN);

            /* Next, advance mut_curr */
            sem_curr = sem_curr->next;

            /* Then, ask for it to be removed */
            if (remove_ht (obj_name, __PVM_SEM))
              die_with_mesg ("pvmsyncd: hash table is inconsistent\n");

            /* At this point, semaphore is removed from hash table */
            /* Decrement counter */
            if (pthread_rwlock_wrlock (&sem_counter_lock))
              die_with_mesg ("pvmsyncd: unable to lock rwlock\n");
            num_sems--;
            unlock_r_die (&sem_counter_lock);
          }
        };
      }
      unlock_r_die (&sem_pool[j].ent_rwlock);
    }

    /* Roughly the current time */
    now = time (NULL);

    /* Lock rwlock for writing */
    /* We don't need to lock the individual mutexes here,
       because write locks are exclusive. */
    for (j = 0; j < NUM_HASH_ENTS; j++) {
      if (pthread_rwlock_wrlock (&cond_pool[j].ent_rwlock))
        die_with_mesg ("pvmsyncd: couldn't lock internal rwlock\n");
      if (cond_pool[j].num_subent) {
        cv_curr = (struct ht_subent_cv *) cond_pool[j].head;
        while (cv_curr) {
          if (cv_curr->pvm_cv.access_time <= now-(*seconds)) {
            /* Remove this thing regardless of it's state */
            /* First, save obj_name */
            (void) strncpy (obj_name, cv_curr->pvm_cv.cond_symbol,
                                                  PVM_MAX_SYMLEN);

            /* Next, advance cv_curr */
            cv_curr = cv_curr->next;

            /* Then, ask for it to be removed */
            if (remove_ht (obj_name, __PVM_CV))
              die_with_mesg ("pvmsyncd: hash table is inconsistent\n");

            /* At this point, condition variable is removed
               from hash table */
            /* Decrement counter */
            if (pthread_rwlock_wrlock (&cond_counter_lock))
              die_with_mesg ("pvmsyncd: unable to lock rwlock\n");
            num_conds--;
            unlock_r_die (&cond_counter_lock);
          }
        };
      }
      unlock_r_die (&cond_pool[j].ent_rwlock);
    }

    /* Roughly the current time */
    now = time (NULL);

    /* Lock rwlock for writing */
    /* We don't need to lock the individual mutexes here,
       because write locks are exclusive. */
    for (j = 0; j < NUM_HASH_ENTS; j++) {
      if (pthread_rwlock_wrlock (&sem_pool[j].ent_rwlock))
        die_with_mesg ("pvmsyncd: couldn't lock internal rwlock\n");
      if (sem_pool[j].num_subent) {
        sem_curr = (struct ht_subent_sem *) sem_pool[j].head;
        while (sem_curr) {
          if (sem_curr->pvm_sem.access_time <= now-(*seconds)) {
            /* Remove this thing regardless of it's state */
            /* First, save obj_name */
            (void) strncpy (obj_name, sem_curr->pvm_sem.sem_symbol,
                                                   PVM_MAX_SYMLEN);

            /* Next, advance sem_curr */
            sem_curr = sem_curr->next;

            /* Then, ask for it to be removed */
            if (remove_ht (obj_name, __PVM_SEM))
              die_with_mesg ("pvmsyncd: hash table is inconsistent\n");

            /* At this point, condition variable is removed
               from hash table */
            /* Decrement counter */
            if (pthread_rwlock_wrlock (&sem_counter_lock))
              die_with_mesg ("pvmsyncd: unable to lock rwlock\n");
            num_sems--;
            unlock_r_die (&sem_counter_lock);
          }
        };
      }
      unlock_r_die (&sem_pool[j].ent_rwlock);
    }

    /* Roughly the current time */
    now = time (NULL);

    /* Lock rwlock for writing */
    if (pthread_rwlock_wrlock (&integer_init_lock)) {
      (void) pthread_mutex_lock (wrlog_mutex);
             report_log ("pvmsyncd: couldn't lock internal rwlock\n");
      (void) pthread_mutex_unlock (wrlog_mutex);
      (void) raise (SIGINT);
      (void) pthread_exit (NULL);
    }

    for (j = 0; j < CONCURRENT_INSTANCES; j++) {
      if ((integer_pool[j].valid) &&
        (integer_pool[j].access_time <= now-(*seconds))) {
        integer_pool[j].data = (pvm_int64_t) 0;
        integer_pool[j].valid = (short int) INVALID;
        (void) memset (integer_pool[j].symbol, 0, SYMBOL_LENGTH);
        integer_pool[j].access_time = 0;
        num_integers--;
      }
    }

    if (pthread_rwlock_unlock (&integer_init_lock)) {
      (void) pthread_mutex_lock (wrlog_mutex);
             report_log ("pvmsyncd: couldn't unlock internal rwlock\n");
      (void) pthread_mutex_unlock (wrlog_mutex);
      (void) raise (SIGINT);
      (void) pthread_exit (NULL);
    }

    /* Roughly the current time */
    now = time (NULL);

    /* Lock rwlock for writing */
    if (pthread_rwlock_wrlock (&float_init_lock)) {
      (void) pthread_mutex_lock (wrlog_mutex);
             report_log ("pvmsyncd: couldn't lock internal rwlock\n");
      (void) pthread_mutex_unlock (wrlog_mutex);
      (void) raise (SIGINT);
      (void) pthread_exit (NULL);
    }

    for (j = 0; j < CONCURRENT_INSTANCES; j++) {
      if ((float_pool[j].valid) &&
        (float_pool[j].access_time <= now-(*seconds))) {
        float_pool[j].data = (pvm_float64_t) 0.0;
        float_pool[j].valid = (short int) INVALID;
        (void) memset (float_pool[j].symbol, 0, SYMBOL_LENGTH);
        float_pool[j].access_time = 0;
        num_floating--;
      }
    }

    if (pthread_rwlock_unlock (&float_init_lock)) {
      (void) pthread_mutex_lock (wrlog_mutex);
             report_log ("pvmsyncd: couldn't unlock internal rwlock\n");
      (void) pthread_mutex_unlock (wrlog_mutex);
      (void) raise (SIGINT);
      (void) pthread_exit (NULL);
    }

    /* Roughly the current time */
    now = time (NULL);

    /* Lock rwlock for writing */
    if (pthread_rwlock_wrlock (&memptr_init_lock)) {
      (void) pthread_mutex_lock (wrlog_mutex);
             report_log ("pvmsyncd: couldn't lock internal rwlock\n");
      (void) pthread_mutex_unlock (wrlog_mutex);
      (void) raise (SIGINT);
      (void) pthread_exit (NULL);
    }

    for (j = 0; j < CONCURRENT_INSTANCES; j++) {
      if ((memptr_pool[j].valid) &&
        (memptr_pool[j].access_time <= now-(*seconds))) {
        free (memptr_pool[j].block);
        memptr_pool[j].block = NULL;
        memptr_pool[j].numbytes = 0;
        memptr_pool[j].valid = (short int) INVALID;
        (void) memset (memptr_pool[j].symbol, 0, SYMBOL_LENGTH);
        memptr_pool[j].access_time = 0;
        num_memblocks--;
      }
    }

    if (pthread_rwlock_unlock (&memptr_init_lock)) {
      (void) pthread_mutex_lock (wrlog_mutex);
             report_log ("pvmsyncd: couldn't unlock internal rwlock\n");
      (void) pthread_mutex_unlock (wrlog_mutex);
      (void) raise (SIGINT);
      (void) pthread_exit (NULL);
    }
  }

/* NOTREACHED */
return NULL;
}


#endif /* __DO_GARBAGE_COLLECTION */


