/*
 *
 * $Id: accept_local.c,v 1.2.4.41 2000/03/16 17:12:20 ajp Exp $
 *
 * Andrew Pitman
 *
 * pvmsync, a distributed synchronization server:  Process
 * requests from the UDP server.  These requests discover
 * whether or not the localhost has a given symbol, and how
 * busy the localhost is in processing requests. 
 *
 * Server that accepts connections (requests) continuously
 * and changes some internal state according to the request.
 *
 * Copyright (C) 1999, 2000 Andrew J. Pitman
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


#ifndef __CONFIG_H
#include <config.h>
#endif

#ifndef __ACCEPT_LOCAL_H
#include <accept_local.h>
#endif


#include <signal.h>
#include <string.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>

#include <sys/types.h>
#include <sys/socket.h>

#include <sys/un.h>

/* extern int errno */
#include <errno.h>

/* Standard error codes for pvmsync */
#ifndef __PVMERRS_H
#include <include/pvmerrs.h>
#endif

#ifndef __LOGGING_H
#include <logging.h>
#endif

/* For RW Locks */
#ifndef __USE_UNIX98
  #define __USE_UNIX98
#endif
#include <pthread.h>
#include <semaphore.h>

#ifndef __TCP_DAEMON_H
#include <tcp_daemon.h>
#endif

/* struct udp_app_dgram */
#ifndef __UDP_DAEMON_H
#include <udp_daemon.h>
#endif


/* File descriptors for accept(2) */
int local_fds[32];

/* What we were called as */
extern char *progname;

/* Internal "pool" of mutexes (mutexi?) */
/* Hash table indexed according to first character */
extern struct ht_ent mutex_pool[NUM_HASH_ENTS];

/* Keep a counter of the number of objects we have of this type */
/* This is protected by the rwlock and is only updated on mutex */
/* creation and destruction. */
extern unsigned int num_mutexes;

/* Protect access to variable num_mutexes */
extern pthread_rwlock_t mutex_counter_lock;

/* Internal "pool" of semaphores */
extern struct ht_ent sem_pool[NUM_HASH_ENTS];

/* Keep a counter of the number of objects we have of this type */
/* This is protected by the rwlock and is only updated on sem   */
/* creation and destruction. */
extern unsigned int num_sems;

/* Protect access to variable num_sems */
extern pthread_rwlock_t sem_counter_lock;

/* Protect access to the integer pool with a rwlock */
extern pthread_rwlock_t integer_init_lock;

/* Internal "pool" of pvm_int64_t's */
extern integer_pool_t integer_pool[CONCURRENT_INSTANCES];

/* Keep a counter of the number of objects we have of this type */
extern unsigned int num_integers;

/* Protect access to the float pool with a rwlock */
extern pthread_rwlock_t float_init_lock;

/* Internal "pool" of pvm_float64_t's. */
extern float_pool_t float_pool[CONCURRENT_INSTANCES];

/* Keep a counter of the number of objects we have of this type */
extern unsigned int num_floating;

/* Protect access to the memptr pool with a rwlock */
extern pthread_rwlock_t memptr_init_lock;
extern pthread_rwlockattr_t *memptr_init_lock_attr;

/* Internal "pool" of pointers to heap-dynamic data. */
extern memptr_pool_t memptr_pool[CONCURRENT_INSTANCES];

/* Keep a counter of the number of objects we have of this type */
extern unsigned int num_memblocks;

/* Internal "pool" of condition variables */
/* Hash table indexed according to first character */
extern struct ht_ent cond_pool[NUM_HASH_ENTS];

/* Keep a counter of the number of objects we have of this type */
/* This is protected by the rwlock and is only updated on condvar */
/* creation and destruction. */
extern unsigned int num_conds;

/* Protect access to variable num_conds */
pthread_rwlock_t cond_counter_lock;


/* Serialize access to run file */
extern pthread_mutex_t *wrlog_mutex;


/* Accept connections from the UDP daemon */
/* over a UNIX domain socket */
void *accept_local (void *disc_fd)
{
int index, status;
int *fd, *curr_index;
char errstring[64];
struct sockaddr_un temp_un;
int temp_un_addrlen;


/* Attributes object to create threads detached */
pthread_attr_t attrib;

/* Dummy thread identifier */
pthread_t thread_id;

  /* Detach process from tty, since we're a daemon */
  ioctl(0, TIOCNOTTY, NULL);

  /* Fill with 0's */
  (void) memset (&temp_un, 0, sizeof (struct sockaddr_un));

  fd = (int *) disc_fd;

  status = pthread_attr_init (&attrib);
  if (status) {
    (void) pthread_mutex_lock (wrlog_mutex);
    snprintf(errstring, 64, "%s: pthread_attr_init: %s\n", progname,
                                                      strerror (status));
           report_log (errstring);
    (void) pthread_mutex_unlock (wrlog_mutex);
    (void) raise (SIGINT);
    (void) pthread_exit (NULL);
  }
  status = pthread_attr_setdetachstate (&attrib, PTHREAD_CREATE_DETACHED);
  if (status) {
    (void) pthread_mutex_lock (wrlog_mutex);
    snprintf(errstring, 64, "%s: pthread_attr_setdetachstate: %s\n", progname,
                                                      strerror (status));
           report_log (errstring);
    (void) pthread_mutex_unlock (wrlog_mutex);
    (void) raise (SIGINT);
    (void) pthread_exit (NULL);
  }

  /* Invalidate all */
  for (index = 0; index < 32; index++) {
    local_fds[index] = -1;
  }

  index = 0;


  while (1) {
    /* This should be the size of our struct, initially */
    temp_un_addrlen = sizeof (struct sockaddr_un);
    local_fds[index] = accept (*fd, &temp_un, &temp_un_addrlen);
    if (local_fds[index] < 0) {
      (void) pthread_mutex_lock (wrlog_mutex);
      (void) snprintf (errstring, 64, "%s: accept: %s\n", progname,
                                                     strerror (errno));
             report_log ((char *) errstring);
      (void) pthread_mutex_unlock (wrlog_mutex);
    } else {
      curr_index = (int *) malloc (sizeof (int));
      *curr_index = index;
      status = pthread_create (&thread_id, &attrib, resource_query, curr_index);
      if (status) {
        (void) pthread_mutex_lock (wrlog_mutex);
        (void) snprintf (errstring, 64, "%s: pthread_create: %s\n", progname,
                                                    strerror (errno));
               report_log ((char *) errstring);
        (void) pthread_mutex_unlock (wrlog_mutex);
      }
    }

    
    while (local_fds[++index] != -1) {
      if (index >= 32)
        index = 0;
    }
  }

return NULL;
}

/* Find out if we have a given symbol, or return our load */
/* depending on the request. */
void *resource_query (void *curr_index)
{
int *index, numbytes, status;
pvm_t obj_type;
struct udp_app_dgram dgram;
char buff[32], errstring[64];
char *symbol = NULL;

  index = (int *) curr_index;
  (void) memset (&dgram,    0, sizeof (struct udp_app_dgram));
  (void) memset (errstring, 0, 64);

  numbytes = read (local_fds[*index], (void *) &dgram,
                      sizeof (struct udp_app_dgram));
  if (numbytes < 0) {
    (void) pthread_mutex_lock (wrlog_mutex);
    (void) snprintf (errstring, 64, "%s: error reading from client: %s\n",
                                              progname, strerror (errno));
           report_log (errstring);
    (void) pthread_mutex_unlock (wrlog_mutex);
  } else {

    /* Here, we extract the version and object type from
       the application layer header, and we know where
       the start of the string containing the name of the
       object is. */

    if (numbytes < 16) {
      (void) snprintf (buff, 32, "NO:%d", EINVAL);
      (void) write (local_fds[*index], buff, strlen (buff));

      /* Close the socket and invalidate the file descriptor */
      close (local_fds[*index]);
      local_fds[*index] = -1;
      free (curr_index);
      return NULL;
    }

    obj_type = (pvm_t) udp_objtype (&dgram);
    symbol = udp_data (&dgram);

    switch (obj_type) {
      case __PVM_MUTEX:
        status = have_mutex (symbol);
        if (status > 0) {
          (void) snprintf (buff, 32, "YES:%d", status);
          (void) write (local_fds[*index], buff, strlen (buff));
        } else {
          (void) snprintf (buff, 32, "NO:%d", EINVAL);
          (void) write (local_fds[*index], buff, strlen (buff));
        }
          break;
      case __PVM_SEM:
        status = have_sem (symbol);
        if (status > 0) {
          (void) snprintf (buff, 32, "YES:%d", status);
          (void) write (local_fds[*index], buff, strlen (buff));
        } else {
          (void) snprintf (buff, 32, "NO:%d", EINVAL);
          (void) write (local_fds[*index], buff, strlen (buff));
        }
          break;
      case __PVM_CV:
        status = have_cv (symbol);
        if (status >= 0) {
          (void) snprintf (buff, 32, "YES:%d", status);
          (void) write (local_fds[*index], buff, strlen (buff));
        } else {
          (void) snprintf (buff, 32, "NO:%d", EINVAL);
          (void) write (local_fds[*index], buff, strlen (buff));
        }
          break;
      case __PVM_SD_INT:
        status = have_int (symbol);
        if (status >= 0) {
          (void) snprintf (buff, 32, "YES:%d", status);
          (void) write (local_fds[*index], buff, strlen (buff));
        } else {
          (void) snprintf (buff, 32, "NO:%d", EINVAL);
          (void) write (local_fds[*index], buff, strlen (buff));
        }
          break;
      case __PVM_SD_FLOAT:
        status = have_float (symbol);
        if (status >= 0) {
          (void) snprintf (buff, 32, "YES:%d", status);
          (void) write (local_fds[*index], buff, strlen (buff));
        } else {
          (void) snprintf (buff, 32, "NO:%d", EINVAL);
          (void) write (local_fds[*index], buff, strlen (buff));
        }
          break;
      case __PVM_SD_MEM:
        status = have_memptr (symbol);
        if (status >= 0) {
          (void) snprintf (buff, 32, "YES:%d", status);
          (void) write (local_fds[*index], buff, strlen (buff));
        } else {
          (void) snprintf (buff, 32, "NO:%d", EINVAL);
          (void) write (local_fds[*index], buff, strlen (buff));
        }
          break;
      case __PVM_GET_LD:
        obj_type = atoi (symbol);
        (void) snprintf (buff, 32, "%d", get_numobjs(obj_type));
        (void) write (local_fds[*index], buff, strlen (buff));
    };

  }

  /* Close the socket and invalidate the file descriptor */
  close (local_fds[*index]);
  local_fds[*index] = -1;
  free (curr_index);

return NULL;
}

/* Does this server have mutex_name? */
/* nonzero if we have it, 0 if we don't, -1 on error */
int have_mutex (char *mutex_name)
{
int index, retnval;

  /* No funny business, make sure the symbol name is valid
     so we won't get an out-of-bounds error for the array */
  if (legal_sym (mutex_name))
    return -1;

  /* Get the index into the hash table */
  index = hash_index (mutex_name);

  /* Lock for reading */
  if (pthread_rwlock_rdlock (&mutex_pool[index].ent_rwlock)) {
    return -1;
  }

  if (search_ht (mutex_name, __PVM_MUTEX) > 0)
    retnval = 1;
  else
    retnval = 0;

  /* Unlock rwlock or die */
  unlock_r_die (&mutex_pool[index].ent_rwlock);

return retnval;
}

/* Does this server have sem_name? */
/* nonzero if we have it, 0 if we don't, -1 on error */
int have_sem (char *sem_name)
{
int index, retnval;

  /* No funny business, make sure the symbol name is valid
     so we won't get an out-of-bounds error for the array */
  if (legal_sym (sem_name))
    return -1;

  /* Get the index into the hash table */
  index = hash_index (sem_name);

  /* Lock for reading */
  if (pthread_rwlock_rdlock (&sem_pool[index].ent_rwlock)) {
    return -1;
  }

  if (search_ht (sem_name, __PVM_SEM) > 0)
    retnval = 1;
  else
    retnval = 0;

  /* Unlock rwlock or die */
  unlock_r_die (&sem_pool[index].ent_rwlock);

return retnval;
}

/* Does this server have int_name? */
int have_int (char *int_name)
{
int count, retnval = -1;

  /* Lock for reading */
  if (pthread_rwlock_rdlock (&integer_init_lock)) {
    return retnval;
  }

  for (count = 0; count < CONCURRENT_INSTANCES; count++) {
    if ((integer_pool[count].valid) &&
       (!strncmp (integer_pool[count].symbol, int_name,
                                            strlen (int_name)))) {

      retnval = count; /* We have it */
      break;
    }
  }
  if (pthread_rwlock_unlock (&integer_init_lock)) {
    (void) pthread_mutex_lock (wrlog_mutex);
           report_log ("pvmsyncd: couldn't unlock internal rwlock\n");
    (void) pthread_mutex_unlock (wrlog_mutex);
    (void) raise (SIGINT);
  }

return retnval;
}

/* Does this server have float_name? */
int have_float (char *float_name)
{
int count, retnval = -1;

  /* Lock for reading */
  if (pthread_rwlock_rdlock (&float_init_lock)) {
    return retnval;
  }

  for (count = 0; count < CONCURRENT_INSTANCES; count++) {
    if ((float_pool[count].valid) &&
       (!strncmp (float_pool[count].symbol, float_name,
                                            strlen (float_name)))) {

      retnval = count; /* We have it */
      break;
    }
  }
  if (pthread_rwlock_unlock (&float_init_lock)) {
    (void) pthread_mutex_lock (wrlog_mutex);
           report_log ("pvmsyncd: couldn't unlock internal rwlock\n");
    (void) pthread_mutex_unlock (wrlog_mutex);
    (void) raise (SIGINT);
  }

return retnval;
}

/* Does this server have memptr_name? */
int have_memptr (char *memptr_name)
{
int count, retnval = -1;

  /* Lock for reading */
  if (pthread_rwlock_rdlock (&memptr_init_lock)) {
    return retnval;
  }

  for (count = 0; count < CONCURRENT_INSTANCES; count++) {
    if ((memptr_pool[count].valid) &&
       (!strncmp (memptr_pool[count].symbol, memptr_name,
                                            strlen (memptr_name)))) {

      retnval = count; /* We have it */
      break;
    }
  }
  if (pthread_rwlock_unlock (&memptr_init_lock)) {
    (void) pthread_mutex_lock (wrlog_mutex);
           report_log ("pvmsyncd: couldn't unlock internal rwlock\n");
    (void) pthread_mutex_unlock (wrlog_mutex);
    (void) raise (SIGINT);
  }

return retnval;
}

/* How many objects of type type does this server have left? */
int get_numobjs (pvm_t type)
{
int retnval = 0;

  switch (type) {
    case __PVM_MUTEX:
      /* Lock for reading */
      if (pthread_rwlock_rdlock (&mutex_counter_lock)) {
        return -1;
      }
      retnval = num_mutexes;
      unlock_r_die (&mutex_counter_lock);
      break;
    case __PVM_SEM:
      /* Lock for reading */
      if (pthread_rwlock_rdlock (&sem_counter_lock)) {
        return -1;
      }
      retnval = num_sems;
      unlock_r_die (&sem_counter_lock);
      break;
    case __PVM_CV:
      /* Lock for reading */
      if (pthread_rwlock_rdlock (&cond_counter_lock)) {
        return -1;
      }
      retnval = num_conds;
      unlock_r_die (&cond_counter_lock);
      break;
    case __PVM_SD_INT:
      if (pthread_rwlock_rdlock (&integer_init_lock)) {
        return -1;
      }
      retnval = CONCURRENT_INSTANCES-num_integers;
      if (pthread_rwlock_unlock (&integer_init_lock)) {
        (void) pthread_mutex_lock (wrlog_mutex);
               report_log ("pvmsyncd: couldn't unlock internal rwlock\n");
        (void) pthread_mutex_unlock (wrlog_mutex);
        (void) raise (SIGINT);
      }
      break;
    case __PVM_SD_FLOAT:
      if (pthread_rwlock_rdlock (&float_init_lock)) {
        return -1;
      }
      retnval = CONCURRENT_INSTANCES-num_floating;
      if (pthread_rwlock_unlock (&float_init_lock)) {
        (void) pthread_mutex_lock (wrlog_mutex);
               report_log ("pvmsyncd: couldn't unlock internal rwlock\n");
        (void) pthread_mutex_unlock (wrlog_mutex);
        (void) raise (SIGINT);
      }
      break;
    case __PVM_SD_MEM:
      if (pthread_rwlock_rdlock (&memptr_init_lock)) {
        return -1;
      }
      retnval = CONCURRENT_INSTANCES-num_memblocks;
      if (pthread_rwlock_unlock (&memptr_init_lock)) {
        (void) pthread_mutex_lock (wrlog_mutex);
               report_log ("pvmsyncd: couldn't unlock internal rwlock\n");
        (void) pthread_mutex_unlock (wrlog_mutex);
        (void) raise (SIGINT);
      }
      break;
    default:
      retnval = -1;
  };

return retnval;
}

/* Does this server have cv_name? */
/* nonzero if we have it, 0 if we don't, -1 on error */
int have_cv (char *cv_name)
{
int index, retnval;

  /* No funny business, make sure the symbol name is valid
     so we won't get an out-of-bounds error for the array */
  if (legal_sym (cv_name))
    return -1;

  /* Get the index into the hash table */
  index = hash_index (cv_name);

  /* Lock for reading */
  if (pthread_rwlock_rdlock (&cond_pool[index].ent_rwlock)) {
    return -1;
  }

  if (search_ht (cv_name, __PVM_CV) > 0)
    retnval = 1;
  else
    retnval = 0;

  /* Unlock rwlock or die */
  unlock_r_die (&cond_pool[index].ent_rwlock);

return retnval;
}


