
/*
 * Objective-C version of Squeak Morphic. Copyright (c) 1999 David Stes.
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Library General Public License as published 
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: BouncingAtomsMorph.m,v 1.4 1999/05/30 17:31:26 stes Exp $
 */

#include "AtomMorph.h"
#include "BouncingAtomsMorph.h"
#include <objpak.h>
#include <Color.h>
#include <MenuMorph.h>

// #include <FillInTheBlank.h>

@implementation BouncingAtomsMorph

- initialize
{
  [super initialize];
  damageReported = NO;
  [self extent:[Point x:400 y:250]];
  [self color:[Color r:0.8 g:1.0 b:0.8]];
  infectionHistory = [OrderedCollection new];
  transmitInfection = NO;
  [self addAtoms:2];
  return self;
}

#define s(x) [String str:(x)]
#define t(x) @selector(x)

- addCustomMenuItems:aMenu hand:aHandMorph
{
  [super addCustomMenuItems:aMenu hand:aHandMorph];
  [aMenu add:s("start infection") action:t(startInfection)];
  [aMenu add:s("set atom count") action:t(setAtomCount)];
  [aMenu add:s("show infection history") action:t(showInfectionHistory:)]; 
  return self;
}

- setAtomCount
{
#if 0
  id num = [FillInTheBlank request:s("Number of atoms?") initialAnswer:[String sprintf:"%i",[self submorphCount]]];
#endif
}

- startInfection
{
  [self submorphsDo:{ :m | [m infected:NO]; }];
  [[self firstSubmorph] infected:YES];
  infectionHistory = [OrderedCollection new];
  transmitInfection = YES;
  [self startStepping];
  return self;
}

- step
{
  int bounces = 0;
  id b = [self bounds];
  id r = [Rectangle origin:[b left]:[b top] corner:[b right]-8:[b bottom]-8];
  [self submorphsDo:{ :m |
    if ([m isMemberOf:AtomMorph]) {
      if ([m bounceIn:r]) bounces++;
    }
    // compute temperature that is proportional to the number of bounces
    // divided by the circumference of the enclosing rectangle
    [self updateTemperature:(10000.0 * bounces) / ([r width] + [r height])];
    if (transmitInfection) [self transmitInfection];
  }];
  return self;
}

//
// As fast as possible ...
//

- stepTime
{
}

//
// Add a bunch of new atoms
//

- addAtoms:(int)n
{
  while (n--) {
    id a = [AtomMorph new];
    [a randomPositionIn:bounds maxVelocity:10];
    [self addMorph:a];
  }

  [self stopStepping];
  return self;
}

//
// -addMorphFront: is sent by Morph's implementation of -addMorph:
// We want non-atoms to go to the back.
//
// Note: normally Morphs do not override methods like -addMorph:,
// -addMorphBack:, -addMorphFront: etc. so this is non-typical
//

- addMorphFront:aMorph
{
  if ([aMorph isMemberOf:AtomMorph]) {
    return [super addMorphFront:aMorph];
  } else {
    return [super addMorphBack:aMorph];
  }
}

- areasRemainingToFill:aRect
{
  if ([color isTranslucent]) {
     return [IdArray with:1,aRect];
  } else {
//     return [aRect areasOutside:[self bounds]];
  }
}

//
// Returns a list of pairs of colliding atoms, which are assumed to be
// circles of a known radius.
//

- collisionPairs
{
}

//
// Clear the damageReported flag when redrawn
//

- drawOn:aCanvas
{
  [super drawOn:aCanvas];
  damageReported = NO;
  return self;
}

//
// Compile with -DNOQUICKREDRAW to get the original invalidRect: behavior
//

- invalidRect:damageRect
{
#ifndef NOQUICKREDRAW
  if (([bounds top]<=[damageRect top] && [bounds left]<=[damageRect left])) {
     if (!damageReported) [super invalidRect:bounds];
     damageReported = YES;
  } else {
     [super invalidRect:damageRect];
  }
#else 
  [super invalidRect:damageRect];
#endif
  return self;
}

//
// Place a graph of the infection history in the world
//

- showInfectionHistory:evt
{
}

- transmitInfection
{
}

//
// Report the current temperature, which is taken to be the number of atoms
// that have bounced in the last cycle.  To avoid too much jitter in the
// reading, the last several readings are averaged.
//

- updateTemperature:x
{
}

@end
 
