
/*
 * Inetpak (c) 1998 by David Stes.  All Rights Reserved.
 * $Id$
 */

/*
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Library General Public License as published 
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "webserver.h"
#include "ocsocket.h"

#include <stdlib.h>
#include <stdio.h>		/* perror */
#include <errno.h>
#include <string.h>
#include <assert.h>
#include <syslog.h>
#include <fcntl.h>
#include <unistd.h>
#include <signal.h>
#include <time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <ocstring.h>
#include <ordcltn.h>

@implementation Webserver

+ new
{
  return [self argc:0 argv:NULL];
}

+ argc:(int)n argv:(char**)v
{
  return [[super new] argc:n argv:v];
}


void 
childdied(int signo, int code, struct sigcontext *ctxt)
{
  while (1) {
    int pid, status;

    pid = waitpid(-1, &status, WNOHANG);
    if (pid <= 0)
      return;
  }
}


- setportnum:(int)num
{
  portnum = num;
  return self;
}

- setdaemon:(BOOL)flag
{
  daemon = flag;
  return self;
}

- setblocking:(BOOL)flag
{
  blocking = flag;
  return self;
}

- argc:(int)n argv:(char**)v
{
  listenfd = -1;
  [self setportnum:80];
  [self setdaemon:YES];
  [self setblocking:YES];

  if (n) {
    n--;
    v++;
    while (n--) {
      char *a = *v++;

#ifdef __PORTABLE_OBJC__
      if (strcmp(a, "-m") == 0) {
	msgFlag = YES;
	continue;
      }
      if (strcmp(a, "-d") == 0) {
	dbgFlag = YES;
	[self setdaemon:NO];
	continue;
      }
#else
      if (strcmp(a, "-d") == 0) {
	[self setdaemon:NO];
	continue;
      }
#endif
      if (strcmp(a, "-p") == 0 && n > 0) {
	[self setportnum:atoi(*v++)];
	n--;
	continue;
      }
      fprintf(stderr, "Usage: %s [-d] [-p portnum]\n", [self name]);
      exit(1);
    }
  }
  return self;
}


- bind
{
  int flag = 1;
  struct sockaddr_in sa;
  struct sigaction na, oa;

  if ((listenfd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
    perror("socket");
    exit(1);
  }
  if (setsockopt(listenfd, SOL_SOCKET, SO_REUSEADDR, &flag, sizeof(flag)) < 0) {
    perror("setsockopt");
    exit(1);
  }
  memset(&sa, 0, sizeof(sa));
  sa.sin_family = AF_INET;
  sa.sin_addr.s_addr = htonl(INADDR_ANY);
  sa.sin_port = htons(portnum);

  if (bind(listenfd, (struct sockaddr *) &sa, sizeof(sa)) < 0) {
    perror("bind");
    exit(1);
  }
  if (listen(listenfd, 32) < 0) {
    perror("listen");
    exit(1);
  }
  na.sa_handler = childdied;
  sigemptyset(&na.sa_mask);
  na.sa_flags = SA_RESTART;
  if (sigaction(SIGCHLD, &na, &oa) < 0) {
    perror("sigaction");
    exit(1);
  }
  if (daemon) {
    [self detach];
  }
  if (blocking == NO) {
    int r;

    r = fcntl(listenfd, F_GETFL, 0);
    if (r < 0)
      perror("fcntl");
    r = fcntl(listenfd, F_SETFL, r | O_NONBLOCK);
    if (r < 0)
      perror("fcntl");
  }
  return self;
}

- halt:msg
{
  if (daemon) {
    syslog(LOG_DEBUG, [msg str]);
  } else {
    fprintf(stderr, "%s\n", [msg str]);
  }
  return self;
}

- detach
{
  int pid = fork();

  if (pid < 0) {
    perror("fork");
    exit(1);
  }
  if (pid > 0)
    exit(0);

  setsid();
  signal(SIGHUP, SIG_IGN);

  pid = fork();
  if (pid < 0) {
    perror("fork");
    exit(1);
  }
  if (pid > 0)
    exit(0);

  chdir("/");
  umask(0);

  close(0);
  close(1);
  close(2);

  openlog([self name], LOG_PID, LOG_USER);

  return self;
}

- errorresponse:(STR)msg
{
  printf("<HTML>\n");
  printf("<TITLE>Error response</TITLE>\n");
  printf("<H1>%s: error while processing request.</H1>\n", [self name]);
  printf("<P>%s</P>\n", msg);
  printf("</HTML>\n");
  return self;
}
- responseheader
{
  time_t x;

  time(&x);
  assert(protocolname != nil);
  printf("%s 200 OK\r\n", [protocolname str]);
  printf("Date: %24.24s\r\n", ctime(&x));
  printf("Server: %s/1.0\r\n", [self name]);
  printf("Content-type: text/html\r\n");
  return self;
}

- (int) readline
{
  int n = 0;

  /* fgets() type of method, returns count of 'lbuf' */

  while (n < RLINEMAX) {
    int c;

    while (cn <= 0) {
      ci = 0;
      cn = read(connfd, cbuf, RLINEMAX);
      if (cn < 0 && errno != EINTR)
	return -1;
      if (cn == 0) {
	lbuf[n] = 0;
	return n;
      }
    }

    cn--;
    c = cbuf[ci++];
    lbuf[n++] = c;
    if (c == '\n')
      break;
  }

  lbuf[n] = 0;
  return n;
}

- parserequest
{
  char *s, *sep = " \t\r\n";

  if ((s = strtok(lbuf, sep))) {
    if (strcmp(s, "GET") == 0)
      method = @selector(get);
    if (strcmp(s, "HEAD") == 0)
      method = @selector(head);
    if (strcmp(s, "POST") == 0)
      method = @selector(post);
  }
  if ((s = strtok(NULL, sep))) {
    request = [String str:s];
  }
  if ((s = strtok(NULL, sep))) {
    if (strcmp(s, "HTTP/1.0") == 0)
      protocolname = [String str:s];
    if (strcmp(s, "HTTP/1.1") == 0)
      protocolname = [String str:s];
    protocolname = [String str:s];
  }
  return self;
}

- parseheader
{
  return self;
}

- (BOOL) iscrnl
{
  return strcmp(lbuf, "\r\n") == 0;
}

- parse
{
  int n = [self readline];

  if (n < 0)
    [self error:"readline %s", strerror(errno)];
  if (n == 0)
    return self;

  [self parserequest];

  if (protocolname) {
    while ([self iscrnl] == 0) {
      n = [self readline];
      dbg(lbuf);
      if (n < 0)
	[self error:"readline %s", strerror(errno)];
      if (n == 0)
	return self;
      [self parseheader];
    }
  }
  return self;
}

- delrequest
{
  if (request)
    request = [request free];
  if (protocolname)
    protocolname = [protocolname free];
  return self;
}

- reply
{
  if (method != NULL && request != nil) {
    [self perform:method];
  } else {
    [self errorresponse:"Error in client request."];
  }

  return self;
}

- get
{
  if (protocolname) {
    [self responseheader];
    printf("\r\n");
    [self contents];
  } else {
    [self contents];
  }

  return self;
}

- head
{
  assert(protocolname != nil);
  return [self responseheader];
}

- post
{
  assert(protocolname != nil);
  return self;
}

- run
{
  while (1) {
#if SERVER_USE_SELECT
    [self select];
#else
    [self poll];
#endif
  }
  return self;
}
#if SERVER_USE_SELECT
- if:(int)fd readable:aBlock
{
  if (!readfds) {
    readfds = [OrdCltn new];
    readblks = [OrdCltn new];
  }
  if (aBlock) {
    [readfds add:[Socket fd:fd]];
    [readblks add:aBlock];
  } else {
    int i, n = [readfds size];

    for (i = 0; i < n; i++)
      if ([[readfds at:i] fd] == fd) {
	[[readfds removeAt:i] free];
	return [readblks removeAt:i];
      }
    return nil;
  }

  return self;
}

- if:(int)fd writable:aBlock
{
  if (!writefds) {
    writefds = [OrdCltn new];
    writeblks = [OrdCltn new];
  }
  if (aBlock) {
    [writefds add:[Socket fd:fd]];
    [writeblks add:aBlock];
  } else {
    int i, n = [writefds size];

    for (i = 0; i < n; i++)
      if ([[writefds at:i] fd] == fd) {
	[[writefds removeAt:i] free];
	return [writefds removeAt:i];
      }
    return nil;
  }
  return self;
}

- (int) maxfd
{
  int i, n, maxfd;

  maxfd = listenfd;
  if (readfds)
    for (i = 0, n = [readfds size]; i < n; i++) {
      int fd = [[readfds at:i] fd];

      if (maxfd < fd)
	maxfd = fd;
    }
  if (writefds)
    for (i = 0, n = [writefds size]; i < n; i++) {
      int fd = [[writefds at:i] fd];

      if (maxfd < fd)
	maxfd = fd;
    }
  if (maxfd >= FD_SETSIZE)
    [self error:"too many file descriptors"];
  return maxfd;
}

- calcreadset:(fd_set *)p
{
  int i, n;

  FD_SET(listenfd, p);

  if (readfds)
    for (i = 0, n = [readfds size]; i < n; i++) {
      int fd = [[readfds at:i] fd];

      FD_SET(fd, p);
    }
  return self;
}

- doreadblks:(fd_set *)p
{
  int i, n;

  if (FD_ISSET(listenfd, p))
    [self poll];

  if (readfds)
    for (i = 0, n = [readfds size]; i < n; i++) {
      int fd = [[readfds at:i] fd];

      if (FD_ISSET(fd, p))
	[[readblks at:i] value];
    }
  return self;
}

- calcwriteset:(fd_set *)p
{
  int i, n;

  if (writefds)
    for (i = 0, n = [writefds size]; i < n; i++) {
      int fd = [[writefds at:i] fd];

      FD_SET(fd, p);
    }
  return self;
}

- dowriteblks:(fd_set *)p
{
  int i, n;

  if (writefds)
    for (i = 0, n = [writefds size]; i < n; i++) {
      int fd = [[writefds at:i] fd];

      if (FD_ISSET(fd, p))
	[[writeblks at:i] value];
    }
  return self;
}

- select
{
  int maxfd;
  fd_set *w = NULL;
  fd_set *r, readset, writeset;

  if (listenfd < 0)
    [self bind];

  maxfd = [self maxfd];

  if (1) {
    r = &readset;
    FD_ZERO(r);
    [self calcreadset:r];
  }
  if (writefds) {
    w = &writeset;
    FD_ZERO(w);
    [self calcwriteset:w];
  }
  while (1) {
    if (select(maxfd + 1, r, w, NULL, NULL) < 0) {
      if (errno == EINTR)
	continue;
      [self error:"select %s", strerror(errno)];
    } else {
      break;
    }
  }

  if (1) {
    [self doreadblks:r];
  }
  if (writefds) {
    [self dowriteblks:w];
  }
  return self;
}

#endif				/* SERVER_USE_SELECT */

- poll
{
  connfd = -1;

  if (listenfd < 0)
    [self bind];

  while (connfd < 0) {
    connfd = accept(listenfd, NULL, NULL);
    if (connfd < 0 && errno != EINTR)
      [self error:"accept %s", strerror(errno)];
  }

  [self parse];

  /* give subclass chance to inspect request in parent */
  [self accepted];

  if (fork() == 0) {
    if (close(listenfd) < 0) {
      [self error:"close listenfd %s", strerror(errno)];
    }
    if (dup2(connfd, 1) < 0) {
      [self error:"dup2 %2", strerror(errno)];
    }
    child = YES;
    [self reply];

    exit(0);
  } else {
    [self delrequest];
    if (close(connfd) < 0)
      [self error:"close connfd %s", strerror(errno)];
  }

  return self;
}

- (int) listenfd
{
  return listenfd;
}


- accepted
{
  return self;
}


- contents
{
  printf("<HTML>\n");
  printf("<TITLE>Default response</TITLE>\n");
  printf("<H1>%s: subclassResponsibility</H1>\n", [self name]);
  printf("<P>Using default implementation of <B>contents</B> message.</P>\n");
  printf("</HTML>\n");
  return self;
}


@end

